%  slirpc++.sl:  Supports generation of S-Lang wrappers for C++ {{{
%
%  This file is part of SLIRP, the (Sl)ang (I)nte(r)face (P)ackage.
%
%  Copyright (c) 2003-2006 Massachusetts Institute of Technology
%  Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu> }}}

require("slirpmaps");

% Front matter: type/variable/forward declarations, etc {{{ 

private define parse_cplusplus_block();

private define activate_cpp_support()
{
   add_tokens(Single_Char_Tokens, "~ ( ) , [ ] < +");
   _macros["static"] = EMPTY;
   SC.funcprefix = "^[~]?[a-zA-Z]+";
   SC.cplusplus = 1;
}

% }}}

% Overloaded type mnemonics % {{{
define map_args_to_type_abbrevs(args)
{
   variable inputs = where( struct_map(Integer_Type, args, "marshal"));
   args = args[inputs];
   variable types = struct_map(String_Type, args, "typeid");
   types = array_map(String_Type, &sltype_abbrev, types);
   variable refs = where(types == "R");
   if (length(refs)) {
	variable refd_types = struct_map(Struct_Type, args[refs], "aux");
	refd_types = struct_map(String_Type, refd_types, "typeid");
	refd_types = array_map(String_Type, &sltype_abbrev, refd_types);
	refd_types = array_map(String_Type, &sltype_ptr_abbrev, refd_types);
	types[refs] = refd_types;
   }
   return types;
}

define make_dispatch_table_entry(overloaded_funcs)
{
   variable entry = EMPTY;
   foreach(overloaded_funcs.head) {

	variable f = ();
	entry = sprintf("\n   { %s, (char*)\"%s\", %d },", f.name,
		 				f.value, strlen(f.value));

	SC.dispatch_table = [ SC.dispatch_table, entry ];
   }
}
% }}}

% Code emission {{{

define emit_destructor(class)
{
   !if (length(class.ancestors)) {

	variable proto = sprintf("void %s%s_delete(void *o)",
					SC.wrapper_prefix, class.name);

	SC.interface.prototypes[proto] = EMPTY;
	!if (SC.cfront)
	   emit("static ");

	emit("%s {delete (%s *)o;}\n", proto, class.name);
   }
}

define emit_dispatch_table()
{
   !if (length(SC.dispatch_table)) return;

   emit("typedef struct _Overloaded_Func {\t%s\n"+
	"   void (*func) (void);\n"+
	"   char *signature;\n"+
	"   int nargs;\n"+
	"} Overloaded_Func;   %s\n", FOLD_OPEN, FOLD_CLOSE);

   emit("\nstatic Overloaded_Func Dispatch_Table[] =%s\n{", FOLD_OPEN);

   foreach(SC.dispatch_table) {
	variable entry = ();
	emit(entry);
   }
   emit("\n   { NULL, NULL, 0}\n};   %s\n\n", FOLD_CLOSE);

   inject_file("dispatch.c");
}

private define field_get_emitter(fmap, argno_ref)
{
   variable retval = fmap.retval, obj = fmap.args[0];
   sprintf("(%s) (%s%s)->%s", retval.type, DeReferer[fmap.vectorized],
		(@obj.referer)(obj, 0, fmap.pop_args), retval.name);

   % S-Lang1 doesn't like @argno_ref += 1
   if (argno_ref != NULL) @argno_ref = @argno_ref + 1;
}

private define field_set_emitter(fmap, argno_ref)
{
   variable obj = fmap.args[0], field = fmap.args[1];
   variable deref = DeReferer[fmap.vectorized];

   sprintf("(%s%s)->%s = (%s) %s%s", deref,
	 (@obj.referer) (obj, 0, fmap.pop_args),
	 field.name, field.type, deref, field.lname);
   if (argno_ref != NULL) @argno_ref = @argno_ref + 2;		% S-Lang1
}
% }}}

% Mapping and support code {{{

define cfront_method_referer(meth, argno_ref)
{
   sprintf("INVOKE_METHOD(%s, OBJECT(arg0), %s)", meth.class.name, meth.name);
   @argno_ref = @argno_ref + 1;
}

private define standard_method_referer(method, argno_ref)
{
   if (argno_ref != NULL) @argno_ref = @argno_ref + 1;
   return "arg0->" + method.name;
}

private define cfront_class_referer(arg, typed, ignored)
{
   if (typed)
	sprintf("void* %s", arg.lname);
   else {
	variable ref = DeReferer[arg.cpp_ref];
	sprintf("%s((%s%s)%s)", ref, arg.type, ref, arg.lname);
   }
}

private define constructor_referer(method, ignore)
{
   return "new " + method.name;
}

static variable method_referer;
static variable class_type_referer;

private define define_class_typemaps(class, parent, destructor)
{
   !if (SC.num_reserved_types) define_reserved_opaques();

   variable ot = _define_opaque(class.name, parent, destructor);
   ot.referer = class_type_referer;
   slirp_map_opaque(class.name+"*", class.name);
   slirp_map_cpp_ref(class.name);
}

private define finalize_method(meth, class)
{
   if (meth== NULL) return;

   meth.class = class;
   meth.language = CPLUSPLUS;
   !if (SC.cfront) meth.pop_args = 1;

   if (meth.name == class.name) {		% constructor
	meth.slname += "_new";
	meth.referer = &constructor_referer;

	if (SC.genstubs) {			% Stubbed constructors should
	   meth.retval.type = EMPTY;		% not reflect a return value,
	   meth.retval.ltype = VOID;		% since that's not legal C++ 
	   meth.gname = class.name + "::" + class.name;
	}
	else
	   meth.gname = meth.slname;
   }
   else if (not SC.genstubs) {			% regular method, so inject
	variable this = @SC.types[class.name];	% "this" as hidden/first arg
	this.type = class.name + "*";
	this.name = EMPTY;
	this.arrayspec = EMPTY;
	this.lname = "arg0";
	this.defval = EMPTY;
	meth.slname = strcat(class.name,"_",meth.slname);
	meth.args = [this, meth.args];
	meth.nargs++;
	meth.referer = method_referer;
	meth.gname = meth.slname;
   }
   else
	meth.gname = meth.retval.type + " " + class.name + "::" + meth.slname;

   !if (SC.cfront)
	annotate(meth);
}

private define queue_public_field_wrapper_generation(class, field)
{
   if (field.typeclass != TYPECL_SCALAR) return;

   % Fabricate phony get/set function prototypes, and replace the
   % default function call emitter (which emit_call_block would
   % normally use to invoke the wrapped function) with an
   % emitter that simply gets/sets the relevant object fielda

   variable f = sprintf("%s %s_get_%s(%s*)", field.type, class.name,
						field.name, class.name);
   f = parse_func_decl(f, 0, 0);
   f.retval.name = field.name;
   f.referer = &field_get_emitter;
   f.callable = 0;

   f = sprintf("void %s_set_%s(%s*, %s %s)", class.name, field.name,
	 				class.name, field.type, field.name);
   f = parse_func_decl(f, 0, 0);
   f.referer = &field_set_emitter;
   f.callable = 0;
}
% }}}

private define parse_destructor(fp, class) % {{{
{
  variable decl = get_func_decl(fp, get_token(fp));
  if (SC.genstubs) {
	variable fmap = parse_func_decl("void ~"+decl, 0, 0);
	fmap.retval.type = EMPTY;		% not reflect a return value,
	fmap.retval.ltype = VOID;		% since that's not legal C++ 
	fmap.gname = class.name + "::~" + class.name;
   }
} % }}}

private define parse_class(fp, nesting) % {{{
{
   variable class = @Class, derived = 0;
   class.name = prep_get_token(fp);
   class.ancestors = String_Type[0];
   class.abstract = 0;
   class.constructors = String_Type[0];
   class.nesting = nesting;
   SC.in_public_interface = 0;

   variable token = prep_get_token(fp);
   if (token[-1] == ';')
	return 1;			% a forward declaration

   variable global_interface = save_interface(SC.interface, 1);
   SC.interface.name = class.name;

   while (token != "{") {

	switch(token)
	  { case ":" or case "," : }
	  { case "public" : derived = 1; }
	  { case "private"  or case "protected" : derived = 0; }

	  {
		% Support inheritance only from known classes
		if (andelse {derived} {SC.types[token] != NULL})
		   class.ancestors = [ class.ancestors, token ];
	  }

	token = prep_get_token(fp);
   }

   variable parent = NULL, destructor = "_delete";
   if (length(class.ancestors))	{		% multiple inheritance
	parent = class.ancestors[0];		% not yet unsupported
	destructor = parent + destructor;
   }
   else
	destructor = class.name + destructor;

   define_class_typemaps(class, parent, SC.wrapper_prefix + destructor);

   parse_cplusplus_block(fp, class);

   class.interface = save_interface(SC.interface, 0);
   restore_interface(global_interface, SC.interface);

   SC.classes[class.name] = class;
   SC.in_public_interface = 1;

} % }}}

private define parse_cplusplus_block(fp, class) % {{{
{
   if (SC.debug)
	tprintf("%s: ENTERED (class %s)", _function_name, class.name);

   SC.tablevel++;

   do { 

	variable fmap, decl, token = prep_get_token(fp);

	if (token == ";") continue;		% forward declaration

	if (SC.debug)
	   tprintf("%s: token=<%S>", _function_name, token);

	switch(token)

	{ case "private" or case "protected":
	   	SC.in_public_interface = 0;
		() = get_token(fp);		% swallow colon
	}
	{ case "public"  :

	   	% Inner classes should never become publicly visible
		SC.in_public_interface = (class.nesting == 0);
		() = get_token(fp);			% swallow colon
	}

	{ case class.name : 

	   % Treat constructors as if they return type of <class_name>*
	   decl = strcat(class.name,"* ", class.name, get_token(fp));
	   decl = get_func_decl(fp, decl);

	   % Strip initialization lists and inline definitions
	   if (string_match(decl, "\\(.+)\\)[ \t]*:.+", 1))
		decl = get_match(decl, 1);

	   if (string_match(decl, "\\(.+)\\)[ \t]*{.+}$", 1))
		decl = get_match(decl, 1);

	   % Defer full treatment of constructors until
	   % we know whether or not class is abstract 
	   class.constructors = [class.constructors, decl];
	}
	   
	{ case "~":	parse_destructor(fp, class); }

	{ case "class": parse_class(fp, class.nesting+1); }

	{ case "virtual":

	   token = get_token(fp);
	   if (token == "~") { parse_destructor(fp, class); continue;}

	   decl = get_func_decl(fp, token);
	   variable pure_virtual = string_match(decl,
					")[ \t]*=[ \t]*0[ \t]*;[ \t]*$", 1);

	   if (pure_virtual) {
		decl = strcat(decl[[0:pure_virtual-1]],";");
		class.abstract = 1;
	   }

	   !if (class.nesting) {
		fmap = parse_func_decl(decl, 0, 0);
		finalize_method(fmap, class);
	   }
	}

	{ case "static" : }

	{
	   if (token == EMPTY) abort("incomplete class definition");

	  !if (macro_or_type_definition(fp, token))
		if (SC.in_public_interface) {
		   if (get_var_decl(fp)) {
			decl = strtrim_end(SC.decl, ";");
			variable tmp = parse_args(_function_name, decl, 0);
			decl = parse_args(_function_name, decl, 0)[0];
			queue_public_field_wrapper_generation(class, decl);
		   }
		   else {
			decl = get_func_decl(fp, SC.decl);
			fmap = parse_func_decl(decl, 0, 0);
			finalize_method(fmap, class);
			SC.decl = EMPTY;
		   }
		}
		else {				% swallow non-public content
		   !if (get_var_decl(fp))
			() = get_func_decl(fp, SC.decl);
		}
	}

   } while (token != "}");

   !if (class.abstract or class.nesting)
	foreach(class.constructors) {
	   decl = ();
	   fmap = parse_func_decl(decl, 0, 0);
	   finalize_method(fmap, class);
	}

   SC.tablevel--;
   if (SC.debug)
	tprintf("%s: EXITED (class %s)\n",_function_name, class.name);

} % }}}

define cplusplus_token(fp) % {{{
{
   variable token = get_token(fp);

   switch(token)
   { case "class": activate_cpp_support;  parse_class(fp, 0); }
   { case "template":

      	activate_cpp_support;
	swallow_to(fp, '>');

	token = get_token(fp);
	if (token == "class")
	   parse_class(fp, 1);
	else
	   swallow_C_block(fp);
   }
   { case "inline" : }				% inlined func defs: supported
   { case "operator" : swallow_C_block(fp); }	% operator defs: ignored
   { case "using":     swallow_to(fp, ';'); }
   { 
      	if ("operator" == next_token(fp))
	   swallow_C_block(fp);
	else {
	   unget_token(fp, token);
	   return 0;
	}
   }

   activate_cpp_support;
   return 1;

} % }}}

private define initialize_cpp_support() % {{{
{
   if (SC.cfront) {
	class_type_referer = &cfront_class_referer;
	method_referer = &cfront_method_referer;
   }
   else {
	class_type_referer = &opaque_referer;
	method_referer = &standard_method_referer;
   }
}
slirp_init_cpp_callback = &initialize_cpp_support;
% }}}

provide("slirpc++");
