
/* SLgtk:  S-Lang bindings for the GTK+ widget set {{{
 *
 * Copyright (C) 2003-2009 Massachusetts Institute of Technology 
 * Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
 *
 * This software was partially developed by the MIT Center for Space
 * Research under contract SV1-61010 from the Smithsonian Institution.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * the supporting documentation, and that the name of the Massachusetts
 * Institute of Technology not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  The Massachusetts Institute of Technology makes
 * no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * }}} */

/* Headers, version info, and other front matter {{{ */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

#define GDK_PIXBUF_ENABLE_BACKEND
#include <gdk-pixbuf/gdk-pixbuf-io.h>
#include <gdk-pixbuf/gdk-pixbuf-animation.h>
#include <gtk/gtk.h>
#include <gtk/gtkmain.h>
#include <gdk/gdkkeysyms.h>

#include "module.h"

#ifdef HAVE_CAIRO
#include <cairo.h>
#ifdef CAIRO_HAS_SVG_SURFACE
#   include <cairo-svg.h>
#endif
#ifdef CAIRO_HAS_PS_SURFACE
#   include <cairo-ps.h>
#endif
#ifdef CAIRO_HAS_PDF_SURFACE
#include <cairo-pdf.h>
#endif
#include <cairo-xlib.h>
#endif

#ifdef HAVE_GTKEXTRA
#include "gtkextra/gtkextra.h"
#endif

#include "slgtkicons.h"

static const unsigned int _gtk_version = 10000*GTK_MAJOR_VERSION+ 
				100*GTK_MINOR_VERSION + GTK_MICRO_VERSION;

#if GTK_MAJOR_VERSION <= 2 && GTK_MINOR_VERSION < 14
#define GTK_OLDER_THAN_214 1
#endif

const int	slgtk_version	  = SLGTK_VERSION;
const char	*slgtk_version_string = SLGTK_VERSION_STRING;
int		slgtk_debug;

SLang_CStruct_Field_Type *GtkAllocation_Layout = GdkRectangle_Layout;

typedef struct _slGFunction {
  SLang_Name_Type *function;
  void		  *widget;
  SLang_Any_Type  **args;
  unsigned int    nargs;
} slGFunction;		/* Used for signal, timer, idle, etc callbacks */
/* }}} */

/* Icon interface {{{ */

static const char *slgtk_icons[] = {
   "SLGTK_STOCK_RECTANGLE",
   "SLGTK_STOCK_ELLIPSE",
   "SLGTK_STOCK_POLYGON",
   "SLGTK_STOCK_LINESEG",
};

static int
add_stock_icon (GtkIconFactory *factory,
		const guchar   *inline_data,
		const gchar    *stock_id)
{
   GdkPixbuf *pixbuf = gdk_pixbuf_new_from_inline (-1,inline_data,FALSE, NULL);
   GtkIconSet *set = gtk_icon_set_new_from_pixbuf(pixbuf);
      
   if (set == NULL)
      return -1;

   gtk_icon_factory_add (factory, stock_id, set);
   gtk_icon_set_unref (set);

   return 0;
}

static int
install_icons(void)
{
   GtkIconFactory *factory = gtk_icon_factory_new ();
   if (factory == NULL)
      return -1;

   if ( add_stock_icon(factory,slgtk_rect_24,slgtk_icons[0])    ||
	add_stock_icon(factory,slgtk_ellipse_24,slgtk_icons[1]) ||
	add_stock_icon(factory,slgtk_poly_24,slgtk_icons[2])    ||
	add_stock_icon(factory,slgtk_lineseg_24,slgtk_icons[3]))
	return -1;

   gtk_icon_factory_add_default(factory);
   return 0;
}
/* }}} */

static void throw_gerror(GError *);

/* GtkObject and GtkWidget manipulators {{{ */

static void slgtk_object_destroyer(GtkObject *obj)
{
   /* if ( GTK_IS_OBJECT(obj) && GTK_OBJECT_FLOATING(obj)) */
   if ( GTK_OBJECT_FLOATING(obj))
	/* An object still floating when it goes out of scope
	 * has not been added to a container, so we have to
	 * clean it up explicitly */
	gtk_object_sink(obj);
}

#include "gtk_glue.c"
#include "faccessors_glue.c"

static void sl_gtk_object_type(Slirp_Opaque *o)
{
   unsigned long type = (unsigned long) GTK_OBJECT_TYPE(o->instance);
   (void) SLang_push_ulong(type);
}

static void sl_gtk_object_type_name(Slirp_Opaque *o)
{
   char *name = (char *) GTK_OBJECT_TYPE_NAME(o->instance);
   (void) SLang_push_string(name);
}

/* The widget flags set/unset macros are wrapped here, to avoid using */
/* the macros (since they cause the emission of compiler warnings)    */
static void sl_gtk_widget_set_flags (Slirp_Opaque* o, gint *arg2)
{

   GTK_WIDGET_FLAGS ( SAFE_DEREF_OPAQUE(o) ) |= (*arg2);
}

static void sl_gtk_widget_unset_flags (Slirp_Opaque* o, gint *arg2)
{
   GTK_WIDGET_FLAGS ( SAFE_DEREF_OPAQUE(o) ) &= ~(*arg2);
}

static void sl_gtk_widget_get_window (Slirp_Opaque *o)
{
   GdkDrawable *d;

   if (GTK_IS_LAYOUT(o->instance))
	d = GTK_LAYOUT(o->instance)->bin_window;
   else
	d = GTK_WIDGET(o->instance)->window;

   /* g_object_ref(G_OBJECT(d)); */
   (void) SLang_push_opaque (GdkDrawable_Type, d, 0);
}

static void sl_gtk_widget_get_allocation(Slirp_Opaque *o)
{
   void *w = o->instance;
   if ( GTK_IS_WIDGET(w) && 0 == SLang_push_cstruct((VOID_STAR)
			&((GtkWidget*)w)->allocation, GdkRectangle_Layout))
	return;
   SLang_verror(SL_USAGE_ERROR, "specified argument is not a GtkWidget*");
}
/* }}} */

/* General signal/idle/timeout callback support {{{ */ 

static int
function_invoke(slGFunction *pf)
{
   unsigned int arg = 0;

   if (SLang_start_arg_list() == -1)
	return -1;

   if (pf->widget && -1 == SLang_push_opaque(GtkWidget_Type,pf->widget,0))
	return -1;

   if (pf->args) {
      for (arg = 0; arg < pf->nargs; arg++)
	  if (SLang_push_anytype(pf->args[arg]) == -1)
	     break;
   }

   if (SLang_get_error() || SLang_end_arg_list() == -1) {
	SLdo_pop_n( (pf->widget != NULL) + arg);
	return -1;
   }

   (void) SLexecute_function(pf->function);

   /* Unrecoverable S-Lang errors should not cause main loop to hang */
   if (SLang_get_error() < 0) {
	error_terminate_main_loop(pf->function->name);
	return -1;
   }

   /* Otherwise try our best to recover */
   if (SLang_get_error()) {
	SLang_restart(0);
	SLang_set_error(0);
   }

   return 0;
}

static void
callback_invoker(GtkWidget *w, gpointer data)
{
   slGFunction *pf = (slGFunction*)data;
   pf->widget = w;
   (void) function_invoke(pf);
}

static void 
function_marshaller (GtkObject *obj, gpointer data, guint n_args,GtkArg *args)
{
   int retval;
   slGFunction *pf = (slGFunction*)data;

   (void) obj; (void) n_args;

   if (-1 == function_invoke(pf))
	return;

   if (SLang_pop_integer(&retval) == -1 || SLang_get_error() < 0) {
	char msg[192];
	strcpy(msg,"could not pop expected boolean return value from: ");
	strncat(msg,pf->function->name,
	      	(size_t) MIN(strlen(pf->function->name),192-strlen(msg)-1));
	error_terminate_main_loop(msg);
   }

   *GTK_RETLOC_BOOL(args[0]) = retval ? TRUE : FALSE;
}

static void
function_destroy (gpointer data)
{
   slGFunction *f = (slGFunction*)data;
   if (f) {
	free_slang_args(f->nargs, f->args);
	SLang_free_function(f->function);
	SLfree((char*)f);
   }
}

static slGFunction*
function_create(void *widget, SLang_Ref_Type **slfunc_ref,
				SLang_Any_Type **args, unsigned int nargs)
{
   slGFunction *f;
   SLang_Name_Type *slfunc;
   
   if ( (slfunc = SLang_get_fun_from_ref(*slfunc_ref)) == NULL)
      return NULL;

   SLang_free_ref(*slfunc_ref);
   *slfunc_ref = NULL;

   if ( (f = (slGFunction*) SLmalloc(sizeof(slGFunction))) == NULL) {
	SLang_free_function(slfunc);
	return NULL;
   }

   f->function	= slfunc;
   f->widget	= (GtkWidget*)widget;
   f->args	= args;
   f->nargs	= nargs;

   return f;
}

static slGFunction*
function_pop(unsigned int num_args_to_omit)
{
   slGFunction *f;
   SLang_Any_Type **args = NULL;
   SLang_Ref_Type *func_ref = NULL;
   unsigned int nargs = SLang_Num_Function_Args - num_args_to_omit - 1;

   if (extract_slang_args(nargs,&args) == 0
      		 && SLang_pop_ref(&func_ref) == 0
		 && (f = function_create(NULL, &func_ref, args, nargs)))
	return f;

   if (args) free_slang_args(nargs,args);
   if (func_ref) SLang_free_ref(func_ref);

   return NULL;
}
/* }}} */

/* Gdk interface {{{ */
#include "slgdk.c"
#include "slgdkpixbuf.c"
/*  }}} */

/* GtkTextIter interface {{{ */

static void sl_gtk_text_buffer_get_iter_at_line_offset ( /*{{{*/
				Slirp_Opaque *textbuf,
                                gint *line_number,
				gint *char_offset)
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL)
	gtk_text_buffer_get_iter_at_line_offset (textbuf->instance, iter,
	      					*line_number,*char_offset);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

static void sl_gtk_text_buffer_get_iter_at_offset ( /*{{{*/
				Slirp_Opaque *textbuf,
				gint *char_offset)
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL) 
	gtk_text_buffer_get_iter_at_offset (textbuf->instance, iter,
	      						*char_offset);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

static void sl_gtk_text_buffer_get_iter_at_line ( /*{{{*/
				Slirp_Opaque *textbuf,
                                gint *line_number)
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL)
	gtk_text_buffer_get_iter_at_line(textbuf->instance, iter,
	      						*line_number);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

static void sl_gtk_text_buffer_get_start_iter  ( Slirp_Opaque *textbuf) /*{{{*/
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL)
	gtk_text_buffer_get_start_iter(textbuf->instance, iter);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

static void sl_gtk_text_buffer_get_end_iter (Slirp_Opaque *textbuf) /*{{{*/
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL)
	gtk_text_buffer_get_end_iter(textbuf->instance, iter);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

static void sl_gtk_text_buffer_get_iter_at_mark (Slirp_Opaque *textbuf, /*{{{*/
      		Slirp_Opaque *mark)
{
   GtkTextIter *iter;
 
   if ( (iter = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) != NULL)
	gtk_text_buffer_get_iter_at_mark(textbuf->instance, iter,
							mark->instance);

   SLang_push_opaque(GtkTextIter_Type, iter, 1);
} /*}}}*/

#if 0
void        gtk_text_buffer_get_iter_at_line_index
                                            (GtkTextBuffer *buffer,
                                             GtkTextIter *iter,
                                             gint line_number,
                                             gint byte_index);
void        gtk_text_buffer_get_iter_at_child_anchor
                                            (GtkTextBuffer *buffer,
                                             GtkTextIter *iter,
                                             GtkTextChildAnchor *anchor);
void        gtk_text_buffer_get_bounds      (GtkTextBuffer *buffer,
                                             GtkTextIter *start,
                                             GtkTextIter *end);
#endif

static void sl_gtk_text_iter_forward_search (void) /*{{{*/
{
   const GtkTextIter* from;		/* where to start search in buffer */
   Slirp_Opaque* from_o = NULL;
   GtkTextSearchFlags flags;		/* how to conduct the search  */
   GtkTextIter *begin, *end;		/* markers delimiting a match */
   gboolean found;			/* whether a match was found  */
   gchar* text;				/* text we're looking for     */

   if (SLang_Num_Function_Args != 3 ||
	SLang_pop_int((int*)&flags) == -1 ||
	SLang_pop_string((char**)&text) == -1 ||
	SLang_pop_opaque(GtkTextIter_Type, (void**)&from, &from_o) == -1 )
	usage_err(99,"(begin,end) = gtk_text_iter_forward_search(from,text,flags)");

   if ( (begin = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) == NULL ||
	(end   = (GtkTextIter *)SLmalloc(sizeof(GtkTextIter))) == NULL)
	return;

   /* FIXME: remove restrictive use of NULL == search until end of buffer */
   found = gtk_text_iter_forward_search(from, text, flags, begin, end, NULL);

   SLang_free_opaque(from_o);
   SLang_free_slstring(text);

   if (!found)
	begin = end = NULL;

   SLang_push_opaque(GtkTextIter_Type, begin, 1);
   SLang_push_opaque(GtkTextIter_Type, end, 1);

} /*}}}*/

static void sl_gtk_text_buffer_create_tag(void) /*{{{*/
{
   GParamSpec *pspec;
   Slirp_Opaque *textbuf_o = NULL;
   char *property_name;
   GtkTextTag *tag = NULL;
   char *tag_name = NULL;
   GtkTextBuffer *textbuf = NULL;
   unsigned int nprops = SLang_Num_Function_Args - 2;
   GValue property_value = _GValue_Initializer;

   if (usage_err(2,"tag = gtk_text_buffer_create_tag(buffer,tag_name,...)"))
      return;

   if (nprops % 2 != 0) {
	SLdo_pop_n(SLang_Num_Function_Args);
	SLang_verror(SL_USAGE_ERROR,"unbalanced name/value property list");
	return;
   }

   (void) SLreverse_stack(SLang_Num_Function_Args);

   if (-1 == SLang_pop_opaque(GObject_Type, (void**)&textbuf, &textbuf_o) ||
       -1 == pop_string_or_null(&tag_name))
	goto cleanup;

   tag = gtk_text_buffer_create_tag(textbuf, tag_name, NULL);
   if (tag == NULL)
	goto cleanup;

   while (nprops) {

	if (-1 == SLang_pop_slstring(&property_name)) {
	   g_object_unref(tag);
	   tag = NULL;
	   break;
	}

	pspec = g_object_class_find_property(G_OBJECT_GET_CLASS(tag),
	      						property_name);
	if (pspec == NULL) {
	   g_object_unref(tag);
	   tag = NULL;
	   break;
	}
	g_value_init(&property_value,G_PARAM_SPEC_VALUE_TYPE(pspec));
	pop_g_value(&property_value);
	g_object_set_property(G_OBJECT(tag),property_name,&property_value);
	g_value_unset(&property_value);

	SLang_free_slstring(property_name);
	nprops -= 2;
   }

   cleanup:

	SLang_free_opaque(textbuf_o);
	SLang_free_slstring(tag_name);
	if (tag != NULL)
	   SLang_push_opaque (GObject_Type, tag, 0);
	else
	   SLang_push_null();
} /*}}}*/

static void sl_gtk_text_buffer_insert_with_tags(void) /*{{{*/
{

   int len, start_offset;
   GtkTextIter start, *iter;
   GtkTextBuffer  *textbuf;
   GtkTextTag  *tag;
   Slirp_Opaque *iter_o = NULL, *textbuf_o = NULL, *tag_o = NULL;
   char *text = NULL;
   unsigned int i, ntags;

   if (usage_err(4,
	"gtk_text_buffer_insert_with_tags(buffer,iter,string,len,...)"))
      return;

   ntags = SLang_Num_Function_Args - 4;
   (void) SLreverse_stack(SLang_Num_Function_Args);

   if (-1 == SLang_pop_opaque(GObject_Type, (void**)&textbuf, &textbuf_o) ||
       -1 == SLang_pop_opaque(GtkTextIter_Type, (void**)&iter, &iter_o) ||
       -1 == SLang_pop_slstring(&text) ||
       -1 == SLang_pop_integer(&len))

	goto cleanup;

   start_offset = gtk_text_iter_get_offset (iter);
   gtk_text_buffer_insert(textbuf, iter, text, len);
   gtk_text_buffer_get_iter_at_offset (textbuf, &start, start_offset);

   for (i=0; i < ntags; i++) {

        if (-1 == SLang_pop_opaque(GObject_Type, (void**)&tag, &tag_o))
	   break;

	gtk_text_buffer_apply_tag (textbuf, tag, &start, iter);
	SLang_free_opaque(tag_o);
   }

   cleanup:

	SLang_free_opaque(textbuf_o);
	SLang_free_opaque(iter_o);
	SLang_free_slstring(text);
} /*}}}*/
/* }}} */

/* GtkMenu interface {{{ */

static void sl_gtk_menu_popup(void)
{
   Slirp_Opaque *menu_o = NULL;
   GtkMenu *menu;
   guint button;
   gulong atime;

   if (usage_err(3,"gtk_menu_popup(menu,mouse_button_num,activate_event_time)"))
      return;

   if ( -1 == SLang_pop_ulong(&atime) ||
	-1 == SLang_pop_uinteger(&button) ||
	-1 == SLang_pop_opaque(GtkWidget_Type, (void**)&menu, &menu_o)) {

	SLang_verror (SL_INTRINSIC_ERROR,
		"Unable to validate arguments to: gtk_menu_popup");
	return;
   }

   gtk_menu_popup(menu, NULL, NULL, NULL, NULL, button, (guint32)atime);
   SLang_free_opaque(menu_o);
}
/* }}} */

/* Tree and List interface {{{ */

static void sl_gtk_tree_iter_new(void)
{
   GtkTreeIter *iter;

   if ( (iter = (GtkTreeIter *)SLmalloc(sizeof(GtkTreeIter))) != NULL) {
	iter->stamp = 0;
	iter->user_data = iter->user_data2 = iter->user_data3 = NULL;
	SLang_push_opaque(GtkOpaque_Type, iter, 1);
   }
   else
	SLang_push_null();
}

static void sl_gtk_list_store_set(void)
{
   GtkListStore *lstore;
   GtkTreeIter  *iter;
   Slirp_Opaque *lstore_o = NULL, *iter_o = NULL;
   unsigned int nvalues = SLang_Num_Function_Args - 2;
   gint colnum;
   GValue value = _GValue_Initializer;

   if (usage_err(2,"gtk_list_store_set(GtkListStore,GtkTreeIter,...)"))
      return;

   if (nvalues % 2 != 0) {
	SLdo_pop_n(SLang_Num_Function_Args);
	SLang_verror(SL_USAGE_ERROR,"unbalanced column/value list");
	return;
   }

   (void) SLreverse_stack(SLang_Num_Function_Args);

   if (	-1 == SLang_pop_opaque(GObject_Type, (void**)&lstore, &lstore_o) ||
	-1 == SLang_pop_opaque(GtkTreeIter_Type, (void**)&iter, &iter_o))
	goto cleanup;

   while (nvalues) {

	if (-1 == SLang_pop_integer(&colnum))
	   break;

	g_value_init(&value, gtk_tree_model_get_column_type(
					GTK_TREE_MODEL(lstore), colnum));

	pop_g_value(&value);

	gtk_list_store_set_value(GTK_LIST_STORE(lstore),iter,colnum,&value);

	g_value_unset(&value);

	nvalues -= 2;
   }

   cleanup:

	SLang_free_opaque(lstore_o);
	SLang_free_opaque(iter_o);
}

static void sl_gtk_list_store_set_value(void)
{
   if (SLang_Num_Function_Args != 4) {
	SLang_verror (SL_USAGE_ERROR,"Usage: gtk_list_store_set_value"
				"(GtkListStore,GtkTreeIter,colnum,value);");
	return;
   }
   sl_gtk_list_store_set();
}

static void sl_gtk_tree_view_column_new_with_attributes(void)
{
   GtkTreeViewColumn *tvcol;
   GtkCellRenderer *cell;
   Slirp_Opaque *cell_o = NULL;
   unsigned int nattr = SLang_Num_Function_Args - 2;
   char *title, *attribute;
   gint colnum;

   if (usage_err(2,"gtk_list_store_set(string,GtkCellRenderer,...)"))
      return;

   if (nattr % 2 != 0) {
	SLdo_pop_n(SLang_Num_Function_Args);
	SLang_verror(SL_USAGE_ERROR,"empty or unbalanced attribute/column list");
	return;
   }

   (void) SLreverse_stack(SLang_Num_Function_Args);

   if (	-1 == SLang_pop_slstring(&title) ||
	-1 == SLang_pop_opaque(GtkCellRenderer_Type, (void**)&cell, &cell_o))
	goto cleanup;

   if ( (tvcol = gtk_tree_view_column_new()) == NULL)
	goto cleanup;

   gtk_tree_view_column_pack_start (tvcol, cell, TRUE);
   gtk_tree_view_column_set_title(tvcol,title);

   while (nattr) {

	attribute = NULL;
   	if (-1 == SLang_pop_slstring(&attribute) || 
	      		-1 == SLang_pop_integer(&colnum)) {

	   if (attribute)
		SLang_free_slstring(attribute);

	   SLang_verror(SL_INTRINSIC_ERROR,
		 		"error popping attribute/column pair");

	   slgtk_object_destroyer( GTK_OBJECT(tvcol) );
	   tvcol = NULL;
	   break;
	}

	gtk_tree_view_column_add_attribute(tvcol,cell,attribute,colnum);

	SLang_free_slstring(attribute);
	nattr -= 2;
   }

   if (tvcol)
	SLang_push_opaque (GtkTreeViewColumn_Type, tvcol, 0);

   cleanup:

	SLang_free_slstring(title);
	SLang_free_opaque(cell_o);
}
/* }}} */

/* GtkToolBar interface {{{ */

static void sl_gtk_toolbar_append_item(void)
{
   GtkWidget *icon;
   GtkToolbar *tb;
   SLang_Ref_Type *function_ref;
   SLang_Name_Type *function;
   SLang_Any_Type **args;
   char *text;
   char *ttip_text;
   char *ttip_private_text;
   GtkWidget *ret;
   unsigned int nargs;
   Slirp_Opaque *tb_o = NULL, *icon_o = NULL;

   if (usage_err(6,"gtk_toolbar_append_item(toolbar,string,string,string,icon,func_ref [,callback_data ...])"))
      return;

   nargs = SLang_Num_Function_Args - 6;
   if (extract_slang_args(nargs,&args) == -1) {
	SLang_push_null();
	return;
   }

   if ( -1 == SLang_pop_ref(&function_ref) ||
	-1 == SLang_pop_opaque(GtkWidget_Type, (void**)&icon, &icon_o) ||
	-1 == pop_string_or_null(&ttip_private_text) ||
	-1 == pop_string_or_null(&ttip_text) ||
	-1 == pop_string_or_null(&text) ||
	-1 == SLang_pop_opaque(GtkWidget_Type, (void**)&tb, &tb_o)) {

	goto fail;
   }

   function = SLang_get_fun_from_ref(function_ref);
   if (function == NULL)
      goto fail;
   SLang_free_ref(function_ref);

   ret = gtk_toolbar_append_item(GTK_TOOLBAR(tb),
   		text, ttip_text, ttip_private_text, icon, NULL, NULL);

   if (ret) {

	(void) g_signal_connect_closure(ret,"clicked",
	                 slg_closure_new(function,args,nargs,NULL),FALSE);

	SLang_push_opaque (GtkWidget_Type, ret, 0);
	SLang_free_mmt(tb_o->mmt);
	SLang_free_mmt(icon_o->mmt);
	return;
   }

   fail:
	SLang_free_opaque(icon_o);
	SLang_free_opaque(tb_o);
	SLang_push_null();
}
/* }}} */

/* Gtk timeout, idle, etc callbacks {{{ */
static unsigned int sl_gtk_timeout_add(void)
{
   unsigned int interval;
   slGFunction *f = NULL;

   if (	usage_err(2,"id = gtk_timeout_add(millis, func_ref [, arg1, ...])")
				|| (f = function_pop(1)) == NULL
				|| -1 == SLang_pop_uinteger(&interval)) {

	function_destroy(f);
	return 0;
   }

   return gtk_timeout_add_full((guint32)interval, NULL, function_marshaller,
	 					f, function_destroy);
}

static unsigned int sl_gtk_idle_add(void)
{
   slGFunction *f = NULL;

   if ( usage_err(1,"id = gtk_idle_add(func_ref [, arg1, ...])")
					|| (f = function_pop(0)) == NULL)
	return 0;

   return gtk_idle_add_full(GTK_PRIORITY_DEFAULT, NULL, function_marshaller,
	 						f, function_destroy);
}

static unsigned int sl_gtk_idle_add_priority(void)
{
   slGFunction *f = NULL;
   int prio;

   if ( usage_err(2,"id = gtk_idle_add_priority(prio, func_ref [, arg1, ...])")
			|| (f = function_pop(1)) == NULL
			|| -1 == SLang_pop_integer(&prio)) 
	return 0;

   return gtk_idle_add_full(prio, NULL, function_marshaller,
	 						f, function_destroy);
}

static unsigned int sl_gtk_quit_add(void)
{
   slGFunction *f = NULL;
   guint main_level;

   if (	usage_err(2,"id = gtk_quit_add(main_level, func_ref [,arg1, ...])")
			|| (f = function_pop(1)) == NULL
			|| -1 == SLang_pop_uinteger(&main_level)) {

	function_destroy(f);
	return 0;
   }

   return gtk_quit_add_full(main_level, NULL, function_marshaller, f,
	 						function_destroy);
}

static void sl_gtk_container_foreach(void)
{
   Slirp_Opaque *c_o = NULL;
   slGFunction *f = NULL;
   GtkContainer *c;

   if (	usage_err(1,
	"id = gtk_container_foreach(container, func_ref [, arg1, ...])")
	 	|| (f = function_pop(1)) == NULL
		|| -1 == SLang_pop_opaque(GtkWidget_Type, (void**)&c, &c_o)) {

	function_destroy(f);
	SLang_free_opaque(c_o);
	return;
   }

   gtk_container_foreach(c, callback_invoker, (gpointer)f);
}

static void sl_gtk_main_quit (void)
{
   /* Automatically clear stack of any args passed in, which helps */
   /* tidy up when called from a connected signal handler/callback */
   SLdo_pop_n(SLang_Num_Function_Args);
   gtk_main_quit();
}

static gboolean main_quit_idle (gpointer data)
{
   (void) data;
   gtk_main_quit();
   return FALSE;
}

static void sl_gtk_main_quit_when_idle (void)
{
   /* Automatically clear stack of any args passed in, which helps */
   /* tidy up when called from a connected signal handler/callback */
   SLdo_pop_n(SLang_Num_Function_Args);
   if (gtk_main_level())
      (void) gtk_idle_add_priority(G_PRIORITY_HIGH_IDLE, main_quit_idle, NULL);
}

/* }}} */

/* Glib interface {{{ */
#include "slglib.c"
/*  }}} */

/* Miscellaneous {{{ */

static void swallow_error (char* msg) { (void)msg; }

static int is_callable(void)
{
   int callable;
   SLang_Ref_Type *ref = NULL;
   SLang_Name_Type *func = NULL;
   void (*save_error_hook)(char *);

   if (SLang_Num_Function_Args == 0)
	return 0;

   /* Don't leave detritus on stack if called incorrectly */
   if (SLang_Num_Function_Args > 1)
	(void) SLdo_pop_n(SLang_Num_Function_Args - 1);

   if (SLang_peek_at_stack() != SLANG_REF_TYPE) {
        (void) SLdo_pop();
	return 0;
   }

   if (SLang_pop_ref(&ref) != 0) {
	SLang_verror (SL_INTRINSIC_ERROR,"unable to pop reference");
	return -1;
   }

   save_error_hook = SLang_Error_Hook;
   SLang_Error_Hook = swallow_error;
   callable = ((func = SLang_get_fun_from_ref(ref)) != NULL);
   SLang_Error_Hook = save_error_hook;

   SLang_free_ref(ref);
   SLang_free_function(func);	/* should quietly accept NULL */

   return callable;
}

static int
eqs_gtkopaque(Slirp_Opaque *o1, Slirp_Opaque *o2)
{
   return (o1->instance == o2->instance);
}

static void recover_unrecoverable_error_hook(char *error_msg)
{
   fputs(error_msg,stderr);	/* support function, which lets scripts */
   fputs("\r\n", stderr);	/* catch "unrecoverable" errors within  */
   fflush (stderr);		/* an ERROR_BLOCK */
   SLang_restart(0);
   SLang_set_error(SL_USAGE_ERROR);

}

static void toggle_error_hook(void)
{
   static void (*previous_error_hook)(char *);

   if (SLang_Error_Hook == recover_unrecoverable_error_hook)
	SLang_Error_Hook = previous_error_hook;
   else {
	previous_error_hook = SLang_Error_Hook;
	SLang_Error_Hook = recover_unrecoverable_error_hook;
   }
}

static void sl_gtk_color_selection_get_current_color (void)
{
   GdkColor color;
   GtkColorSelection *csel;
   Slirp_Opaque *csel_o = NULL;

   if (SLang_Num_Function_Args != 1) {
	SLang_verror (SL_USAGE_ERROR,
	"Usage: GdkColor = gtk_color_selection_get_current_color(GtkWidget);");
	return;
   }

   if ( SLang_pop_opaque(GtkWidget_Type, (void**)&csel, &csel_o) == -1 ) {
	SLang_verror (SL_INTRINSIC_ERROR,"Unable to validate arguments to: gtk_color_selection_get_current_color");
	SLang_free_opaque(csel_o);
	return;
   }

   gtk_color_selection_get_current_color( csel, &color);
   if ( SLang_push_cstruct((VOID_STAR)&color,GdkColor_Layout) != 0)
	(void) SLang_push_null();

   SLang_free_opaque(csel_o);
   return;
}

static void sl_gtk_adjustment_get_value (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->value);
}

#ifdef GTK_OLDER_THAN_214
static void sl_gtk_adjustment_get_lower (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->lower);
}


static void sl_gtk_adjustment_get_upper (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->upper);
}


static void sl_gtk_adjustment_get_step_increment (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->step_increment);
}


static void sl_gtk_adjustment_get_page_increment (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->page_increment);
}


static void sl_gtk_adjustment_get_page_size (Slirp_Opaque* o)
{
   (void) SLang_push_float( GTK_ADJUSTMENT (o->instance)->page_size);
}
#endif
/* }}} */

/* Intrinsic function and variable tables {{{ */

static SLang_Intrin_Fun_Type GtkWidget_Funcs[] =
{

  MAKE_INTRINSIC_1("gtk_widget_get_window",sl_gtk_widget_get_window,V,O),
  MAKE_INTRINSIC_1("gtk_widget_get_allocation",sl_gtk_widget_get_allocation,V,O),
  MAKE_INTRINSIC_0("gtk_toolbar_append_item",sl_gtk_toolbar_append_item,V),

  MAKE_INTRINSIC_0("gtk_timeout_add",sl_gtk_timeout_add,UI),
  MAKE_INTRINSIC_0("gtk_idle_add",sl_gtk_idle_add,UI),
  MAKE_INTRINSIC_0("gtk_idle_add_priority",sl_gtk_idle_add_priority,UI),
  MAKE_INTRINSIC_0("gtk_quit_add",sl_gtk_quit_add,UI),
  MAKE_INTRINSIC_0("gtk_main_quit",sl_gtk_main_quit,V),
  MAKE_INTRINSIC_0("gtk_container_foreach",sl_gtk_container_foreach,V),
  MAKE_INTRINSIC_0("_gtk_main_quit_when_idle",sl_gtk_main_quit_when_idle, V),
  MAKE_INTRINSIC_0("gtk_color_selection_get_current_color",
			sl_gtk_color_selection_get_current_color, V),

  MAKE_INTRINSIC_2("gtk_widget_set_flags",sl_gtk_widget_set_flags,V,O,I),
  MAKE_INTRINSIC_2("gtk_widget_unset_flags",sl_gtk_widget_unset_flags,V,O,I),

  MAKE_INTRINSIC_0("gtk_menu_popup",sl_gtk_menu_popup,V),
  MAKE_INTRINSIC_0("_is_callable",is_callable,I),

#include "faccessors_ftable.c"

  MAKE_INTRINSIC_0("_toggle_error_hook",toggle_error_hook,V),

  SLANG_END_INTRIN_FUN_TABLE
};

static SLang_Intrin_Fun_Type GtkObject_Funcs[] =
{
  MAKE_INTRINSIC_1("gtk_object_type",sl_gtk_object_type,V,O),
  MAKE_INTRINSIC_1("gtk_object_type_name",sl_gtk_object_type_name,V,O),
  MAKE_INTRINSIC_1("gtk_adjustment_get_value",sl_gtk_adjustment_get_value,V,O),
#ifdef GTK_OLDER_THAN_214
  MAKE_INTRINSIC_1("gtk_adjustment_get_lower",sl_gtk_adjustment_get_lower, V,O),
  MAKE_INTRINSIC_1("gtk_adjustment_get_upper",sl_gtk_adjustment_get_upper, V,O),
  MAKE_INTRINSIC_1("gtk_adjustment_get_step_increment",sl_gtk_adjustment_get_step_increment, V, O),
  MAKE_INTRINSIC_1("gtk_adjustment_get_page_increment",sl_gtk_adjustment_get_page_increment, V, O),
  MAKE_INTRINSIC_1("gtk_adjustment_get_page_size",sl_gtk_adjustment_get_page_size, V, O),
#endif

  SLANG_END_INTRIN_FUN_TABLE
};

static SLang_Intrin_Fun_Type GObject_Funcs[] =
{
  MAKE_INTRINSIC_2("gtk_text_buffer_get_iter_at_offset",
		  	sl_gtk_text_buffer_get_iter_at_offset,V,O,I),
  MAKE_INTRINSIC_3("gtk_text_buffer_get_iter_at_line_offset",
			sl_gtk_text_buffer_get_iter_at_line_offset,V,O,I,I),
  MAKE_INTRINSIC_2("gtk_text_buffer_get_iter_at_line",
			sl_gtk_text_buffer_get_iter_at_line,V,O,I),
  MAKE_INTRINSIC_2("gtk_text_buffer_get_iter_at_mark",
			sl_gtk_text_buffer_get_iter_at_mark,V,O,O),
  MAKE_INTRINSIC_1("gtk_text_buffer_get_start_iter",
		  	sl_gtk_text_buffer_get_start_iter,V,O),
  MAKE_INTRINSIC_1("gtk_text_buffer_get_end_iter",
		  	sl_gtk_text_buffer_get_end_iter,V,O),
  MAKE_INTRINSIC_0("gtk_text_iter_forward_search",
			sl_gtk_text_iter_forward_search,V),
  MAKE_INTRINSIC_0("gtk_text_buffer_create_tag",
			sl_gtk_text_buffer_create_tag,V),
  MAKE_INTRINSIC_0("gtk_text_buffer_insert_with_tags",
			sl_gtk_text_buffer_insert_with_tags,V),

  MAKE_INTRINSIC_0("gtk_tree_view_column_new_with_attributes",
			sl_gtk_tree_view_column_new_with_attributes,V),
  MAKE_INTRINSIC_0("gtk_list_store_set", sl_gtk_list_store_set,V),
  MAKE_INTRINSIC_0("gtk_list_store_set_value", sl_gtk_list_store_set_value,V),
  MAKE_INTRINSIC_0("gtk_tree_iter_new", sl_gtk_tree_iter_new,V),

  MAKE_INTRINSIC_2("__eqs_gtkopaque",eqs_gtkopaque,I,O,O),

  SLANG_END_INTRIN_FUN_TABLE
};

#undef V 

static SLang_IConstant_Type Generic_IConsts [] =
{

  MAKE_ICONSTANT("TRUE", TRUE),
  MAKE_ICONSTANT("FALSE", FALSE),

  SLANG_END_ICONST_TABLE
};

#if defined(__CYGWIN__)
static unsigned int _gtk_major_version;
static unsigned int _gtk_minor_version;
static unsigned int _gtk_micro_version;
#endif

static SLang_Intrin_Var_Type Global_Intrin_Vars[] =
{
   MAKE_VARIABLE("_gtk_module_version", &slgtk_version, I, 1),
   MAKE_VARIABLE("_gtk_module_version_string", &slgtk_version_string,S, 1),
   SLANG_END_INTRIN_VAR_TABLE
};

static SLang_Intrin_Var_Type Local_Intrin_Vars[] =
{
#if defined(__CYGWIN__)
  MAKE_VARIABLE("_gtk_major_version", &_gtk_major_version,UI,1),
  MAKE_VARIABLE("_gtk_minor_version", &_gtk_minor_version,UI,1),
  MAKE_VARIABLE("_gtk_micro_version", &_gtk_micro_version,UI,1),
#else
  MAKE_VARIABLE("_gtk_major_version", &gtk_major_version,UI,1),
  MAKE_VARIABLE("_gtk_minor_version", &gtk_minor_version,UI,1),
  MAKE_VARIABLE("_gtk_micro_version", &gtk_micro_version,UI,1),
#endif

  MAKE_VARIABLE("_slgtk_debug", &slgtk_debug,I,0),
  MAKE_VARIABLE("_gtk_version", &_gtk_version,I,0),

  MAKE_VARIABLE("SLGTK_STOCK_RECTANGLE",&slgtk_icons[0],S,1),
  MAKE_VARIABLE("SLGTK_STOCK_ELLIPSE",&slgtk_icons[1],S,1),
  MAKE_VARIABLE("SLGTK_STOCK_POLYGON",&slgtk_icons[2],S,1),
  MAKE_VARIABLE("SLGTK_STOCK_LINESEG",&slgtk_icons[3],S,1),

  SLANG_END_INTRIN_VAR_TABLE
};

/* }}} */

/* Module initialization {{{ */

static int init_sl_gtk(SLang_NameSpace_Type *ns)
{
   patch_ftable(gtk_Funcs, O, GtkWidget_Type);
   patch_ftable(GtkWidget_Funcs, O, GtkWidget_Type);
   patch_ftable(GtkObject_Funcs, O, GtkObject_Type);
   patch_ftable(GObject_Funcs,   O, GObject_Type);

   if (-1 == SLns_add_intrin_fun_table (ns, gtk_Funcs, "__GTK__"))
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, GtkWidget_Funcs, NULL))
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, GtkObject_Funcs, NULL))
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, GObject_Funcs, NULL))
	return -1;

   if (install_icons() == -1)
	return -1;

   if (-1 == SLns_add_iconstant_table (ns, Generic_IConsts, NULL)
		|| -1 == SLns_add_iconstant_table (ns, gtk_IConsts, NULL))
	return -1;

#ifdef __CYGWIN__
   _gtk_major_version = gtk_major_version;
   _gtk_minor_version = gtk_minor_version;
   _gtk_micro_version = gtk_micro_version;
#endif

   if (-1 == SLns_add_intrin_var_table (ns, Local_Intrin_Vars, NULL) ||
			-1 == SLns_add_intrin_var_table (ns, gtk_IVars, NULL))
	return -1;

   /* Unlike above, where we load the various intrinsics into the */
   /* given namespace, here we add them to the global namespace   */
   if (SLang_is_defined("Global->_gtk_module_version") == 0 &&
	SLadd_intrin_var_table (Global_Intrin_Vars, NULL) == -1)
	return -1;

#ifdef SLGTK_STATIC_MODULE
   if (SLdefine_for_ifdef ((char*)"GTK_MODULE_STATIC_BINARY") != 0)
	return -1;
#endif

   slgtk_debug = 0;
   return 0;
}

SLANG_MODULE(gtk);
int init_gtk_module_ns(char *ns_name)
{
   int argc = 1;
   char *ARGV[] = { MODULE_NAME, NULL};
   char **argv = ARGV;
   SLang_NameSpace_Type *ns = NULL;

   if (slang_abi_mismatch()) return -1;

   if (SLang_is_defined("min") == 0) {
	if (SLang_init_array_extra() == -1)
	   return -1;
   }

   if (ns_name != NULL) {
	ns = SLns_create_namespace (ns_name);
	if (ns == NULL || (slns = SLmalloc(strlen(ns_name)+1)) == NULL)
	   return -1;
   }

   slirp_debug_pause(MODULE_NAME);

   if (!GtkOpaque_Type) {			/* Already initialized */

	/* Support lite module testing w/out connecting to display system */
	int needs_display = getenv("SLGTK_NO_DISPLAY") == NULL;

	if (allocate_reserved_opaque_types() == -1 ||
	 				allocate_gtk_opaque_types() == -1)
	   return -1;

	/* FIXME: is this still necessary? */
	putenv("GTK_IM_MODULE_FILE=/dev/null");

	if (gtk_init_check(&argc,&argv) != TRUE && needs_display) {
	   SLang_verror(SLEI, "could not initialize Gtk (check $DISPLAY, etc)");
	   return -1; 
	}
   }

   /* Initialize each subsystem */
   if (init_sl_glib(ns) == -1)
      return -1;

   if (init_sl_gdk(ns) == -1)
      return -1;

   if (init_sl_gdkpixbuf(ns) == -1)
      return -1;

   if (init_sl_gtk(ns) == -1)
      return -1;

   if ( gtk_major_version != GTK_MAJOR_VERSION ||
	gtk_minor_version != GTK_MINOR_VERSION ||
	gtk_micro_version != GTK_MICRO_VERSION)
	fprintf(stderr, "Warning: SLgtk compiled with Gtk %d.%d.%d but runtime linked to version %d.%d.%d\n",
		GTK_MAJOR_VERSION, GTK_MINOR_VERSION, GTK_MICRO_VERSION,
		gtk_major_version, gtk_minor_version, gtk_micro_version);

   return 0;
}
/* }}} */
