#include "WipeTowerPrusaMM.hpp"

#include <assert.h>
#include <math.h>
#include <fstream>
#include <iostream>

#ifdef __linux
#include <strings.h>
#endif /* __linux */

#ifdef _MSC_VER 
#define strcasecmp _stricmp
#endif

namespace Slic3r
{

namespace PrusaMultiMaterial {

class Writer
{
public:
	Writer() : 
		m_current_pos(std::numeric_limits<float>::max(), std::numeric_limits<float>::max()),
		m_current_z(0.f),
		m_current_feedrate(0.f),
		m_extrusion_flow(0.f) {}

	Writer& 			 set_initial_position(const WipeTower::xy &pos) { m_current_pos = pos; return *this; }

	Writer&				 set_z(float z) 
		{ m_current_z = z; return *this; }

	Writer& 			 set_extrusion_flow(float flow)
		{ m_extrusion_flow = flow; return *this; }

	Writer& 			 feedrate(float f)
	{
		if (f != m_current_feedrate)
			m_gcode += "G1" + set_format_F(f) + "\n";
		return *this;
	}

	const std::string&   gcode() const { return m_gcode; }
	float                x()     const { return m_current_pos.x; }
	float                y()     const { return m_current_pos.y; }
	const WipeTower::xy& pos()   const { return m_current_pos; }

	// Extrude with an explicitely provided amount of extrusion.
	Writer& extrude_explicit(float x, float y, float e, float f = 0.f) 
	{
		if (x == m_current_pos.x && y == m_current_pos.y && e == 0.f && (f == 0.f || f == m_current_feedrate))
			return *this;
		m_gcode += "G1";
		if (x != m_current_pos.x)
			m_gcode += set_format_X(x);
		if (y != m_current_pos.y)
			m_gcode += set_format_Y(y);
		if (e != 0.f)
			m_gcode += set_format_E(e);
		if (f != 0.f && f != m_current_feedrate)
			m_gcode += set_format_F(f);
		m_gcode += "\n";
		return *this;
	}

	Writer& extrude_explicit(const WipeTower::xy &dest, float e, float f = 0.f) 
		{ return extrude_explicit(dest.x, dest.y, e, f); }

	// Travel to a new XY position. f=0 means use the current value.
	Writer& travel(float x, float y, float f = 0.f)
		{ return extrude_explicit(x, y, 0.f, f); }

	Writer& travel(const WipeTower::xy &dest, float f = 0.f) 
		{ return extrude_explicit(dest.x, dest.y, 0.f, f); }

	// Extrude a line from current position to x, y with the extrusion amount given by m_extrusion_flow.
	Writer& extrude(float x, float y, float f = 0.f)
	{
		float dx = x - m_current_pos.x;
		float dy = y - m_current_pos.y;
		return extrude_explicit(x, y, sqrt(dx*dx+dy*dy) * m_extrusion_flow, f);
	}

	Writer& extrude(const WipeTower::xy &dest, const float f = 0.f) 
		{ return extrude(dest.x, dest.y, f); }

	Writer& load(float e, float f = 0.f)
	{
		if (e == 0.f && (f == 0.f || f == m_current_feedrate))
			return *this;
		m_gcode += "G1";
		if (e != 0.f)
			m_gcode += set_format_E(e);
		if (f != 0.f && f != m_current_feedrate)
			m_gcode += set_format_F(f);
		m_gcode += "\n";
		return *this;
	}

	// Derectract while moving in the X direction.
	// If |x| > 0, the feed rate relates to the x distance,
	// otherwise the feed rate relates to the e distance.
	Writer& load_move_x(float x, float e, float f = 0.f)
		{ return extrude_explicit(x, m_current_pos.y, e, f); }

	Writer& retract(float e, float f = 0.f)
		{ return load(-e, f); }

	// Elevate the extruder head above the current print_z position.
	Writer& z_hop(float hop, float f = 0.f)
	{ 
		m_gcode += std::string("G1") + set_format_Z(m_current_z + hop);
		if (f != 0 && f != m_current_feedrate)
			m_gcode += set_format_F(f);
		m_gcode += "\n";
		return *this;
	}

	// Lower the extruder head back to the current print_z position.
	Writer& z_hop_reset(float f = 0.f) 
		{ return z_hop(0, f); }

	// Move to x1, +y_increment,
	// extrude quickly amount e to x2 with feed f.
	Writer& ram(float x1, float x2, float dy, float e0, float e, float f)
	{
		return  extrude_explicit(x1, m_current_pos.y + dy, e0, f)
			   .extrude_explicit(x2, m_current_pos.y, e);
	}

	// Let the end of the pulled out filament cool down in the cooling tube
	// by moving up and down and moving the print head left / right
	// at the current Y position to spread the leaking material.
	Writer& cool(float x1, float x2, float e1, float e2, float f)
	{
		return  extrude_explicit(x1, m_current_pos.y, e1, f)
			   .extrude_explicit(x2, m_current_pos.y, e2);
	}

	Writer& set_tool(int tool) 
	{
		char buf[64];
		sprintf(buf, "T%d\n", tool);
		m_gcode += buf;
		return *this;
	}

	// Set extruder temperature, don't wait by default.
	Writer& set_extruder_temp(int temperature, bool wait = false)
	{
		char buf[128];
		sprintf(buf, "M%d S%d\n", wait ? 109 : 104, temperature);
		m_gcode += buf;
		return *this;
	};

	// Set speed factor override percentage.
	Writer& speed_override(int speed) 
	{
		char buf[128];
		sprintf(buf, "M220 S%d\n", speed);
		m_gcode += buf;
		return *this;
	};

	// Set digital trimpot motor
	Writer& set_extruder_trimpot(int current) 
	{
		char buf[128];
		sprintf(buf, "M907 E%d\n", current);
		m_gcode += buf;
		return *this;
	};

	Writer& flush_planner_queue() 
	{ 
		m_gcode += "G4 S0\n"; 
		return *this;
	}

	// Reset internal extruder counter.
	Writer& reset_extruder()
	{ 
		m_gcode += "G92 E0\n";
		return *this;
	}

	Writer& comment_with_value(const char *comment, int value)
	{
		char strvalue[64];
		sprintf(strvalue, "%d", value);
		m_gcode += std::string(";") + comment + strvalue + "\n";
		return *this;
	};

	Writer& comment_material(WipeTowerPrusaMM::material_type material)
	{
		m_gcode += "; material : ";
		switch (material)
		{
		case WipeTowerPrusaMM::PVA:
			m_gcode += "#8 (PVA)";
			break;
		case WipeTowerPrusaMM::SCAFF:
			m_gcode += "#5 (Scaffold)";
			break;
		case WipeTowerPrusaMM::FLEX:
			m_gcode += "#4 (Flex)";
			break;
		default:
			m_gcode += "DEFAULT (PLA)";
			break;
		}
		m_gcode += "\n";
		return *this;
	};

	Writer& append(const char *text) { m_gcode += text; return *this; }

private:
	WipeTower::xy m_current_pos;
	float    	  m_current_z;
	float 	  	  m_current_feedrate;
	float 	  	  m_extrusion_flow;
	std::string   m_gcode;

	std::string   set_format_X(float x) {
		char buf[64];
		sprintf(buf, " X%.3f", x);
		m_current_pos.x = x;
		return buf;
	}

	std::string   set_format_Y(float y) {
		char buf[64];
		sprintf(buf, " Y%.3f", y);
		m_current_pos.y = y;
		return buf;
	}

	std::string   set_format_Z(float z) {
		char buf[64];
		sprintf(buf, " Z%.3f", z);
		return buf;
	}

	std::string   set_format_E(float e) {
		char buf[64];
		sprintf(buf, " E%.4f", e);
		return buf;
	}

	std::string   set_format_F(float f) {
		char buf[64];
		sprintf(buf, " F%.0f", f);
		m_current_feedrate = f;
		return buf;
	}
};

} // namespace PrusaMultiMaterial

WipeTowerPrusaMM::material_type WipeTowerPrusaMM::parse_material(const char *name)
{
	if (strcasecmp(name, "PLA") == 0)
		return PLA;
	if (strcasecmp(name, "ABS") == 0)
		return ABS;
	if (strcasecmp(name, "PET") == 0)
		return PET;
	if (strcasecmp(name, "HIPS") == 0)
		return HIPS;
	if (strcasecmp(name, "FLEX") == 0)
		return FLEX;
	if (strcasecmp(name, "SCAFF") == 0)
		return SCAFF;
	if (strcasecmp(name, "EDGE") == 0)
		return EDGE;
	if (strcasecmp(name, "NGEN") == 0)
		return NGEN;
	if (strcasecmp(name, "PVA") == 0)
		return PVA;
	return INVALID;
}

std::pair<std::string, WipeTower::xy> WipeTowerPrusaMM::tool_change(int tool, Purpose purpose)
{
	// Either it is the last tool unload,
	// or there must be a nonzero wipe tower partitions available.
	assert(tool < 0 || it_layer_tools->wipe_tower_partitions > 0);

	if (m_idx_tool_change_in_layer == (unsigned int)(-1)) {
		// First layer, prime the extruder.
		return toolchange_Brim(purpose);
	}

	box_coordinates cleaning_box(
		m_wipe_tower_pos + xy(0.f, m_current_wipe_start_y + 0.5f * m_perimeter_width),
		m_wipe_tower_width, 
		m_wipe_area - m_perimeter_width);

	PrusaMultiMaterial::Writer writer;
	writer.set_extrusion_flow(m_extrusion_flow)
		  .set_z(m_z_pos)
		  .append(";--------------------\n"
			 	  "; CP TOOLCHANGE START\n")
		  .comment_with_value(" toolchange #", m_num_tool_changes)
		  .comment_material(m_current_material)
		  .append(";--------------------\n")
		  .speed_override(100);

    xy initial_position = ((m_current_shape == SHAPE_NORMAL) ? cleaning_box.ld : cleaning_box.lu) + 
    	xy(m_perimeter_width, ((m_current_shape == SHAPE_NORMAL) ? 1.f : -1.f) * m_perimeter_width);

	if (purpose == PURPOSE_MOVE_TO_TOWER || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE) {
		// Scaffold leaks terribly, reduce leaking by a full retract when going to the wipe tower.
		float initial_retract = ((m_current_material == SCAFF) ? 1.f : 0.5f) * m_retract;
		writer 	// Lift for a Z hop.
		  	  	.z_hop(m_zhop, 7200)
		  		// Additional retract on move to tower.
		  		.retract(initial_retract, 3600)
		  		// Move to a starting position, one perimeter width inside the cleaning box.
		  		.travel(initial_position, 7200)
		  		// Unlift for a Z hop.
		  		.z_hop_reset(7200)
		  		// Additional retract on move to tower.
		  		.load(initial_retract, 3600)
		  		.load(m_retract, 1500);
	} else {
		// Already at the initial position.
		writer.set_initial_position(initial_position);
	}

	if (purpose == PURPOSE_EXTRUDE || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE) {
		// Increase the extruder driver current to allow fast ramming.
		writer.set_extruder_trimpot(750);
		// Ram the hot material out of the melt zone, retract the filament into the cooling tubes and let it cool.
		toolchange_Unload(writer, cleaning_box, m_current_material,
			m_is_first_layer ? m_first_layer_temperature[tool]  : m_temperature[tool]);

		if (tool >= 0) {
			// This is not the last change.
			// Change the tool, set a speed override for soluble and flex materials.
			toolchange_Change(writer, tool, m_material[tool]);
			toolchange_Load(writer, cleaning_box);
			// Wipe the newly loaded filament until the end of the assigned wipe area.
			toolchange_Wipe(writer, cleaning_box);
			// Draw a perimeter around cleaning_box and wipe.
			box_coordinates box = cleaning_box;
			if (m_current_shape == SHAPE_REVERSED) {
				std::swap(box.lu, box.ld);
				std::swap(box.ru, box.rd);
			}
			// Draw a perimeter around cleaning_box.
			writer.travel(box.lu, 7000)
				  .extrude(box.ld, 3200).extrude(box.rd)
				  .extrude(box.ru).extrude(box.lu);
			// Wipe the nozzle.
			if (purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE)
				writer.travel(box.ru, 7200)
			  		  .travel(box.lu);
		}

		// Reset the extruder current to a normal value.
		writer.set_extruder_trimpot(550)
			  .feedrate(6000)
			  .flush_planner_queue()
			  .reset_extruder()
			  .append("; CP TOOLCHANGE END\n"
		 		      ";------------------\n"
					  "\n\n");

	    ++ m_num_tool_changes;
	    ++ m_idx_tool_change_in_layer;
	    m_current_wipe_start_y += m_wipe_area;
	}

	return std::pair<std::string, xy>(writer.gcode(), writer.pos());
}

std::pair<std::string, WipeTower::xy> WipeTowerPrusaMM::toolchange_Brim(Purpose purpose, bool sideOnly, float y_offset)
{
	const box_coordinates wipeTower_box(
		m_wipe_tower_pos,
		m_wipe_tower_width,
		m_wipe_area * float(m_max_color_changes) - m_perimeter_width / 2);

	PrusaMultiMaterial::Writer writer;
	writer.set_extrusion_flow(m_extrusion_flow * 1.1f)
		  // Let the writer know the current Z position as a base for Z-hop.
		  .set_z(m_z_pos)
		  .append(
			";-------------------------------------\n"
			"; CP WIPE TOWER FIRST LAYER BRIM START\n");

	xy initial_position = wipeTower_box.lu - xy(m_perimeter_width * 10.f, 0);

	if (purpose == PURPOSE_MOVE_TO_TOWER || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE)
		// Move with Z hop.
		writer.z_hop(m_zhop, 7200)
			  .travel(initial_position, 6000)
			  .z_hop_reset(7200);
	else 
		writer.set_initial_position(initial_position);

	if (purpose == PURPOSE_EXTRUDE || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE) {
		// Prime the extruder 10*m_perimeter_width left along the vertical edge of the wipe tower.
		writer.extrude_explicit(wipeTower_box.ld - xy(m_perimeter_width * 10.f, 0), m_retract, 2400);

		// The tool is supposed to be active and primed at the time when the wipe tower brim is extruded.
		// toolchange_Change(writer, int(tool), m_material[tool]);

		if (sideOnly) {
			float x_offset = m_perimeter_width;
			for (size_t i = 0; i < 4; ++ i, x_offset += m_perimeter_width)
				writer.travel (wipeTower_box.ld + xy(- x_offset,   y_offset), 7000)
					  .extrude(wipeTower_box.lu + xy(- x_offset, - y_offset), 2100);
			writer.travel(wipeTower_box.rd + xy(x_offset, y_offset), 7000);
			x_offset = m_perimeter_width;
			for (size_t i = 0; i < 4; ++ i, x_offset += m_perimeter_width)
				writer.travel (wipeTower_box.rd + xy(x_offset,   y_offset), 7000)
					  .extrude(wipeTower_box.ru + xy(x_offset, - y_offset), 2100);
		} else {
			// Extrude 4 rounds of a brim around the future wipe tower.
			box_coordinates box(wipeTower_box);
			//FIXME why is the box shifted in +Y by 0.5f * m_perimeter_width?
			box.translate(0.f, 0.5f * m_perimeter_width);
			box.expand(0.5f * m_perimeter_width);
			for (size_t i = 0; i < 4; ++ i) {
				writer.travel (box.ld, 7000)
					  .extrude(box.lu, 2100).extrude(box.ru)
					  .extrude(box.rd      ).extrude(box.ld);
				box.expand(m_perimeter_width);
			}
		}

		// Move to the front left corner.
		writer.travel(wipeTower_box.ld, 7000);

		if (purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE)
			// Wipe along the front edge.
			writer.travel(wipeTower_box.rd)
			      .travel(wipeTower_box.ld);
			  
	    writer.append("; CP WIPE TOWER FIRST LAYER BRIM END\n"
				      ";-----------------------------------\n");

		// Mark the brim as extruded.
		m_idx_tool_change_in_layer = 0;
	}

	return std::pair<std::string, xy>(writer.gcode(), writer.pos());
}

// Ram the hot material out of the melt zone, retract the filament into the cooling tubes and let it cool.
void WipeTowerPrusaMM::toolchange_Unload(
	PrusaMultiMaterial::Writer &writer,
	const box_coordinates 	&cleaning_box,
	const material_type		 current_material,
	const int 				 new_temperature)
{
	float xl = cleaning_box.ld.x + 0.5f * m_perimeter_width;
	float xr = cleaning_box.rd.x - 0.5f * m_perimeter_width;
	float y_step = ((m_current_shape == SHAPE_NORMAL) ? 1.f : -1.f) * m_perimeter_width;

	writer.append("; CP TOOLCHANGE UNLOAD\n");

	// Ram the hot material out of the extruder melt zone.
	// Current extruder position is on the left, one perimeter inside the cleaning box in both X and Y.
	float e0 = m_perimeter_width * m_extrusion_flow;
	float e = (xr - xl) * m_extrusion_flow;
	switch (current_material)
	{
	case ABS:
   		// ramming          start                    end                  y increment     amount feedrate
		writer.ram(xl + m_perimeter_width * 2, xr - m_perimeter_width,     y_step * 0.2f, 0,  1.2f * e,  4000)
			  .ram(xr - m_perimeter_width,     xl + m_perimeter_width,     y_step * 1.2f, e0, 1.6f * e,  4600)
			  .ram(xl + m_perimeter_width * 2, xr - m_perimeter_width * 2, y_step * 1.2f, e0, 1.8f * e,  5000)
			  .ram(xr - m_perimeter_width * 2, xl + m_perimeter_width * 2, y_step * 1.2f, e0, 1.8f * e,  5000);
		break;
	case PVA:
		writer.ram(xl + m_perimeter_width * 2, xr - m_perimeter_width,     y_step * 0.2f, 0,  3,     4000)
			  .ram(xr - m_perimeter_width,     xl + m_perimeter_width,     y_step * 1.5f, 0,  3,     4500)
			  .ram(xl + m_perimeter_width * 2, xr - m_perimeter_width * 2, y_step * 1.5f, 0,  3,     4800)
			  .ram(xr - m_perimeter_width,     xl + m_perimeter_width,     y_step * 1.5f, 0,  3,     5000);
		break;
	case SCAFF:
		writer.ram(xl + m_perimeter_width * 2, xr - m_perimeter_width,     y_step * 2.f,  0,  3,     4000)
			  .ram(xr - m_perimeter_width,     xl + m_perimeter_width,     y_step * 3.f,  0,  4,     4600)
			  .ram(xl + m_perimeter_width * 2, xr - m_perimeter_width * 2, y_step * 3.f,  0,  4.5,   5200);
		break;
	default:
		writer.ram(xl + m_perimeter_width * 2, xr - m_perimeter_width,     y_step * 0.2f, 0,  1.6f  * e, 4000)
			  .ram(xr - m_perimeter_width,     xl + m_perimeter_width,     y_step * 1.2f, e0, 1.65f * e, 4600)
			  .ram(xl + m_perimeter_width * 2, xr - m_perimeter_width * 2, y_step * 1.2f, e0, 1.74f * e, 5200);
	}

	// Pull the filament end into a cooling tube.
	writer.retract(15, 5000).retract(50, 5400).retract(15, 3000).retract(12, 2000);

	if (new_temperature != 0)
		// Set the extruder temperature, but don't wait.
		writer.set_extruder_temp(new_temperature, false);

	// In case the current print head position is closer to the left edge, reverse the direction.
	if (std::abs(writer.x() - xl) < std::abs(writer.x() - xr))
		std::swap(xl, xr);
	// Horizontal cooling moves will be performed at the following Y coordinate:
	writer.travel(xr, writer.y() + y_step * 0.8f, 7200);
	switch (current_material)
	{
	case ABS:
		writer.cool(xl, xr, 3, -5, 1600)
			  .cool(xl, xr, 5, -5, 2000)
			  .cool(xl, xr, 5, -5, 2400)
			  .cool(xl, xr, 5, -3, 2400);
		break;
	case PVA:
		writer.cool(xl, xr, 3, -5, 1600)
			  .cool(xl, xr, 5, -5, 2000)
			  .cool(xl, xr, 5, -5, 2200)
			  .cool(xl, xr, 5, -5, 2400)
			  .cool(xl, xr, 5, -5, 2400)
			  .cool(xl, xr, 5, -3, 2400);
		break;
	case SCAFF:
		writer.cool(xl, xr, 3, -5, 1600)
			  .cool(xl, xr, 5, -5, 2000)
			  .cool(xl, xr, 5, -5, 2200)
			  .cool(xl, xr, 5, -5, 2200)
			  .cool(xl, xr, 5, -3, 2400);
		break;
	default:
		writer.cool(xl, xr, 3, -5, 1600)
			  .cool(xl, xr, 5, -5, 2000)
			  .cool(xl, xr, 5, -5, 2400)
			  .cool(xl, xr, 5, -3, 2400);
	}

	writer.flush_planner_queue();
}

// Change the tool, set a speed override for solube and flex materials.
void WipeTowerPrusaMM::toolchange_Change(
	PrusaMultiMaterial::Writer &writer,
	const int 		new_tool, 
	material_type 	new_material)
{
	// Speed override for the material. Go slow for flex and soluble materials.
	int speed_override;
	switch (new_material) {
	case PVA:   speed_override = (m_z_pos < 0.80f) ? 60 : 80; break;
	case SCAFF: speed_override = 35; break;
	case FLEX:  speed_override = 35; break;
	default:    speed_override = 100;
	}
	writer.set_tool(new_tool)
	      .speed_override(speed_override)
	      .flush_planner_queue();
	m_current_material = new_material;
}

void WipeTowerPrusaMM::toolchange_Load(
	PrusaMultiMaterial::Writer &writer,
	const box_coordinates  &cleaning_box)
{
	float xl = cleaning_box.ld.x + m_perimeter_width;
	float xr = cleaning_box.rd.x - m_perimeter_width;

	writer.append("; CP TOOLCHANGE LOAD\n")
	// Load the filament while moving left / right,
	// so the excess material will not create a blob at a single position.
		  .load_move_x(xr, 20, 1400)
		  .load_move_x(xl, 40, 3000)
		  .load_move_x(xr, 20, 1600)
		  .load_move_x(xl, 10, 1000);

	// Extrude first five lines (just three lines if colorInit is set).
	writer.extrude(xr, writer.y(), 1600);
	bool   colorInit = false;
	size_t pass = colorInit ? 1 : 2;
	float  dy   = ((m_current_shape == SHAPE_NORMAL) ? 1.f : -1.f) * m_perimeter_width * 0.85f;
	for (int i = 0; i < pass; ++ i) {
		writer.travel (xr, writer.y() + dy, 7200);
		writer.extrude(xl, writer.y(), 		2200);
		writer.travel (xl, writer.y() + dy, 7200);
	 	writer.extrude(xr, writer.y(), 		2200);
	}

	// Reset the extruder current to the normal value.
	writer.set_extruder_trimpot(550);
}

// Wipe the newly loaded filament until the end of the assigned wipe area.
void WipeTowerPrusaMM::toolchange_Wipe(
	PrusaMultiMaterial::Writer &writer,
	const box_coordinates  &cleaning_box)
{
	// Increase flow on first layer, slow down print.
	writer.set_extrusion_flow(m_extrusion_flow * (m_is_first_layer ? 1.18f : 1.f))
		  .append("; CP TOOLCHANGE WIPE\n");
	float wipe_coeff = m_is_first_layer ? 0.5f : 1.f;
	float xl = cleaning_box.ld.x + 2.f * m_perimeter_width;
	float xr = cleaning_box.rd.x - 2.f * m_perimeter_width;
	// Wipe speed will increase up to 4800.
	float wipe_speed 	 = 4200.f;
	float wipe_speed_inc = 50.f;
	float wipe_speed_max = 4800.f;
	// Y increment per wipe line.
	float dy = ((m_current_shape == SHAPE_NORMAL) ? 1.f : -1.f) * m_perimeter_width * 0.8f;
	for (bool p = true; ; p = ! p) {
		wipe_speed = std::min(wipe_speed_max, wipe_speed + wipe_speed_inc);
		if (p) {
			writer.extrude(xl - m_perimeter_width / 2, writer.y() + dy, wipe_speed * wipe_coeff);
			writer.extrude(xr + m_perimeter_width,     writer.y());
		} else {
			writer.extrude(xl - m_perimeter_width,     writer.y() + dy, wipe_speed * wipe_coeff);
			writer.extrude(xr + m_perimeter_width * 2, writer.y());
		}
		wipe_speed = std::min(wipe_speed_max, wipe_speed + wipe_speed_inc);
		writer.extrude(xr + m_perimeter_width, writer.y() + dy, wipe_speed * wipe_coeff);
		writer.extrude(xl - m_perimeter_width, writer.y());
		if ((m_current_shape == SHAPE_NORMAL) ?
			(writer.y() > cleaning_box.lu.y - m_perimeter_width) :
			(writer.y() < cleaning_box.ld.y + m_perimeter_width))
			// Next wipe line does not fit the cleaning box.
			break;
	}
	// Reset the extrusion flow.
	writer.set_extrusion_flow(m_extrusion_flow);
}

std::pair<std::string, WipeTower::xy> WipeTowerPrusaMM::finish_layer(Purpose purpose)
{
	// This should only be called if the layer is not finished yet.
	// Otherwise the caller would likely travel to the wipe tower in vain.
	assert(! this->layer_finished());

	PrusaMultiMaterial::Writer writer;
	writer.set_extrusion_flow(m_extrusion_flow)
		  .set_z(m_z_pos)
		  .append(";--------------------\n"
				  "; CP EMPTY GRID START\n")
		  // m_num_layer_changes is incremented by set_z, so it is 1 based.
		  .comment_with_value(" layer #", m_num_layer_changes - 1);

	// Slow down on the 1st layer.
	float speed_factor = m_is_first_layer ? 0.5f : 1.f;

	box_coordinates fill_box(m_wipe_tower_pos + xy(0.f, float(m_idx_tool_change_in_layer) * m_wipe_area),
		m_wipe_tower_width, float(m_max_color_changes - m_idx_tool_change_in_layer) * m_wipe_area);
	fill_box.expand(0.f, - 0.5f * m_perimeter_width);
	{
		float firstLayerOffset = 0.f;
		fill_box.ld.y += firstLayerOffset;
		fill_box.rd.y += firstLayerOffset;
	}

	if (purpose == PURPOSE_MOVE_TO_TOWER || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE) {
		if (m_idx_tool_change_in_layer == 0) {
			// There were no tool changes at all in this layer.
			writer.retract(m_retract * 1.5f, 3600)
				  // Jump with retract to fill_box.ld + a random shift in +x.
				  .z_hop(m_zhop, 7200)
				  .travel(fill_box.ld + xy(5.f + 15.f * float(rand()) / RAND_MAX, 0.f), 7000)
				  .z_hop_reset(7200)
				  // Prime the extruder.
				  .load_move_x(fill_box.ld.x, m_retract * 1.5f, 3600);
		} else {
			// Otherwise the extruder is already over the wipe tower.
		}
	} else {
		// The print head is inside the wipe tower. Rather move to the start of the following extrusion.
		// writer.set_initial_position(fill_box.ld);
		writer.travel(fill_box.ld, 7000);
	}

	if (purpose == PURPOSE_EXTRUDE || purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE) {
		// Extrude the first perimeter.
		box_coordinates box = fill_box;
		writer.extrude(box.lu, 2400 * speed_factor)
			  .extrude(box.ru)
			  .extrude(box.rd)
			  .extrude(box.ld + xy(m_perimeter_width / 2, 0));

		// Extrude second perimeter.
		box.expand(- m_perimeter_width / 2);
		writer.extrude(box.lu, 3200 * speed_factor)
			  .extrude(box.ru)
			  .extrude(box.rd)
			  .extrude(box.ld + xy(m_perimeter_width / 2, 0))
			  .extrude(box.ld + xy(m_perimeter_width / 2, m_perimeter_width / 2));

		// Extrude an inverse U at the left of the region.
		writer.extrude(fill_box.ld + xy(m_perimeter_width * 3,   m_perimeter_width), 2900 * speed_factor)
		      .extrude(fill_box.lu + xy(m_perimeter_width * 3, - m_perimeter_width))
			  .extrude(fill_box.lu + xy(m_perimeter_width * 6, - m_perimeter_width))
			  .extrude(fill_box.ld + xy(m_perimeter_width * 6,   m_perimeter_width));

		if (fill_box.lu.y - fill_box.ld.y > 4.f) {
			// Extrude three zig-zags.
			float step = (m_wipe_tower_width - m_perimeter_width * 12.f) / 12.f;
			for (size_t i = 0; i < 3; ++ i) {
				writer.extrude(writer.x() + step, fill_box.ld.y + m_perimeter_width * 8, 3200 * speed_factor);
				writer.extrude(writer.x()       , fill_box.lu.y - m_perimeter_width * 8);
				writer.extrude(writer.x() + step, fill_box.lu.y - m_perimeter_width    );
				writer.extrude(writer.x() + step, fill_box.lu.y - m_perimeter_width * 8);
				writer.extrude(writer.x()       , fill_box.ld.y + m_perimeter_width * 8);
				writer.extrude(writer.x() + step, fill_box.ld.y + m_perimeter_width    );
			}
		}

			   // Extrude an inverse U at the left of the region.
		writer.extrude(fill_box.ru + xy(- m_perimeter_width * 6, - m_perimeter_width), 2900 * speed_factor)
			  .extrude(fill_box.ru + xy(- m_perimeter_width * 3, - m_perimeter_width))
			  .extrude(fill_box.rd + xy(- m_perimeter_width * 3,   m_perimeter_width))
			  .extrude(fill_box.rd + xy(- m_perimeter_width,       m_perimeter_width));

		if (purpose == PURPOSE_MOVE_TO_TOWER_AND_EXTRUDE)
	       	// Wipe along the front side of the current wiping box.
			writer.travel(fill_box.ld + xy(  m_perimeter_width, m_perimeter_width / 2), 7200)
			  	  .travel(fill_box.rd + xy(- m_perimeter_width, m_perimeter_width / 2));
		else
			writer.feedrate(7200);

		writer.append("; CP EMPTY GRID END\n"
				      ";------------------\n\n\n\n\n\n\n");

		// Indicate that this wipe tower layer is fully covered.
	    m_idx_tool_change_in_layer = m_max_color_changes;
	}

	return std::pair<std::string, xy>(writer.gcode(), writer.pos());
}

}; // namespace Slic3r
