/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of version 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_tcptable.h"
#include "slid_fmt_xlong0.h"
#include "slid_log.h"
#include "slid_read.h"
#include <open.h>
#include <byte.h>
#include <scan.h>
#include <buffer.h>
#include <str.h>
#include <fmt.h>
#include <unistd.h>

#ifdef __linux__

#define MAX_TCPTABLE_BUFFLEN 4096

#ifndef NDEBUG
#define SLIDENTD_PARSE_DEBUG_MESSAGES 1
#endif

static char tcptable_buffer[MAX_TCPTABLE_BUFFLEN + 1];
static int tcptable;
static buffer tcpbuff;

static struct tcptable_layout {
	int conn_status_col;
	int uid_col;
	int local_addr_col;
	int local_port_col;
	int remote_port_col;
	int remote_addr_col;
	int parsed_first_line;
} tcptable_layout __attribute__ ((__aligned__));

/*

Here's what the linux /proc/net/tcp table looks like: 

  sl  local_address rem_address   st tx_queue rx_queue tr tm->when retrnsmt   uid  timeout inode 
   0: 00000000:08AE 00000000:0000 0A 00000000:00000000 00:00000000 00000000   500        0 807345
   1: 00000000:177B 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 805915
   2: 0200000A:0016 0300000A:0412 01 00000000:00000000 00:00000000 00000000     0        0 805905
   3: 00000000:177A 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 803388
   4: 0200000A:0016 0300000A:0411 01 00000014:00000000 01:00000014 00000000     0        0 803377
   5: 00000000:0015 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800694
   6: 00000000:0050 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800688
   7: 7D8B9DC3:0035 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800676
   8: 0200000A:0035 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800630
   9: 0100007F:0035 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800626
  10: 00000000:0019 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800602
  11: 00000000:0274 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800574
  12: 00000000:0071 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800570
  13: 00000000:006E 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 800568
  14: 00000000:0C38 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 483750
  15: 00000000:0224 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 1012  
  16: 00000000:0203 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 685   
  17: 00000000:0016 00000000:0000 0A 00000000:00000000 00:00000000 00000000     0        0 664   

*/

static int
slid_index_find(const char *haystack, const char *needle)
{
	return abs(haystack - strstr(haystack, needle));
}

static void
slid_parse_tcptable_headers(char *s, int len)
{
	tcptable_layout.local_addr_col = slid_index_find(s, "local_address ");
	tcptable_layout.remote_addr_col = slid_index_find(s, "rem_address ");
	tcptable_layout.conn_status_col = slid_index_find(s, "st ");
	tcptable_layout.uid_col = slid_index_find(s, "  uid ");

#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Got local addr at ");
		buffer_putlong(log_buffer, tcptable_layout.local_addr_col);
		buffer_puts(log_buffer, " rem addr at ");
		buffer_putlong(log_buffer, tcptable_layout.remote_addr_col);
		buffer_puts(log_buffer, " status at ");
		buffer_putlong(log_buffer, tcptable_layout.conn_status_col);
		buffer_puts(log_buffer, " uid at ");
		buffer_putlong(log_buffer, tcptable_layout.uid_col);
		buffer_putsflush(log_buffer, "\n");
	}
#endif
}

static void
slid_parse_tcptable_firstline(char *s, int len)
{
	int pos = tcptable_layout.local_addr_col
	    + scan_noncharsetnskip(s + tcptable_layout.local_addr_col, ":",
				   len - tcptable_layout.local_addr_col)
	    + 1;
	tcptable_layout.local_port_col = pos;

	pos = tcptable_layout.remote_addr_col
	    + scan_noncharsetnskip(s + tcptable_layout.remote_addr_col, ":",
				   len - tcptable_layout.remote_addr_col)
	    + 1;
	tcptable_layout.remote_port_col = pos;

	tcptable_layout.parsed_first_line = 1;
#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Got local port at ");
		buffer_putlong(log_buffer, tcptable_layout.local_port_col);
		buffer_puts(log_buffer, " rem port at ");
		buffer_putlong(log_buffer, tcptable_layout.remote_port_col);
		buffer_puts(log_buffer, " Buffer[");
		buffer_put(log_buffer, s, len);
		buffer_putsflush(log_buffer, "]\n");
	}
#endif
}

void
slid_open_tcptable(void)
{

	char linebuf[MAX_TCPTABLE_BUFFLEN + 1];
	int len;

	tcptable = open_read(SLIDENTD_TCPTABLE_PATH);
	if (tcptable < 0 || errno) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to open kernel tcp table at ");
		buffer_puts(log_buffer, SLIDENTD_TCPTABLE_PATH);
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}

	buffer_init(&tcpbuff, read, tcptable, tcptable_buffer,
		    sizeof tcptable_buffer);

	len = buffer_getline(&tcpbuff, linebuf, sizeof linebuf - 1);
	if (len < 0) {
		slid_die("Unable to read headers from kernel tcp table");
	}
	slid_parse_tcptable_headers(linebuf, len);

}

void
slid_close_tcptable(void)
{
	close(tcptable);
}

/* get the uid from a tcptable line */
static int
slid_get_uid_fromline(char *buf, int len)
{
	int idx;
	int uid;
	idx =
	    scan_whitenskip(buf + tcptable_layout.uid_col,
			    len - tcptable_layout.uid_col) +
	    tcptable_layout.uid_col;
#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "getting uid from col ");
		buffer_putlong(log_buffer, (long) idx);
		buffer_puts(log_buffer, " limit ");
		buffer_putlong(log_buffer, (long) len);
		buffer_puts(log_buffer, " number starts [");
		buffer_puts(log_buffer, buf + idx);
		buffer_putsflush(log_buffer, "]\n");
	}
#endif
	if (!scan_int(buf + idx, &uid)) {
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to get sensible uid from ");
		buffer_puts(log_buffer, buf + idx);
		buffer_puts(log_buffer, "\n");
		slid_dies();
	}
#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "I get [");
		buffer_putlong(log_buffer, (long) uid);
		buffer_putsflush(log_buffer, "]\n");
	}
#endif
	return (uid);
}

static int
slid_match(const char *a, const char *b, size_t len)
{

#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "comparing [");
		buffer_put(log_buffer, a, len);
		buffer_puts(log_buffer, "] with [");
		buffer_put(log_buffer, b, len);
		buffer_puts(log_buffer, "] Len: ");
		buffer_putlong(log_buffer, (long) len);
		buffer_putsflush(log_buffer, "\n");
	}
#endif
	return (str_diffn(a, b, len) ? 0 : 1);
}

int
slid_search_tcptable(int local_port, int remote_port)
{
	int found_port_pair;
	int is_active;
	char search_local_port[5];
	char search_remote_port[5];
	char search_remote_addr[9];
	struct in_addr a;
	int len;
	stralloc *remote_ip;
	char linebuf[MAX_TCPTABLE_BUFFLEN];

	remote_ip = slid_get_remote_ip();

	if (inet_aton(remote_ip->s, &a) == 0) {
		slid_die("Unable to decode remote address\n");
	}
	slid_fmt_xlong0(search_local_port, (long) local_port,
			sizeof (search_local_port) - 1);
	search_local_port[4] = '\0';
	slid_fmt_xlong0(search_remote_addr, a.s_addr,
			sizeof (search_remote_addr) - 1);
	search_remote_addr[8] = '\0';
	slid_fmt_xlong0(search_remote_port, (long) remote_port,
			sizeof (search_remote_port) - 1);
	search_remote_port[4] = '\0';

#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
	{
		buffer *log_buffer = slid_get_log_buffer();
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "localport: ");
		buffer_putlong(log_buffer, local_port);
		buffer_puts(log_buffer, " remoteport: ");
		buffer_putlong(log_buffer, remote_port);
		buffer_puts(log_buffer, " Remote: ");
		buffer_puts(log_buffer, search_remote_addr);
		buffer_puts(log_buffer, " ");
		buffer_puts(log_buffer, search_remote_port);
		buffer_puts(log_buffer, " Local: ");
		buffer_puts(log_buffer, search_local_port);
		buffer_putsflush(log_buffer, "\n");
	}
#endif

	for (;;) {
		int rem_addr_ok;
		int rem_port_ok;
		int lcl_port_ok;

		len = buffer_getline(&tcpbuff, linebuf, sizeof linebuf);
		if (len <= 0) {
			slid_die("Unable to match port pair");
		}
#ifdef SLIDENTD_PARSE_DEBUG_MESSAGES
		{
			buffer *log_buffer = slid_get_log_buffer();
			slid_put_datestamp(log_buffer);
			buffer_puts(log_buffer, " read ");
			buffer_putlong(log_buffer, len);
			buffer_puts(log_buffer, " chars [");
			buffer_put(log_buffer, linebuf, len);
			buffer_putsflush(log_buffer, "]\n");
		}
#endif

		if (!tcptable_layout.parsed_first_line) {
			slid_parse_tcptable_firstline(linebuf, len);
		}
		rem_addr_ok = slid_match(linebuf +
					 tcptable_layout.remote_addr_col,
					 search_remote_addr, 8);
		rem_port_ok = slid_match(linebuf +
					 tcptable_layout.remote_port_col,
					 search_remote_port, 4);
		lcl_port_ok = slid_match(linebuf +
					 tcptable_layout.local_port_col,
					 search_local_port, 4);

		found_port_pair = (rem_addr_ok && rem_port_ok && lcl_port_ok);
		is_active = slid_match(linebuf +
				       tcptable_layout.conn_status_col,
				       "01", 2);

		if (found_port_pair && is_active) {
			break;
		}
	}

	return slid_get_uid_fromline(linebuf, len);
}

#else
#error System dependant tcptable routines not yet written for your platform
#endif
