package SliMP3::Decoder;

# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License, 
# version 2.

# The two micronas chips have a lot in common wrt volume/tone/balance 
# controls, so all the shared stuff goes here.

use strict;

use SliMP3::mas3507d;
use SliMP3::i2c;
use SliMP3::Misc;
use SliMP3::Stream;

# The codes come from the MAS3507D datasheet 

my @trebleCodes =(
    'b2c00',    
    'bb400',    
    'c1800',    
    'c6c00',    
    'cbc00',
    'd0400',
    'd5000',
    'd9800',
    'de000',
    'e2800',    
    'e7e00',    
    'ec000',
    'f0c00',
    'f5c00',
    '0fac00',
    '00000',    # midpoint (50% or no adjustment)
    '05400',
    '0ac00',
    '10400',
    '16000',
    '1c000',
    '22000',
    '28400',
    '2ec00',
    '35400',
    '3c000',
    '42c00',
    '49c00',
    '51800',
    '58400',
    '5f800',
);



my @bassCodes =(
    '9e400',    
    'a2800',    
    'a7400',    
    'ac400',    
    'b1800',    
    'b7400',    
    'bd400',    
    'c3c00',    
    'ca400',    
    'd1800',    
    'd8c00',    
    'e0400',    
    'e8000',    
    'efc00',
    'f7c00',
    '00000',  # midpoint (50% or no adjustment)
    '00800',
    '10000',
    '17c00',
    '1f800',
    '27000',
    '2e400',
    '35800',
    '3c000',
    '42800',
    '48800',
    '4e400',
    '53800',
    '58800',
    '5d400',
    '61800',
);


my @prefactorCodes =(
    'e9400',
    'e6800',
    'e3400',
    'dfc00',
    'dc000',
    'd7800',
    'd25c0',
    'cd000',
    'c6c00',
    'bfc00',
    'b8000',
    'af400',
    'a5800',
    '9a400',
    '8e000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000',
    '80000'    
);


#
# initialize the MAS3507D and tell the client to start a new stream
#

sub reset {
	my $client = shift;
	my $pcm = shift;

	if ($client->decoder eq 'mas3507d') {

		SliMP3::i2c::send($client,
			 SliMP3::mas3507d::masWrite('config','00002')
			.SliMP3::mas3507d::masWrite('loadconfig')
			.SliMP3::mas3507d::masWrite('plloffset48','5d9e9')
			.SliMP3::mas3507d::masWrite('plloffset44','cecf5')
			.SliMP3::mas3507d::masWrite('setpll')
		);

		# sleep for .05 seconds
		# this little kludge is to prevent overflowing cs8900's RX buf because the client
		# is still processing the i2c commands we just sent.
		select(undef,undef,undef,.05);

	}
}	


sub volume {
	my ($client, $volume) = @_;

	# volume squared seems to correlate better with the linear scale.  I'm sure there's something
	# optimal, but this is better.
	my $level = sprintf('%X', 0xFFFFF - 0x7FFFF * $volume * $volume);

	if ($client->decoder eq 'mas3507d') {
		
		SliMP3::i2c::send($client,
			 SliMP3::mas3507d::masWrite('ll', $level)
			.SliMP3::mas3507d::masWrite('rr', $level)
		);

	}
}

#
# set the MAS3507D treble in the range of -1 to 1
#

sub treble {
	my ($client, $treble) = @_;

	my $index = int($treble / $SliMP3::Control::maxTreble * (scalar(@trebleCodes)-1) + 0.5);

	if ($client->decoder eq 'mas3507d') {	
		SliMP3::i2c::send($client,
			SliMP3::mas3507d::masWrite('treble', $trebleCodes[$index])
		);	
		$::d_control && msg("setting new treble value of $treble (index $index)\n"); 

#		SliMP3::i2c::send($client,
#			SliMP3::mas3507d::masWrite('prefactor', $prefactorCodes{$treble})
#		);	

	}

}

#
# set the MAS3507D bass in the range of -1 to 1
#

sub bass {
	my ($client, $bass) = @_;
	
	my $index = int($bass / $SliMP3::Control::maxTreble * (scalar(@bassCodes)-1) + 0.5);

	if ($client->decoder eq 'mas3507d') {
	
		SliMP3::i2c::send($client,
			SliMP3::mas3507d::masWrite('bass', $bassCodes[$index])
		);	
	
#		$::d_control && msg("setting new bass value of $bass (index $index)\n"); 
#		SliMP3::i2c::send($client,
#			SliMP3::mas3507d::masWrite('prefactor', $prefactorCodes{$bass})
#		);	
	}

}


1;

__END__
