/*
** Copyright (C) 2000, 2001 Rene Puls <rpuls@gmx.net>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <getopt.h>
#include <ctbcs.h>

#include "smartcard.h"

/* This is used for deciding which "action" to take (read, write, etc). */

enum what_t { NOT_SET, ERROR_MULTI, 
	      SHOWINFO, READCARD, WRITECARD, CLEARCARD };

/* This prints the usage information for our program. Usually called
   when the user asks for --help. PROGNAME should be argv[0]. */

void print_usage(char const * const progname)
{
  printf("Usage: %s [ACTION] [OPTION]...\n", progname);
  printf("Reads and writes data on a smart card, or displays ");
  printf("information about a\nsmart card and its terminal.\n\n");
  printf("Actions:\n");
  printf("  -r, --read              read data\n");
  printf("  -w, --write             write data\n");
  printf("  -e, --erase             clear card (erases all data)\n");
  printf("  -i, --information       display some device information\n");
  printf("\n");
  printf("Other Options:\n");
  printf("  -c, --count=NUM         number of bytes to read/write\n");
  printf("  -o, --offset=NUM        start reading/writing at position NUM\n");
  printf("      --force-atr         analyze the ATR even if it seems fake\n");
  printf("      --debug             display debugging information\n");
  printf("\n");
  printf("Communication Options:\n");
  printf("  -p, --port=NUM          port number (0=COM1, 1=COM2, ...)\n");
  printf("\n");
  printf("Standard Options:\n");
  printf("  -h, --help              usage information\n");
  printf("  -v, --version           display the program version number\n");
  printf("\n");
  printf("Currently, the standard input/output channels are used by ");
  printf("all options.\n");
}

/* This prints the version and package name of our program and is
   called when the user supplies --version as an argument. */

void print_version(char const * const progname)
{
  printf("%s " VERSION "\n", PACKAGE);
  printf("Copyright (C) 2000, 2001 Ren Puls\n\n");
  printf("This is free software; see the source for copying conditions. ");
  printf("There is NO\nwarranty; not even for MERCHANTABILITY or FITNESS ");
  printf("FOR A PARTICULAR PURPOSE.\n\n");
  printf("Please send comments and bug reports to <rpuls@gmx.net>.\n");
}

/* These variables are used to store the options parsed from the
   command line: */

static int force_read_atr = 0;	/* option: read ATR even if it seems fake */
static int portnum = 0;		/* option: port number to use */
static int offset = 0;		/* option: offset for reading/writing */
static int size = 255;		/* option: number of bytes to read */
int debug = 0;			/* option: enable debugging */

static struct option long_options[] = {    /* long option definitions */
  {"help", 0, 0, 0},
  {"version", 0, 0, 0},
  {"read", 0, 0, 0},
  {"write", 0, 0, 0},
  {"information", 0, 0, 0},
  {"force-atr", 0, &force_read_atr, 1},
  {"port", 1, 0, 0},
  {"offset", 1, 0, 0},
  {"count", 1, 0, 0},
  {"clear", 0, 0, 0},
  {"debug", 0, &debug, 1},
  {0, 0, 0, 0}
};

/* Our big, baaad main function. :) */

int main(int argc, char *argv[])
{
  int c;			/* temporary variable */
  int option_index;		/* for getopt_long */
  enum what_t what = NOT_SET;	/* what to do */
  int cnr = 0;			/* handle for the CT_xxx functions */
  int count;			/* number of bytes read/written */
  char *buf = NULL;		/* buffer for reading/writing data */

  /* Get the COM port from the environment, if available. */

  if (getenv("SMARTCARD_PORT"))
    portnum = atoi(getenv("SMARTCARD_PORT"));

  /* Parse the command line arguments. */

  while ((c = getopt_long(argc, argv, "hvirwep:o:c:", 
			  long_options, &option_index)) != -1)

    switch (c) {
    case 0:			/* long option */
      if (strcmp(long_options[option_index].name, "help") == 0) {
	print_usage(argv[0]);
	exit(0);
	break;
      }
      if (strcmp(long_options[option_index].name, "version") == 0) {
	print_version(argv[0]);
	exit(0);
	break;
      }
      else if (strcmp(long_options[option_index].name, "port") == 0) {
	portnum = atoi(optarg);
	break;
      }
      else if (strcmp(long_options[option_index].name, "count") == 0) {
	size = atoi(optarg);
	break;
      }
      else if (strcmp(long_options[option_index].name, "offset") == 0) {
	offset = atoi(optarg);
	break;
      }
      else if (strcmp(long_options[option_index].name, "read") == 0) {
	if (what == NOT_SET)
	  what = READCARD;
	else
	  what = ERROR_MULTI;
	break;
      }
      else if (strcmp(long_options[option_index].name, "write") == 0) {
	if (what == NOT_SET)
	  what = WRITECARD;
	else
	  what = ERROR_MULTI;
	break;
      }
      else if (strcmp(long_options[option_index].name, "information") == 0) {
	if (what == NOT_SET)
	  what = SHOWINFO;
	else
	  what = ERROR_MULTI;
	break;
      }
      else if (strcmp(long_options[option_index].name, "erase") == 0) {
	if (what == NOT_SET)
	  what = CLEARCARD;
	else
	  what = ERROR_MULTI;
	break;
      }
      break;
      
    case 'h':			/* short options start here */
      print_usage(argv[0]);
      exit(0);
      break;
    case 'v':
      print_version(argv[0]);
      exit(0);
      break;
    case 'i':
      if (what == NOT_SET)
	what = SHOWINFO;
      else
	what = ERROR_MULTI;
      break;
    case 'r':
      if (what == NOT_SET)
	what = READCARD;
      else
	what = ERROR_MULTI;
      break;
    case 'w':
      if (what == NOT_SET)
	what = WRITECARD;
      else
	what = ERROR_MULTI;
      break;
    case 'e':
      if (what == NOT_SET)
	what = CLEARCARD;
      else
	what = ERROR_MULTI;
      break;
    case 'p':
      portnum = atoi(optarg);
      break;
    case 'o':
      offset = atoi(optarg);
      break;
    case 'c':
      size = atoi(optarg);
      break;
    }

  /* Bail out if multiple read/write/info options were given. */
  if (what == ERROR_MULTI) {
    fprintf(stderr, "%s: multiple actions given, but only one allowed "
	    "(try --help)\n",
	    argv[0]);
    exit(1);
  }
  
  /* The same if no action was specified at all. */
  if (what == NOT_SET) {
    fprintf(stderr, "%s: no action specified, try --help\n",
	    argv[0]);
    exit(1);
  }

  /* Initialize the card terminal. */
  c = smartcard_init_terminal(cnr, portnum);

  if (c != 0) {
    fprintf(stderr, "%s: error initializing smart card terminal: %s\n", 
	    argv[0], smartcard_ctapi_error_msg(c));
    exit(1);
  }

  if (debug)
    fprintf(stderr, "debug: size is %d\n", size);

  /* Now check what we should do. */
  switch (what) {

  case READCARD:		/* read data from the card */
    c = smartcard_activate_card(cnr);
    if (c > 0) {
      fprintf(stderr, "%s: asynchronous cards not supported\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    else if (c < 0) {
      fprintf(stderr, "%s: no card in terminal\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    buf = malloc(size);
    count = smartcard_read_data(cnr, buf, size, offset);
    if (debug)
      fprintf(stderr, "debug: smartcard_read_data() returned %d\n", count);

    count = fwrite(buf, sizeof(buf[0]), count, stdout);
    if (debug)
      fprintf(stderr, "debug: fwrite() returned %d\n", count);

    free(buf);
    break;

  case WRITECARD:		/* write data to the card */
    c = smartcard_activate_card(cnr);
    if (c > 0) {
      fprintf(stderr, "%s: asynchronous cards not supported\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    else if (c < 0) {
      fprintf(stderr, "%s: no card in terminal\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    buf = malloc(size);

    /* This will read exactly SIZE bytes, and no more. If you happen
       to type more on the console, the characters you typed will
       probably appear on your prompt. This is normal. */

    count = fread(buf, sizeof(buf[0]), size, stdin);
    if (debug)
      fprintf(stderr, "debug: fread() returned %d\n", count);

    if (!ferror(stdin)) {
      fprintf(stderr, "debug: writing %d bytes of data\n", count);
      smartcard_write_data(cnr, buf, count, offset);
    } else {
      free(buf);
      perror("%s: error reading data");
      smartcard_close_terminal(cnr);
      exit(1);
    }

    free(buf);
    break;

  case CLEARCARD:		/* erase the card */
    c = smartcard_activate_card(cnr);
    if (c > 0) {
      fprintf(stderr, "%s: asynchronous cards not supported\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    else if (c < 0) {
      fprintf(stderr, "%s: no card in terminal\n", argv[0]);
      smartcard_close_terminal(cnr);
      exit(1);
    }
    buf = malloc(size);
    memset(buf, 0, size);
    c = smartcard_write_data(cnr, buf, size, offset);
    
    if (c != 0)
      fprintf(stderr, "%s: error writing data: %s\n",
	      argv[0], smartcard_ctapi_error_msg(c));
    
    free(buf);
    break;
    
  case SHOWINFO:		/* display some information */
    smartcard_display_info(cnr, force_read_atr);
    break;

  case NOT_SET:			/* should never happen */
  case ERROR_MULTI:
    fprintf(stderr, "Hey, you broke something!\n");
    break;
  }

  /* Close the card terminal. */
  c = smartcard_close_terminal(cnr);
  
  if (c != 0)
    fprintf(stderr, "%s: error closing terminal: %s\n",
	    smartcard_ctapi_error_msg(c));

  return 0;
}
