-- See the Copyright notice at the end of this file.
--
class XMLNS_NODE
	--
	-- A node in an XMLNS_TREE
	--

creation {XMLNS_TREE}
	make

feature {ANY}
	namespace: STRING
			-- The namespace of the node

	name: STRING
			-- The name of the node

	line, column: INTEGER
			-- The position of the node in its source file

	attribute_namespace (index: INTEGER): STRING is
			-- The namespace of the i'th attribute
		require
			index.in_range(1, attributes_count)
		local
			i, d, n: INTEGER
		do
			from
				i := attributes.lower
				d := index
			until
				Result /= Void
			loop
				n := attributes.key(i).count
				if d < n then
					Result := attributes.key(i)
				else
					d := d - n
					i := i + 1
				end
			end
		end

	attribute_name (index: INTEGER): STRING is
			-- The name of the i'th attribute
		require
			index.in_range(1, attributes_count)
		local
			i, d, n: INTEGER
		do
			from
				i := attributes.lower
				d := index
			until
				Result /= Void
			loop
				n := attributes.key(i).count
				if d < n then
					Result := attributes.item(i).key(d)
				else
					d := d - n
					i := i + 1
				end
			end
		end

	attribute_value (index: INTEGER): STRING is
			-- The value of the i'th attribute
		require
			index.in_range(1, attributes_count)
		local
			i, d, n: INTEGER
		do
			from
				i := attributes.lower
				d := index
			until
				Result /= Void
			loop
				n := attributes.key(i).count
				if d < n then
					Result := attributes.item(i).item(d)
				else
					d := d - n
					i := i + 1
				end
			end
		end

	attribute_at (a_attribute_namesoace, a_attribute_name: STRING): STRING is
			-- The value of the attribute given by its name; Void if not set
		local
			att: HASHED_DICTIONARY[STRING, STRING]
		do
			att := attributes.reference_at(a_attribute_namesoace)
			if att /= Void then
				Result := att.reference_at(a_attribute_name)
			end
		end

	attributes_count: INTEGER is
			-- The number of attributes
		local
			i: INTEGER
		do
			from
				i := attributes.lower
			until
				i > attributes.upper
			loop
				Result := Result + attributes.item(i).count
				i := i + 1
			end
		end

	child (index: INTEGER): XMLNS_NODE is
			-- The i'th child
		require
			index.in_range(1, children_count)
		do
			Result := children.item(index - 1)
		end

	children_count: INTEGER is
			-- The number of children
		do
			Result := children.count
		end

	parent: XMLNS_NODE
			-- The parent of the node, Void if it is the root

	data: STRING
			-- Data in the node, if there is some.

feature {XMLNS_TREE}
	set_attribute (a_attribute_namespace, a_attribute_name, a_attribute_value: STRING) is
		require
			a_attribute_namespace /= Void
			a_attribute_name /= Void
			a_attribute_value /= Void
		local
			att: HASHED_DICTIONARY[STRING, STRING]
		do
			att := attributes.reference_at(a_attribute_namespace)
			if att = Void then
				create att.make
				attributes.add(att, a_attribute_namespace)
			end
			att.add(a_attribute_value, a_attribute_name)
		ensure
			attribute_at(a_attribute_namespace, a_attribute_name) = a_attribute_value
		end

	add_child (node: XMLNS_NODE) is
		require
			node /= Void
			node.parent = Void
		do
			children.add_last(node)
			node.set_parent(Current)
		ensure
			node.parent = Current
			child(children_count) = node
		end

	set_data (a_data: like data) is
		require
			a_data /= Void
			children_count = 0
		do
			data := a_data
		end

feature {XMLNS_NODE}
	set_parent (a_parent: like parent) is
		require
			a_parent /= Void
		do
			parent := a_parent
		ensure
			parent = a_parent
		end

feature {}
	attributes: HASHED_DICTIONARY[HASHED_DICTIONARY[STRING, STRING], STRING]

	children: FAST_ARRAY[XMLNS_NODE]

	make (a_namespace: like namespace; a_name: like name; a_line: like line; a_column: like column) is
		require
			a_namespace /= Void
			a_name /= Void
			a_line > 0
			a_column > 0
		do
			namespace := a_namespace
			name := a_name
			line := a_line
			column := a_column
			create attributes.make
			create children.make(0)
		end

invariant
	name /= Void
	namespace /= Void

end -- class XMLNS_NODE
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
