#!/bin/sh
#
# Simple script to build SmartEiffel without using the ugly interactive installer
#
# Usage:
# make_release.sh [build|install] {plain}
#

cd $(dirname $0)
export LOG=$(pwd)/make_release.log

C2C_OPTIONS='-boost -no_gc -verbose'
EXE_SUFFIX=''
OS=UNIX
if test -x /usr/bin/cygpath.exe; then
    # Cygwin specific
    EXE_SUFFIX=".exe"
    OS=Cygwin
fi

# Tools using the AST. Those tools are split. Does not include compile_to_c.
LARGE_TOOLS="short class_check pretty eiffeldoc finder extract_internals" # compile_to_jvm

# Small helper tools. Those tools are not split. Does not include se, compile, clean.
HELPER_TOOLS="eiffeltest ace_check" # print_jvm_class

if [ "$2" = plain ]; then
    plain=TRUE
else
    plain=FALSE
    eval `tset -s`
    tput init
fi

bold() {
    test $plain = TRUE && tput bold
}

normal() {
    test $plain = TRUE && tput sgr0
}

italic() {
    test $plain = TRUE && tput sitm
}

underline() {
    test $plain = TRUE && tput smul
}

foreground() {
    if test $plain = FALSE; then
	case $1 in
	    black)
		tput setaf 0
		;;
	    red)
		tput setaf 1
		;;
	    green)
		tput setaf 2
		;;
	    yellow)
		tput setaf 3
		;;
	    blue)
		tput setaf 4
		;;
	    magenta)
		tput setaf 5
		;;
	    cyan)
		tput setaf 6
		;;
	    white)
		tput setaf 7
		;;
	esac
    fi
}

background() {
    if test $plain = FALSE; then
	case $1 in
	    black)
		tput setab 0
		;;
	    red)
		tput setab 1
		;;
	    green)
		tput setab 2
		;;
	    yellow)
		tput setab 3
		;;
	    blue)
		tput setab 4
		;;
	    magenta)
		tput setab 5
		;;
	    cyan)
		tput setab 6
		;;
	    white)
		tput setab 7
		;;
	esac
    fi
}

title() {
    if test $plain = TRUE; then
	echo "$1"
    else
	tput el
	foreground blue
	bold
	echo "$1"
	foreground black
	normal
    fi
}

show_status() {
    if test $plain = TRUE; then
	if test $1 = 0; then
	    echo "    OK"
	else
	    echo "    KO"
	fi
    else
	tput el
	if test $1 = 0; then
	    tput setaf 2
	    echo "    OK"
	else
	    tput bold
	    tput setaf 1
	    echo "    KO"
	fi
	tput setaf 0
	tput sgr0
	echo
    fi
}

progress() {
    label="$4"
    if test $plain = TRUE; then
	echo "$label"
    else
	col=$(($(tput cols) - $1 - 11))
	tput setaf 0
	tput sgr0
	size=$1
	current=$2
	max=$3
	awk 'BEGIN {
                fill=int('$size' * '$current' / '$max' + .5);
                printf(" '$(tput bold)'%3.1f%%'$(tput sgr0)'\t'$(tput setab 6)'", 100*'$current'/'$max');
                for (i=0;    i<fill; i++) printf(" ");
                printf("'$(tput setab 4)'");
                for (i=fill; i<'$size'; i++) printf(" ");
                printf("'$(tput sgr0)' '$(tput setaf 5)'%-'$col'.'$col's'$(tput sgr0)' \r", "'"$label"'");
            }' </dev/zero >/dev/tty
    fi
}

error() {
    if test $plain = FALSE; then
	tput el
	tput setaf 1
	tput bold
    fi
    echo "$2 failed with status $1"
    echo "Please look at $LOG"
    if test $plain = FALSE; then
	tput setaf 0
	tput sgr0
    fi
}

run() {
    echo "$@" >> $LOG
    if eval "$@" >>$LOG 2>&1; then
	status=0
    else
	s=$?
	error $s "$1"
	status=1
    fi
    return $status
}

# ---------------------------------------------------------------------------------------------------------- #

generate_default_config() {
    if [ ! -d $SmartEiffel ]; then
	if [ -e $SmartEiffel ]; then
            mkdir ${SmartEiffel}.new
            mv ${SmartEiffel} ${SmartEiffel}.new/vanilla.se
            mv ${SmartEiffel}.new $SmartEiffel
	else
            mkdir -p $SmartEiffel
	fi
    fi

    SE_BIN=${SE_BIN:-$LIB}
    SE_SYS=${SE_SYS:-$LIB}
    SE_SHORT=${SE_SHORT:-$LIB}
    SE_TUTORIAL=${SE_TUTORIAL:-$DOC}
    SE_TOOLS=${SE_TOOLS:-$LIB}
    SE_LIB=${SE_LIB:-$LIB}

    test -r $SmartEiffel/vanilla.se || cat > ${SmartEiffel}/vanilla.se <<EOF
[General]
bin: ${SE_BIN}/bin/
sys: ${SE_SYS}/sys/
short: ${SE_SHORT}/short/
os: $OS
flavor: generic
tag: 3
jobs: 4

[Environment]
path_lib: ${SE_LIB}/lib/

[Loadpath]
lib: \${path_lib}loadpath.se

[Tools]
c: compile
c2c: compile_to_c
clean: clean
pretty: pretty
short: short
find: finder
ace_check: ace_check
class_check: class_check
doc: eiffeldoc
test: eiffeltest
x_int: extract_internals

[boost]
c_compiler_type: gcc
c_compiler_options: -pipe -Os
cpp_compiler_type: g++
cpp_compiler_options: -pipe -Os

[no_check]
c_compiler_type: gcc
c_compiler_options: -pipe -O1
cpp_compiler_type: g++
cpp_compiler_options: -pipe -O1

[require_check]
c_compiler_type: gcc
c_compiler_options: -pipe
cpp_compiler_type: g++
cpp_compiler_options: -pipe

[ensure_check]
c_compiler_type: gcc
c_compiler_options: -pipe
cpp_compiler_type: g++
cpp_compiler_options: -pipe

[invariant_check]
c_compiler_type: gcc
c_compiler_options: -pipe
cpp_compiler_type: g++
cpp_compiler_options: -pipe

[loop_check]
c_compiler_type: gcc
c_compiler_options: -pipe
cpp_compiler_type: g++
cpp_compiler_options: -pipe

[all_check]
c_compiler_type: gcc
c_compiler_options: -pipe
cpp_compiler_type: g++
cpp_compiler_options: -pipe

[debug_check]
c_compiler_type: gcc
c_compiler_options: -pipe -g
cpp_compiler_type: g++
cpp_compiler_options: -pipe -g
smarteiffel_options: -no_strip

[release]
c_compiler_type: gcc
c_compiler_options: -pipe -O3 -fomit-frame-pointer
cpp_compiler_type: g++
cpp_compiler_options: -pipe -O3 -fomit-frame-pointer
smarteiffel_options: -no_split
EOF

    test -r $SmartEiffel/tools.se || cat > ${SmartEiffel}/tools.se <<EOF
[Environment]
path_tools: ${SE_TOOLS}/tools/

[Loadpath]
tools: \${path_tools}loadpath.se
EOF

    test -r $SmartEiffel/doc.se || cat > ${SmartEiffel}/doc.se <<EOF
[Environment]
path_tutorial: ${SE_TUTORIAL}/tutorial/

[Loadpath]
tutorial: \${path_tutorial}loadpath.se
EOF

#     test -r $SmartEiffel/java.se || cat > ${SmartEiffel}/java.se <<EOF
# [Tools]
# java: compile_to_jvm
# javap: print_jvm_class
# 
# [Java]
# jar: jar
# jvm: java
# java_compiler: javac
# EOF

}

# ---------------------------------------------------------------------------------------------------------- #

do_build() {
    echo

    se=$SmartEiffel
    export SmartEiffel=${SmartEiffel:-$HOME/.serc}
    if [ "$SmartEiffel" != "$se" -a "$SmartEiffel" != /etc/serc -a "$SmartEiffel" != $HOME/.serc ]; then
	echo
	echo '*** The SmartEiffel location is not standard.'
	echo '*** To use SmartEiffel, please set the SmartEiffel variable to'
	echo '*** "'"$SmartEiffel"'".'
	echo
    fi
    LIB=$(pwd)
    DOC=$LIB
    generate_default_config

    if [ -x bin/compile_to_c$EXE_SUFFIX ]; then
	foreground magenta
	echo "Using the existing compile_to_c as bootstrap compiler"
	foreground black
	echo
    else
	title "Bootstrapping compile_to_c from the subversion germ"
	run gcc -pipe -O0 -lm -o bin/compile_to_c$EXE_SUFFIX work/germ/compile_to_c.c
	show_status $?
    fi

    cd bin/

    title "Building the compiler"
    progress 30 0 10 compile_to_c
    run ./compile_to_c$EXE_SUFFIX "$@" $C2C_OPTIONS compile_to_c -o compile_to_c$EXE_SUFFIX
    s=$?
    n=$(cat ./compile_to_c.make | wc -l)
    i=8
    n=$(($n*10+8))
    cat ./compile_to_c.make | grep -v '^#' | grep -vE '^[[:space:]]*$' | while read line; do
    	progress 30 $i $n "$line"
    	run $line
	s=$(($s + $?))
    	i=$(($i+8))
    done

    progress 30 8 10 compile
    run ./compile_to_c$EXE_SUFFIX "$@" $C2C_OPTIONS -no_split compile -o compile$EXE_SUFFIX
    s=$(($s + $?))
    n=$(cat ./compile.make | wc -l)
    i=$(($n*8+1))
    n=$(($n*10+1))
    cat ./compile.make | grep -v '^#' | grep -vE '^[[:space:]]*$' | while read line; do
    	progress 30 $i $n "$line"
    	eval "$line"
    	i=$(($i+1))
    done

    progress 30 9 10 clean
    run ./compile_to_c$EXE_SUFFIX "$@" $C2C_OPTIONS -no_split clean -o clean$EXE_SUFFIX
    s=$(($s + $?))
    n=$(cat ./clean.make | wc -l)
    i=$(($n*9+1))
    n=$(($n*10+1))
    cat ./clean.make | grep -v '^#' | grep -vE '^[[:space:]]*$' | while read line; do
    	progress 30 $i $n "$line"
    	run $line
	s=$(($s + $?))
    	i=$(($i+1))
    done

    progress 30 3 3 'Cleaning...'
    run ./clean$EXE_SUFFIX compile_to_c
    s=$(($s + $?))
    run ./clean$EXE_SUFFIX compile
    s=$(($s + $?))
    run ./clean$EXE_SUFFIX clean
    s=$(($s + $?))

    show_status $s

    title "Building large tools (using the SmartEiffel parser and AST)"
    n=$(echo $LARGE_TOOLS | awk '{printf("%d\n",NF)}')
    i=0
    s=0
    for exe in $LARGE_TOOLS; do
    	progress 30 $i $n $exe
    	run ./compile$EXE_SUFFIX "$@" $C2C_OPTIONS -clean $exe -o $exe$EXE_SUFFIX
	s=$(($s + $?))
    	i=$(($i+1))
    done

    show_status $s

    title "Building helper tools"
    n=$(($(echo $HELPER_TOOLS | awk '{printf("%d\n",NF)}') + 1))
    i=0
    s=0
    for exe in se $HELPER_TOOLS; do
    	progress 30 $i $n $exe
    	run ./compile$EXE_SUFFIX "$@" $C2C_OPTIONS -no_split -clean $exe -o $exe$EXE_SUFFIX
	s=$(($s + $?))
    	i=$(($i+1))
    done

    show_status $s

    title "Generating API docs"
    cd ../work/html/site
    if test $plain = TRUE; then
	run ./new_libs
	s=$?
    else
	tput sgr0
	tput setaf 5
	col=$(($(tput cols) - 8)) # remove so many characters because SmartEiffel outputs tabs which count as one char only for awk
	if ./new_libs 2>>$LOG | tee -a $LOG | awk '{printf("'"$(tput el)"'%-'${col}.${col}'s\r", $0)}'; then
	    s=0
	else
	    s=$?
	    error $s ./new_libs
	fi
    fi

    show_status $s
}

do_install() {
    PREFIX=${USRDIR:-/usr/local}
    BIN=${PREFIX}/bin
    LIB=${PREFIX}/lib/smarteiffel
    DOC=${DOCDIR:-${PREFIX}/doc}/smarteiffel
    CONF=${ETCDIR:-${USRDIR:-}/etc}
    export SmartEiffel=${CONF}/serc
    if [ -n "$SE_PREFIX" ]; then
	# useful for boxed installs
	SE_BIN=${SE_BIN:-${LIB#$SE_PREFIX}}
	SE_SYS=${SE_SYS:-${LIB#$SE_PREFIX}}
	SE_SHORT=${SE_SHORT:-${LIB#$SE_PREFIX}}
	SE_TUTORIAL=${SE_TUTORIAL:-${DOC#$SE_PREFIX}}
	SE_TOOLS=${SE_TOOLS:-${LIB#$SE_PREFIX}}
	SE_LIB=${SE_LIB:-${LIB#$SE_PREFIX}}
    fi
    if [ $SmartEiffel != /etc/serc -a $SmartEiffel != $HOME/.serc ]; then
	foreground red
	bold
	echo
	echo '*** The SmartEiffel location is not standard.'
	echo '*** To use SmartEiffel, please set the SmartEiffel variable to'
	echo '*** "'"$SmartEiffel"'".'
	echo
	normal
	foreground black
    fi
    generate_default_config

    title "Installing programs:"
    echo " - se (in ${BIN})"
    install -D -m 755 bin/se$EXE_SUFFIX ${BIN}/se$EXE_SUFFIX
    for exe in clean compile compile_to_c $LARGE_TOOLS $HELPER_TOOLS; do
	echo " - $exe (in ${LIB}/bin)"
	install -D -m 755 bin/${exe}$EXE_SUFFIX ${LIB}/bin/${exe}$EXE_SUFFIX
    done

    title "Installing libraries:"
    for dir in lib tools sys; do
	echo " - $dir (in ${LIB}/$dir)"
	n=$(find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | wc -l)
	i=0
	find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | while read f; do
	    progress 30 $i $n $(basename $f)
	    install -D -m 644 $f ${LIB}/$f
	    i=$(($i+1))
	done
    done

    ok

    title "Installing data files:"
    for dir in short misc; do
	echo " - $dir (in ${LIB}/$dir)"
	n=$(find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | wc -l)
	i=0
	find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | while read f; do
	    progress 30 $i $n $(basename $f)
	    install -D -m 644 $f ${LIB}/$f
	    i=$(($i+1))
	done
    done

    ok

    title "Installing doc:"
    for dir in tutorial; do
	echo " - $dir (in ${DOC}/$dir)"
	n=$(find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | wc -l)
	i=0
	find ${dir}/ -name .svn -prune -o '(' -type f -print ')' | while read f; do
	    progress 30 $i $n $(basename $f)
	    install -D -m 644 $f ${DOC}/$f
	    i=$(($i+1))
	done
    done
    cd work/html/site/export/
    for api in libraries tools; do
	echo " - $api API (in ${DOC}/api/$api)"
	#find $api/ '(' -name \*.html -o -name \*.css -o -name \*.js ')' -exec install -D -m 644 '{}' ${DOC}/api/'{}' \;
	n=$(find $api/ '(' -name \*.html -o -name \*.css -o -name \*.js ')' -print | wc -l)
	i=0
	find $api/ '(' -name \*.html -o -name \*.css -o -name \*.js ')' -print | while read f; do
	    progress 30 $i $n $(basename $f)
	    install -D -m 644 $f ${DOC}/api/$f
	    i=$(($i+1))
	done
    done

    progress 30 1 1 index.html
    install -D -m 644 api.html ${DOC}/api/index.html

    ok
}

### MAIN ###

case "$1" in
    build|install)
	rm -f $LOG
	touch $LOG
	do_$1
	title "Done."
	exit 0
	;;
    *)
	foreground red
	echo "Usage: $0 [build|install]" >&2
	echo "For install, you may use the USRDIR, DOCDIR, and ETCDIR environment variables" >&2
	echo "You may also use SE_PREFIX which is removed from the above variables" >&2
	echo "when creating the configuration files." >&2
	foreground black
	normal
	exit 1
	;;
esac
