class LOADPATH

inherit
	CLUSTERS

create {CLASSES_TREE_FACTORY}
	make

feature {ANY}
	to_string: STRING is
		do
			Result := once ""
			Result.copy(path)
		end

feature {LOADPATH, CLASSES_TREE_FACTORY}
	name: STRING
	path: STRING

feature {} -- Open loadpath files
	connect (a_system_path: STRING; loadpath: LOADPATH): TEXT_FILE_READ is
		local
			sp: like a_system_path
		do
			if connected.has(path) then
				error_cycle
			end
			if a_system_path /= Void then
				sp := a_system_path
			else
				sp := classes_path_to_system_path(path)
			end
			if files.is_empty then
				create Result.make
			else
				Result := files.last
				check
					not Result.is_connected
				end
				files.remove_last
			end
			echo.tfr_connect(Result, sp)
			if not Result.is_connected then
				error_handler.append("Unknown loadpath")
				if loadpath /= Void then
					error_handler.append(" in ")
					error_handler.append(loadpath.path)
				else
					error_handler.append(" (not in a loadpath)")
				end
				error_handler.append(": %"")
				error_handler.append(path)
				error_handler.append("%" (resolved as %"")
				error_handler.append(sp)
				error_handler.append("%").")
				error_handler.print_as_fatal_error
			end
			connected.put(Result, path)
		ensure
			not files.fast_has(Result)
			Result.is_connected
			Result.path.same_as(a_system_path)
			connected.at(path) = Result
		end

	disconnect (in: TEXT_FILE_READ) is
		require
			in.is_connected
			not files.fast_has(in)
			connected.at(in.path) = in
		do
			connected.remove(in.path)
			in.disconnect
			files.add_last(in)
		ensure
			not in.is_connected
			files.fast_has(in)
			connected.fast_occurrences(in) = 0
		end

	files: FAST_ARRAY[TEXT_FILE_READ] is
		once
			create Result.make(0)
		end

	connected: DICTIONARY[TEXT_FILE_READ, STRING] is
		once
			create {HASHED_DICTIONARY[TEXT_FILE_READ, STRING]} Result.make
		end

	error_cycle is
		do
			error_handler.append(once "Cycle detected:%N")
			show_cycle(Current)
			error_handler.print_as_fatal_error
		end

	show_cycle (tree: CLASSES) is
		local
			p: CLUSTERS; l: LOADPATH
		do
			p := tree.parent
			if p = Current then
				error_handler.append("%T%"")
				error_handler.append(path)
				error_handler.append("%"%N")
			else
				show_cycle(p)
			end
			if l ?:= tree then
				l ::= tree
				error_handler.append("%T%"")
				error_handler.append(l.path)
				error_handler.append("%"%N")
			end
		end

	show_name is
		do
			echo.put_character('"')
			echo.put_string(path)
			echo.put_character('"')
		end

feature {}
	make (a_distance: like distance; a_name: like name; a_path, a_system_path: like path; loadpath: LOADPATH) is
		require
			a_distance > 0
			not a_name.is_empty
			not a_path.is_empty
			is_classes_path(a_path)
			classes_notation.is_absolute_path(a_path)
			not (create {FILE_TOOLS}).is_directory(a_path)
			string_aliaser.registered_one(a_path)
		local
			directory_path, loadpath_file: STRING
		do
			distance := a_distance
			name := a_name
			directory_path := strings.new_twin(a_path)
			classes_notation.to_parent_directory(directory_path)
			if a_system_path /= Void then
				loadpath_file := strings.new_twin(a_system_path)
			else
				loadpath_file := strings.new_twin(classes_path_to_system_path(a_path))
			end
			path := a_path
			create classeses.with_capacity(16)
			create_subtree(loadpath_file, directory_path, loadpath)
			strings.recycle(directory_path)
			strings.recycle(loadpath_file)
		end

	create_subtree (loadpath_file, a_path: STRING; loadpath: LOADPATH) is
		require
			system_notation.is_valid_path(loadpath_file)
			is_classes_path(a_path)
		local
			tfr: TEXT_FILE_READ; entry, a_system_path, system_entry: STRING; i: INTEGER
			factory: CLASSES_TREE_FACTORY; classes: CLASSES
			sp, cp: STRING
		do
			tfr := connect(loadpath_file, loadpath)
			from
				i := 1
			until
				tfr.end_of_input
			loop
				tfr.read_line
				if not tfr.last_string.is_empty then
					entry := tfr.last_string
					system_tools.environment_variable_substitution(path, entry)
					if system_notation.is_absolute_path(entry) then
						sp := string_aliaser.string(entry)
						cp := string_aliaser.string(system_path_to_classes_path(entry))
					elseif classes_notation.is_absolute_path(entry) then
						sp := Void
						cp := string_aliaser.string(entry)
					else
						system_entry := classes_notation.to_notation(entry, system_notation)
						if a_system_path = Void then
							a_system_path := strings.new_twin(loadpath_file)
							system_notation.to_parent_directory(a_system_path)
						end
						sp := new_path(system_notation, a_system_path, system_entry)
						cp := new_path(classes_notation, a_path, entry)
					end
					check
						cp /= Void
					end
					classes := factory.classes(1, sp, cp, new_name(entry, i), Current)
					if classes /= Void and then classes.parent = Void then
						add_classes(classes)
					end
				end
				i := i + 1
			end
			tfr.disconnect
			if classeses.is_empty then
				error_handler.append(once "Empty loadpath: %"")
				error_handler.append(path)
				error_handler.append("%".")
				error_handler.print_as_warning
			end
			if a_system_path /= Void then
				strings.recycle(a_system_path)
			end
		end

	new_path (notation: DIRECTORY_NOTATION; a_path, a_entry: STRING): STRING is
		require
			notation.is_valid_path(a_path)
			--|*** TO BE CHECKED: notation.is_valid_file_name(a_entry)
			not notation.is_absolute_path(a_entry)
		local
			buffer: STRING
		do
			buffer := once ""
			buffer.copy(a_path)
			notation.to_subpath_with(buffer, a_entry)
			Result := string_aliaser.string(buffer)
		ensure
			Result /= Void
			string_aliaser.registered_one(Result)
		end

	new_name (a_name: STRING; index: INTEGER): STRING is
		local
			buffer: STRING
		do
			buffer := once ""
			buffer.copy(a_name)
			if buffer.is_empty then
				buffer.add_last('#')
				index.append_in(buffer)
			elseif buffer.has_prefix(once "./") then
				buffer.remove_head(2)
			end
			Result := once ""
			Result.copy(name)
			Result.add_last(':')
			Result.append(buffer)
			Result := string_aliaser.string(Result)
		ensure
			Result /= Void
			string_aliaser.registered_one(Result)
		end

invariant
	distance > 0

end
