-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class ADDRESS_OF
	--
	-- For the special address form notation : $name
	--

inherit
	EXPRESSION
	
creation {ANY}
	with_local, with_feature_name

feature {ANY}
	local_name: EXPRESSION
			-- When it is the address of some local variable.

	feature_name: FEATURE_NAME
			-- When it is the address of some feature.

	is_writable, is_current, is_implicit_current: BOOLEAN is False

	is_static: BOOLEAN is False

	is_result: BOOLEAN is False

	is_void: BOOLEAN is False

	is_manifest_string: BOOLEAN is False

	use_current (type: TYPE): BOOLEAN is
		do
			Result := True
		end

	extra_bracket_flag: BOOLEAN is False

	declaration_type: TYPE is
		once
			Result := smart_eiffel.type_pointer
		end

	resolve_in (type: TYPE): TYPE is
		do
			Result := declaration_type
		end

	side_effect_free (type: TYPE): BOOLEAN is
		do
			Result := True
		end

	adapt_for (type: TYPE): like Current is
		local
			ln: like local_name; rf: like run_feature
		do
			if local_name /= Void then
				ln := local_name.adapt_for(type)
				Result := current_or_twin_init(ln, feature_stamp, rf)
			else
				rf := feature_stamp.run_feature_for(type)
				if run_feature = Void then
					run_feature := rf
				end
				Result := current_or_twin_init(ln, feature_stamp, rf)
				address_of_pool.register_for(Result)
			end
		end

	safety_check (type: TYPE) is
		do
		end

	compile_target_to_jvm, compile_to_jvm (type: TYPE) is
		do
			error_handler.add_position(start_position)
			error_handler.append(fz_jvm_error)
			error_handler.print_as_fatal_error
		end

	jvm_branch_if_false, jvm_branch_if_true (type: TYPE): INTEGER is
		do
			check
				False
			end
		end

	compile_to_c (type: TYPE) is
		do
			if local_name /= Void then
				cpp.pending_c_function_body.append(once "/*$*/&")
				local_name.compile_to_c(type)
			else
				cpp.pending_c_function_body.append(once "/*$*/((void*)")
				run_feature.address_of_c_mapping
				cpp.pending_c_function_body.extend(')')
			end
		end

	mapping_c_target (type, target_formal_type: TYPE) is
		do
			compile_to_c(type)
		end

	mapping_c_arg (type: TYPE) is
		do
			compile_to_c(type)
		end

	feature_stamp: FEATURE_STAMP

	specialize_in (type: TYPE): like Current is
		local
			ln: like local_name; fs: like feature_stamp
		do
			if local_name /= Void then
				ln := local_name.specialize_in(type)
			else
				fs := type.search(feature_name)
				if fs = Void then
					error_handler.add_position(feature_name.start_position)
					error_handler.append(fz_vuar4)
					error_handler.append(" Feature `")
					error_handler.append(feature_name.to_string)
					error_handler.append("' not found.")
					error_handler.print_as_fatal_error
				end
				if feature_stamp = Void then
					feature_stamp := fs
				end
			end
			Result := current_or_twin_init(ln, fs, run_feature)
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			ln: like local_name; fs: like feature_stamp
		do
			if local_name /= Void then
				ln := local_name.specialize_thru(parent_type, parent_edge, new_type)
			else
				fs := feature_stamp.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(ln, fs, run_feature)
		ensure then
			Result /= Current implies Result.feature_stamp /= feature_stamp
		end

	specialize_2 (type: TYPE): like Current is
		local
			af: ANONYMOUS_FEATURE; cst: CST_ATT
		do
			if local_name /= Void then
				Result := Current
			else
				af := feature_stamp.anonymous_feature(type)
				if cst ?:= af then
					error_handler.add_position(feature_name.start_position)
					error_handler.append(fz_vuar4)
					error_handler.print_as_fatal_error
				end
				Result := Current
			end
		end

	has_been_specialized: BOOLEAN is
		do
			if feature_name /= Void then
				Result := feature_stamp /= Void
			else
				Result := local_name.has_been_specialized
			end
		end

	non_void_no_dispatch_type (type: TYPE): TYPE is
		do
			Result := resolve_in(type)
		end

	simplify (type: TYPE): ADDRESS_OF is
		do
			Result := Current
		end

	precedence: INTEGER is
		do
			Result := atomic_precedence
		end

	start_position: POSITION is
		do
			if local_name /= Void then
				Result := local_name.start_position
			else
				Result := feature_name.start_position
			end
		end

	pretty (indent_level: INTEGER) is
		do
			pretty_printer.put_character('%D')
			if local_name /= Void then
				local_name.pretty(indent_level)
			else
				feature_name.pretty(indent_level)
			end
		end

	pretty_target (indent_level: INTEGER) is
		do
			check
				False
			end
		end

	bracketed_pretty (indent_level: INTEGER) is
		do
			pretty_printer.put_character('(')
			pretty(indent_level)
			pretty_printer.put_character(')')
		end

	short (type: TYPE) is
		do
			short_printer.put_character('%D')
			if local_name /= Void then
				local_name.short(type)
			else
				feature_name.short(type)
			end
		end

	short_target (type: TYPE) is
		do
			bracketed_short(type)
			short_printer.put_dot
		end

	jvm_assign_creation, jvm_assign (type: TYPE) is
		do
			check
				False
			end
		end

	collect (type: TYPE): TYPE is
		do
			if feature_name /= Void then
				Result := smart_eiffel.collect(type, feature_stamp, False)
			end
			Result := declaration_type
			smart_eiffel.collect_create(Result)
		end

	accept (visitor: ADDRESS_OF_VISITOR) is
		do
			visitor.visit_address_of(Current)
		end

feature {ADDRESS_OF}
	set_local_name (ln: like local_name) is
		do
			local_name := ln
		end

	set_feature_stamp (fs: like feature_stamp) is
		do
			feature_stamp := fs
		end

	set_run_feature (rf: like run_feature) is
		do
			run_feature := rf
		end

feature {ADDRESS_OF_POOL}
	run_feature: RUN_FEATURE
			-- Corresponding one when runnable.

feature {}
	current_or_twin_init (ln: like local_name; fs: like feature_stamp; rf: like run_feature): like Current is
		do
			if ln = local_name and then fs = feature_stamp and then rf = run_feature then
				Result := Current
			else
				Result := twin
				Result.set_local_name(ln)
				Result.set_feature_stamp(feature_stamp)
				Result.set_run_feature(rf)
			end
		end
	
	with_local (ln: like local_name) is
		require
			ln /= Void
		do
			local_name := ln
		ensure
			local_name = ln
		end

	with_feature_name (fn: like feature_name) is
		require
			fn /= Void
		do
			feature_name := fn
		ensure
			feature_name = fn
		end

invariant
	local_name /= Void xor feature_name /= Void

end -- class ADDRESS_OF
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
