-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class E_CHECK
	--
	-- Instruction "check ... end;".
	--

inherit
	INSTRUCTION

creation {ANY}
	make

feature {ANY}
	end_mark_comment: BOOLEAN is True

	side_effect_free (type: TYPE): BOOLEAN is
		do
			Result := ace.boost
		end

	start_position: POSITION is
			-- Of keyword "check".
		do
			if check_invariant /= Void then
				Result := check_invariant.start_position
			end
		end

	specialize_in (type: TYPE): like Current is
		local
			ci: like check_invariant
		do
			if check_invariant /= Void then
				ci := check_invariant.specialize_in(type)
			end
			Result := current_or_twin_init(ci)
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		local
			ci: like check_invariant
		do
			if check_invariant /= Void then
				ci := check_invariant.specialize_thru(parent_type, parent_edge, new_type)
			end
			Result := current_or_twin_init(ci)
		end

	specialize_2 (type: TYPE): like Current is
		local
			ci: like check_invariant
		do
			if check_invariant /= Void then
				ci := check_invariant.specialize_2(type)
			end
			Result := current_or_twin_init(ci)
		end

	has_been_specialized: BOOLEAN is
		do
			if check_invariant = Void then
				Result := True
			else
				Result := check_invariant.has_been_specialized
			end
		end

	safety_check (type: TYPE) is
		do
			check_invariant.safety_check(type)
		end

	simplify (type: TYPE): INSTRUCTION is
		do
			check
				simplify_is_only_made_for_boost: not must_be_generated(type)
			end
			smart_eiffel.magic_count_increment
		end

	compile_to_c (type: TYPE) is
		do
			if must_be_generated(type) then
				check_invariant.compile_to_c(type)
			end
		end

	compile_to_jvm (type: TYPE) is
		do
			if must_be_generated(type) then
				check_invariant.compile_to_jvm(type, True)
				code_attribute.opcode_pop
			end
		end

	use_current (type: TYPE): BOOLEAN is
		do
			if not ace.boost then
				Result := check_invariant.use_current(type)
			end
		end

	pretty (indent_level: INTEGER) is
		do
			if check_invariant /= Void then
				check_invariant.pretty(indent_level)
			end
		end

	accept (visitor: E_CHECK_VISITOR) is
		do
			visitor.visit_e_check(Current)
		end

	collect (t: TYPE) is
		do
			if check_invariant /= Void then
				check_invariant.collect(t)
			end
		end

	adapt_for (type: TYPE): like Current is
		do
			if must_be_generated(type) then
				Result := current_or_twin_init(check_invariant.adapt_for(type))
			else
				Result := Current
			end
		end

feature {E_CHECK}
	init (ci: like check_invariant) is
		do
			check_invariant := ci
		ensure
			check_invariant = ci
		end

feature {E_CHECK_VISITOR}
	check_invariant: CHECK_INVARIANT
			-- (May be Void when there is only a comment after a "check" keyword.)

feature {}
	make (sp: like start_position; hc: COMMENT; l: FAST_ARRAY[ASSERTION]) is
		require
			not sp.is_unknown
		do
			if hc /= Void or else l /= Void then
				create check_invariant.make(sp, hc, l)
			end
		end

	must_be_generated (type: TYPE): BOOLEAN is
		do
			if check_invariant /= Void then
				Result := type.class_text.all_check
			end
		end

	current_or_twin_init (ci: like check_invariant): like Current is
		do
			if check_invariant = ci then
				Result := Current
			else
				Result := twin
				Result.init(ci)
			end
		end

end -- class E_CHECK
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
