-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class WHEN_CLAUSE
	--
	-- To store a when clause of an inspect instruction.
	--

inherit
	VISITABLE
insert
	GLOBALS

creation {EIFFEL_PARSER, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	make

creation {INTROSPECTION_HANDLER}
	make_strippable

feature {ANY}
	start_position: POSITION
			-- Of the first character of keyword "when".

	header_comment: COMMENT
			-- Of the when clause.

	compound: INSTRUCTION
			-- Of the when clause if any.

	use_current (type: TYPE): BOOLEAN is
		do
			if compound /= Void then
				Result := compound.use_current(type)
			end
		end

	accept (visitor: WHEN_CLAUSE_VISITOR) is
		do
			visitor.visit_when_clause(Current)
		end

feature {INSPECT_STATEMENT}
	validity_check (original_inspect: INSPECT_STATEMENT; type: TYPE) is
		require
			original_inspect.first_one.item = original_inspect
		local
			i: INTEGER
		do
			from
				i := list.lower
			until
				i > list.upper
			loop
				list.item(i).validity_check(original_inspect)
				i := i + 1
			end
		end

	validity_check_continued (occurrence_1: WHEN_ITEM): BOOLEAN is
		require
			occurrence_1 /= Void
		local
			i: INTEGER
		do
			from
				i := list.lower
			until
				i > list.upper or else list.item(i).validity_check_continued(occurrence_1)
			loop
				i := i + 1
			end
		end

	collect (t: TYPE; condition_live_type: LIVE_TYPE) is
		require
			has_been_specialized
			t.feature_collection_done
			smart_eiffel.status.is_collecting
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER
		do
			if compound /= Void then
				if strip_condition /= Void then
					condition_live_type.forbid_collecting(strip_condition)
				end
				compound.collect(t)
				if strip_condition /= Void then
					condition_live_type.allow_collecting(strip_condition)
				end
			end
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).collect(t)
				i := i - 1
			end
		end

	adapt_for (t: TYPE): like Current is
		local
			i: INTEGER; c: like compound; l: like list; wi1, wi2: WHEN_ITEM
		do
			if compound /= Void then
				c := compound.adapt_for(t)
			end
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.adapt_for(t)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).adapt_for(t), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		end

feature {}
	make (inspect_statement: INSPECT_STATEMENT; sp: like start_position; hc: like header_comment) is
		require
			inspect_statement /= Void
			not sp.is_unknown
		do
			inspect_statement.add_when(Current)
			start_position := sp
			header_comment := hc
			create list.with_capacity(4)
		ensure
			start_position = sp
		end

	make_strippable (inspect_statement: INSPECT_STATEMENT; condition: like strip_condition) is
		require
			inspect_statement /= Void
			condition /= Void
			inspect_statement.condition_type /= Void
		do
			strip_condition := condition
			make(inspect_statement, inspect_statement.start_position, Void)
		ensure
			strip_condition = condition
		end

feature {INSPECT_STATEMENT}
	safety_check (type: TYPE) is
		do
			if compound /= Void then
				compound.safety_check(type)
			end
		end

feature {INSPECT_STATEMENT}
	share_values_of (first: like Current) is
		require
			first /= Void
		do
			values := first.values
		ensure
			values /= Void
		end

	compile_to_jvm (type: TYPE; else_position: POSITION; remainder: INTEGER) is
			-- Where `remainder' is the number of WHEN_CLAUSE after Current.
		local
			point3, point4, bi: INTEGER; must_test: BOOLEAN; ca: like code_attribute
		do
			ca := code_attribute
			if remainder > 0 then
				must_test := True
			elseif ace.no_check then
				must_test := True
			else
				-- boost :
				must_test := not else_position.is_unknown
			end
			points1.clear_count
			if must_test then
				from
					bi := values.lower
				until
					bi > values.upper
				loop
					if values.item(bi) = values.item(bi + 1) then
						ca.opcode_dup
						ca.opcode_push_integer(values.item(bi))
						points1.add_last(ca.opcode_if_icmpeq)
					else
						ca.opcode_dup
						ca.opcode_push_integer(values.item(bi))
						point3 := ca.opcode_if_icmplt
						ca.opcode_dup
						ca.opcode_push_integer(values.item(bi + 1))
						points1.add_last(ca.opcode_if_icmple)
						ca.resolve_u2_branch(point3)
					end
					bi := bi + 2
				end
				point4 := ca.opcode_goto
			end
			ca.resolve_with(points1)
			if compound /= Void then
				compound.compile_to_jvm(type)
			end
			point2 := ca.opcode_goto
			if must_test then
				ca.resolve_u2_branch(point4)
			end
		end

	compile_to_jvm_resolve_branch is
		do
			code_attribute.resolve_u2_branch(point2)
		end

	compile_to_c (type: TYPE) is
		local
			bi: INTEGER
		do
			cpp.pending_c_function_body.append(once "%Nif(")
			from
				bi := values.lower
			until
				bi > values.upper
			loop
				cpp.pending_c_function_body.extend('(')
				if values.item(bi) = values.item(bi + 1) then
					values.item(bi).append_in(cpp.pending_c_function_body)
					cpp.pending_c_function_body.append(once "==")
					cpp.put_inspect_tmp
				else
					cpp.pending_c_function_body.extend('(')
					values.item(bi).append_in(cpp.pending_c_function_body)
					cpp.pending_c_function_body.append(once "<=")
					cpp.put_inspect_tmp
					cpp.pending_c_function_body.append(once ")&&(")
					cpp.put_inspect_tmp
					cpp.pending_c_function_body.append(once "<=")
					values.item(bi + 1).append_in(cpp.pending_c_function_body)
					cpp.pending_c_function_body.extend(')')
				end
				cpp.pending_c_function_body.extend(')')
				--
				bi := bi + 2
				if bi < values.upper then
					cpp.pending_c_function_body.append(once "||")
				end
			end
			cpp.pending_c_function_body.append(once "){%N")
			if compound /= Void then
				compound.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "}%N")
		end

	compile_to_c_switch (type: TYPE) is
		local
			bi, v: INTEGER
		do
			from
				bi := values.lower
			until
				bi > values.upper
			loop
				from
					v := values.item(bi)
				until
					v > values.item(bi + 1)
				loop
					cpp.pending_c_function_body.append(once "case ")
					v.append_in(cpp.pending_c_function_body)
					cpp.pending_c_function_body.append(once ":%N")
					v := v + 1
				end
				bi := bi + 2
			end
			if compound /= Void then
				compound.compile_to_c_with_internal_c_local_saving(type)
			end
			cpp.pending_c_function_body.append(once "break;%N")
		end

	pretty (indent_level: INTEGER) is
		local
			i: INTEGER
		do
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.keyword(once "when")
			if header_comment /= Void then
				header_comment.pretty(indent_level + 1)
			end
			if list /= Void then
				from
					i := list.lower
				until
					i > list.upper
				loop
					list.item(i).pretty(indent_level + 1)
					i := i + 1
					if i <= list.upper then
						pretty_printer.put_character(',')
						if not pretty_printer.zen_mode then
							pretty_printer.put_character(' ')
						end
					end
				end
			end
			pretty_printer.keyword(once "then")
			pretty_printer.set_indent_level(indent_level + 1)
			if compound /= Void then
				compound.pretty(indent_level + 1)
			end
			pretty_printer.set_indent_level(0)
		end

	includes_integer (v: INTEGER): BOOLEAN is
		local
			i: INTEGER
		do
			if values /= Void and then v >= values.item(values.lower) and then v <= values.item(values.upper) then
				from
					i := values.lower
				until
					Result or else i > values.upper
				loop
					Result := values.item(i) <= v and then v <= values.item(i + 1)
					i := i + 2
				end
			end
		end

	includes_integer_between (l, u: INTEGER): BOOLEAN is
		require
			invalid_range: l < u
		local
			i: INTEGER
		do
			if values /= Void and then u >= values.item(values.lower) and then l <= values.item(values.upper) then
				from
					i := values.lower
				until
					Result or else i > values.upper
				loop
					Result := values.item(i) <= u and then l <= values.item(i + 1)
					i := i + 2
				end
			end
		end

	largest_range_of_values: INTEGER is
			-- The largest number of consecutive values - returns 0 if none
		local
			i, range: INTEGER
		do
			if values /= Void then
				from
					i := values.lower
				until
					i > values.upper
				loop
					range := values.item(i + 1) - values.item(i) + 1
					if range > Result then
						Result := range
					end
					i := i + 2
				end
			end
		end

feature {INSPECT_STATEMENT, WHEN_CLAUSE}
	specialize_in (new_type: TYPE): like Current is
		require
			new_type /= Void
		local
			i: INTEGER; c: like compound; l: like list; wi1, wi2: WHEN_ITEM
		do
			if compound /= Void then
				c := compound.specialize_in(new_type)
			end
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.specialize_in(new_type)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_in(new_type), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		ensure
			Result.has_been_specialized
		end

	specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		require
			parent_type /= Void
			parent_edge /= Void
			new_type /= Void
			new_type /= parent_type
			has_been_specialized
		local
			i: INTEGER; c: like compound; l: like list; wi1, wi2: WHEN_ITEM
		do
			if compound /= Void then
				c := compound.specialize_thru(parent_type, parent_edge, new_type)
			end
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.specialize_thru(parent_type, parent_edge, new_type)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2_character (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER; c: like compound; l: like list; wi1, wi2: WHEN_ITEM
		do
			if compound /= Void then
				c := compound.specialize_2(type)
			end
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.specialize_2_character(type)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_2_character(type), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2_integer (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			c: like compound
		do
			if compound /= Void then
				c := compound.specialize_2(type)
			end
			Result := specialize_2_when_items_integer(type, c)
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2_when_items_integer (type: TYPE; c: like compound): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER; l: like list; wi1, wi2: WHEN_ITEM
		do
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.specialize_2_integer(type)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_2_integer(type), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	specialize_2_string (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER; c: like compound; l: like list; wi1, wi2: WHEN_ITEM
		do
			if compound /= Void then
				c := compound.specialize_2(type)
			end
			from
				i := list.upper
			until
				i < list.lower or else wi1 /= wi2
			loop
				wi1 := list.item(i)
				wi2 := wi1.specialize_2_string(type)
				i := i - 1
			end
			if c = compound and then wi1 = wi2 then
				Result := Current
			else
				Result := twin
				if wi1 = wi2 then
					Result.set_compound(c)
				else
					from
						l := list.twin
						l.put(wi2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_2_string(type), i)
						i := i - 1
					end
					Result.init(c, l)
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			Result := compound /= Void implies compound.has_been_specialized
			if Result then
				from
					i := list.upper
				until
					i < list.lower or else not Result
				loop
					Result := list.item(i).has_been_specialized
					i := i - 1
				end
			end
		ensure
			Result
		end

feature {WHEN_ITEM}
	always_selected (container: INSPECT_STATEMENT) is
		do
			not_yet_implemented
			--|*** if compound /= Void then
			--|*** compound := compound.simplify_2
			--|*** end
			--|*** container.always_selected(compound)
		end

feature {INSPECT_STATEMENT}
	compute_values is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).compute_values(Current)
				i := i - 1
			end
		end

	force_internal_integer_values (type: TYPE) is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).force_internal_integer_values(type, Current)
				i := i - 1
			end
		end

	force_internal_character_values (type: TYPE) is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				i < list.lower
			loop
				list.item(i).force_internal_character_values(type, Current)
				i := i - 1
			end
		end

	match_value (v: INTEGER): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := list.upper
			until
				Result or else i < list.lower
			loop
				Result := list.item(i).match_value(v)
				i := i - 1
			end
		end

	strip_or_set_value (v: INTEGER; specialize_type, condition_type: TYPE): like Current is
		local
			l: like list; ic: INTEGER_CONSTANT
		do
			if condition_type = Void then
				Result := Current
			elseif is_live(condition_type) then
				create ic.make(v, start_position)
				if list.is_empty then
					add_value(ic)
					Result := Current
				else
					create l.with_capacity(1)
					Result := twin
					Result.init(compound, l)
					Result.add_value(ic)
				end
				Result := Result.specialize_2_when_items_integer(specialize_type, compound)
			end
		end

feature {}
	is_live (condition_type: TYPE): BOOLEAN is
		do
			Result := strip_condition = Void or else condition_type.live_type.collected(strip_condition)
		end

feature {WHEN_ITEM_1}
	compute_values_with_item_1 (wi1: WHEN_ITEM_1) is
		require
			wi1 /= Void
		local
			i, v: INTEGER
		do
			v := wi1.expression_value
			if values = Void then
				create values.make(501, 502)
				values.put(v, values.lower)
				values.put(v, values.upper)
			else
				i := locate_in_values(v)
				if i = values.lower then
					-- v is lower than lowest value
					if v = values.item(i) - 1 then
						-- just change lower
						values.put(v, i)
					else
						values.resize(values.lower - 2, values.upper)
						values.put(v, values.lower)
						values.put(v, values.lower + 1)
					end
				elseif i > values.upper then
					-- v is higher than highest value
					if v = values.item(i - 1) + 1 then
						-- just change upper
						values.put(v, i - 1)
					else
						values.resize(values.lower, values.upper + 2)
						values.put(v, values.upper - 1)
						values.put(v, values.upper)
					end
				else
					if v = values.item(i - 1) + 1 and then v = values.item(i) - 1 then
						values.put(values.item(i + 1), i - 1)
						from
						until
							i > values.upper - 2
						loop
							values.put(values.item(i + 2), i)
							values.put(values.item(i + 3), i + 1)
							i := i + 2
						end
						values.resize(values.lower, values.upper - 2)
					elseif v = values.item(i - 1) + 1 then
						-- just change upper
						values.put(v, i - 1)
					elseif v = values.item(i) - 1 then
						-- just change lower
						values.put(v, i)
					else
						values.resize(values.lower, values.upper + 2)
						from
							i := values.upper - 1
						until
							v > values.item(i - 1)
						loop
							values.put(values.item(i - 2), i)
							values.put(values.item(i - 1), i + 1)
							i := i - 2
						end
						values.put(v, i)
						values.put(v, i + 1)
					end
				end
			end
		end

feature {WHEN_ITEM_2}
	compute_values_with_item_2 (wi2: WHEN_ITEM_2) is
		require
			wi2 /= Void
		local
			l, u, i: INTEGER
		do
			l := wi2.lower_value
			u := wi2.upper_value
			if values = Void then
				create values.make(501, 502)
				values.put(l, values.lower)
				values.put(u, values.upper)
			else
				i := locate_in_values(l)
				if i = values.lower then
					-- l and u are lower than lowest value
					if u = values.item(i) - 1 then
						-- just change lower
						values.put(l, i)
					else
						values.resize(values.lower - 2, values.upper)
						values.put(l, values.lower)
						values.put(u, values.lower + 1)
					end
				elseif i > values.upper then
					-- l and u are higher than highest
					if l = values.item(i - 1) + 1 then
						-- just change upper
						values.put(u, i - 1)
					else
						values.resize(values.lower, values.upper + 2)
						values.put(l, values.upper - 1)
						values.put(u, values.upper)
					end
				else
					if l = values.item(i - 1) + 1 and then u = values.item(i) - 1 then
						values.put(values.item(i + 1), i - 1)
						from
						until
							i > values.upper - 2
						loop
							values.put(values.item(i + 2), i)
							values.put(values.item(i + 3), i + 1)
							i := i + 2
						end
						values.resize(values.lower, values.upper - 2)
					elseif l = values.item(i - 1) + 1 then
						-- just change upper
						values.put(u, i - 1)
					elseif u = values.item(i) - 1 then
						-- just change lower
						values.put(l, i)
					else
						values.resize(values.lower, values.upper + 2)
						from
							i := values.upper - 1
						until
							l > values.item(i - 1)
						loop
							values.put(values.item(i - 2), i)
							values.put(values.item(i - 1), i + 1)
							i := i - 2
						end
						values.put(l, i)
						values.put(u, i + 1)
					end
				end
			end
		end

feature {EIFFEL_PARSER, WHEN_CLAUSE, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	add_value (v: EXPRESSION) is
		require
			v /= Void
		local
			element: WHEN_ITEM
		do
			create {WHEN_ITEM_1} element.make(v)
			list.add_last(element)
		end

	add_slice (min, max: EXPRESSION) is
		require
			min /= Void
			max /= Void
		local
			element: WHEN_ITEM
		do
			no_manifest_string_slice_check(min)
			no_manifest_string_slice_check(max)
			create {WHEN_ITEM_2} element.make(min, max)
			list.add_last(element)
		end

feature {WHEN_CLAUSE, INTROSPECTION_HANDLER}
	init (c: like compound; l: like list) is
		require
			l /= Void
		do
			compound := c
			list := l
		ensure
			compound = c
			list = l
		end

feature {WHEN_CLAUSE, INSPECT_STATEMENT, EIFFEL_PARSER, INTROSPECTION_HANDLER, MANIFEST_STRING_INSPECTOR}
	set_compound (c: like compound) is
		do
			compound := c
		ensure
			compound = c
		end

feature {MANIFEST_STRING_INSPECTOR, WHEN_CLAUSE_VISITOR}
	list: FAST_ARRAY[WHEN_ITEM]

feature {WHEN_CLAUSE}
	values: ARRAY[INTEGER]
			-- To store pairs of range values.

feature {}
	points1: FAST_ARRAY[INTEGER] is
			-- To reach the `compound'.
		once
			create Result.with_capacity(12)
		end

	point2: INTEGER
			-- To go outside the INSPECT_STATEMENT.

	strip_condition: FEATURE_STAMP

	no_manifest_string_slice_check (expression: EXPRESSION) is
		do
			if expression.is_manifest_string then
				error_handler.add_position(expression.start_position)
				error_handler.append("Cannot use '..' with manifest strings.")
				error_handler.print_as_fatal_error
			end
		end

	locate_in_values (v: INTEGER): INTEGER is
			-- Returns index in values table where `v' would be inserted.
		require
			values /= Void
		do
			from
				Result := values.lower
			until
				Result > values.upper or else v < values.item(Result)
			loop
				Result := Result + 2
			end
		ensure
			(Result - values.lower) \\ 2 = 0
			Result < values.upper implies v < values.item(Result)
			Result > values.upper implies v > values.item(Result - 1)
		end

	em1: STRING is "Bad when clause."

	em2: STRING is "Empty when clause in inspect."

invariant
	list /= Void

end -- class WHEN_CLAUSE
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
