-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class SMART_EIFFEL
	--
	-- Singleton object to handle general purpose information about the
	-- SmartEiffel global compilation process. (This singleton is shared via
	-- the GLOBALS.`smart_eiffel' once function.)
	--

inherit
	SINGLETON

insert
	GLOBALS

feature {ANY}
	copyright: STRING is
		once
			Result := "SmartEiffel The GNU Eiffel Compiler, Eiffel tools and libraries%N"
			Result.append(release_number)
			Result.append("[
								Copyright (C), 1994-2002 - INRIA - LORIA - ESIAL UHP Nancy 1 - FRANCE
								Copyright (C), 2003-2005 - INRIA - LORIA - IUT Charlemagne Nancy 2 - FRANCE
								D.COLNET, P.RIBET, C.ADRIAN, V.CROIZIER F.MERIZEN - SmartEiffel@loria.fr
								http://SmartEiffel.loria.fr
								
								]")
		end

	release_number: STRING is "Release 2.3 (???? February ??th 2007) [Antoine-Auguste Parmentier]%N"

	status: STATUS is
		once
			create Result.make
		end

	is_ready: BOOLEAN is
			-- Is all the live code already gathered
		do
			Result := status.is_safety_checking or else status.is_generating
		end

	short_or_class_check_flag: BOOLEAN
			-- True when command `short' or command `class_check' is running.

	pretty_flag: BOOLEAN
			-- True when command `pretty' is running.

	no_id: BOOLEAN
			-- True when the ids file has not to be read.

	cluster_of (class_name: CLASS_NAME; report_error: BOOLEAN): CLUSTER is
			-- Retrieve the cluster of the class name. If more than one class exists with the same name, the
			-- closest to where the class name is written is returned.
		require
			class_name /= Void
		do
			Result := ace.cluster_of(class_name, report_error)
		ensure
			report_error implies Result /= Void
		end

	cluster_named (cluster_name: STRING): CLUSTER is
			-- Retrieve the cluster by its name
		require
			not cluster_name.is_empty
			string_aliaser.registered_one(cluster_name)
		do
			Result := ace.cluster_named(cluster_name)
		ensure
			Result /= Void implies Result.name.is_equal(cluster_name)
		end

	class_text (class_name: CLASS_NAME; report_error: BOOLEAN): CLASS_TEXT is
			-- Retrieve the corresponding memorized one or launch the `eiffel_parser' if the `class_name'
			-- class is not yet loaded. When the `report_error' flag is False, no
			-- error occurs even when the class is not found.
			-- (See also `loaded_class_text'.)
		require
			class_name /= Void
		do
			Result := ace.class_text(class_name, report_error, True, Void)
		ensure
			report_error implies Result /= Void
		end

	loaded_class_text (class_name: CLASS_NAME): CLASS_TEXT is
			-- Retrieve the corresponding memorized one if already loaded. Do not launch the `eiffel_parser'
			-- if the `class_name' class is not yet loaded.
			-- (See also `class_text'.)
		require
			class_name /= Void
		do
			Result := ace.class_text(class_name, False, False, Void)
		end

	class_text_in_cluster (class_name: CLASS_NAME; report_error: BOOLEAN; cluster: CLUSTER): CLASS_TEXT is
			-- Retrieve the corresponding memorized one or launch the `eiffel_parser' if the `class_name'
			-- class is not yet loaded. When the `report_error' flag is False, no
			-- error occurs even when the class is not found. `cluster' is the client cluster to start the
			-- search from.
			-- (See also `class_text', `loaded_class_text'.)
		require
			class_name /= Void
		do
			Result := ace.class_text(class_name, report_error, True, cluster)
		ensure
			report_error implies Result /= Void
		end

	same_base_feature (up_rf: RUN_FEATURE; run_time_set: RUN_TIME_SET): BOOLEAN is
			-- True when all `dynamic' features of the `run_time_set' have
			-- excately the same final name and refer exactely to the same
			-- `base_feature'.
			--|*** still valid ???
		require
			up_rf /= Void
			is_ready and run_time_set.count > 1
		local
			i: INTEGER; f: ANONYMOUS_FEATURE; dyn_rf: RUN_FEATURE; lt: LIVE_TYPE
		do
			from
				Result := True
				i := run_time_set.count
				f := up_rf.base_feature
			until
				not Result or else i = 0
			loop
				lt := run_time_set.item(i)
				dyn_rf := lt.dynamic_feature(up_rf)
				if f = dyn_rf.base_feature then
					if dyn_rf.name.to_string /= up_rf.name.to_string then
						Result := False
					end
				else
					Result := False
				end
				i := i - 1
			end
		end

	stupid_switch (t: TYPE_MARK; run_time_set: RUN_TIME_SET): BOOLEAN is
			-- True when `t' drives exactely to the same `t.run_type' for all
			-- element of the `run_time_set'.
			--|*** NOT CLEAR AT ALL ***
			-- Do not remember the goal of this method
			--|*** TO BE REVISITED ***
		require
			is_ready
		do
			--|*** 
			Result := False --|*** SHOULD NOT BREAK ... JUST SLOW DOWN.
		end

	generating_type_used: BOOLEAN
			-- When ANY `generating_type' is used.

	generator_used: BOOLEAN
			-- When ANY `generator' is used.

	deep_twin_used: BOOLEAN
			-- When `deep_twin' support is necessary.

	accumulating_type: TYPE is
			-- Service provided for debugging purpose only
		do
			debug
				Result := feature_accumulator.context_type
			end
		ensure
			debugging_only: Result /= Void
		end

	unknown_feature_fatal_error (target_expression: EXPRESSION; target_type: TYPE; fn: FEATURE_NAME) is
			-- (This feature is placed here to standardize all error messages of this kind.)
		require
			target_expression /= Void
			target_type /= Void
			fn /= Void
		do
			error_handler.add_position(fn.start_position)
			if fn.is_infix_name then
				error_handler.append("Unknown infix operator %"")
				error_handler.append(fn.to_string)
				error_handler.append("%" in type ")
			elseif fn.is_prefix_name then
				error_handler.append("Unknown prefix operator %"")
				error_handler.append(fn.to_string)
				error_handler.append("%" in type ")
			else
				error_handler.append("Unknown feature `")
				error_handler.append(fn.to_string)
				error_handler.append("' in type ")
			end
			error_handler.add_type(target_type)
			error_handler.append(".")
			if target_expression.is_implicit_current then
				error_handler.append(" (For this call, the target is the implicit non written `Current' %
                                 %which is of type ")
				error_handler.add_type(target_type)
				error_handler.append(".)")
			else
				error_handler.add_position(target_expression.start_position)
				error_handler.append(" (For this call, the target ")
				error_handler.add_expression(target_expression)
				error_handler.append(" is of type ")
				error_handler.add_type(target_type)
				error_handler.append(".)")
			end
			error_handler.print_as_fatal_error
		end

feature {CLUSTER}
	analyze_class (cn: CLASS_NAME; c: CLUSTER): CLASS_TEXT is
		do
			Result := eiffel_parser.analyse_class(cn, c)
		end

feature {}
	free_inline_memo: FAST_ARRAY[INLINE_MEMO] is
		once
			create Result.with_capacity(128)
		end

feature {ANY}
	get_inline_memo: INLINE_MEMO is
		do
			if free_inline_memo.is_empty then
				create Result.make
			else
				Result := free_inline_memo.last
				free_inline_memo.remove_last
				Result.clear
			end
		ensure
			Result.cleared
		end

	dispose_inline_memo (im: INLINE_MEMO) is
		require
			im /= Void
		do
			free_inline_memo.add_last(im)
		end

feature {RESULT, ONCE_FUNCTION}
	vffd7_fatal_error (sp: POSITION) is
		do
			error_handler.add_position(sp)
			error_handler.append("Result type of a once function must not involve formal generic names nor anchored types (VFFD.8).")
			error_handler.print_as_fatal_error
		end

feature {GC_HANDLER, CODE_PRINTER}
	root_procedure: RUN_FEATURE_3
			-- The root procedure of the system to compile.

feature {PRETTY}
	class_text_for_pretty (file_path: STRING; class_name: CLASS_NAME): CLASS_TEXT is
		require
			(create {FILE_TOOLS}).is_readable(file_path)
			class_name /= Void
		local
			c: CLUSTER
		do
			parser_buffer.load_file(file_path)
			create c.for_pretty
			parser_buffer.set_cluster(c)
			Result := eiffel_parser.analyse_class(class_name, c)
		end

	remove_loaded_class (ct: CLASS_TEXT) is
			-- Removed the already loaded `ct' in order to allow `pretty' to parse again the generated file to check that the newly
			-- created file is syntactically correct.
		require
			ct /= Void
		do
			ace.remove(ct)
		end

feature {FINDER}
	find_paths_for (class_name: HASHED_STRING): FAST_ARRAY[CLASS_TEXT] is
			-- Finds the path to any class having the given `class_name'
		require
			class_name /= Void
		do
			Result := ace.all_class_texts(create {CLASS_NAME}.unknown_position(class_name))
		ensure
			Result /= Void
		end

feature {CLASS_CHECKER}
	tuple_related_classes_in (parent_list: FAST_ARRAY[TYPE]) is
		require
			parent_list.is_empty
		local
			--|*** ct, tuple_ct: CLASS_TEXT; tuple_type: TYPE; cn: CLASS_NAME; stop: BOOLEAN; i, tuple_index: INTEGER
		do
			not_yet_implemented
			--|*** from
			--|*** 	i := 1
			--|*** until
			--|*** 	stop
			--|*** loop
			--|*** 	if i > class_text_dictionary.count then
			--|*** 		if tuple_type = Void then
			--|*** 			-- No more TUPLE class text.
			--|*** 			stop := True
			--|*** 		else
			--|*** 			tuple_type := Void
			--|*** 		end
			--|*** 		i := 1
			--|*** 	else
			--|*** 		ct := class_text_dictionary.item(i)
			--|*** 		cn := ct.name
			--|*** 		if cn.is_tuple_related then
			--|*** 			if ct.formal_generic_list = Void then
			--|*** 				if tuple_index = 0 then
			--|*** 					tuple_ct := ct
			--|*** 				end
			--|*** 			elseif ct.formal_generic_list.count = tuple_index then
			--|*** 				tuple_ct := ct
			--|*** 			end
			--|*** 			if tuple_ct /= Void then
			--|*** 				tuple_type := tuple_ct.declaration_type_of_like_current
			--|*** 				parent_list.add_last(tuple_type)
			--|*** 				tuple_index := tuple_index + 1
			--|*** 				i := 1
			--|*** 				tuple_ct := Void
			--|*** 				tuple_type := Void
			--|*** 			end
			--|*** 		end
			--|*** 		i := i + 1
			--|*** 	end
			--|*** end
		end

feature {CLASS_CHECKER, EXTERNAL_TOOL}
	set_short_or_class_check_flag is
		do
			short_or_class_check_flag := True
			set_no_id
		end

feature {PRETTY}
	set_pretty_flag is
		do
			pretty_flag := True
			set_no_id
		end

feature {}
	set_no_id is
		do
			no_id := True
		end

feature {ANY}
	class_text_count: INTEGER is
			-- Total number of class text actually loaded.
		do
			Result := ace.class_text_count
		end

feature {COMMAND_LINE_TOOLS}
	compile (backend: CODE_PRINTER) is
			-- Produce some code for `root_class'/`procedure'.
		local
			root_class, procedure: STRING
		do
			initialize_any_tuple
			root_class := ace.root_class_name.to_string
			procedure := ace.root_procedure_name
			get_started(root_class, procedure)
			backend.compile
			very_last_information
		end

feature {CODE_PRINTER}
	customize_runtime is
		local
			i: INTEGER; plugin: NATIVE_PLUG_IN
		do
			if collected_plug_in /= Void then
				from
					i := collected_plug_in.lower
				until
					i > collected_plug_in.upper
				loop
					plugin := collected_plug_in.item(i)
					plugin.customize_runtime
					i := i + 1
				end
			end
		end

	run_features: FAST_ARRAY[RUN_FEATURE]
			-- All the known features just prior the generation itself

	class_invariants: FAST_ARRAY[LIVE_TYPE]
			-- All the known types having an invariant during the generation

	agent_creations: FAST_ARRAY[AGENT_CREATION]
			-- All the known agent creations during the generation

	agent_switches: FAST_ARRAY[TYPE]
			-- All the known agent switches during the generation

	type_dictionary: DICTIONARY[TYPE, HASHED_STRING] is
			-- When looking for a TYPE using it's name (ie. FOO[BAR] is stored at key whith name "FOO[BAR]").
		once
			create {HASHED_DICTIONARY[TYPE, HASHED_STRING]} Result.with_capacity(1024)
		end

feature {ANY}
	magic_count: INTEGER
			-- Grow each time a new run class is added, each time a new class is
			-- loaded, each time a new feature is checked, each time another
			-- expression is optimized, etc. ...
			-- Thus when `magic_count' stops growing, we are really ready :-).

	magic_count_increment is
		do
			magic_count := magic_count + 1
		end

feature {ASSIGNMENT_HANDLER}
	simplify_done: BOOLEAN
			-- After optimizations, some code may be turned into invalid 
			-- eiffel code. Has to be very uncommon.

feature {GC_HANDLER, CODE_PRINTER, INTROSPECTION_HANDLER}
	live_type_map: FAST_ARRAY[LIVE_TYPE] is
			--|*** an attribute created in init_phase would be more
			--|*** efficient than this once function. (in default_create)
			--|*** (PR 29/05/04) this once should be precomputed
		once
			create Result.with_capacity(2048)
		end

feature {CALL_SUPPORT, WRITABLE_ATTRIBUTE_NAME, MANIFEST_STRING_POOL, CREATION_CLAUSE, ADDRESS_OF, ONCE_FUNCTION, MANIFEST_GENERIC}
	collect (type: TYPE; fs: FEATURE_STAMP; at_run_time: BOOLEAN): TYPE is
			-- The `Result' is not Void when `fs' is actually a function (see ensure).
		require
			fs.has_anonymous_feature_for(type)
		local
			lt: LIVE_TYPE; af: ANONYMOUS_FEATURE; tm: TYPE_MARK; tmp_fs: FEATURE_STAMP
		do
			--|*** PH(03/03/04) To be removed
			lt := collect_one_type(type, at_run_time)
			--|*** PH(03/03/04) should be "af := fs.anonymous_feature(type)",
			--but in order to quickly solve a hard problem I added one
			--extra resolution.
			--
--			tmp_fs := fs.resolve_dynamic_binding_for(type)
--***
			tmp_fs := fs
-- *** But we must now have two types... *** Dom march 16th 2006 ***
			af := tmp_fs.anonymous_feature(type)
			tm := af.result_type
			if tm /= Void then
				Result := tm.resolve_in(type)
			end
			lt.collect(tmp_fs)
		ensure
			fs.anonymous_feature(type).result_type /= Void implies Result = fs.anonymous_feature(type).result_type.resolve_in(type)
		end

feature {LOCAL_VAR_LIST, INTERNAL_LOCAL_LIST, ANONYMOUS_FEATURE}
	collect_local_expanded (type: TYPE) is
			-- Make live the given `type'.
		require
			type.is_expanded
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, True)
			--|*** May be we have to launch here the collect on the
			--| corresponding creation procedure without argument ?
			--| (Dom. 8/9/2003)
		ensure
			type.live_type /= Void
		end

feature {BASE_TYPE_CONSTANT}
	collect_constant (type: TYPE) is
			-- Make live the given `type' (of a constant)
		require
			type /= Void
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, True)
		ensure
			type.live_type /= Void
		end

feature {EXTERNAL_FUNCTION}
	collect_external (type: TYPE) is
			-- Make live the given type (of an external function)
		require
			type /= Void
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, True)
		ensure
			type.live_type /= Void
		end

feature {ASSERTION}
	collect_assertion (type: TYPE) is
			-- Make live the given type (of an assertion)
		require
			type.is_boolean
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, True)
		ensure
			type.live_type /= Void
		end

feature {AGENT_CREATION, ADDRESS_OF}
	collect_create (type: TYPE) is
		require
			type /= Void
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, True)
		ensure
			type.live_type.at_run_time
		end

feature {LIVE_TYPE}
	collect_generic (type: TYPE) is
		require
			type /= Void
		local
			lt: LIVE_TYPE
		do
			lt := collect_one_type(type, False)
		ensure
			type.live_type /= Void
		end

	collect_se_atexit (type: TYPE) is
		require
			type /= Void
			no_extra_computation: se_atexit_stamp = Void
		local
			dummy: TYPE
		do
			se_atexit_stamp := type.feature_stamp_of(se_atexit_name)
			dummy := collect(type, se_atexit_stamp, False)
			se_atexit_id := type.id
		ensure
			se_atexit_stamp /= Void
		end

feature {CODE_PRINTER}
	se_atexit_stamp: FEATURE_STAMP

	se_atexit_id: INTEGER

feature {}
	collected_external_functions: FAST_ARRAY[NON_VOID_NO_DISPATCH] is
		once
			create Result.with_capacity(3)
		end

feature {NON_VOID_NO_DISPATCH}
	collect_external_function (non_void_no_dispatch: NON_VOID_NO_DISPATCH; fs: FEATURE_STAMP; type: TYPE) is
		require
			non_void_no_dispatch /= Void
			type /= Void
			fs /= Void
		local
			dummy: TYPE
		do
			if not collected_external_functions.fast_has(non_void_no_dispatch) then
				dummy := collect(type, fs, False)
				collected_external_functions.add_last(non_void_no_dispatch)
			end
		end

feature {ONCE_ROUTINE_POOL}
	collect_precomputable (type: TYPE; fs: FEATURE_STAMP) is
		require
			type /= Void
			fs /= Void
		local
			dummy: TYPE
		do
			dummy := collect(type, fs, False)
		end

feature {CECIL_ENTRY}
	collect_cecil_entry (type: TYPE; fs: FEATURE_STAMP; is_creation: BOOLEAN) is
		require
			type /= Void
			fs /= Void
		local
			dummy: TYPE
		do
			dummy := collect(type, fs, is_creation)
		end

feature {CALL, PROC_CALL, PRECURSOR_CALL, AGENT_CREATION}
	argument_count_check (call_site: POSITION; af: ANONYMOUS_FEATURE; actual_args: EFFECTIVE_ARG_LIST) is
			-- Check that the number of arguments of `af' is compatible with `actual_args'. Only the number of
			-- arguments is checked here. (The complete conformance test is performed in EFFECTIVE_ARG_LIST.)
		require
			not call_site.is_unknown
			af /= Void
		local
			formal_args: FORMAL_ARG_LIST; actual, formal: INTEGER
		do
			formal_args := af.arguments
			if formal_args /= Void then
				if actual_args /= Void then
					formal := formal_args.count
					actual := actual_args.count
					if actual /= formal then
						error_handler.append("The feature called has ")
						error_handler.append_integer(formal)
						error_handler.append(" formal argument")
						if formal > 1 then
							error_handler.extend('s')
						end
						error_handler.append(" while the actual argument list has ")
						error_handler.append_integer(actual)
						error_handler.append(" argument")
						if actual > 1 then
							error_handler.extend('s')
						end
						error_handler.append(".")
						error_handler.add_position(formal_args.start_position)
						error_handler.add_position(actual_args.start_position)
						error_handler.print_as_fatal_error
					end
				else
					formal := formal_args.count
					error_handler.append("The feature called has ")
					error_handler.append_integer(formal)
					error_handler.append(" formal argument")
					if formal > 1 then
						error_handler.extend('s')
					end
					error_handler.append(" while there is no actual argument list in the call.")
					error_handler.add_position(formal_args.start_position)
					error_handler.add_position(call_site)
					error_handler.print_as_fatal_error
				end
			elseif actual_args /= Void then
				actual := actual_args.count
				error_handler.append("The feature called has no formal argument while the actual argument list has ")
				error_handler.append_integer(actual)
				error_handler.append(" argument")
				if actual > 1 then
					error_handler.extend('s')
				end
				error_handler.append(".")
				error_handler.add_position(af.start_position)
				error_handler.add_position(actual_args.start_position)
				error_handler.print_as_fatal_error
			end
		end

feature {TYPE_MARK}
	long_type_name (type_name: HASHED_STRING; type_cluster: CLUSTER): HASHED_STRING is
		require
			type_name /= Void
			type_cluster /= Void
		local
			ln: STRING
		do
			ln := once ""
			ln.copy(type_name.to_string)
			ln.extend('@')
			ln.append(type_cluster.name)
			Result := string_aliaser.hashed_string(ln)
		ensure
			Result /= Void
		end

feature {AGENT_POOL, CREATE_SUPPORT, ASSIGNMENT, ASSIGNMENT_ATTEMPT, CREATE_WRITABLE, EFFECTIVE_ARG_LIST}
	collect_one_type (type: TYPE; at_run_time: BOOLEAN): LIVE_TYPE is
			-- Make live the given `type'.
		require
			type /= Void
		local
			rt: BOOLEAN
		do
			Result := type.live_type
			if Result = Void then
				create Result.make(type)
				check
					not type.is_expanded implies not Result.at_run_time
					type_dictionary.reference_at(type.long_name).live_type = Result
				end
			end
			rt := at_run_time or else Result.is_expanded and then Result.name.to_string /= as_integer_general
			if rt and then not Result.at_run_time then
				debug
					if not at_run_time then
						-- sedb_breakpoint -- type will become alive, is it really necessary?
						--|*** I have noticed that it is often NATIVE_ARRAY here ...
						--|*** NATIVE_ARRAY must be made alive when {NATIVE_ARRAY}.malloc is collected !!
						--|*** (Dom. april 22th 2004) ***
					end
				end
				Result.set_at_run_time
			end
		ensure
			is_alive: type.live_type /= Void
		end

feature {ANONYMOUS_FEATURE, RUN_FEATURE, ARGUMENT_NAME2, LOCAL_NAME2, RESULT, E_OLD, INSPECT_STATEMENT}
	context_feature: ANONYMOUS_FEATURE is
		do
			Result := context_feature_stack.top
		end

feature {ANONYMOUS_FEATURE, RUN_FEATURE, FEATURE_STAMP}
	push_context (af: ANONYMOUS_FEATURE) is
		do
			context_feature_stack.push(af)
		end

	pop_context (af: ANONYMOUS_FEATURE) is
		require
			context_feature = af
		do
			context_feature_stack.pop
		end

feature {}
	context_feature_stack: STACK[ANONYMOUS_FEATURE] is
			-- We really need a stack. As an example, it is necessary to follow the calling graph during
			-- collect.
		once
			create Result.with_capacity(4)
		end

feature {LIVE_TYPE}
	is_tagged (lt: LIVE_TYPE): BOOLEAN is
		require
			is_ready
			lt.at_run_time
			lt.is_reference
			ace.boost
		local
			i: INTEGER; run_time_set: RUN_TIME_SET
		do
			from
				i := live_type_map.upper
			until
				Result or else i < 0
			loop
				run_time_set := live_type_map.item(i).run_time_set
				if run_time_set.count = 0 then
				elseif run_time_set.has(lt) then
					Result := run_time_set.count > 1
				end
				i := i - 1
			end
		end

feature {CODE_PRINTER}
	weak_reference_used: BOOLEAN is
			-- Is the WEAK_REFERENCE class used?
			--
			-- This function is conservative: when it returns False, you are guaranteed that no WEAK_REFERENCE 
			-- is used. There is no guarantee when it returns True.
		require
			status.collecting_done
--| 		local
--| 			hs: HASHED_STRING
		do
--|*** The following code can't work because it doesn't look for the right WEAK_REFERENCE class <FM-11/12/2006>
--| 			hs := string_aliaser.hashed_string(as_weak_reference)
--| 			Result := loaded_class_text(create {CLASS_NAME}.unknown_position(hs)) /= Void
--| 			-- Yes, there is a memory leak, but this feature is probably called only once.
			Result := True
		end

feature {LOCAL_NAME2, EFFECTIVE_ROUTINE, EXTERNAL_ROUTINE, INTROSPECTION_HANDLER}
	specializing_feature_local_var_list: LOCAL_VAR_LIST

feature {EFFECTIVE_ROUTINE, EXTERNAL_ROUTINE, INTROSPECTION_HANDLER}
	set_specializing_feature_variables (lvl: LOCAL_VAR_LIST) is
		do
			specializing_feature_local_var_list := lvl
		end

feature {}
	type_to_be_created: HASHED_STRING

	create_type (static_type: TYPE_MARK): TYPE is
			-- Create a type. Type creation is locked until `register_type' has been called.
		require
			static_type.is_static
			lock_type_creation(static_type.long_name)
			not has_type(static_type)
		do
			create Result.make(static_type)
		ensure
			Result /= Void
			has_type(static_type)
		end

	lock_type_creation (t: like type_to_be_created): BOOLEAN is
		do
			Result := type_to_be_created = Void
			type_to_be_created := t
		ensure
			Result
		end

feature {}
	type_any_memory: TYPE

	type_boolean_memory: TYPE

	type_character_memory: TYPE

	type_string_memory: TYPE

	type_unicode_string_memory: TYPE

	type_pointer_memory: TYPE

	type_integer_8_memory: TYPE

	type_integer_16_memory: TYPE

	type_integer_32_memory: TYPE

	type_integer_64_memory: TYPE

	type_real_32_memory: TYPE

	type_real_64_memory: TYPE

	type_real_extended_memory: TYPE
	
feature {ANY} -- To get a TYPE:
	--|*** TYPE creation can be quite recursive, so these cannot be once functions <FM-14/10/2004>

	--|*** Those types are basic types; there is no cluster information in their long name hence no call to `long_type_name'

	type_any: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_any_memory = Void then
				hs := string_aliaser.hashed_string(as_any)
				type_any_memory := type_dictionary.fast_reference_at(hs)
				if type_any_memory = Void then
					type_any_memory := get_type(create {ANY_TYPE_MARK}.make(unknown_position))
				end
			end
			Result := type_any_memory
		ensure
			type_any_memory /= Void
			Result /= Void
		end

	type_boolean: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_boolean_memory = Void then
				hs := string_aliaser.hashed_string(as_boolean)
				type_boolean_memory := type_dictionary.fast_reference_at(hs)
				if type_boolean_memory = Void then
					type_boolean_memory := get_type(create {BOOLEAN_TYPE_MARK}.make(unknown_position))
				end
			end
			Result := type_boolean_memory
		ensure
			type_boolean_memory /= Void
			Result /= Void
		end

	type_character: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_character_memory = Void then
				hs := string_aliaser.hashed_string(as_character)
				type_character_memory := type_dictionary.fast_reference_at(hs)
				if type_character_memory = Void then
					type_character_memory := get_type(create {CHARACTER_TYPE_MARK}.make(unknown_position))
				end
			end
			Result := type_character_memory
		ensure
			type_character_memory /= Void
			Result /= Void
		end

	type_string: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_string_memory = Void then
				hs := string_aliaser.hashed_string(as_string)
				type_string_memory := type_dictionary.fast_reference_at(hs)
				if type_string_memory = Void then
					type_string_memory := get_type(create {STRING_TYPE_MARK}.make(unknown_position))
				end
			end
			Result := type_string_memory
		ensure
			type_string_memory /= Void
			Result /= Void
		end

	type_unicode_string: TYPE is
		require
			not eiffel_parser.is_running
		local
			hs: HASHED_STRING; unknown_position: POSITION
			unicode_class_text: CLASS_TEXT cn: CLASS_NAME
		do
			if type_unicode_string_memory = Void then
				hs := string_aliaser.hashed_string(as_unicode_string)
				type_unicode_string_memory := type_dictionary.fast_reference_at(hs)
				if type_unicode_string_memory = Void then
					create cn.make(hs, unknown_position)
					unicode_class_text := class_text(cn, True)
					check
						unicode_class_text /= Void
					end
					create cn.make(hs, unicode_class_text.name.start_position)
					type_unicode_string_memory := get_type(create {CLASS_TYPE_MARK}.make(cn))
				end
			end
			Result := type_unicode_string_memory
		ensure
			type_unicode_string_memory /= Void
			Result /= Void
		end

	type_pointer: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_pointer_memory = Void then
				hs := string_aliaser.hashed_string(as_pointer)
				type_pointer_memory := type_dictionary.fast_reference_at(hs)
				if type_pointer_memory = Void then
					type_pointer_memory := get_type(create {POINTER_TYPE_MARK}.make(unknown_position))
				end
			end
			Result := type_pointer_memory
		ensure
			type_pointer_memory /= Void
			Result /= Void
		end

	type_integer_8: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_integer_8_memory = Void then
				hs := string_aliaser.hashed_string(as_integer_8)
				type_integer_8_memory := type_dictionary.fast_reference_at(hs)
				if type_integer_8_memory = Void then
					type_integer_8_memory := get_type(create {INTEGER_TYPE_MARK}.integer_8(unknown_position))
				end
			end
			Result := type_integer_8_memory
		ensure
			type_integer_8_memory /= Void
			Result /= Void
		end

	type_integer_16: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_integer_16_memory = Void then
				hs := string_aliaser.hashed_string(as_integer_16)
				type_integer_16_memory := type_dictionary.fast_reference_at(hs)
				if type_integer_16_memory = Void then
					type_integer_16_memory := get_type(create {INTEGER_TYPE_MARK}.integer_16(unknown_position))
				end
			end
			Result := type_integer_16_memory
		ensure
			type_integer_16_memory /= Void
			Result /= Void
		end

	type_integer_32: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_integer_32_memory = Void then
				hs := string_aliaser.hashed_string(as_integer_32)
				type_integer_32_memory := type_dictionary.fast_reference_at(hs)
				if type_integer_32_memory = Void then
					type_integer_32_memory := get_type(create {INTEGER_TYPE_MARK}.integer(unknown_position))
				end
			end
			Result := type_integer_32_memory
		ensure
			type_integer_32_memory /= Void
			Result /= Void
		end

	type_integer_64: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_integer_64_memory = Void then
				hs := string_aliaser.hashed_string(as_integer_64)
				type_integer_64_memory := type_dictionary.fast_reference_at(hs)
				if type_integer_64_memory = Void then
					type_integer_64_memory := get_type(create {INTEGER_TYPE_MARK}.integer_64(unknown_position))
				end
			end
			Result := type_integer_64_memory
		ensure
			type_integer_64_memory /= Void
			Result /= Void
		end

	type_real_32: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_real_32_memory = Void then
				hs := string_aliaser.hashed_string(as_real_32)
				type_real_32_memory := type_dictionary.fast_reference_at(hs)
				if type_real_32_memory = Void then
					type_real_32_memory := get_type(create {REAL_TYPE_MARK}.real_32(unknown_position))
				end
			end
			Result := type_real_32_memory
		ensure
			type_real_32_memory /= Void
			Result /= Void
		end

	type_real_64: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_real_64_memory = Void then
				hs := string_aliaser.hashed_string(as_real_64)
				type_real_64_memory := type_dictionary.fast_reference_at(hs)
				if type_real_64_memory = Void then
					type_real_64_memory := get_type(create {REAL_TYPE_MARK}.real(unknown_position))
				end
			end
			Result := type_real_64_memory
		ensure
			type_real_64_memory /= Void
			Result /= Void
		end

	type_real_extended: TYPE is
		local
			hs: HASHED_STRING; unknown_position: POSITION
		do
			if type_real_extended_memory = Void then
				hs := string_aliaser.hashed_string(as_real_extended)
				type_real_extended_memory := type_dictionary.fast_reference_at(hs)
				if type_real_extended_memory = Void then
					type_real_extended_memory := get_type(create {REAL_TYPE_MARK}.real_extended(unknown_position))
				end
			end
			Result := type_real_extended_memory
		ensure
			type_real_extended_memory /= Void
			Result /= Void
		end

	unlock_type_creation (t: like type_to_be_created): BOOLEAN is
		do
			Result := type_to_be_created = t
			type_to_be_created := Void
		ensure
			Result
		end

	has_type (static_type: TYPE_MARK): BOOLEAN is
		require
			static_type.is_static
		do
			Result := type_dictionary.fast_has(static_type.long_name)
		end

	get_type (static_type: TYPE_MARK): TYPE is
		require
			static_type.is_static
		local
			name: HASHED_STRING; tm: TYPE_MARK; gl: ARRAY[TYPE_MARK]; i: INTEGER
		do
			name := static_type.long_name
			Result := type_dictionary.fast_reference_at(name)
			if Result = Void then
				-- Before creating the TYPE of `static_type', we must be sure that it's generic 
				-- argument, if any, are created first:
				if static_type.is_generic then
					from
						gl := static_type.generic_list
						i := gl.lower
					until
						i > gl.upper
					loop
						tm := gl.item(i)
						if type_dictionary.fast_reference_at(tm.long_name) = Void then
							-- We must create the corresponding generic argument:
							Result := get_type(tm)
						end
						i := i + 1
					end
				end
				-- Because it is possible that the previous computation actually resulted in creating the 
				-- type of `static_type' itself:
				Result := type_dictionary.fast_reference_at(name)
				if Result = Void then
					Result := create_type(static_type)
				end
			end
		ensure
			Result /= Void
			has_type(static_type)
		end

	get_array_type (gen_type: TYPE; pos: POSITION): TYPE is
			-- Returns the ARRAY[gen_type] type
		local
			name: STRING; hs: HASHED_STRING; atm: ARRAY_TYPE_MARK
		do
			name := once ""
			name.copy(once "ARRAY[")
			name.append(gen_type.name.to_string)
			name.extend(']')
			hs := string_aliaser.hashed_string(name)
			Result := type_dictionary.fast_reference_at(hs)
			if Result = Void then
				create atm.make(pos, gen_type.canonical_type_mark)
				Result := atm.type
			end
		ensure
			Result /= Void
		end

	tuple_class_not_found_fatal_error (class_name: CLASS_NAME) is
			-- Because there is a special trick for TUPLE related classes.
		require
			class_name.is_tuple_related
		local
			tuple_file_path: STRING; stop: BOOLEAN; cn: CLASS_NAME; ct: CLASS_TEXT; max_tuple: INTEGER
		do
			from
				check
					max_tuple = 0
				end
			until
				stop
			loop
				create cn.unknown_position(string_aliaser.hashed_string("TUPLE " + (max_tuple + 1).to_string))
				ct := ace.class_text(cn, False, False, Void)
				if ct = Void then
					stop := True
				else
					max_tuple := max_tuple + 1
					if tuple_file_path = Void then
						tuple_file_path := ct.path
					end
				end
			end
			error_handler.add_position(class_name.start_position)
			if tuple_file_path = Void then
				error_handler.append("Unable to load class %"")
				error_handler.append(class_name.to_string)
				error_handler.append("%".")
			else
				error_handler.append("No such TUPLE definition in file %"")
				error_handler.append(tuple_file_path)
				error_handler.append("%".%NToo long TUPLE (the TUPLE you want has ")
				error_handler.append_integer(class_name.tuple_count)
				error_handler.append(" formal generic arguments while the maximum allowed is ")
				error_handler.append_integer(max_tuple)
				error_handler.append(").%N")
			end
			error_handler.print_as_fatal_error
		end
	
feature {CLASS_TYPE_MARK}
	get_type_for_non_generic  (non_generic_static_type: TYPE_MARK): TYPE is
			-- Just an optimized version of `get_type'.
		require
			non_generic_static_type.is_static
			not non_generic_static_type.is_generic
		do
			Result := type_dictionary.fast_reference_at(non_generic_static_type.long_name)
			if Result = Void then
				Result := create_type(non_generic_static_type)
			end
		ensure
			Result = get_type(non_generic_static_type)
		end
	
feature {CODE_PRINTER, LIVE_TYPE}
	dispatch_special (type: TYPE; feature_stamp: FEATURE_STAMP): STRING is
			-- For some very special cases, we must not use switch. A non Void Result indicates that
			-- we must not use a switch for this feature.
		local
			fs: FEATURE_STAMP
			ta: TYPE
		do
			-- I am not really happy with this function which is called
			-- very often during code generation. May be we'll have
			-- another solution with the new inlined dispatch
			-- strategy... for the next release.
			--|*** (Dom. june 18th 2004) ***
			fs := any_c_inline_c_feature_stamp
			ta := type_any
			if feature_stamp = fs or else feature_stamp = fs.resolve_static_binding_for(ta, type) then
				Result := as_c_inline_c
			end
			if Result = Void then
				fs := any_c_inline_h_feature_stamp
				if feature_stamp = fs or else feature_stamp = fs.resolve_static_binding_for(ta, type) then
					Result := as_c_inline_h
				end
			end
		end

feature {}
	any_c_inline_c_feature_stamp: FEATURE_STAMP is
		once
			Result := feature_stamp_of(as_any, as_c_inline_c)
		end

	any_c_inline_h_feature_stamp: FEATURE_STAMP is
		once
			Result := feature_stamp_of(as_any, as_c_inline_h)
		end

	feature_stamp_of (class_name, feature_name: STRING): FEATURE_STAMP is
		require
			string_aliaser.registered_one(class_name)
			string_aliaser.registered_one(feature_name)
			string_aliaser.hashed_string(feature_name).is_simple_feature_name
		local
			hs: HASHED_STRING; tm: TYPE
		do
			hs := string_aliaser.hashed_string(class_name)
			tm := type_dictionary.fast_reference_at(hs)
			Result := tm.feature_stamp_of(string_aliaser.hashed_string(feature_name))
		ensure
			Result /= Void
		end

feature {LIVE_TYPE}
	memory_dispose_stamp: FEATURE_STAMP is
			-- Feature stamp of {DISPOSABLE}.dispose
		require
			status.is_collecting -- Because the call can make DISPOSABLE live
		local
			hs: HASHED_STRING; cn: CLASS_NAME; t: TYPE; c: CLUSTER
		once
			hs := string_aliaser.hashed_string(as_disposable)
			create cn.unknown_position(hs)
			c := cluster_of(cn, True)
			hs := long_type_name(hs, c)
			t := type_dictionary.fast_reference_at(hs)
			if t = Void then
				t := get_type(create {CLASS_TYPE_MARK}.make(cn))
				check
					type_dictionary.has(hs)
				end
			end
			if t.has_simple_feature_name(dispose_name) then
				Result := t.feature_stamp_of(dispose_name)
			else
				error_handler.append("Class DISPOSABLE must have a `dispose' feature.")
				error_handler.add_position(t.class_text.name.start_position)
				error_handler.print_as_fatal_error
			end
		ensure
			Result /= Void
		end

feature {MANIFEST_GENERIC}
	manifest_creation_name: FEATURE_NAME is
			-- feature name of {ANY}.manifest_creation
		require
			status.is_collecting -- Overkill ? <FM-17/12/2004>
		local
			t: TYPE; hs: HASHED_STRING
		once
			t := type_any
			hs := string_aliaser.hashed_string(as_manifest_creation)
			if t.has_simple_feature_name(hs) then
				Result := t.feature_stamp_of(hs).anonymous_feature(t).first_name
			else
				error_handler.append("Class ANY must have a `manifest_creation' feature.")
				error_handler.add_position(t.class_text.name.start_position)
				error_handler.print_as_fatal_error
			end
		ensure
			Result /= Void
		end

feature {TYPE}
	register_type (t: TYPE) is
		require
			t /= Void
			unlock_type_creation(t.long_name)
		local
			ln: HASHED_STRING
		do
			ln := t.long_name
			if type_dictionary.fast_has(ln) then
				error_handler.append("You seem to have too classes named ")
				error_handler.append(t.canonical_type_mark.class_text_name.to_string)
				error_handler.append(". Yours is in the cluster %"")
				error_handler.append(t.class_text.cluster.directory_path)
				error_handler.append("%". This is not possible as this class is basically used by SmartEiffel internals. Please pick another name.")
				error_handler.print_as_fatal_error
			else
				type_dictionary.add(t, ln)
			end
		end

feature {LIVE_TYPE}
	register_live_type (lt: LIVE_TYPE) is
		require
			lt /= Void
--		local
--			min, max, n: INTEGER; stop: BOOLEAN; slt, s: STRING
		do
--			if ace.boost or else live_type_map.is_empty then
				live_type_map.add_last(lt)
--|*** Nice try, but this breaks the loop in `do_one_collect_cycle' (the one with the comment"
--|***We go this way to collect new incoming live-types") <FM-17/02/2006>
--			else
				-- Insert it sorted by class name (useful to help recompiling the least possible C chunks)
				-- (Binary algorithm taken and adapted from ABSTRACT_SORTER.index_of)
--				from
--					slt := lt.class_text_name.to_string
--					min := live_type_map.lower
--					max := live_type_map.upper
--				variant
--					max - min
--				until
--					stop
--				loop
--					if min = max then
--						stop := True
--						if live_type_map.item(min).class_text_name.to_string < slt then
--							n := min + 1
--						else
--							n := min
--						end
--					else
--						check
--							min < max
--						end
--						n := (min + max) #// 2
--						s := live_type_map.item(n).class_text_name.to_string
--						if s < slt then
--							min := n + 1
--						else
--							max := n
--						end
--					end
--				end
--				live_type_map.add(lt, n)
--			end
		end

feature {CLASS_CHECK}
	very_last_information is
		do
			assignment_handler.echo_information
			feature_accumulator.echo_information
			status.info
			eiffel_parser.show_nb_warnings
			eiffel_parser.show_nb_errors
			echo.put_string(once "Done.%N")
		end

feature {ID_PROVIDER}
	id_extra_information (tfw: TEXT_FILE_WRITE; name: HASHED_STRING; cluster: CLUSTER) is
		require
			cluster /= Void
		local
			ct: CLASS_TEXT; lt: LIVE_TYPE; path: STRING; t: TYPE
		do
			t := type_dictionary.fast_reference_at(long_type_name(name, cluster))
			if t /= Void then
				lt := t.live_type
			end
			if lt /= Void then
				ct := lt.class_text
			else
				ct := ace.class_text(create {CLASS_NAME}.unknown_position(name), False, False, cluster)
			end
			tfw.put_character('%N')
			if ct /= Void then
				tfw.put_string(once "class-path: %"")
				path := ct.path
				tfw.put_string(path)
				tfw.put_character('%"')
				tfw.put_character('%N')
				ct.id_extra_information(tfw)
			end
			if lt /= Void then
				lt.id_extra_information(tfw)
			end
		end

feature {CALL_SUPPORT}
	covariance_check (call_site: POSITION; up_rf: RUN_FEATURE; run_time_set: RUN_TIME_SET) is
		require
			not call_site.is_unknown
			up_rf.arguments.count >= 1
			not run_time_set.is_empty
		local
			dyn_rf: RUN_FEATURE; up_args, dyn_args: FORMAL_ARG_LIST; r, a: INTEGER
			original_tm, redefined_tm: TYPE_MARK; original, redefined: TYPE
		do
			from
				r := 1
				up_args := up_rf.arguments
			until
				r > run_time_set.count
			loop
				dyn_rf := run_time_set.item(r).dynamic_feature(up_rf)
				from
					a := 1
					dyn_args := dyn_rf.arguments
				until
					a > up_args.count
				loop
					original_tm := up_args.type_mark(a)
					original := original_tm.resolve_in(up_rf.type_of_current)
					redefined_tm := dyn_args.type_mark(a)
					redefined := redefined_tm.resolve_in(dyn_rf.type_of_current)
					if not original.can_be_assigned_to(redefined) and then
						original.collected_feature_count > 0 and then
						original.collected_feature_count /= redefined.collected_feature_count
					 then
						error_handler.add_position(call_site)
						error_handler.append("Unsafe call site (see also next warning).")
						error_handler.print_as_warning						
						error_handler.append("Unsafe covariant redefinition of argument number ")
						error_handler.append_integer(a)
						error_handler.append(" (type %"")
						error_handler.append(original.name.to_string)
						error_handler.append("%" redefined as %"")
						error_handler.append(redefined.name.to_string)
						error_handler.append("%").")
						error_handler.add_position(up_rf.base_feature.start_position)
						error_handler.add_position(dyn_rf.base_feature.start_position)
						error_handler.print_as_warning
						status.set_safety_check_failed
					end
					a := a + 1
				end
				r := r + 1
			end
		end

feature {E_OLD}
	register_old (e_old: E_OLD) is
		require
			e_old /= Void
		do
			if old_list = Void then
				create old_list.with_capacity(2)
			end
			old_list.add_last(e_old)
		ensure
			old_list.fast_has(e_old)
		end

feature {}
	old_list: FAST_ARRAY[E_OLD]
			-- Non Void when some E_OLD have been gathered.

feature {RUN_FEATURE}
	get_and_clear_old_list: FAST_ARRAY[E_OLD] is
		do
			Result := old_list
			old_list := Void
		ensure
			old_list = Void
		end

feature {RUN_FEATURE, SWITCH, LIVE_TYPE, CLASS_INVARIANT, AGENT_CREATION, AGENT_ARGS}
	local_profile is
		require
			ace.profile
			cpp.pending_c_function
		do
			cpp.pending_c_function_body.append(once "se_local_profile_t local_profile;%N")
		end

	start_profile (rf: RUN_FEATURE) is
		require
			ace.profile
			cpp.pending_c_function
			rf /= Void
		do
			cpp.pending_c_function_body.append(once "local_profile.profile=profile+")
			run_features.fast_first_index_of(rf).append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nstart_profile(parent_profile, &local_profile);%N")
		end

	start_profile_class_invariant (t: LIVE_TYPE) is
		require
			ace.profile
			cpp.pending_c_function
			t /= Void
		do
			register_class_invariant(t)
			cpp.pending_c_function_body.append(once "local_profile.profile=inv_profile+")
			class_invariants.fast_first_index_of(t).append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nstart_profile(parent_profile, &local_profile);%N")
		end

	start_profile_agent_creation (ac: AGENT_CREATION) is
		require
			ace.profile
			cpp.pending_c_function
			ac /= Void
		do
			register_agent_creation(ac)
			cpp.pending_c_function_body.append(once "local_profile.profile=agent_profile+")
			agent_creations.fast_first_index_of(ac).append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nstart_profile(parent_profile, &local_profile);%N")
		end

	start_profile_agent_switch (t: TYPE) is
		require
			ace.profile
			cpp.pending_c_function
			t /= Void
		do
			register_agent_switch(t)
			cpp.pending_c_function_body.append(once "local_profile.profile=agent_switch_profile+")
			agent_switches.fast_first_index_of(t).append_in(cpp.pending_c_function_body)
			cpp.pending_c_function_body.append(once ";%Nstart_profile(parent_profile, &local_profile);%N")
		end

	stop_profile is
		require
			ace.profile
			cpp.pending_c_function
		do
			cpp.pending_c_function_body.append(once "stop_profile(parent_profile, &local_profile);%N")
		end

feature {}
	collected_plug_in: SET[NATIVE_PLUG_IN] is
		once
			create {HASHED_SET[NATIVE_PLUG_IN]} Result.make
		end

feature {NATIVE_PLUG_IN}
	register_plug_in (native_plug_in: NATIVE_PLUG_IN) is
		require
			native_plug_in /= Void
		do
			collected_plug_in.add(native_plug_in)
		end

feature {LIVE_TYPE}
	set_deep_twin_used is
		do
			deep_twin_used := True
		end

feature {RUN_FEATURE_8, EXTERNAL_FUNCTION, GENERATOR_GENERATING_TYPE} --|*** remove RUN_FEATURE_8
	set_generating_type_used is
		do
			generating_type_used := True
		end

	set_generator_used is
		do
			generator_used := True
		end

feature {CLUSTER}
	parse_include (include_name: STRING) is
		require
			include_name /= Void
		local
			ct: CLASS_TEXT
		do
			check
				parser_buffer.is_ready
			end
			echo.put_string(once "Handling include of %"")
			echo.put_string(include_name)
			echo.put_string(once "%" from ACE file. (Parsing %"")
			echo.put_string(parser_buffer.path)
			echo.put_string(once "%".)%N")
			ct := eiffel_parser.analyse_buffer
			check
				ct /= Void implies ace.has(ct.name.hashed_name)
			end
		end

feature {RUN_FEATURE}
	register_run_feature (rf: RUN_FEATURE) is
		require
			not_registered: not registered(rf)
		do
			if run_features = Void then
				create run_features.with_capacity(16)
			end
			run_features.add_last(rf)
		ensure
			run_features.last = rf
		end

feature {RUN_FEATURE, LIVE_TYPE}
	registered (rf: RUN_FEATURE): BOOLEAN is
		do
			Result := run_features /= Void and then run_features.fast_has(rf)
		end

feature {COMMAND_LINE_TOOLS}
	initialize_any_tuple is
			-- Some tools have to call this `initialize_any_tuple' once routine.
			--	Actually, `initialize_any_tuple' forces the creation of ANY and TUPLE first.
			-- Note, this is not in the creation of `smart_eiffel' itself because, not all tools are
			-- supposed to load Eiffel classes.
		local
			hashed_string: HASHED_STRING; ct: CLASS_TEXT
		once
			-- Forcing first creation of ANY in order to initialize the machinery:
			hashed_string := string_aliaser.hashed_string(as_any)
			ct := class_text(create {CLASS_NAME}.unknown_position(hashed_string), True)

			-- Finally, forcing creation of TUPLE:
			hashed_string := string_aliaser.hashed_string(as_tuple)
			ct := class_text(create {CLASS_NAME}.make(hashed_string, ct.name.start_position), True)

			-- Note: even the simple HELLO_WORLD, normally has to load TUPLE.
		end

feature {}
	register_class_invariant (t: LIVE_TYPE) is
		do
			if class_invariants = Void then
				create class_invariants.with_capacity(16)
			end
			if not class_invariants.fast_has(t) then
				class_invariants.add_last(t)
			end
		end

	register_agent_creation (ac: AGENT_CREATION) is
		do
			if agent_creations = Void then
				create agent_creations.with_capacity(16)
			end
			if not agent_creations.fast_has(ac) then
				agent_creations.add_last(ac)
			end
		end

	register_agent_switch (t: TYPE) is
		do
			if agent_switches = Void then
				create agent_switches.with_capacity(16)
			end
			if not agent_switches.fast_has(t) then
				agent_switches.add_last(t)
			end
		end

feature {E_FUNCTION, CALL_INFIX_POWER}
	simplify_integer_infix_power (call_site: POSITION; target, exponent: EXPRESSION): INTEGER_CONSTANT is
			-- Static simplification of {INTEGER_GENERAL}.infix "^" is here to be shared for `static_simplify' and 
			-- `simplify'.
		local
			ic1, ic2: INTEGER_CONSTANT; v1, v2, r, i: INTEGER_64; overflow: BOOLEAN
		do
			ic1 ?= target
			if ic1 /= Void then
				ic2 ?= exponent
				if ic2 /= Void then
					v1 := ic1.value_memory
					v2 := ic2.value_memory
					if v2 < 0 then
						error_handler.append("Exponent of infix %"^%" must be a positive INTEGER. %
													%Exponent actual value is %"")
						error_handler.append(v2.to_string)
						error_handler.append("%".")
						error_handler.add_position(call_site)
						error_handler.print_as_fatal_error
					end
					-- We now computing the result of `v1^v2' in `r' slowly, but carefully whitout 
					-- using infix "^" itself:
					if v2 = 0 then
						r := 1
					else
						from
							r := v1
							i := v2
						until
							overflow or else i = 1 
						loop
							i := i - 1
							if ((r #* v1) #// v1) /= r then
								overflow := True
							end
							r := r * v1
						end
					end
					if overflow then
						error_handler.append("Overflow while computing %"")
						error_handler.append(v1.to_string)
						error_handler.append("^")
						error_handler.append(v2.to_string)
						error_handler.append("%"")
						error_handler.add_position(call_site)
						error_handler.print_as_fatal_error
					end
					-- Now checking that the result is identical to what's actually in INTEGER_GENERAL: 
					if r /= v1 ^ v2 then
						error_handler.append("Internal compiler error. Definition of infix %"^%" of %
													%INTEGER_GENERAL is not coherent with compiler builtin %
												   %simplifications.")
						error_handler.add_position(call_site)
						error_handler.print_as_fatal_error
					end
					create {INTEGER_CONSTANT} Result.make(r, call_site)
				end
			end
		end

feature {CODE_PRINTER}
	is_at_run_time (a_class_name: STRING): BOOLEAN is
			-- Pure query to know if the STRING type is used and is `at_run_time'?
			-- The class name must be the one of a basic type because the cluster information is not appended
		require
			is_ready
			string_aliaser.registered_one(a_class_name)
		local
			t: TYPE; hashed_string: HASHED_STRING; lt: LIVE_TYPE
		do
			hashed_string := string_aliaser.hashed_string(a_class_name)
			if hashed_string /= Void then
				t := type_dictionary.fast_reference_at(hashed_string)
				if t /= Void then
					lt := t.live_type
				end
				if lt /= Void then
					Result := lt.at_run_time
				end
			end
		end

feature {}
	echo_magic_count (msg: STRING) is
		require
			msg /= Void
		do
			echo.put_string(msg)
			echo.put_string(once " (magic_count=")
			echo.put_integer(magic_count)
			echo.put_string(once ").%N")
		end

	get_started (root_class_name, root_procedure_name: STRING) is
			-- Get started to compile using creation procedure
			-- `root_procedure_name' of class text `root_class_name'.
		require
			not root_class_name.is_empty
			not root_procedure_name.is_empty
		local
			root_fn: FEATURE_NAME; root: CLASS_TEXT; root_type: TYPE; root_name: CLASS_NAME
			hashed_root_class_name: HASHED_STRING; i: INTEGER; lt: LIVE_TYPE
			fs: FEATURE_STAMP; af: ANONYMOUS_FEATURE
		do
			hashed_root_class_name := string_aliaser.hashed_string(root_class_name)
			if ace.no_check then
				set_generator_used
				set_generating_type_used
			end
			ace.parse_include
			create root_name.unknown_position(hashed_root_class_name)
			root := class_text(root_name, True)
			if root = Void then
				error_handler.append("Cannot load root class ")
				error_handler.append(root_class_name)
				error_handler.append(once ". ")
				error_handler.print_as_error
			elseif root.is_expanded then
				error_handler.add_position(root.name.start_position)
				error_handler.append("The root class must not be expanded (sorry, but this is a limitation of the compiler).")
				error_handler.print_as_fatal_error
			elseif root.is_generic then
				error_handler.append(root.name.to_string)
				error_handler.append(" cannot be a root class since it is a generic class.")
				error_handler.print_as_fatal_error
			else
				root_fn := root.root_creation_search(root_procedure_name)
				root_type := root.declaration_type_of_like_current
				if root_type.is_deferred then
					error_handler.append(root.name.to_string)
					error_handler.append(" cannot be a root class since it is a deferred class.")
					error_handler.print_as_warning
				end
				fs := root_type.lookup(root_fn)
				af := fs.anonymous_feature(root_type)
				if af.arguments /= Void then
					error_handler.add_position(af.start_position)
					error_handler.append("The main procedure must not have arguments.")
					error_handler.print_as_fatal_error
				end
			end
			if nb_errors = 0 then
				if cecil_pool /= Void then
					cecil_pool.parse_cecil_files
				end
				collect_from_root(root_type, root_type.lookup(root_fn))
			end
			ace.generic_formal_arguments_check
			if nb_errors = 0 and then not pretty_flag and then not short_or_class_check_flag then
				optimize(root_type, root_type.lookup(root_fn))
			end
			if nb_errors = 0 then
				status.set_adapting
				from
					i := live_type_map.upper
				until
					i < live_type_map.lower
				loop
					live_type_map.item(i).make_run_features
					i := i - 1
				end
				if nb_errors = 0 then
					from
						i := live_type_map.upper
					until
						i < live_type_map.lower
					loop
						lt := live_type_map.item(i)
						lt.adapt_run_features_and_class_invariant
						lt.make_switch_sites
						i := i - 1
					end
					if nb_errors = 0 then
						--|*** PH(03/03/04) Next lines may be removed I think.
						root_procedure ?= root_type.live_type.at(root_fn)
						check
							root_is_run_feature_3: root_procedure /= Void
						end
					end
				end
			end
			if nb_errors = 0 then
				safety_check
				echo.print_count(once "Loaded Classe", ace.class_text_count)
			end
		ensure
			nb_errors = 0 implies root_procedure /= Void
		end

	collect_from_root (root_type: TYPE; root_feature: FEATURE_STAMP) is
		local
			string_type, tmp: TYPE; i, magic: INTEGER
		do
			status.set_collecting
			assignment_handler.reset
			manifest_string_pool.reset
			manifest_generic_pool.reset
			once_routine_pool.reset
			switch_collection.reset
			agent_pool.reset
			assignment_test_pool.reset
			--|*** (PR 30/05/04) Reset is not yet complete. Should be
			--|*** reseted: switches (waiting they are replaced by
			--|*** inspects), manifest [unicode] strings,
			--|*** and maybe some other pools such as tuples, ...
			se_atexit_stamp := Void
			from
				i := live_type_map.upper
			until
				i < live_type_map.lower
			loop
				live_type_map.item(i).forget_previous_collect
				i := i - 1
			end
			echo_magic_count(once "Starting collect")
			-- Collect all the live features:
			if ace.no_check then
				-- STRINGs are needed for runtime support:
				string_type := type_string
				manifest_string_pool.collect_string(string_type)
			end
			from
				magic := -1 -- Iterate at least once
				-- Mark the root procedure as alive:
				tmp := collect(root_type, root_feature, True)
				check
					tmp = Void
				end
				-- Also mark possible -cecil features as alive:
				if cecil_pool /= Void then
					cecil_pool.collect
				end
			until
				magic = magic_count
			loop
				magic := magic_count
				echo_magic_count(once "Before collect cycle")
				do_one_collect_cycle
			end
			if deep_twin_used then
				collect_deep_features
			end
			status.set_collecting_done
			echo_magic_count(once "Collecting done")
			echo.put_string(once "Live_type_map size=")
			echo.put_integer(live_type_map.count)
			echo.put_new_line
		ensure
			status.collecting_done
		end

	do_one_collect_cycle is
		local
			i: INTEGER
		do
			from
				i := live_type_map.upper
			until
				i < live_type_map.lower
			loop
				live_type_map.item(i).propagate_features
				i := i - 1
			end
			from
				i := live_type_map.lower
			until
				i > live_type_map.upper
			loop
				live_type_map.item(i).do_collect
				i := i + 1 -- We go this way to collect new incoming live-types
			end
			assignment_handler.recompute_all_run_time_sets
		end

	collect_deep_features is
		require
			deep_twin_used
		local
			i: INTEGER
		do
			from
				i := live_type_map.upper
			until
				i < live_type_map.lower
			loop
				live_type_map.item(i).do_collect_is_deep_equal
				i := i - 1
			end
		end

	safety_check is
			-- Start final whole system analysis to decide whether this
			-- system is safe or not.
		require
			status.is_adapting
		local
			old_magic_count, i: INTEGER; do_it: BOOLEAN; lt: LIVE_TYPE
		do
			debug
				old_magic_count := magic_count
			end
			do_it := ace.safety_check
			status.set_safety_checking
			if do_it then
				echo_magic_count(once "Starting type safety check")
				from
					i := live_type_map.upper
				until
					i < live_type_map.lower
				loop
					lt := live_type_map.item(i)
					if lt.at_run_time then
						lt.safety_check
					end
					i := i - 1
				end
				status.set_safety_check_ok
			end
			debug
				check
					old_magic_count = magic_count
				end
			end
		ensure
			status.is_safety_checking
			ace.safety_check implies status.safety_check_done
		end

	optimize (root_type: TYPE; root_feature: FEATURE_STAMP) is
		require
			not pretty_flag
			not short_or_class_check_flag
			status.collecting_done
		local
			prev_magic_count: INTEGER
		do
			if ace.boost then
				echo.put_string(once "Starting optimization%N")
				if ace.boost1 then
					simplify
					collect_from_root(root_type, root_feature)
				elseif ace.boost2 then
					from
					until
						prev_magic_count = magic_count
					loop
						simplify
						prev_magic_count := magic_count
						--|*** save run_time_sets
						--|*** full collect collect
						--|*** if run_time_sets = old run_time_sets then prev_magic_count := magic_count
					end
				else
					from
					until
						prev_magic_count = magic_count
					loop
						contextual_simplify
						prev_magic_count := magic_count
						--|*** save run_time_sets
						--|*** full collect collect
						--|*** if run_time_sets = old run_time_sets then prev_magic_count := magic_count
					end
				end
				echo.put_string(once "Finished optimization (")
				echo.put_integer(magic_count)
				echo.put_string(once ")]%N")
			end
		end

	set_simplify_done is
		do
			simplify_done := True
		ensure
			simplify_done
		end
	
	simplify is
		require
			not pretty_flag
			not short_or_class_check_flag
			status.collecting_done
			ace.boost1 or ace.boost2
		local
			lt: LIVE_TYPE; i: INTEGER; max_loop, prev_magic_count: INTEGER
		do
			echo_magic_count(once "Starting simplify")
			from
				max_loop := 50 -- (Is actually more than necessary, but avoids infinite inlinings.)
			until
				max_loop = 0 or else prev_magic_count = magic_count
			loop
				prev_magic_count := magic_count
				echo_magic_count(once "Before simplify cycle")
				from
					i := live_type_map.upper
				until
					i < 0
				loop
					lt := live_type_map.item(i)
					lt.simplify
					i := i - 1
				end
				max_loop := max_loop - 1
			end
			if max_loop = 0 then
				echo_magic_count(once "Symplify interrupted (infinite inlining ... or infinite recursion detected.).")
			end
			echo_magic_count(once "Symplify done")
			set_simplify_done
		end

	contextual_simplify is
		require
			not pretty_flag
			not short_or_class_check_flag
			status.collecting_done
			ace.boost3
		local
			lt: LIVE_TYPE; i: INTEGER; prev_magic_count: INTEGER
		do
			echo.put_string(once "[Starting context_simplify (")
			echo.put_integer(magic_count)
			echo.put_string(once "): ")
			from
			until
				prev_magic_count = magic_count
			loop
				prev_magic_count := magic_count
				echo.put_character('.')
				from
					i := live_type_map.upper
				until
					i < 0
				loop
					lt := live_type_map.item(i)
					lt.contextual_simplify
					i := i - 1
				end
			end
			echo.put_string(once " finished context_simplify (")
			echo.put_integer(magic_count)
			echo.put_string(once ")]%N")
			set_simplify_done
		end

end -- class SMART_EIFFEL
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
