/***************************************************************************
    smb4kauthinfo.cpp  -  This class provides a container for the
    authentication data.
                             -------------------
    begin                : Sa Feb 28 2004
    copyright            : (C) 2004-2009 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// application specific includes
#include <smb4kauthinfo.h>


Smb4KAuthInfo::Smb4KAuthInfo( const Smb4KHost *host )
: m_type( Host ), m_workgroup( host->workgroup() ), m_host_name( host->name() ),
  m_share_name( QString() ), m_homes_share( false ), m_login( QByteArray() ),
  m_password( QByteArray() ), m_homes_users( QStringList() )
{
}


Smb4KAuthInfo::Smb4KAuthInfo( const Smb4KShare *share )
: m_type( Share ), m_workgroup( share->workgroup() ), m_host_name( share->host() ),
  m_share_name( share->name() ), m_homes_share( share->isHomesShare() ),
  m_login( QByteArray() ), m_password( QByteArray() ), m_homes_users( share->homesUsers() )
{
}


Smb4KAuthInfo::Smb4KAuthInfo()
: m_type( Unknown ), m_workgroup( QString() ), m_host_name( QString() ),
  m_share_name( QString() ), m_homes_share( false ), m_login( QByteArray() ),
  m_password( QByteArray() ), m_homes_users( QStringList() )
{
}


Smb4KAuthInfo::Smb4KAuthInfo( Smb4KAuthInfo &i )
: m_type( i.type() ), m_workgroup( i.workgroupName() ), m_host_name( i.hostName() ),
  m_share_name( i.shareName() ), m_homes_share( i.isHomesShare() ), m_login( i.login() ),
  m_password( i.password() ), m_homes_users( i.homesUsers() )
{
}


Smb4KAuthInfo::~Smb4KAuthInfo()
{
}


void Smb4KAuthInfo::setHost( Smb4KHost *host )
{
  Q_ASSERT( host );

  m_type        = Host;
  m_workgroup   = host->workgroup();
  m_host_name   = host->name();
  m_share_name  = QString();
  m_homes_share = false;
  m_login       = QByteArray();
  m_password    = QByteArray();
  m_homes_users = QStringList();
}


void Smb4KAuthInfo::setShare( Smb4KShare *share )
{
  Q_ASSERT( share );

  m_type        = Share;
  m_workgroup   = share->workgroup();
  m_host_name   = share->host();
  m_share_name  = share->name();
  m_homes_share = share->isHomesShare();
  m_login       = QByteArray();
  m_password    = QByteArray();
  m_homes_users = share->homesUsers();
}


void Smb4KAuthInfo::setWorkgroupName( const QString &workgroup )
{
  m_workgroup = workgroup;
}


void Smb4KAuthInfo::setUNC( const QString &unc )
{
  // Check that a valid UNC was passed to this function.
  if ( !unc.startsWith( "//" ) && !unc.startsWith( "smb:" ) )
  {
    // The UNC is malformatted.
    return;
  }
  else
  {
    // Do nothing
  }

  if ( unc.count( "/" ) == 3 )
  {
    // Set the type
    m_type = Share;

    // Extract the login, if available.
    if ( unc.contains( "@" ) )
    {
      m_login = unc.section( "/", 2, 2 ).section( "@", 0, 0 ).trimmed().toLocal8Bit();
    }
    else
    {
      // Do nothing
    }

    // Extract the host name.
    if ( unc.contains( "@" ) )
    {
      m_host_name = unc.section( "@", 1, 1 ).section( "/", 0, 0 ).trimmed();
    }
    else
    {
      m_host_name = unc.section( "/", 2, 2 ).trimmed();
    }

    // If there a port is given, dump it. We do not need it here.
    if ( m_host_name.contains( ":" ) )
    {
      m_host_name = m_host_name.section( ":", 0, 0 ).trimmed();
    }
    else
    {
      // Do nothing
    }

    // Extract the share name.
    m_share_name = unc.section( "/", 3, 3 ).trimmed();

    // Determine whether this is a homes share.
    m_homes_share = (QString::compare( m_share_name, "homes", Qt::CaseSensitive ) == 0);
  }
  else if ( unc.count( "/" ) == 2 )
  {
    m_type = Host;

    // Extract the login, if available.
    if ( unc.contains( "@" ) )
    {
      m_login = unc.section( "/", 2, 2 ).section( "@", 0, 0 ).trimmed().toLocal8Bit();
    }
    else
    {
      // Do nothing
    }

    // Extract the host name.
    if ( unc.contains( "@" ) )
    {
      m_host_name = unc.section( "@", 1, 1 ).trimmed();
    }
    else
    {
      m_host_name = unc.section( "/", 2, 2 ).trimmed();
    }

    // If there a port is given, dump it. We do not need it here.
    if ( m_host_name.contains( ":" ) )
    {
      m_host_name = m_host_name.section( ":", 0, 0 ).trimmed();
    }
    else
    {
      // Do nothing
    }
  }
  else
  {
    m_type        = Unknown;
    m_host_name   = QString();
    m_share_name  = QString();
    m_homes_share = false;
  }
}


QString Smb4KAuthInfo::unc( int format ) const
{
  // We use a string list to assemble the UNC and start with the
  // minimal UNC (//HOST/SHARE). The entries of the list are as
  // follows:
  // 0 - smb:
  // 1 - //
  // 2 - USER@ (if wanted; mandatory under FreeBSD)
  // 3 - HOST
  // 4 - /
  // 5 - SHARE
  QStringList unc_list;

  switch ( m_type )
  {
    case Host:
    {
      unc_list << "";
      unc_list << "//";
      unc_list << "";
      unc_list << m_host_name;
      break;
    }
    case Share:
    {
      unc_list << "";
      unc_list << "//";
      unc_list << "";
      unc_list << m_host_name;
      unc_list << "/";
      unc_list << m_share_name;
      break;
    }
    default:
    {
      return QString();
    }
  }

  if ( (format & UNCWithLogin) )
  {
    if ( !m_login.isEmpty() )
    {
      unc_list[2] = QString::fromLocal8Bit( m_login )+"@";
    }
    else
    {
      // Do nothing
    }
  }
  else
  {
    // Do nothing
  }

  if ( (format & UNCMinimal) )
  {
    // Remove user
    unc_list[2] = "";
  }
  else
  {
    // Do nothing
  }

  if ( (format & UNCHostOnly) )
  {
    // Remove the share entry and the slash.
    unc_list[4] = "";
    unc_list[5] = "";
  }
  else
  {
    // Do nothing
  }

  if ( (format & UNCWithProtocol) )
  {
    unc_list[0] = "smb:";
  }
  else
  {
    // Do nothing
  }

  (void) unc_list.removeAll( "" );

  return unc_list.join( "" );
}


void Smb4KAuthInfo::setLogin( const QString &login )
{
  m_login = login.toLocal8Bit();
}


void Smb4KAuthInfo::setPassword( const QString &passwd )
{
  m_password = passwd.toLocal8Bit();
}


void Smb4KAuthInfo::setHomesUser( const QString &user )
{
  if ( m_homes_share )
  {
    m_share_name = user;
  }
  else
  {
    // Do nothing
  }
}


void Smb4KAuthInfo::setDefaultAuthInfo()
{
  m_type = Default;
}


void Smb4KAuthInfo::setHomesUsers( const QStringList &users )
{
  if ( m_homes_share )
  {
    m_homes_users = users;
  }
  else
  {
    // Do nothing
  }
}
