/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt includes
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

// KDE includes
#include <kmessagebox.h>
#include <klocale.h>
#include <kapplication.h>

// system includes
#include <stdlib.h>
#include <unistd.h>

// application specific includes
#include "smb4kfileio.h"
#include "smb4kdefs.h"


Smb4KFileIO::Smb4KFileIO( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_config->setGroup( "System" );
  m_uid = m_config->readNumEntry( "UID" );
  m_gid = m_config->readNumEntry( "GID" );

  m_proc = new KProcess( this, "FileIOProcess" );
  m_proc->setUseShell( true );
  
  m_state = Idle;

  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ), SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ), SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KFileIO::~Smb4KFileIO()
{
}


/****************************************************************************
   Writes entries to the super.tab file.
****************************************************************************/

bool Smb4KFileIO::writeSuperUserEntries( bool force, bool useSuper, bool overwrite )
{
  if ( !QFile::exists( "/tmp/smb4k.lock" ) || overwrite )
  {
    // First generate the lock file.
    QFile file( "smb4k.lock" );
    QDir::setCurrent( "/tmp" );
    if ( file.open( IO_WriteOnly ) )
    {
      file.putch( 0 );
      file.close();
    }

    // There are two known locations for the super.tab file.
    // Linux: /etc/super.tab
    // FreeBSD: /usr/local/super.tab.    
    // Search those paths for the file.
    
    file.setName( "super.tab" );
    
    if ( QFile::exists( "/etc/super.tab" ) )
      QDir::setCurrent( "/etc" );
    else if ( QFile::exists( "/usr/local/etc/super.tab" ) )
      QDir::setCurrent( "/usr/local/etc" );
      
    if ( file.exists() )
    {
      // Open super.tab and read it.
      QStringList contents;

      if ( file.open( IO_ReadOnly ) )
      {
        QTextStream ts( &file );

        while ( !ts.atEnd() )
          contents.append( ts.readLine() );

        file.close();
      }

      // We want to determine whether the super.tab already contains entries made
      // by Smb4K or not.
      if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 )
      {
        // Let's check, if there are already entries, that are similar to the ones, that will be
        // entered.

#ifndef __FreeBSD__
        if ( contents.grep( "umount" ).count() == 0 && contents.grep( "mount" ).count() == 0 && contents.grep( "kill" ).count() == 0 )
#else
        if ( contents.grep( "umount" ).count() == 0 && contents.grep( "mount_smbfs" ).count() == 0 && contents.grep( "kill" ).count() == 0 )
#endif         
        {
          // Now write the new entry.
          contents.append( "# Entries for Smb4K users." );
          contents.append( "# Generated by Smb4K. Please do not modify!" );

#ifndef __FreeBSD__
          // Force unmounting of shares.
          if ( force && !useSuper )
          {
            contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "umount\t/bin/umount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
          }
          // Mounting and unmounting of shares using super user rights.
          else if ( useSuper || ( force && useSuper ) )
          {
            contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "umount\t/bin/umount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "mount\t/bin/mount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
          }
#else
          // Force unmounting of shares.
          if ( force && !useSuper )
          {
            contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "umount\t/sbin/umount\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
          }
          // Mounting and unmounting of shares using super user rights.
          else if ( useSuper || ( force && useSuper ) )
          {
            contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "umount\t/sbin/umount\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
            contents.append( QString( "mount_smbfs\t/sbin/mount_smbfs\t%1\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME" ).arg( getenv( "USER" ) ) );
          }
#endif

          contents.append( "# End of Smb4K user entries." );

          // Now write the data to the temp file.
          file.setName( "super.tab.smb4k" );
          QDir::setCurrent( "/tmp" );

          if ( file.open( IO_WriteOnly ) )
          {
            QTextStream ts( &file );

            for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
            {
              ts << *it << endl;
            }
            
            file.close();
          }

          // We are ready to move the temp file to its new position. Let's use normal shell commands
          // and KDESU to gain root privileges.
          m_state = writeSU;

#ifndef __FreeBSD__          
          QString command( "chown root:root /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /etc/super.tab" );
#else
          QString command( "chown root:wheel /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /usr/local/etc/super.tab" );
#endif          
          
          m_proc->operator<<( QString( "kdesu %1" ).arg( KProcess::quote( command ) ) );
          m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
        }
        else
        {
          // Smb4K won't modify anything automatically, if there are custom entries, that might collide with those,
          // that Smb4K would write. Emit an error signal. The user can overwrite these entries by telling Smb4K to
          // do so.
          // REMARK: The case that the checkbox m_mount is check alone is impossible, thus it is not in the if query.
          QStringList entries;

          if ( useSuper || ( force && useSuper ) )
          {
            entries = contents.grep( "umount" ).grep( getenv( "USER" ) );
#ifndef __FreeBSD__
            entries.operator+=( contents.grep( "mount" ).grep( getenv( "USER" ) ) );
#else
            entries.operator+=( contents.grep( "mount_smbfs" ).grep( getenv( "USER" ) ) );
#endif            
            entries.operator+=( contents.grep( "kill" ).grep( getenv( "USER" ) ) );
          }
          else if ( force )
          {
            entries = contents.grep( "umount" ).grep( getenv( "USER" ) );
            entries.operator+=( contents.grep( "kill" ).grep( getenv( "USER" ) ) );
          }
          
          // Remove all duplicates.
          uint index( 0 );
          while( index < entries.count() )
          {
            if ( entries.grep( entries[index] ).count() > 1 )
              entries[index] = QString::null;

            index++;
          }

          entries.remove( QString::null );

          if ( !overwrite && entries.count() != 0 )
          {
            if ( KMessageBox::questionYesNoList( (QWidget *)this, i18n( "The file super.tab contains entries, that might conflict with those, that are needed by Smb4K. They are listed below for you to check. You should overwrite them to enable Smb4K to work properly.\nDo you want to overwrite them now?" ), entries ) == KMessageBox::Yes )            
            {
              writeSuperUserEntries( force, useSuper, true );
            }
            else
            {
              emit noSuper();
            }          
          }
          else if ( overwrite || ( entries.count() == 0 && !overwrite ) )
          {
            // First remove the lines, that might conflict with Smb4K's entries from the string list.
            for ( QStringList::ConstIterator it = entries.begin(); it != entries.end(); ++it )
              contents.remove( *it );

            // Now write the new entries.
            contents.append( "# Entries for Smb4K users." );
            contents.append( "# Generated by Smb4K. Please do not modify!" );

            // Force unmounting of shares.
#ifndef __FreeBSD__
            if ( force && !useSuper )
            {
              contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "umount\t/bin/umount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
            }
            // Mounting and unmounting of shares using super user rights.
            else if ( useSuper || ( force && useSuper ) )
            {
              contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "umount\t/bin/umount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "mount\t/bin/mount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) ) );
            }
#else
            if ( force && !useSuper )
            {
              contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "umount\t/sbin/umount\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
            }
            // Mounting and unmounting of shares using super user rights.
            else if ( useSuper || ( force && useSuper ) )
            {
              contents.append( QString( "kill\t/bin/kill\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "umount\t/sbin/umount\t%1\tuid=root\tgid=wheel" ).arg( getenv( "USER" ) ) );
              contents.append( QString( "mount_smbfs\t/sbin/mount_smbfs\t%1\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME" ).arg( getenv( "USER" ) ) );
            }
#endif
            
            contents.append( "# End of Smb4K user entries." );

            // Now write the data to the temp file.
            file.setName( "super.tab.smb4k" );
            QDir::setCurrent( "/tmp" );

            if ( file.open( IO_WriteOnly ) )
            {
              QTextStream ts( &file );

              for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
              {
                ts << *it << endl;
              }              

              file.close();
            }

            // We are ready to move the temp file to its new position. Let's use normal shell commands
            // and KDESU to gain root privileges.
            m_state = writeSU;

#ifndef __FreeBSD__
            QString command( "chown root:root /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /etc/super.tab" );
#else
            QString command( "chown root:wheel /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /usr/local/etc/super.tab" );
#endif
            m_proc->operator<<( QString( "kdesu %1" ).arg( KProcess::quote( command ) ) );
            m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
          }
        }
      }
      else
      {
        bool write = true;
        int entries = 0;

        // First we need to find the Smb4K section. If there are already some entries, they will be checked
        // whether the user is already in the list. If not he/she is added.
        for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
        {
          if ( (*it).stripWhiteSpace().startsWith( "kill" ) )
          {
            // The user section is needed.
            if ( (*it).section( "\t", 2, 2 ).contains( getenv( "USER" ), true ) == 0 )
            {
              (*it).replace( (*it).section( "\t", 2, 2 ), (*it).section( "\t", 2, 2 ).append( QString( ",%1" ).arg( getenv( "USER" ) ) ) );
              entries++;
            }
            else
            {
              entries++;
              write = false;
            }
          }
          else if ( (*it).stripWhiteSpace().startsWith( "umount" ) )
          {
            // The user section is needed.
            if ( (*it).section( "\t", 2, 2 ).contains( getenv( "USER" ), true ) == 0 )
            {
              (*it).replace( (*it).section( "\t", 2, 2 ), (*it).section( "\t", 2, 2 ).append( QString( ",%1" ).arg( getenv( "USER" ) ) ) );
              entries++;
            }
            else
            {
              entries++;
              write = false;
            }
          }
#ifndef __FreeBSD__
          if ( (*it).stripWhiteSpace().startsWith( "mount" ) )
#else
          if ( (*it).stripWhiteSpace().startsWith( "mount_smbfs" ) )
#endif           
          {
            // The user section is needed.
            if ( (*it).section( "\t", 2, 2 ).contains( getenv( "USER" ), true ) == 0 )
            {
              (*it).replace( (*it).section( "\t", 2, 2 ), (*it).section( "\t", 2, 2 ).append( QString( ",%1" ).arg( getenv( "USER" ) ) ) );
              entries++;
            }
            else
            {
              entries++;
              write = false;
            }
          }
          else if ( (*it).contains( "# End of Smb4K user entries." ) )
            break;
        }

        // Now we have to check whether all entries are in the list, that should be there. If there are
        // less than 3 entries, only the 'mount' entry can be missing.
        if ( entries < 3 && useSuper )
        {
          QStringList::Iterator it = contents.find( "# End of Smb4K user entries." );
#ifndef __FreeBSD__          
          QString entry = QString( "mount\t/bin/mount\t%1\tuid=root\tgid=root" ).arg( getenv( "USER" ) );
#else
          QString entry = QString( "mount_smbfs\t/sbin/mount_smbfs\t%1\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME" ).arg( getenv( "USER" ) );
#endif
          contents.insert( it, entry );
          write = true;
        }

        if ( write )
        {
          // Now write the data to the temp file.
          file.setName( "super.tab.smb4k" );
          QDir::setCurrent( "/tmp" );

          if ( file.open( IO_WriteOnly ) )
          {
            QTextStream ts( &file );

            for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
            {
              ts << *it << endl;
            }

            file.close();
          }

          // We are ready to move the temp file to its new position. Let's use normal shell commands
          // and KDESU to gain root privileges.
          m_state = writeSU;

#ifndef __FreeBSD__
          QString command( "chown root:root /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /etc/super.tab" );
#else
          QString command( "chown root:wheel /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /usr/local/etc/super.tab" );
#endif
          
          m_proc->operator<<( QString( "kdesu %1" ).arg( KProcess::quote( command ) ) );
          m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
        }
        else
          emit doneSUPWriting();
      }
    }
    else
    {
      emit error( ERROR_FILE_NOT_FOUND, "super.tab" );
      emit noSuper();
    }

    // Remove the lock file.
    QFile::remove( "/tmp/smb4k.lock" );

    return true;
  }
  else
  {
    emit error( ERROR_LOCKED, QString::null );
    return false;
  }
}


/****************************************************************************
   Removes all Smb4K specific entries of the user from /etc/super.tab.
****************************************************************************/

bool Smb4KFileIO::removeSuperUserEntries()
{
  if ( !QFile::exists( "/tmp/smb4k.lock" ) )
  {
    // First generate the lock file.
    QFile file( "smb4k.lock" );
    QDir::setCurrent( "/tmp" );
    if ( file.open( IO_WriteOnly ) )
    {
      file.putch( 0 );
      file.close();
    }

    // Now turn to the super.tab file.
    file.setName( "super.tab" );
    
    if ( QFile::exists( "/etc/super.tab" ) )
    {
      QDir::setCurrent( "/etc" );
    }
    else if ( QFile::exists( "/usr/local/etc/super.tab" ) )
    {
      QDir::setCurrent( "/usr/local/etc" );
    }
    
    // First locate the super.tab file. Normally it is located in
    // /etc/super.tab
    if ( file.exists() )
    {
      // Open super.tab and read it.
      QStringList contents;

      if ( file.open( IO_ReadOnly ) )
      {
        QTextStream ts( &file );

        while ( !ts.atEnd() )
          contents.append( ts.readLine() );

        file.close();
      }

      // Remove the entries of the user from the /etc/super.tab.
      if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 )
      {
        for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
        {
#ifndef __FreeBSD__          
          if ( (*it).startsWith( "kill", true ) || (*it).startsWith( "umount", true ) || (*it).startsWith( "mount", true ) )
#else
          if ( (*it).startsWith( "kill", true ) || (*it).startsWith( "umount", true ) || (*it).startsWith( "mount_smbfs", true ) )
#endif
          {
            QString test_string = (*it).section( "\t", 2, 2 );

            if ( test_string.contains( ",", true ) == 0 )
            {
              (*it) = QString::null;
            }
            else if ( test_string.contains( ",", true ) != 0 )
            {
              int pos = (*it).find( getenv( "USER" ) );
              
              if ( (*it).at( pos-1 ) == ',' )
                (*it).remove( QString( ",%1" ).arg( getenv( "USER" ) ) );
              else
                (*it).remove( QString( "%1," ).arg( getenv( "USER" ) ) );
            }
          }
          else if ( (*it).contains( "# End of Smb4K user entries." ) )
            break;
        }

        // Remove the null strings from the list.
        contents.remove( QString::null );

        // Post-processing.
        int lines = 0;

        for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
        {
          if ( (*it).contains( "# End of Smb4K user entries." ) )
          {
            lines++;
            break;
          }
          else
            lines++;
        }

        if ( lines == 3 )
        {
          contents.remove( contents.find( "# Entries for Smb4K users." ) );
          contents.remove( contents.find( "# Generated by Smb4K. Please do not modify!" ) );
          contents.remove( contents.find( "# End of Smb4K user entries." ) );
        }

        // Now write the data to the temp file.
        file.setName( "super.tab.smb4k" );
        QDir::setCurrent( "/tmp" );

        if ( file.open( IO_WriteOnly ) )
        {
          QTextStream ts( &file );

          uint index( 0 );
          while ( index < contents.count() )
            ts << contents[index++] << endl;

          file.close();
        }

        // We are ready to move the temp file to its new position. Let's use normal shell commands
        // and KDESU to gain root privileges.
#ifndef __FreeBSD__
        QString command( "chown root:root /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /etc/super.tab" );
#else
        QString command( "chown root:wheel /tmp/super.tab.smb4k && chmod go-wx /tmp/super.tab.smb4k && mv /tmp/super.tab.smb4k /usr/local/etc/super.tab" );
#endif

        m_proc->operator<<( QString( "kdesu %1" ).arg( KProcess::quote( command ) ) );
        m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
      }
    }
    else
    {
      emit error( ERROR_FILE_NOT_FOUND, "super.tab" );
      emit noSuper();
    }

    QFile::remove( "/tmp/smb4k.lock" );

    return true;
  }
  else
  {
    emit error( ERROR_LOCKED, QString::null );

    return false;
  }
}


/****************************************************************************
   Extracts the non-system users from the /etc/passwd and /etc/group
****************************************************************************/


const QValueList<Smb4KUser *> Smb4KFileIO::getUsers()
{
  QFile file( "passwd" );
  QDir::setCurrent( "/etc" );

  QStringList list;

  if ( QFile::exists( "/etc/passwd" ) )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );

      while ( !ts.atEnd() )
        list.append( ts.readLine() );

      file.close();
    }
  }
  else
    emit error( ERROR_FILE_NOT_FOUND, "/etc/passwd" );

  QValueList<Smb4KUser *> users;
      
  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    if ( !(*it).stripWhiteSpace().startsWith( "#" ) && ( (*it).section( ":", 2, 2 ).toInt() >= 500 || (*it).section( ":", 2, 2 ).toInt() == m_uid ) )
    {
      users.append( new Smb4KUser( (*it).section( ":", 2, 2 ).toInt(), (*it).section( ":", 3, 3 ).toInt() ) );    
    }
    else
      continue;
  }
  
  return users;
}


/****************************************************************************
   Detects all avaiable (global) options in the samba.conf
****************************************************************************/

const QMap<QString, QString> Smb4KFileIO::getSMBOptions()
{
  QMap<QString, QString> map;
  QFile file( "smb.conf" );

  if ( QFile::exists( "/etc/smb.conf" ) )
    QDir::setCurrent( "/etc" );
  else if ( QFile::exists( "/etc/samba/smb.conf" ) )
    QDir::setCurrent( "/etc/samba" );
  else if ( QFile::exists( "/usr/local/etc/smb.conf" ) )
    QDir::setCurrent( "/usr/local/etc" );
  else if ( QFile::exists( "/usr/local/etc/samba/smb.conf" ) )
    QDir::setCurrent( "/usr/local/etc/samba" );
  else
    emit error( ERROR_FILE_NOT_FOUND, "smb.conf" );


  QStringList list;

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );

    while ( !ts.atEnd() )
      list.append( ts.readLine().stripWhiteSpace() );

    file.close();
  }

  // Clean the list from everything that is not needed. Especially all
  // non-global sections will be removed.
  // REMARK: We assume that the global entries are not scattered through
  // the whole config file but are written en bloc.
  for ( QStringList::Iterator it = list.erase( list.begin(), ++(list.find( "[global]" )) ); it != list.end(); ++it )
  {
    if ( (*it).startsWith( "#" ) )
      *it = QString::null;
    else if ( (*it).startsWith( "[" ) )
    {
      list.erase( it, list.end() );
      break;
    }
  }

  list.remove( QString::null );

  // Write all options into the map:
  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    QString key = (*it).section( "=", 0, 0 ).stripWhiteSpace().lower();
    map[ key ] = QString( (*it).section( "=", 1, 1 ).stripWhiteSpace().upper() );
  }

  // Post-processing. Some values should be entered with their defaults, if they are
  // not already present.
  if ( !map.contains( "netbios name" ) )
  {
    char *hostname;
    size_t hostnamelen = 255;
    hostname = (char *)malloc( hostnamelen );

    if ( gethostname( hostname, hostnamelen ) == 0 )
      map[ "netbios name" ] = ( QString( "%1" ).arg( hostname ) ).upper();
    else
      emit error( ERROR_GETTING_HOSTNAME, QString::null );

    free(hostname);
  }
  
  return map;
}


/****************************************************************************
   Returns the paper size defined on the system.
****************************************************************************/

const QString Smb4KFileIO::getPaperSize()
{
  QString ps;
  
  if ( QFile::exists( "/etc/papersize" ) )
  {
    QFile file( "papersize" );
    QDir::setCurrent( "/etc" );
    QStringList temp;
    
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );

      while ( !ts.atEnd() )
        temp.append( ts.readLine().stripWhiteSpace() );

      file.close();
    }
    
    ps = temp.join( " " ).stripWhiteSpace();
  }
  else
    ps = "a4";
    
  return ps;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KFileIO::slotReceivedStderr( KProcess *, char *buf, int len )
{
  QString buffer( QString::fromLocal8Bit( buf, len ) );
  
  if ( buffer.contains( "KIconLoader" ) == 0 )
    emit error( ERROR_UNKNOWN, buffer );
}


void Smb4KFileIO::slotProcessExited( KProcess * )
{
  if ( m_state == writeSU )
    emit doneSUPWriting();

  m_state = Idle;
  m_proc->clearArguments();
}

#include "smb4kfileio.moc"
