/***************************************************************************
    smb4ksharewidget.cpp  -  The widget, that displays the mounted shares.
                             -------------------
    begin                : Sam Mr 1 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qregexp.h>

// KDE includes
#include <klocale.h>
#include <kprocess.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kapplication.h>

// application specific includes
#include "smb4ksharewidget.h"
#include "../smb4k.h"


Smb4KShareWidget::Smb4KShareWidget( QWidget *parent, const char *name ) : KIconView( parent, name )
{
  m_config = kapp->config();

  // We do not need to put this into the readOptions() function since the
  // UID and GID will not change.
  m_config->setGroup( "System" );
  m_uid = m_config->readNumEntry( "UID" );
  m_gid = m_config->readNumEntry( "GID" );
  
  setFrameShape( KIconView::PopupPanel );
  setFrameShadow( KIconView::Sunken );
  setSelectionMode( KIconView::Single );
  setResizeMode( KIconView::Adjust );
  setShowToolTips( true );
  setAutoArrange( true );
  setSorting( true, true );
  
  m_read = false;

  // Internal connections
  connect( this, SIGNAL( mouseButtonPressed( int, QIconViewItem *, const QPoint & ) ), SLOT( slotMouseButtonPressed(  int, QIconViewItem *, const QPoint & ) ) );
  connect( this, SIGNAL( selectionChanged( QIconViewItem * ) ), SLOT( slotSelectionChanged( QIconViewItem * ) ) );
  connect( this, SIGNAL( executed( QIconViewItem * ) ), SLOT( slotOpenFilemanager() ) );

  // External connections
  connect( sapp->core()->mounter(), SIGNAL( importedShares() ), this,                   SLOT( slotMountedShares() ) );
  connect( sapp->core()->mounter(), SIGNAL( mountedShare(Smb4KShare *) ), this,         SLOT( slotMountedShare(Smb4KShare *) ) );
  connect( sapp->core()->mounter(), SIGNAL( unmountedShare(QString, QString) ), this,   SLOT( slotUnmountedShare() ) );
  connect( sapp->core()->mounter(), SIGNAL( unmountedAllShares() ), this,               SLOT( slotUnmountedAllShares() ) );
}


Smb4KShareWidget::~Smb4KShareWidget()
{
}

/****************************************************************************
   Reads the options for the share view widget.
****************************************************************************/

void Smb4KShareWidget::readOptions()
{
  // Lock the import function.
  m_read = true;

  // Which appearance does the user want?
  m_config->setGroup( "Appearance" );
  m_display = m_config->readEntry( "Show Shares", "icons" );
  
  if ( m_display == "icons" )
  {
    setItemTextPos( KIconView::Bottom );
    setMaxItemWidth( 150 );
    setArrangement( KIconView::LeftToRight );
    setWordWrapIconText( true );
  }
  else if ( m_display == "list" )
  {
    setItemTextPos( KIconView::Right );
    setMaxItemWidth( 250 );
    setArrangement( KIconView::TopToBottom );
    setWordWrapIconText( false );
  }

  m_mountpoint = m_config->readBoolEntry( "Show Mount Point", false );

  // Should all share be shown, that were mounted on the system?
  m_showExternal = m_config->readBoolEntry( "Show All Shares", false );

  // Now the items are going to be processed. First clear the whole icon view.
  clear();

  // Read the string list and put the items back in the icon view.
  QValueList<Smb4KShare *> shares = sapp->core()->mounter()->getShares();
  
  for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
  {
    QString share = (*it)->getName();
    QString path = (*it)->getPath();
    int share_uid = (*it)->getUID();
    int share_gid = (*it)->getGID();
    bool externalMount = false;

    if ( share_uid != m_uid && share_gid != m_gid )
      externalMount = true;

    // Put the items into the icon view.
    if ( m_showExternal )
    {
      if ( !m_mountpoint )
      {
        if ( m_display == "icons" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->name() );
        }
        else if ( m_display == "list" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->name() );
        }
      }
      else
      {
        if ( m_display == "icons" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->mountpoint() );
        }
        else if ( m_display == "list" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->mountpoint() );
        }
      }
    }
    else
    {
      if ( !m_mountpoint && !externalMount )
      {
        if ( m_display == "icons" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid,share_gid  );
          item->setText( item->name() );
        }
        else if ( m_display == "list" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->name() );
        }
      }
      else if ( m_mountpoint && !externalMount )
      {
        if ( m_display == "icons" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->mountpoint() );
        }
        else if ( m_display == "list" )
        {
          Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
          item->setText( item->mountpoint() );
        }
      }
    }
  }

  // Mount options.
  m_config->setGroup( "Mount Options" );

  if ( m_config->readBoolEntry( "Super" ) )
    m_forcedUnmount = m_config->readBoolEntry( "Force Unmount", false );
  else
    m_forcedUnmount = false;


  // Unlock the import function.
  m_read = false;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KShareWidget::slotMouseButtonPressed( int button, QIconViewItem *item, const QPoint &pos )
{
  // First build the menu:
  m_menu = new KPopupMenu( this );

  m_menu->insertTitle( SmallIcon( "hdd_mount" ), i18n( "Shares" ), Title, 0 );

  // Unmount current share.      
  QIconSet unmountSet( SmallIcon( "hdd_unmount" ), QIconSet::Automatic );
  m_menu->insertItem( unmountSet, i18n( "Unmount" ), this, SLOT( slotUnmountShare() ), CTRL+Key_U, Unmount, 1 );

  // Unmount all shares
  QIconSet unmountallSet( SmallIcon( "gear" ), QIconSet::Automatic );
  m_menu->insertItem( unmountallSet, i18n( "Unmount All" ), this, SLOT( slotUnmountAllShares() ), CTRL+Key_N, UnmountAll, 2 );

  // Force unmounting
  QIconSet forceUnmountSet( SmallIcon( "hdd_unmount" ), QIconSet::Automatic );
  m_menu->insertItem( forceUnmountSet, i18n( "Force Unmounting" ), this, SLOT( slotForceUnmountShare() ), CTRL+Key_F, ForcedUnmount, 3 );

  m_menu->insertSeparator( 4 );

  // Konqueror.
  QIconSet filemanagerSet( SmallIcon( "kfm_home" ), QIconSet::Automatic );
  m_menu->insertItem( filemanagerSet, i18n( "Konqueror" ), this, SLOT( slotOpenFilemanager() ), CTRL+Key_K, Konqueror, 5 );  
  
  if ( item )
  {
    // Enable/disable some actions:
    sapp->sharesUnmount->setEnabled( true );
    sapp->sharesStartFilemanager->setEnabled( true );
    
    if ( m_forcedUnmount )
      sapp->sharesForceUnmount->setEnabled( true );
        
    if ( button == KIconView::RightButton )
    {
      if ( this->count() != 0 )
      {
        m_menu->setItemEnabled( Unmount, true );
        m_menu->setItemEnabled( Konqueror, true );
        m_menu->setItemEnabled( UnmountAll, true );
        
        if ( m_forcedUnmount )
          m_menu->setItemEnabled( ForcedUnmount, true );
        else
          m_menu->setItemEnabled( ForcedUnmount, false );      
      }
      else
      {
        m_menu->setItemEnabled( Unmount, false );
        m_menu->setItemEnabled( Konqueror, false );
        m_menu->setItemEnabled( UnmountAll, false );
        m_menu->setItemEnabled( ForcedUnmount, false );
      } 
      
      m_menu->exec( pos, 0 );     
    }
  }
  else
  {
    sapp->sharesUnmount->setEnabled( false );
    sapp->sharesStartFilemanager->setEnabled( false );
    sapp->sharesForceUnmount->setEnabled( false );
  
    if ( button == KIconView::LeftButton )
    {
      sapp->view()->shareTab()->init( NULL );
      sapp->core()->shellIO()->getDiskUsage( QString::null );
      clearSelection();    
    }
    else if ( button == KIconView::RightButton )
    {
      if ( this->count() != 0 )
      {
        m_menu->setItemEnabled( Unmount, false );
        m_menu->setItemEnabled( Konqueror, false );
        m_menu->setItemEnabled( ForcedUnmount, false );      
      }
      else
      {
        m_menu->setItemEnabled( Unmount, false );
        m_menu->setItemEnabled( Konqueror, false );
        m_menu->setItemEnabled( UnmountAll, false );
        m_menu->setItemEnabled( ForcedUnmount, false );
      } 
      
      m_menu->exec( pos, 0 );     
    }
  }
}


void Smb4KShareWidget::slotSelectionChanged( QIconViewItem *item )
{
  // The info about the currently highlighted share has to be 
  // passed to the program. Do this now.
  // When NO share is under the mouse, the signal will be send
  // by slotMouseButtonPressed().
  if ( item )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)item;
    sapp->view()->shareTab()->init( new Smb4KShare( i->name(), i->mountpoint(), i->uid(), i->gid() ) );
    sapp->core()->shellIO()->getDiskUsage( i->mountpoint() );

    // Enable the actions:
    sapp->sharesUnmount->setEnabled( true );
    sapp->sharesStartFilemanager->setEnabled( true );
  }
}


/****************************************************************************
   External slots.
****************************************************************************/

void Smb4KShareWidget::slotMountedShares()
{
  QValueList<Smb4KShare *> shares = sapp->core()->mounter()->getShares();
  
  if ( !shares.isEmpty() && !m_read )
  {
    // First remove those, that are not mounted anymore:
    for ( QIconViewItem *item = this->firstItem(); item; item = item->nextItem() )
    {
      if( !sapp->core()->mounter()->findShareByPath( ((Smb4KShareWidgetItem *)item)->mountpoint() ) )
      {
        delete item;
        this->sort( this->sortDirection() );
      }
    }

    // Now process the entries in the list:
    for ( QValueListIterator<Smb4KShare *> it = shares.begin(); it != shares.end(); ++it )
    {
      QString share = (*it)->getName();
      QString path = (*it)->getPath();
      int share_uid = (*it)->getUID();
      int share_gid = (*it)->getGID();

      // Check, whether the share is already in the list. Look for the
      // mount point.
      bool alreadyInView = false;
      for ( QIconViewItem *item = this->firstItem(); item; item = item->nextItem() )
      {
        if ( ((Smb4KShareWidgetItem *)item)->mountpoint() == path )
        {
          alreadyInView = true;
          break;
        }
        else
          continue;
      }

      if ( !alreadyInView )
      {
        // Test whether the imported share belongs to the user.
        bool externalMount = false;

        if ( share_uid != m_uid && share_gid != m_gid )
          externalMount = true;

        if ( !m_mountpoint )
        {
          if ( m_display == "icons" && !m_showExternal && externalMount )
            continue;
          else if ( m_display == "icons" )
          {
            Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
            item->setText( item->name() );
          }

          if ( m_display == "list" && !m_showExternal && externalMount )
            continue;
          else if ( m_display == "list" )
          {
            Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
            item->setText( item->name() );
          }
        }
        else
        {
          if ( m_display == "icons" && !m_showExternal && externalMount )
            continue;
          else if ( m_display == "icons" )
          {
            Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
            item->setText( item->mountpoint() );
          }

          if ( m_display == "list" && !m_showExternal && externalMount )
            continue;
          else if ( m_display == "list" )
          {
            Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share, path, share_uid, share_gid );
            item->setText( item->mountpoint() );
          }
        }
        // Do the sorting:
        this->sort( this->sortDirection() );
      }
    }
  }
  else if ( shares.isEmpty() && !m_read )
  {
    clear();
    sapp->view()->shareTab()->clear();
  }

  // Enable actions:
  if ( count() != 0 && !sapp->sharesUnmountAll->isEnabled() )
    sapp->sharesUnmountAll->setEnabled( true );
}


void Smb4KShareWidget::slotMountedShare(Smb4KShare *share)
{
  // We know, that only the first entry of the string list is full:
  if ( !m_mountpoint )
  {
    Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share->getName(), share->getPath(), m_uid, m_gid );
    item->setText( item->name() );
  }
  else
  {
    Smb4KShareWidgetItem *item = new Smb4KShareWidgetItem( this, m_display, share->getName(), share->getPath(), m_uid, m_gid );
    item->setText( item->mountpoint() );
  }

  // Do the sorting:
  this->sort( this->sortDirection() );

  // Enable actions:
  sapp->sharesUnmountAll->setEnabled( true );
}


void Smb4KShareWidget::slotUnmountedShare()
{
  delete currentItem();

  // Do the sorting:
  this->sort( this->sortDirection() );

  sapp->view()->shareTab()->clear();

  // Enabled/disable actions:
  if ( count() != 0 )
  {
    sapp->sharesUnmount->setEnabled( false );
    sapp->sharesUnmountAll->setEnabled( true );
    sapp->sharesStartFilemanager->setEnabled( false );
  }
  else
  {
    sapp->sharesUnmount->setEnabled( false );
    sapp->sharesUnmountAll->setEnabled( false );
    sapp->sharesStartFilemanager->setEnabled( false );
  }
}


void Smb4KShareWidget::slotUnmountedAllShares()
{
  clear();

  sapp->view()->shareTab()->clear();

  // Disable selections:
  sapp->sharesUnmount->setEnabled( false );
  sapp->sharesUnmountAll->setEnabled( false );
  sapp->sharesStartFilemanager->setEnabled( false );

  slotMountedShares();
}


/***************************************************************************
   Slots for the menu.
***************************************************************************/


void Smb4KShareWidget::slotOpenFilemanager()
{
  if ( currentItem() )
  {
    sapp->core()->open( ((Smb4KShareWidgetItem *)currentItem())->mountpoint() );
  }
}


void Smb4KShareWidget::slotUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    sapp->core()->mounter()->unmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


void Smb4KShareWidget::slotUnmountAllShares()
{
  sapp->core()->mounter()->unmountAllShares();
}


void Smb4KShareWidget::slotForceUnmountShare()
{
  if ( currentItem() )
  {
    Smb4KShareWidgetItem *i = (Smb4KShareWidgetItem *)currentItem();
    sapp->core()->mounter()->forceUnmountShare( i->mountpoint(), QString( "%1" ).arg( i->uid() ), QString( "%1" ).arg( i->gid() ) );
  }
}


/***************************************************************************
 *                                                                         *
 *   The Smb4KShareWidgetItem class.                                       *
 *                                                                         *
 ***************************************************************************/


Smb4KShareWidgetItem::Smb4KShareWidgetItem( Smb4KShareWidget *parent, const QString &alignment, const QString &name, const QString &mountpoint, int uid, int gid ) : KIconViewItem( parent, QString::null ), m_uid( uid ), m_gid( gid ), m_sharename( name ), m_mountpoint( mountpoint )
{
  // Get the UID and the GID from the config file:
  kapp->config()->setGroup( "System" );
  int u = kapp->config()->readNumEntry( "UID", 0 );
  int g = kapp->config()->readNumEntry( "GID", 0 );
  
  if ( uid != u && gid != g )
    m_foreignMount = true;
  else
    m_foreignMount = false;

  if ( !m_foreignMount && alignment == "icons" )
  {
    setPixmap( DesktopIcon( "hdd_mount" ) );
  }
  else if ( !m_foreignMount && alignment == "list" )
  {
    setPixmap( SmallIcon( "hdd_mount" ) );
  }
  else if ( m_foreignMount && alignment == "icons" )
  {
    setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Desktop, 0, KIcon::DisabledState, 0L, false ) );
  }
  else if ( m_foreignMount && alignment == "list" )
  {
    setPixmap( kapp->iconLoader()->loadIcon( "hdd_mount", KIcon::Small, 0, KIcon::DisabledState, 0L, false ) );
  }
}


Smb4KShareWidgetItem::~Smb4KShareWidgetItem()
{
}

/****************************************************************************
   Paints an item.
****************************************************************************/


void Smb4KShareWidgetItem::paintItem( QPainter *p, const QColorGroup &cg )
{
  QColorGroup colorgrp( cg );
  
  if ( m_foreignMount )
  {
/*
    // Change the font:
    QFont f( p->font() );
    f.setItalic( true );
    p->setFont( f );
*/
    colorgrp.setColor( QColorGroup::Text, Qt::gray );
  }
    
  QIconViewItem::paintItem( p, colorgrp );
}


#include "smb4ksharewidget.moc"

