/***************************************************************************
    smb4kipfinder.cpp  -  This class takes care of the IP address detection
    of the hosts in the browser window.
                             -------------------
    begin                : Son Feb 22 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// KDE includes
#include <kapplication.h>
#include <kmessagebox.h>

// application specific includes
#include "smb4kipfinder.h"
#include "smb4kglobal.h"

using namespace Smb4K_Global;


Smb4KIPFinder::Smb4KIPFinder( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_timer = new QTimer( this );
  m_timer->start( 50, false );

  m_working = false;

  m_proc = new KProcess( this, "IPFinderProcess" );
  m_proc->setUseShell( true );

  m_queue.setAutoDelete( true );

  connect( m_timer, SIGNAL( timeout() ), this,                                 SLOT( getIPAddress() ) );
  connect( m_proc,  SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,  SIGNAL( processExited( KProcess* ) ), this,                SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KIPFinder::~Smb4KIPFinder()
{
  abort();
}


/****************************************************************************
   Aborts the network scan.
****************************************************************************/

void Smb4KIPFinder::abort()
{
  if ( m_proc->isRunning() )
  {
    m_proc->kill();
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KIPFinder::slotInput( const QValueList<Smb4KHostItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      m_queue.enqueue( new Smb4KHostItem( (*it)->workgroup(), (*it)->name(), (*it)->comment(), (*it)->ip() ) );
    }
  }
}


void Smb4KIPFinder::getIPAddress()
{
  if ( !m_queue.isEmpty() && !m_working )
  {
    m_working = true;
    emit running( IPFINDER_WORKING, m_working );

    QString wins = getWINSServer();

    m_workgroup = (m_queue.head())->workgroup();

    QString command;

    while ( m_queue.head() && QString::compare( (m_queue.head())->workgroup(), m_workgroup ) == 0 )
    {
      Smb4KHostItem *item = m_queue.dequeue();

      command.append( "nmblookup" );

      QString nmblookup_options = getNmblookupOptions();

      if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
      {
        command.append( nmblookup_options );
      }

      // If there is a WINS server, it will be queried for the host.
      if ( !wins.isEmpty() )
      {
        command.append( " -R -U "+KProcess::quote( wins )+" -W "+KProcess::quote( item->workgroup() )+" -- "+KProcess::quote( item->name() )+" | grep '<00>' | sed -e '/nmb_name/d'" );
      }
      else
      {
        command.append( " -W "+KProcess::quote( item->workgroup() )+" -- "+KProcess::quote( item->name() )+" | grep '<00>' | sed -e '/nmb_name/d'" );
      }

      command.append( " ; " );

      delete item;
    }

    command.truncate( command.length() - 3 );

    m_proc->operator<<( command );
    m_proc->start( KProcess::NotifyOnExit, KProcess::Stdout );
  }
}


void Smb4KIPFinder::slotReceivedStdout( KProcess *, char *buf, int len )
{
  QString buffer = QString::fromLocal8Bit( buf, len ).stripWhiteSpace();

  QString ip = buffer.section( " ", 0, 0 ).stripWhiteSpace();
  QString host = buffer.section( " ", 1, 1 ).section( "<00>", 0, 0 ).stripWhiteSpace();

  if ( !host.isEmpty() && !ip.isEmpty() )
  {
    Smb4KHostItem *item = new Smb4KHostItem( m_workgroup, host, QString::null, ip );

    emit ip_address( item );

    delete item;
  }
}


void Smb4KIPFinder::slotProcessExited( KProcess * )
{
  m_proc->clearArguments();
  m_working = false;
  emit running( IPFINDER_STOP, false );
}

#include "smb4kipfinder.moc"
