/***************************************************************************
    smb4kpasswordhandler  -  This class handles the passwords for Smb4K.
                             -------------------
    begin                : So Jan 16 2005
    copyright            : (C) 2005 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt includes
#include <qframe.h>
#include <qlayout.h>
#include <qstring.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qdir.h>
#include <qfile.h>
#include <qstringlist.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>

#ifdef __FreeBSD__
#include <kprocess.h>
#include <sys/stat.h>
#endif

// application specific includes
#include "smb4kpasswordhandler.h"
#include "smb4kdefs.h"
#include "smb4kglobal.h"

using namespace Smb4K_Global;


Smb4KPasswordHandler *Smb4KPasswordHandler::m_this_class = 0L;


Smb4KPasswordHandler::Smb4KPasswordHandler( QObject *parent, const char *name )
: QObject( parent, name )
{
  m_this_class = this;

  config()->setGroup( "Authentication" );

  m_auth = NULL;
  m_dlg = NULL;
  m_wallet = NULL;
  m_temp_auth = NULL;

  if ( !config()->hasKey( "Use Wallet" ) )
  {
    config()->writeEntry( "Use Wallet", true );
  }

  config()->sync();

//   open_close_wallet();
}


Smb4KPasswordHandler::~Smb4KPasswordHandler()
{
  m_wallet->disconnectApplication( KWallet::Wallet::NetworkWallet(), kapp->name() );
  m_wallet->closeWallet( KWallet::Wallet::NetworkWallet(), false );
  delete m_wallet;
}


void Smb4KPasswordHandler::open_close_wallet()
{
  config()->setGroup( "Authentication" );

  if ( config()->readBoolEntry( "Use Wallet", true ) )
  {
    if ( !m_wallet )
    {
      m_wallet = KWallet::Wallet::openWallet( KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous );

      if ( m_wallet )
      {
        if ( !m_wallet->hasFolder( "Smb4K" ) )
        {
          m_wallet->createFolder( "Smb4K" );
        }

        m_wallet->setFolder( "Smb4K" );

        import();
      }
      else
      {
        KMessageBox::sorry( 0, i18n( "Opening the wallet failed! KWallet support will be disabled." ) );

        delete m_wallet;
        m_wallet = NULL;

        config()->writeEntry( "Use Wallet", false );
        config()->sync();

        import();
      }
    }
  }
  else
  {
    if ( m_wallet )
    {
      m_wallet->disconnectApplication( KWallet::Wallet::NetworkWallet(), kapp->name() );
      m_wallet->closeWallet( KWallet::Wallet::NetworkWallet(), false );

      delete m_wallet;
      m_wallet = NULL;
    }
  }
}


bool Smb4KPasswordHandler::askpass( const QString &workgroup, const QString &host, const QString &share, int desc )
{
  m_auth = readAuth( workgroup, host, share );

  m_dlg = new AskPass::AskPass( m_auth, desc, 0, "AskPassDlg" );

  if ( QString::compare( share, "homes" ) != 0 )
  {
    m_dlg->userEdit()->setText( m_auth->user() );
    m_dlg->passwdEdit()->setText( m_auth->password() );

    if ( m_auth->user().isEmpty() )
    {
      m_dlg->userEdit()->setFocus();
    }
    else
    {
      m_dlg->passwdEdit()->setFocus();
    }
  }
  else
  {
    config()->setGroup( "Homes Shares" );

    QStringList list = config()->readListEntry( host.upper() );

    m_dlg->userCombo()->insertStringList( list );
    m_dlg->userCombo()->setCurrentText( QString::null );

    connect( m_dlg->userCombo(), SIGNAL( activated( const QString & ) ), this, SLOT( slotGetPassword( const QString & ) ) );

    m_dlg->userCombo()->setFocus();
  }

  bool ok = false;

  if ( m_dlg->exec() == KDialogBase::Accepted )
  {
    if ( QString::compare( share, "homes" ) != 0 )
    {
      QString user = m_dlg->userEdit()->text();
      QString pass = m_dlg->passwdEdit()->text();

      m_auth->setUser( user );
      m_auth->setPassword( pass );

      writeAuth( m_auth );
    }
    else
    {
      QString user = m_dlg->userCombo()->currentText();
      QString pass = m_dlg->passwdEdit()->text();

      m_auth->setUser( user );
      m_auth->setPassword( pass );

      writeAuth( m_auth );
    }

    ok = true;
  }

  delete m_dlg;
  m_dlg = NULL;

  delete m_auth;
  m_auth = NULL;

  return ok;
}


Smb4KAuthInfo *Smb4KPasswordHandler::readAuth( const QString &workgroup, const QString &host, const QString &share )
{
  open_close_wallet();

  config()->setGroup( "Authentication" );
  bool use_default = config()->readBoolEntry( "Default Authentication" );

  Smb4KAuthInfo *auth = new Smb4KAuthInfo( workgroup, host, share );

  if ( m_wallet && m_wallet->isOpen() )
  {
    QStringList full_list = m_wallet->entryList();
    QStringList host_list = full_list.grep( host, false );

    if ( !host_list.isEmpty() )
    {
      for ( QStringList::ConstIterator it = host_list.begin(); it != host_list.end(); ++it )
      {
         if ( !share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":"+share.upper() ) != 0 )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          break;
        }
        else if ( !share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":*" ) != 0 )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          continue;
        }
        else if ( share.isEmpty() && (*it).upper().contains( ":"+host.upper()+":" ) )
        {
          QString user = (*it).section( ":", 3, 3 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          break;
        }
        else if ( share.isEmpty() && (*it).upper().startsWith( "DEFAULT:" ) && use_default )
        {
          QString user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
          QString pass;

          m_wallet->readPassword( *it, pass );

          auth->setUser( user );
          auth->setPassword( pass );

          continue;
        }
        else
        {
          continue;
        }
      }
    }
    else
    {
      config()->setGroup( "Authentication" );

      if ( config()->readBoolEntry( "Default Authentication", false ) )
      {
        QStringList default_list = full_list.grep( "DEFAULT", false );
        QString user, pass;

        for ( QStringList::ConstIterator it = default_list.begin(); it != default_list.end(); ++it )
        {
          if ( (*it).startsWith( "DEFAULT:" ) )
          {
            user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
            m_wallet->readPassword( *it, pass );

            auth->setUser( user );
            auth->setPassword( pass );

            break;
          }
          else
          {
            continue;
          }
        }
      }
    }
  }
  else
  {
    config()->setGroup( "Authentication" );
    bool remember = config()->readBoolEntry( "Remember Passwords", true );

    if ( remember && !m_auth_list.isEmpty() )
    {
      for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_auth_list.begin(); it != m_auth_list.end(); ++it )
      {
        if ( (QString::compare( (*it)->workgroup().upper(), workgroup.upper() ) == 0 &&
              QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
              QString::compare( (*it)->share().upper(), share.upper() ) == 0) ||
             (QString::compare( (*it)->workgroup().upper(), "*" ) == 0 &&
              QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
              QString::compare( (*it)->share().upper(), share.upper() ) == 0) )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );

          break;
        }
        else if ( (QString::compare( (*it)->workgroup().upper(), workgroup.upper() ) == 0 &&
                   QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
                   QString::compare( (*it)->share().upper(), "*" ) == 0) ||
                  (QString::compare( (*it)->workgroup().upper(), "*" ) == 0 &&
                   QString::compare( (*it)->host().upper(), host.upper() ) == 0 &&
                   QString::compare( (*it)->share().upper(), "*" ) == 0) )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );

          continue;
        }
        else
        {
          continue;
        }
      }
    }
    else if ( !remember )
    {
      if ( m_temp_auth )
      {
        auth->setUser( m_temp_auth->user() );
        auth->setPassword( m_temp_auth->password() );

        delete m_temp_auth;
        m_temp_auth = NULL;
      }
    }
  }

#ifdef __FreeBSD__

  writeToSMBConfFile( auth );

#endif

  return auth;
}


void Smb4KPasswordHandler::writeAuth( Smb4KAuthInfo *auth )
{
  open_close_wallet();

  if ( m_wallet && m_wallet->isOpen() )
  {
    QString key;
    key.append( auth->workgroup().isEmpty() ? "*" : auth->workgroup().upper() );
    key.append( ":" );
    key.append( auth->host().upper() );
    key.append( ":" );
    key.append( auth->share().isEmpty() ? "*" : auth->share().upper() );
    key.append( ":" );
    key.append( auth->user() );

    m_wallet->writePassword( key, auth->password() );
    m_wallet->sync();
  }
  else
  {
    config()->setGroup( "Authentication" );

    if ( config()->readBoolEntry( "Remember Passwords", true ) )
    {
      bool is_included = false;

      for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_auth_list.begin(); it != m_auth_list.end(); ++it )
      {
        if ( (QString::compare( (*it)->workgroup().upper(), auth->workgroup().upper() ) == 0 ||
              QString::compare( (*it)->workgroup().upper(), "*" ) == 0) &&
             QString::compare( (*it)->host().upper(), auth->host().upper() ) == 0 &&
             QString::compare( (*it)->share().upper(), auth->share().upper() ) == 0 )
        {
          delete *it;
          *it = new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() );

          is_included = true;

          break;
        }
      }

      if ( !is_included )
      {
        m_auth_list.append( new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() ) );
      }
    }
    else
    {
      if ( !m_temp_auth )
      {
        m_temp_auth = new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() );
      }
    }
  }

#ifdef __FreeBSD__

  writeToSMBConfFile( auth );

#endif
}


Smb4KAuthInfo *Smb4KPasswordHandler::readDefaultAuth()
{
  open_close_wallet();

  Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null );

  if ( m_wallet && m_wallet->isOpen() )
  {
    QStringList full_list = m_wallet->entryList();
    QStringList default_list = full_list.grep( "DEFAULT" );

    QString user, pass;

    for ( QStringList::ConstIterator it = default_list.begin(); it != default_list.end(); ++it )
    {
      if ( (*it).startsWith( "DEFAULT:" ) )
      {
        user = (*it).section( ":", 1, 1 ).stripWhiteSpace();
        m_wallet->readPassword( *it, pass );

        auth->setUser( user );
        auth->setPassword( pass );

        break;
      }
      else
      {
        continue;
      }
    }
  }
  else
  {
    delete auth;
    auth = NULL;
  }

  return auth;
}


void Smb4KPasswordHandler::writeDefaultAuth( Smb4KAuthInfo *auth )
{
  open_close_wallet();

  if ( m_wallet && m_wallet->isOpen() )
  {
    QString key = "DEFAULT:"+auth->user();

    m_wallet->writePassword( key, auth->password() );
  }
}


void Smb4KPasswordHandler::import()
{
  config()->setGroup( "Authentication" );

  int key = 0;

  if ( config()->hasKey( "Random Number" ) )
  {
    key = config()->readNumEntry( "Random Number" );
    config()->deleteEntry( "Random Number" );
  }

  QDir::setCurrent( locateLocal( "appdata", QString::null, KGlobal::instance() ) );

  QFile file( "passwords" );

  if ( file.exists() && key != 0 )
  {
    KConfig *pw = new KConfig( "passwords", false, true, "appdata" );

    if ( pw->hasGroup( "Default" ) )
    {
      pw->setGroup( "Default" );

      QString default_user = pw->readEntry( "User" );
      QString default_pass = pw->readEntry( "Password" );

      QString dp;
      uint index( 0 );

      while ( index < default_pass.length() )
      {
        dp[index] = (char)( (int)default_pass[index].unicode()-key%62 );
        index++;
      }

      if ( m_wallet && m_wallet->isOpen() )
      {
        m_wallet->writePassword( "DEFAULT:"+default_user, dp );
      }
    }

    if ( pw->hasGroup( "Logins" ) )
    {
      pw->setGroup( "Logins" );

      for ( int i = 0; ; i++ )
      {
        QStringList list = pw->readListEntry( QString( "%1" ).arg( i ).stripWhiteSpace(), ',' );

        if ( list.isEmpty() )
        {
          break;
        }
        else
        {
          QString p;
          uint index( 0 );

          while( index < list[4].length() )
          {
            p[index] = (char)( (int)list[4][index].unicode()-key%62 );
            index++;
          }

          QString entry = list[0].upper()+":"+list[1].upper()+":"+list[2].upper()+":"+list[3];

          if ( m_wallet && m_wallet->isOpen() )
          {
            m_wallet->writePassword( entry, p );
          }
        }
      }
    }

    if ( m_wallet && m_wallet->isOpen() )
    {
      m_wallet->sync();
    }

    delete pw;

    file.remove();
  }

  config()->sync();
}


#ifdef __FreeBSD__

void Smb4KPasswordHandler::writeToSMBConfFile( Smb4KAuthInfo *auth )
{
  m_nsmbrc_auth = *auth;

  KProcess *p = new KProcess();
  p->setUseShell( true );

  connect( p, SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivePassword( KProcess *, char *, int ) ) );
  connect( p, SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotWritePassword( KProcess * ) ) );

  *p << QString( "smbutil crypt %1" ).arg( m_nsmbrc_auth.password() );

  p->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}

#endif


/////////////////////////////////////////////////////////////////////////////
//  SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


void Smb4KPasswordHandler::slotGetPassword( const QString &username )
{
  if ( m_dlg && m_auth )
  {
    Smb4KAuthInfo *auth = readAuth( m_auth->workgroup().upper(), m_auth->host().upper(), username );
    m_dlg->passwdEdit()->setText( auth->password() );
    delete auth;

    m_auth->setShare( username );
  }
}


void Smb4KPasswordHandler::slotReceivePassword( KProcess *, char *buffer, int buflen )
{
#ifdef __FreeBSD__
  m_buffer.append( QString::fromLocal8Bit( buffer, buflen ) );
#endif
}


void Smb4KPasswordHandler::slotWritePassword( KProcess *proc )
{
#ifdef __FreeBSD__
  delete proc;

  QString pass = m_buffer.remove( "\n" ).stripWhiteSpace();
  m_buffer = QString::null;

  QDir::setCurrent( QDir::homeDirPath() );

  QFile file( ".nsmbrc" );

  QStringList contents;

  if ( file.exists() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );

      while ( !ts.atEnd() )
      {
        contents.append( ts.readLine().stripWhiteSpace() );
      }

      file.close();
    }
  }

  // Check if the [default] section is present.
  if ( contents.find( "[default]" ) == contents.end() &&
       contents.find( "[DEFAULT]" ) == contents.end() )
  {
    QMap<QString,QString> map = readGlobalSMBOptions();
    QString workgroup = map["workgroup"];
    QString wins = getWINSServer();

    QStringList::Iterator it = contents.prepend( "[default]" );
    it++;

    if ( !workgroup.isEmpty() )
    {
      it = contents.insert( it, "workgroup="+workgroup );
    }

    if ( !wins.isEmpty() )
    {
      it = contents.insert( it, "nbns="+wins );
    }

    // FIXME: Should we write the charsets here, too??
  }

  QString section;

  if ( m_nsmbrc_auth.share().isEmpty() )
  {
    section.append( QString( "[%1:%2]" ).arg( m_nsmbrc_auth.host().upper(), m_nsmbrc_auth.user().isEmpty() ? "GUEST" : m_nsmbrc_auth.user().upper() ) );
  }
  else
  {
    section.append( QString( "[%1:%2:%3]" ).arg( m_nsmbrc_auth.host().upper(), m_nsmbrc_auth.user().isEmpty() ? "GUEST" : m_nsmbrc_auth.user().upper(), m_nsmbrc_auth.share().upper() ) );
  }

  QStringList::Iterator it = contents.find( section );

  if ( it == contents.end() )
  {
    if ( contents.last() != QString::null )
    {
      contents.append( QString::null );
    }

    contents.append( section );
    contents.append( "password="+pass );

    if ( !m_nsmbrc_auth.workgroup().isEmpty() )
    {
      contents.append( "workgroup="+m_nsmbrc_auth.workgroup() );
    }
  }
  else
  {
    for ( QStringList::Iterator i = ++it; i != contents.end(); ++i )
    {
      if ( (*i).contains( "password=" ) )
      {
        QString old_pass = (*i).section( "password=", 1, 1 ).stripWhiteSpace();

        if ( QString::compare( pass, old_pass ) == 0 )
        {
          // The passwords are identical, so stop here.
          return;
        }
        else
        {
          *i = "password="+pass;
          break;
        }
      }
      else if ( (*i).startsWith( "[" ) )
      {
        break;
      }
      else
      {
        continue;
      }
    }
  }

  QDir::setCurrent( QDir::homeDirPath() );

  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &file );

    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
    {
      ts << *it << endl;
    }

    file.close();
  }
  else
  {
    emit error( ERROR_WRITING_FILE, "~/.nsmbrc" );

    return;
  }

  // Get minimal security: Fix permissions.
  QString path = QDir::homeDirPath()+"/"+file.name();

  if ( chmod( path.ascii(), 00600 ) == -1 )
  {
    // FIXME: Do we need to emit an error here at all?
  }
#endif
}




/***************************************************************************
 *                                                                         *
 *  The Smb4KPasswordHandler::AskPass class.                               *
 *                                                                         *
 ***************************************************************************/

Smb4KPasswordHandler::AskPass::AskPass( Smb4KAuthInfo *auth, int desc, QWidget *parent, const char *name )
: KDialogBase( Plain, i18n( "Authentication" ), Ok|Cancel, Ok, parent, name, false, true )
{
  // WARNING: Do not set the Qt::WDestructiveClose flag! You'll get a crash in the Smb4KPasswordHandler::askpass()
  // function.

  QFrame *frame = plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 10 );
  layout->setMargin( 0 );

  QString message;

  switch ( desc )
  {
    case NewData:
      break;
    case AccessDenied:
      message = i18n( "The access was denied. " );
      break;
    case BadPassword:
      message = i18n( "The password is not correct. " );
      break;
    case PermDenied:
      message = i18n( "The permission was denied. " );
      break;
    case AuthError:
      message = i18n( "An authentication error occurred. " );
      break;
    case LogonFailure:
      message = i18n( "The logon failed. " );
      break;
    default:
      break;
  }

  message.append( i18n( "Please enter authentication data for %1." ).arg( auth->share().stripWhiteSpace().isEmpty() ? auth->host() : "//"+auth->host()+"/"+auth->share() ) );

  QGroupBox *message_box = new QGroupBox( 1, Qt::Horizontal, i18n( "Information" ), frame, "AskPassInfoBox" );
  QLabel *message_label = new QLabel( message_box );
  message_label->setText( message.stripWhiteSpace() );
  message_label->setTextFormat( Qt::RichText );

  layout->addWidget( message_box, 0, 0, 0 );

  QGroupBox *input_box = new QGroupBox( 1, Qt::Horizontal, i18n( "Input" ), frame, "AskPassInputBox" );
  QWidget *input_widget = new QWidget( input_box );
  QGridLayout *input_widget_layout = new QGridLayout( input_widget );
  input_widget_layout->setSpacing( 10 );
  input_widget_layout->setMargin( 0 );

  QLabel *user_label = new QLabel( i18n( "User:" ), input_widget );
  input_widget_layout->addWidget( user_label, 0, 0, 0 );

  m_user_edit = NULL;
  m_user_combo = NULL;

  if ( QString::compare( auth->share(), "homes" ) != 0 )
  {
    m_user_edit = new KLineEdit( input_widget, "AskPassUserEdit" );
    m_user_edit->setMinimumWidth( 250 );
    input_widget_layout->addWidget( m_user_edit, 0, 1, 0 );
  }
  else
  {
    m_user_combo = new KComboBox( input_widget, "AskPassUserCombo" );
    m_user_combo->setEditable( true );
    m_user_combo->setMinimumWidth( 250 );
    input_widget_layout->addWidget( m_user_combo, 0, 1, 0 );
  }

  QLabel *password_label = new QLabel( i18n( "Password:" ), input_widget );
  input_widget_layout->addWidget( password_label, 1, 0, 0 );

  m_pass_edit = new KLineEdit( input_widget, "AskPassPasswordEdit" );
  m_pass_edit->setEchoMode( KLineEdit::Password );
  m_pass_edit->setMinimumWidth( 250 );
  input_widget_layout->addWidget( m_pass_edit, 1, 1, 0 );

  layout->addWidget( input_box, 1, 0, 0 );

  message_box->setFixedWidth( input_box->sizeHint().width() );  // Do not change this! You'll get layout probems in the case you do.

  setMainWidget( frame );

  setFixedSize( sizeHint() );
}


Smb4KPasswordHandler::AskPass::~AskPass()
{
}


#include "smb4kpasswordhandler.moc"
