/***************************************************************************
    smb4kmounter.h  -  The core class that mounts the shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KMOUNTER_H
#define SMB4KMOUNTER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstringlist.h>
#include <qptrqueue.h>
#include <qfile.h>

// KDE includes
#include <kprocess.h>

// application specific includes
#include "smb4kpasswordhandler.h"
#include "smb4kshare.h"
#include "smb4kdefs.h"
#include "smb4kdataitem.h"


/**
 * This is a core class. It mounts the shares.
 */
class Smb4KMounter : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KMounter( QObject *parent = 0, const char *name = 0 );

    /**
     * The destructor.
     */
    ~Smb4KMounter();

    /**
     * Aborts a running process.
     */
    void abort();

    /**
     * Enumeration for the state of the process. The members are also
     * used in the results() signal as return values.
     */
    enum State{ MountRecent, Import, Mount, Unmount, UnmountAll, ForceUnmount, Idle };

    /**
     * Unmounts a share.
     *
     * @param share       The share object that should be unmounted.
     *
     * @param noMessage   Determines whether this function should emit an error code in case of an error.
     *                    The default value is FALSE.
     */
    void unmountShare( Smb4KShare *share, bool noMessage = false );

    /**
     * Unmounts all shares at once.
     */
    void unmountAllShares();

    /**
     * Mounts a share.
     */
    void mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share );

    /**
     * Unmounts a dead share using super/sudo and 'umount -l' command.
     */
    void forceUnmountShare( Smb4KShare *share );

    /**
     * Returns the unsorted list of mounted shares.
     */
    const QValueList<Smb4KShare *> &getShares() { return m_mounted_shares; };

    /**
     * Find a share in the list with its path.
     */
    Smb4KShare* findShareByPath( const QString &path );

    /**
     * Find a share in the list with its name
     */
    Smb4KShare* findShareByName( const QString &name );

    /**
     * This function returns TRUE if a share is mounted and FALSE otherwise.
     *
     * @param name The name of the share. It has to look like this: //HOST/SHARE.
     *
     * @return TRUE if the share is mounted and FALSE otherwise.
     */
    bool isMounted( const QString &name );

    /**
     * This function reports if the mounter is running or not.
     *
     * @returns             TRUE if the scanner is running and FALSE otherwise.
     */
    bool isRunning() { return m_working; }

    /**
     * This function returns the list of broken shares.
     *
     * @returns             The list of broken shares.
     */
    const QValueList<Smb4KShare *> getBrokenShares();

  signals:
    /**
     * This signal emits the run state.
     *
     * @param state           The so-called run state. There are several defined
     *                        in the smb4kdefs.h header file.
     */
    void state( int state );

    /**
     * Tells the program, that the shares list has been updated.
     */
    void updated();

    /**
     * This signal is emitted if a share has successfully been mounted. It is
     * not used by the main program, but may be used by plugins, etc.
     *
     * @param mountpoint      The mount point of the share.
     */
    void mountedShare( const QString &mountpoint );

    /**
     * This signal is emitted if the share is already mounted.
     *
     * @param mountpoint      The mount point of the already mounted share.
     */
    void alreadyMountedShare( const QString &mountpoint );

  protected:
    /**
     * Starts a process. It takes an interger, that determines,
     * which process has to be started.
     */
    void startProcess( int state );

    /**
     * Is called, when the process ended.
     */
    void endProcess();

    /**
     * Finishes the mounting of shares.
     */
    void processMount();

    /**
     * Finishes the unmounting of a single share.
     */
    void processUnmount();

  protected slots:
    /**
     * Is called, when the process exits.
     */
    void slotProcessExited( KProcess * );

    /**
     * Is called, if output is received at Stdout.
     */
    void slotReceivedStdout( KProcess *, char *buf, int len );

    /**
     * Is called, if output is received a Stderr.
     */
    void slotReceivedStderr( KProcess *, char *buf, int len );

    /**
     * Initiates the processing of the queue.
     */
    void init();

    /**
     * This slot is called by the KApplication::shutDown() signal.
     * Is does everything that has to be done before the program
     * really exits.
     */
    void slotShutdown();

  private:
    /**
     * Mount recently used shares.
     */
    void mountRecent();

    /**
     * Imports mounted shares.
     */
    void import();

    /**
     * Mounts a selected share.
     */
    void mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share );

    /**
     * Unmounts the selected item.
     */
    void unmount( const QString &mountpoint, bool noMessage );

    /**
     * Unmounts dead shares.
     */
    void forceUnmount( const QString &mountpoint );

    /**
     * Unmounts all shares at once.
     */
    void unmountAll();

    /**
     * This function returns the advanced option string that is needed to
     * mount the shares. The contents is determined by the choice of the
     * user (config dialog).
     */
    const QString getMountOptions();

    /**
     * The KProcess object.
     */
    KProcess *m_proc;

    /**
     * The buffer.
     */
    QString m_buffer;

    /**
     * The queue, where the incoming requests are stored.
     */
    QPtrQueue<QString> m_queue;

    /**
     * Determines, whether the mounter is running or not.
     */
    bool m_working;

    /**
     * Holds the list of currently mounted shares as a pointer list.
     */
    QValueList<Smb4KShare *> m_mounted_shares;

    /**
     * Makes sure that the error message concerning the missing of
     * the file /proc/mounts is only shown once.
     */
    bool m_proc_error;

    /**
     * This is the password handler for this class.
     */
    Smb4KPasswordHandler *m_password_handler;

    /**
     * The internal state of the process. Do not mix this up with
     * the state that's emitted to notify the application about changes.
     */
    int m_state;

#ifndef __FreeBSD__
    /**
     * This file object points to /proc/mounts.
     */
    QFile m_proc_mounts;
#endif
};

#endif
