/***************************************************************************
    smb4kscanner.cpp  -  The network scan core class of Smb4K.
                             -------------------
    begin                : Sam Mai 31 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>

// KDE includes
#include <klocale.h>
#include <kapplication.h>
#include <kdebug.h>

// Application specific includes.
#include "smb4kscanner.h"
#include "smb4kauthinfo.h"
#include "smb4kglobal.h"

using namespace Smb4KGlobal;


class Smb4KScannerPrivate
{
  public:
    Smb4KScannerPrivate()
    {
      timerTicks = 0;
      retry = false;
      mainData = NULL;
      bgData = NULL;
    }
    ~Smb4KScannerPrivate()
    {
      if ( mainData )
      {
        delete mainData;
      }

      if ( bgData )
      {
        delete bgData;
      }
    }
    int timerTicks;
    bool retry;
    Smb4KDataItem *mainData;
    Smb4KDataItem *bgData;
};

Smb4KScannerPrivate sp;



Smb4KScanner::Smb4KScanner( QObject *parent, const char *name )
: QObject( parent, name )
{
  m_main_proc = new KProcess( this, "ScannerMainProcess" );
  m_main_proc->setUseShell( true );

  m_bg_proc = new KProcess( this, "ScannerBackgroundProcess" );
  m_bg_proc->setUseShell( true );

  m_password_handler = new Smb4KPasswordHandler( this, "ScannerPasswordHandler" );

  m_working = false;
  m_bg_proc_working = false;

  m_queue.setAutoDelete( true );

  QString *input =  new QString( QString( "%1:" ).arg( Init ) );
  m_queue.enqueue( input );

  connect_timer( true );

  connect( m_main_proc, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,        SLOT( slotReceivedMainProcessStdout( KProcess *, char *, int ) ) );
  connect( m_main_proc, SIGNAL( processExited( KProcess* ) ),
           this,        SLOT( slotMainProcessExited( KProcess * ) ) );
  connect( m_main_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,        SLOT( slotReceivedMainProcessStderr( KProcess *, char *, int ) ) );
  connect( m_bg_proc,   SIGNAL( receivedStdout( KProcess *, char *, int ) ),
           this,        SLOT( slotReceivedBackgroundProcessStdout( KProcess *, char *, int ) ) );
   connect( m_bg_proc,   SIGNAL( receivedStderr( KProcess *, char *, int ) ),
           this,        SLOT( slotReceivedBackgroundProcessStderr( KProcess *, char *, int ) ) );
  connect( m_bg_proc,   SIGNAL( processExited( KProcess * ) ),
           this,        SLOT( slotBackgroundProcessExited( KProcess * ) ) );
  connect( timer(),     SIGNAL( timeout() ),
           this,        SLOT( start() ) );
}


Smb4KScanner::~Smb4KScanner()
{
  abort();

  for ( QValueList<Smb4KWorkgroupItem *>::Iterator it = m_workgroup_list.begin(); it != m_workgroup_list.end(); ++it )
  {
    delete *it;
  }

  m_workgroup_list.clear();

  for ( QValueList<Smb4KHostItem *>::Iterator it = m_hosts_list.begin(); it != m_hosts_list.end(); ++it )
  {
    delete *it;
  }

  m_hosts_list.clear();
}


/****************************************************************************
   This function initiates the scanning for workgroups
****************************************************************************/

void Smb4KScanner::rescan()
{
  connect_timer( true );

  m_queue.enqueue( new QString( QString( "%1:" ).arg( Init ) ) );
}


/****************************************************************************
   Scans for workgroup members. (public part)
****************************************************************************/

void Smb4KScanner::getWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip )
{
  connect_timer( true );

  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Hosts ).arg( workgroup, master, ip ) ) );
}


/****************************************************************************
   Scans for shares on a selected host. (public part)
****************************************************************************/

void Smb4KScanner::getShares( const QString &workgroup, const QString &host, const QString &ip, const QString &protocol )
{
  connect_timer( true );

  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5" ).arg( Shares ).arg( workgroup, host, ip ).arg( protocol ) ) );
}


/****************************************************************************
   Gets more info on a selected host. (public part)
****************************************************************************/

void Smb4KScanner::getInfo( const QString &workgroup, const QString &host, const QString &ip )
{
  connect_timer( true );

  Smb4KHostItem *item = getHost( host, workgroup );

  if ( item && item->infoChecked() )
  {
    emit info( item );

    return;
  }

  m_queue.enqueue( new QString( QString( "%1:%2:%3:%4" ).arg( Info ).arg( workgroup, host, ip ) ) );
}


/****************************************************************************
   Searches for a host. (public part)
****************************************************************************/

void Smb4KScanner::makeSearch( const QString &host )
{
  connect_timer( true );

  m_queue.enqueue( new QString( QString( "%1:%2" ).arg( Search ).arg( host ) ) );
}


/****************************************************************************
   Gets the preview of a share. (public part)
****************************************************************************/

const QString Smb4KScanner::getPreview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path )
{
  connect_timer( true );

  QString share_name = QString::null;

  if ( QString::compare( share, "homes" ) == 0 )
  {
    share_name = specifyUser( host );
  }
  else
  {
    share_name = share;
  }

  if ( !share_name.stripWhiteSpace().isEmpty() )
  {
    m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5:%6" ).arg( Preview ).arg( workgroup, host, ip ).arg( share_name, path ) ) );
  }
  else
  {
    share_name = QString::null;
  }

  return share_name;
}


/****************************************************************************
   Aborts any process that is running.
****************************************************************************/

void Smb4KScanner::abort()
{
  m_queue.clear();

  if ( m_main_proc->isRunning() )
  {
    m_main_proc->kill();
  }

  if ( m_bg_proc->isRunning() )
  {
    m_bg_proc->kill();
  }
}


/****************************************************************************
   This function scans for the workgroups - private part
****************************************************************************/

void Smb4KScanner::init()
{
  abort();

  config()->setGroup( "Browse Options" );
  QString method = config()->readEntry( "Browse List", "lookup" );
  QString host_to_query = config()->readEntry( "Query Host", QString::null );

  config()->setGroup( "Samba" );
  QString workgroup = config()->readEntry( "Domain", *(globalSMBOptions().find( "workgroup" )) );

  sp.mainData = new Smb4KDataItem();

  QString command;

  // Look up the workgroups/domains and their master browsers.
  // At the moment we have three methods:
  // (1) lookup: This method is the most reliable one. It uses
  //     nmblookup and will only find *active* workgroup master
  //     browsers and thus active domains.
  // (2) master: This method will query the current master browser
  //     of the local workgroup/domain. This method is not as
  //     reliable as the first one, because you might get wrong (i.
  //     e. outdated) master browsers for the workgroups or even
  //     empty workgroups.
  // (3) host: This method is similar to the second one, but the
  //     user has defined a fixed host name or IP address.

  if ( QString::compare( method, "lookup" ) == 0 )
  {
    QString wins = winsServer();
    QString nmblookup_options = nmblookupOptions();

    command = QString( "nmblookup -M" );

    if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
    {
      command.append( nmblookup_options );
    }

    command.append( " -- - | grep '<01>' | awk '{print $1}'" );

    // Query the WINS server, if available
    if ( wins.isEmpty() )
    {
      command.append( " | xargs nmblookup -A" );
    }
    else
    {
      command.append( QString( " | xargs nmblookup -R -U %1 -A" ).arg( wins ) );
    }

    *m_main_proc << command;

    startMainProcess( Workgroups );
  }
  else if ( QString::compare( method, "master" ) == 0 )
  {
    command.append( "net " );
    command.append( netOptions( "lookup master "+KProcess::quote( workgroup ) ) );
    command.append( " -U % | xargs net " );
    command.append( netOptions( "domain" ) );
    command.append( " -U % -S" );

    *m_main_proc << command;

    startMainProcess( QueryHost );
  }
  else if ( QString::compare( method, "host" ) == 0 )
  {

    command.append( "net " );
    command.append( netOptions( "lookup host" ) );
    command.append( " -U guest% -S "+KProcess::quote( host_to_query ) );
    command.append( " | xargs net " );
    command.append( netOptions( "domain" ) );
    command.append( " -U guest% -S "+KProcess::quote( host_to_query )+" -I " );

    *m_main_proc << command;

    startMainProcess( QueryHost );
  }
}


/****************************************************************************
   Scans for workgroup members. (private part)
****************************************************************************/

void Smb4KScanner::scanForWorkgroupMembers( const QString &workgroup, const QString &master, const QString &ip )
{
  sp.mainData = new Smb4KDataItem( workgroup, master, QString::null, ip );

  QString command;

  if ( !ip.isEmpty() )
  {
    command.append( "IPADDR="+KProcess::quote( ip )+" &&" );
  }
  else
  {
    command.append( "IPADDR=`net "+netOptions( "lookup host "+KProcess::quote( master ) )+" -S "+KProcess::quote( master )+" -w "+KProcess::quote( workgroup )+" -U %` &&" );
  }

  command.append( " echo \"*** "+master+": $IPADDR ***\" &&" );
  command.append( " net "+netOptions( "server domain" ) );
  command.append( " -w "+KProcess::quote( workgroup ) );
  command.append( " -S "+KProcess::quote( master ) );
  command.append( " -I $IPADDR" );

  // NOTE: Do not set the environment variables USER and PASSWD
  // here, because this won't work at all (Why?).
  command.append( " -U %" );

  *m_main_proc << command;

  startMainProcess( Hosts );
}


/****************************************************************************
   Scans for shares on a selected host. (private part)
****************************************************************************/

void Smb4KScanner::scanForShares( const QString &workgroup, const QString &host, const QString &ip, const QString &protocol )
{
  sp.mainData = new Smb4KDataItem( workgroup, host, QString::null, ip );

  Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, host, QString::null );

  QString command;

  if ( protocol.isEmpty() )
  {
    command = QString( "net %1 -w %2 -S %3" ).arg( netOptions( "share" ), KProcess::quote( workgroup ), KProcess::quote( host ) );
  }
  else
  {
    command = QString( "net %1 %2 -w %3 -S %4" ).arg( protocol, netOptions( "share_no_protocol" ) ).arg( KProcess::quote( workgroup ), KProcess::quote( host ) );
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  if ( !auth->user().isEmpty() )
  {
    command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

    if ( !auth->password().isEmpty() )
    {
      m_main_proc->setEnvironment( "PASSWD", auth->password() );
    }
  }
  else
  {
    command.append( " -U guest%" );
  }

  delete auth;

  *m_main_proc << command;

  startMainProcess( Shares );
}


/****************************************************************************
   Gets more info on a selected host. (private part)
****************************************************************************/

void Smb4KScanner::scanForInfo( const QString &workgroup, const QString &host, const QString &ip )
{
  sp.mainData = new Smb4KDataItem( workgroup, host, QString::null, ip );

  QString smbclient_options = smbclientOptions();

  QString command = QString( "smbclient -d1 -U guest% -W %1 -L %2" ).arg( KProcess::quote( workgroup ) ).arg( KProcess::quote( host ) );

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  if ( !smbclient_options.stripWhiteSpace().isEmpty() )
  {
    command.append( smbclient_options );
  }

  *m_main_proc << command;

  startMainProcess( Info );
}


/****************************************************************************
   Searches for a host. (private part)
****************************************************************************/
#include <kmessagebox.h>

void Smb4KScanner::searchForHost( const QString &host )
{
  config()->setGroup( "Browse Options" );
  QString search_method = config()->readEntry( "Network Search", "nmblookup" );

  if ( QString::compare( search_method, "smbclient" ) == 0 &&
       host.stripWhiteSpace().contains( "." ) == 3 &&
       host.stripWhiteSpace()[0].isNumber() &&
       host.stripWhiteSpace()[host.stripWhiteSpace().length()-1].isNumber() )
  {
    showCoreError( ERROR_IP_CANNOT_BE_USED, QString::null );
    m_working = false;
    emit state( SCANNER_STOP );
    return;
  }

  sp.mainData = new Smb4KDataItem( QString::null, host, QString::null );

  QString wins = winsServer();
  QString nmblookup_options = nmblookupOptions();
  QString smbclient_options = smbclientOptions();

  QString command;

  if ( QString::compare( search_method, "nmblookup" ) == 0 )
  {
    command = QString( "nmblookup" );

    if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
    {
      command.append( nmblookup_options );
    }

    if ( host.contains( '.', true ) != 3 )
    {
      if ( !wins.isEmpty() )
      {
        command.append( QString( " -R -U %1 %2 -S | grep '<00>' | sed -e 's/<00>.*//'" ).arg( wins ).arg( sp.mainData->host() ) );
      }
      else
      {
        command.append( QString( " %1 -S | grep '<00>' | sed -e 's/<00>.*//'" ).arg( sp.mainData->host() ) );
      }
    }
    else
    {
      if ( !wins.isEmpty() )
      {
        command.append( QString( " -R -U %1 %2 -A | grep '<00>' | sed -e 's/<00>.*//'" ).arg( wins ).arg( sp.mainData->host() ) );
      }
      else
      {
        command.append( QString( " %1 -A | grep '<00>' | sed -e 's/<00>.*//'" ).arg( sp.mainData->host() ) );
      }
    }
  }
  else
  {
    command = QString( "smbclient -d2 -U % -L %1" ).arg( sp.mainData->host() );

    if ( !smbclient_options.stripWhiteSpace().isEmpty() )
    {
      command.append( smbclient_options );
    }
  }

  *m_main_proc << command;

  startMainProcess( Search );
}


/****************************************************************************
   Gets the preview of a share. (private part)
****************************************************************************/

void Smb4KScanner::preview( const QString &workgroup, const QString &host, const QString &ip, const QString &share, const QString &path )
{
  sp.mainData = new Smb4KDataItem( workgroup, host, share, ip, path );

  QString smbclient_options = smbclientOptions();

  QString command;

  // Assemble command:
  command = QString( "smbclient //%1/%2 -d1 -W %3 -c 'ls" ).arg( KProcess::quote( host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( workgroup ) );

  if ( !path.isEmpty() )
  {
    // To be able to display directories, that contain umlauts, we have to do
    // two things:
    // (1) replace all slashes by backslashes;
    // (2) convert the directory name to local 8 bit.
    QString proc_path = path;
    proc_path = proc_path.replace( QChar( '/' ), QChar( '\\' ) ).local8Bit();
    command.append( " \"" ).append( proc_path ).append( "*\"" );
  }

  command.append( "'" );

  if ( !ip.isEmpty() )
  {
    command.append( QString( " -I %1" ).arg( KProcess::quote( ip ) ) );
  }

  if ( !smbclient_options.stripWhiteSpace().isEmpty() )
  {
    command.append( smbclient_options );
  }

  Smb4KAuthInfo *auth = m_password_handler->readAuth( workgroup, host, share );


  if ( !auth->user().isEmpty() )
  {
    command.append( QString( " -U %1" ).arg( KProcess::quote( auth->user() ) ) );

    if ( !auth->password().isEmpty() )
    {
      m_main_proc->setEnvironment( "PASSWD", auth->password() );
    }
  }
  else
  {
    command.append( " -U guest%" );
  }

  delete auth;

  *m_main_proc << command;
  startMainProcess( Preview );
}


/****************************************************************************
   Starts the scanning for IP addresses
****************************************************************************/

void Smb4KScanner::getIPAddresses()
{
  bool start = false;
  QString wins = winsServer();
  QString command = QString::null;

  sp.bgData = new Smb4KDataItem();

  for ( QValueList<Smb4KHostItem *>::ConstIterator it = m_hosts_list.begin(); it != m_hosts_list.end(); ++it )
  {
    if ( (*it)->ip().stripWhiteSpace().isEmpty() && !(*it)->ipAddressChecked() )
    {
      if ( sp.bgData->workgroup().isEmpty() ||
           QString::compare( sp.bgData->workgroup(), (*it)->workgroup() ) == 0 )
      {
        sp.bgData->setWorkgroup( (*it)->workgroup() );

        start = true;

        (*it)->setIPAddressChecked( true );

        command.append( "nmblookup" );

        QString nmblookup_options = nmblookupOptions();

        if ( !nmblookup_options.stripWhiteSpace().isEmpty() )
        {
          command.append( nmblookup_options );
        }

        // If there is a WINS server, it will be queried for the host.
        if ( !wins.isEmpty() )
        {
          command.append( " -R -U "+KProcess::quote( wins )+" -W "+KProcess::quote( (*it)->workgroup() )+" -- "+KProcess::quote( (*it)->name() )+" | grep '<00>' | sed -e '/nmb_name/d'" );
        }
        else
        {
          command.append( " -W "+KProcess::quote( (*it)->workgroup() )+" -- "+KProcess::quote( (*it)->name() )+" | grep '<00>' | sed -e '/nmb_name/d'" );
        }

        command.append( " ; " );

        continue;
      }
      else
      {
        continue;
      }
    }
    else
    {
      continue;
    }
  }

  command.truncate( command.length() - 3 );

  if ( start )
  {
    *m_bg_proc << command;
    startBackgroundProcess( IPAddresses );
  }
}


/****************************************************************************
   Starts the main process of the scanner.
****************************************************************************/

void Smb4KScanner::startMainProcess( int state )
{
  m_main_state = state;
  m_main_buffer = QString::null;

  if ( state != Info )
  {
    QApplication::setOverrideCursor( waitCursor );
  }

  m_main_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   Starts the background process of the scanner.
****************************************************************************/

void Smb4KScanner::startBackgroundProcess( int state )
{
  // We do not want to notify the user, that the scanner is doing
  // something.
  m_bg_state = state;
  m_bg_proc_working = true;
  m_bg_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
}


/****************************************************************************
   End the main process and tell, what to do with the data.
****************************************************************************/

void Smb4KScanner::endMainProcess()
{
  switch ( m_main_state )
  {
    case Workgroups:
    case QueryHost:
      processWorkgroups();
      break;
    case Hosts:
      processHosts();
      break;
    case Shares:
      processShares();
      break;
    case Info:
      processInfo();
      break;
    case Preview:
      processPreview();
      break;
    case Search:
      processSearch();
      break;
    default:
      break;
  }

  m_main_state = Idle;

  delete sp.mainData;
  sp.mainData = NULL;

  QApplication::restoreOverrideCursor();

  if ( m_queue.isEmpty() )
  {
    connect_timer( false );
  }

  m_main_proc->clearArguments();

  m_working = false;
  emit state( SCANNER_STOP );
}


/****************************************************************************
   End the background process
****************************************************************************/

void Smb4KScanner::endBackgroundProcess()
{
  m_bg_buffer = QString::null;
  m_bg_proc->clearArguments();

  delete sp.bgData;
  sp.bgData = NULL;

  m_bg_proc_working = false;
}


/****************************************************************************
   Process the list of workgroups.
****************************************************************************/

void Smb4KScanner::processWorkgroups()
{
  QStringList list = QStringList::split( '\n', m_main_buffer, false );

  for ( QValueList<Smb4KWorkgroupItem *>::Iterator it = m_workgroup_list.begin(); it != m_workgroup_list.end(); ++it )
  {
    delete *it;
  }

  for ( QValueList<Smb4KHostItem *>::Iterator it = m_hosts_list.begin(); it != m_hosts_list.end(); ++it )
  {
    delete *it;
  }

  m_workgroup_list.clear();
  m_hosts_list.clear();

  if ( m_main_state == Workgroups )
  {
    QString workgroup, master, ip;

    for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
    {
      if ( (*it).stripWhiteSpace().startsWith( "Looking" ) )
      {
        ip = (*it).section( "of", 1, 1 ).stripWhiteSpace();
        continue;
      }
      else if ( (*it).contains( "<00>" ) != 0 && (*it).contains( "<GROUP>" ) == 0 )
      {
        if ( workgroup.isEmpty() && master.isEmpty() && !ip.isEmpty() )
        {
          master = (*it).section( "<00>", 0, 0 ).stripWhiteSpace();
        }

        continue;
      }
      else if ( (*it).contains( "<00>" ) != 0 && (*it).contains( "<GROUP>" ) != 0 )
      {
        if ( workgroup.isEmpty() && !master.isEmpty() && !ip.isEmpty() )
        {
          workgroup = (*it).left( (*it).find( "<00>" ) ).stripWhiteSpace();

          m_workgroup_list.append( new Smb4KWorkgroupItem( workgroup, master, ip ) );

          Smb4KHostItem *master_item = new Smb4KHostItem( workgroup, master, QString::null, ip );
          master_item->setMaster( true );

          m_hosts_list.append( master_item );

          workgroup = QString::null;
          master = QString::null;
          ip = QString::null;
        }

        continue;
      }
    }
  }
  else if ( m_main_state == QueryHost )
  {
    bool process = false;

    for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      QString line = (*it).stripWhiteSpace();

      if ( line.startsWith( "-------------" ) )
      {
        process = true;
        continue;
      }

      if ( process && !line.isEmpty() )
      {
        QString workgroup = line.section( "   ", 0, 0 ).stripWhiteSpace();
        QString master = line.section( "   ", 1, -1 ).stripWhiteSpace();

        m_workgroup_list.append( new Smb4KWorkgroupItem( workgroup, master, QString::null ) );

        Smb4KHostItem *master_item = new Smb4KHostItem( workgroup, master );
        master_item->setMaster( true );

        m_hosts_list.append( master_item );

        continue;
      }
      else
      {
        continue;
      }
    }
  }

  emit workgroups( m_workgroup_list );
  emit hostListChanged();
}


/****************************************************************************
   Process the memeber list of a workgroup.
****************************************************************************/

void Smb4KScanner::processHosts()
{
  QStringList list = QStringList::split( '\n', m_main_buffer, false );

  if ( (m_main_buffer.contains( "Could not connect to server", true ) != 0 &&
        m_main_buffer.contains( "The username or password was not correct.", true ) == 0) ||
       m_main_buffer.contains( "Unable to find a suitable server" ) != 0 )
  {
    // We remove the first line, because the IP address of the master
    // is reported there.
    list.remove( list.first() );

    showCoreError( ERROR_GETTING_MEMBERS, list.join( "\n" ) );

    return;
  }

  QValueList<Smb4KHostItem *> hosts;

  bool process = false;
  QString ip;

  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    QString line = (*it).stripWhiteSpace();

    if ( line.startsWith( "***" ) && line.endsWith( "***" ) )
    {
      ip = line.section( ":", 1, 1 ).section( "***", 0, 0 ).stripWhiteSpace();
    }

    if ( line.startsWith( "-------------" ) )
    {
      process = true;

      continue;
    }

    if ( process && !line.isEmpty() )
    {
      QString host, comment;

      if ( line.contains( "   " ) == 0 )
      {
        host = line;
      }
      else
      {
        host = line.section( "   ", 0, 0 ).stripWhiteSpace();
        comment = line.section( "   ", 1, -1 ).stripWhiteSpace();
      }

      Smb4KHostItem *item = new Smb4KHostItem( sp.mainData->workgroup(), host, comment );

      if ( QString::compare( item->name(), sp.mainData->host() ) == 0 )
      {
        if ( sp.mainData->ip().isEmpty() )
        {
          getWorkgroup( sp.mainData->workgroup() )->setMasterIPAddress( ip );
        }

        item->setIPAddress( ip );
        item->setMaster( true );
      }

      hosts.append( item );
    }
  }

  // If the list is empty, put the master in.
  if ( hosts.isEmpty() )
  {
    Smb4KHostItem *item = new Smb4KHostItem( sp.mainData->workgroup(), sp.mainData->host() );
    item->setMaster( true );

    hosts.append( item );
  }

  emit members( hosts );

  // Now put the hosts in m_hosts_list:
  for ( QValueList<Smb4KHostItem *>::Iterator it = m_hosts_list.begin(); it != m_hosts_list.end(); ++it )
  {
    if ( QString::compare( (*it)->workgroup(), sp.mainData->workgroup() ) == 0 )
    {
      bool found = false;

      for ( QValueList<Smb4KHostItem *>::Iterator i = hosts.begin(); i != hosts.end(); ++i )
      {
        if ( *i && QString::compare( (*i)->name(), (*it)->name() ) == 0 )
        {
          found = true;

          // The only thing that might be missing is the comment:
          (*it)->setComment( (*i)->comment() );

          delete *i;
          *i = NULL;

          break;
        }
        else
        {
          continue;
        }
      }

      if ( !found )
      {
        delete *it;
        *it = NULL;
      }
    }
    else
    {
      continue;
    }
  }

  m_hosts_list.remove( NULL );
  hosts.remove( NULL );

  m_hosts_list += hosts;

  emit hostListChanged();
}


/****************************************************************************
   Process the share list of a host.
****************************************************************************/

void Smb4KScanner::processShares()
{
  // Error handling
  if ( m_main_buffer.contains( "The username or password was not correct.", true ) != 0 )
  {
    if ( m_password_handler->askpass( sp.mainData->workgroup(), sp.mainData->host(), QString::null, Smb4KPasswordHandler::AccessDenied ) )
    {
      m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5" ).arg( Shares ).arg( sp.mainData->workgroup(), sp.mainData->host(), sp.mainData->ip(), QString::null ) ) );
    }

    return;
  }
  else if ( m_main_buffer.contains( "could not obtain sid for domain", true ) != 0 )
  {
    // FIXME: Does this error only occur when we scan a server that is
    // only capable of the RAP protocol or also under other conditions?
    // Please report!
    m_queue.enqueue( new QString( QString( "%1:%2:%3:%4:%5" ).arg( Shares ).arg( sp.mainData->workgroup(), sp.mainData->host(), sp.mainData->ip(), "rap" ) ) );

    sp.retry = true;

    return;
  }
  else if ( (m_main_buffer.contains( "Could not connect to server", true ) != 0 &&
             m_main_buffer.contains( "The username or password was not correct.", true ) == 0) ||
            m_main_buffer.contains( "Unable to find a suitable server" ) != 0 )
  {
    showCoreError( ERROR_GETTING_SHARES, m_main_buffer );

    return;
  }

  QStringList list = QStringList::split( '\n', m_main_buffer, false );

  QValueList<Smb4KShareItem *> share_list;

  config()->setGroup( "Programs" );

  bool process = false;

  for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    if ( (*it).startsWith( "---" ) )
    {
      process = true;
    }

    if ( process )
    {
      QString name, type, comment;

      if ( (*it).contains( " Disk     ", true ) != 0 )
      {
        name = (*it).section( " Disk     ", 0, 0 ).stripWhiteSpace();
        type = "Disk";
        comment = (*it).section( " Disk     ", 1, 1 ).stripWhiteSpace();
      }
      else if ( (*it).contains( " Print    ", true ) != 0 )
      {
        name = (*it).section( " Print    ", 0, 0 ).stripWhiteSpace();
        type = "Printer";
        comment = (*it).section( " Print    ", 1, 1 ).stripWhiteSpace();
      }
      else if ( (*it).contains( " IPC      ", true ) != 0 )
      {
        name = (*it).section( " IPC      ", 0, 0 ).stripWhiteSpace();
        type = "IPC";
        comment = (*it).section( " IPC      ", 1, 1 ).stripWhiteSpace();
      }
      else
      {
        continue;
      }

      share_list.append( new Smb4KShareItem( sp.mainData->workgroup(), sp.mainData->host(), name, type, comment ) );
    }
    else
    {
      continue;
    }
  }

  emit shares( share_list );
}


/****************************************************************************
   Process the preview.
****************************************************************************/

void Smb4KScanner::processPreview()
{
  QStringList list = QStringList::split( '\n', m_main_buffer, false );

  QValueList<Smb4KPreviewItem *> preview;

  // Now do the rest:
  if ( list.grep( "NT_STATUS" ).count() != 0 || list.grep( "Connection to" ).count() != 0 )
  {
    if ( list.grep( "NT_STATUS" ).count() != 0  )
    {
      QString errmsg = list.grep( "NT_STATUS" ).first().stripWhiteSpace().section( " ", 0, 0 );

      // The error output of smbclient is a little bit inconsistent:
      if ( errmsg.contains( "NT_STATUS" ) == 0 )
      {
        errmsg = list.grep( "NT_STATUS" ).first().stripWhiteSpace().section( " ", -1, -1 );
      }

      if ( QString::compare( errmsg, "NT_STATUS_ACCESS_DENIED" ) == 0 ||
           QString::compare( errmsg, "NT_STATUS_LOGON_FAILURE" ) == 0 )
      {
        int state = Smb4KPasswordHandler::None;

        if ( QString::compare( errmsg, "NT_STATUS_ACCESS_DENIED" ) == 0 )
        {
          state = Smb4KPasswordHandler::AccessDenied;
        }
        else if ( QString::compare( errmsg, "NT_STATUS_LOGON_FAILURE" ) == 0 )
        {
          state = Smb4KPasswordHandler::LogonFailure;
        }

        if ( m_password_handler->askpass( sp.mainData->workgroup(), sp.mainData->host(), sp.mainData->share(), state ) )
        {
          QString *input = new QString( QString( "%1:%2:%3:%4:%5:%6" ).arg( Preview ).arg( sp.mainData->workgroup() ).arg( sp.mainData->host() ).arg( sp.mainData->ip() ).arg( sp.mainData->share() ).arg( sp.mainData->path() ) );
          m_queue.enqueue( input );
        }
      }
      else
      {
        showCoreError( ERROR_GETTING_PREVIEW, list.grep( "NT_STATUS" ).first().stripWhiteSpace() );
      }
    }
    else
    {
      showCoreError( ERROR_GETTING_PREVIEW, list.grep( "Connection to" ).first().stripWhiteSpace() );
    }
  }
  else if ( list.grep( "Error returning browse list:" ).count() != 0 && list.grep( "NT_STATUS" ).count() == 0 )
  {
    showCoreError( ERROR_GETTING_PREVIEW, list.grep( "Error returning browse list:" ).first().stripWhiteSpace() );
  }
  else
  {
    for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( (*it).stripWhiteSpace().startsWith( "Domain" ) ||
           (*it).stripWhiteSpace().startsWith( "OS" ) ||
           (*it).stripWhiteSpace().startsWith( "Anonymous" ) )
      {
        continue;
      }
      else if ( (*it).contains( "blocks of size" ) != 0 )
      {
        continue;
      }
      else
      {
        QString tmp = (*it).stripWhiteSpace().section( " ", 0, -9 ).stripWhiteSpace();

        QString item = tmp.section( "  ", 0, -2 ).stripWhiteSpace();

        bool isFile = true;

        if ( !item.isEmpty() && tmp.section( "  ", -1, -1 ).contains( "D" ) != 0 )
          isFile = false;
        else if ( item.isEmpty() )
          item = tmp;

        Smb4KPreviewItem *i = new Smb4KPreviewItem( sp.mainData->host(), sp.mainData->share(), sp.mainData->path(), item, isFile );

        preview.append( i );
      }
    }
  }

  emit previewResult( preview );
}


/****************************************************************************
   Process the search data.
****************************************************************************/

void Smb4KScanner::processSearch()
{
  // Stop right here if the user searched for illegal
  // strings like #, ', () etc.
  if ( m_main_buffer.contains( "Usage:", true ) != 0 || m_main_buffer.contains( "/bin/sh:", true ) != 0 )
  {
    emit searchResult( new Smb4KHostItem() );

    return;
  }

  config()->setGroup( "Browse Options" );
  QString search_method = config()->readEntry( "Network Search", "nmblookup" );

  QStringList data = QStringList::split( "\n", m_main_buffer.stripWhiteSpace(), false );

  if ( QString::compare( search_method, "nmblookup" ) == 0 )
  {
    if ( !data.isEmpty() )
    {
      // The last entry in the list is the workgroup:
      QString workgroup = data.last().stripWhiteSpace();
      QString host, ip;

      if ( sp.mainData->host().contains( ".", true ) != 3 )
      {
        // The IP address is in the first entry:
        ip = data.first().stripWhiteSpace().section( " ", 0, 0 );
        // The host.
        host = sp.mainData->host().upper();
      }
      else
      {
        ip = sp.mainData->host();
        host = data[0].stripWhiteSpace();
      }

      emit searchResult( new Smb4KHostItem( workgroup, host, QString::null, ip ) );
    }
    else
      emit searchResult( new Smb4KHostItem() );
  }
  else
  {
    if ( data.count() > 1 && !data[1].isEmpty() )
    {
      if ( m_main_buffer.contains( QString( "Connection to %1 failed" ).arg( sp.mainData->host() ) ) != 0 )
      {
        emit searchResult( new Smb4KHostItem() );
      }
      else
      {
        QString workgroup = data.grep( "Domain" ).first().section( "Domain=[", 1, 1 ).section( "]", 0, 0 );
        QString ip = data.grep( "Got a positive name query" ).first().section( "(", 1, 1 ).section( ")", 0, 0 ).stripWhiteSpace();

        emit searchResult( new Smb4KHostItem( workgroup, sp.mainData->host().upper(), QString::null, ip ) );
      }
    }
    else
    {
      emit searchResult( new Smb4KHostItem() );
    }
  }
}


/****************************************************************************
   Process the output of the IP address look-up
****************************************************************************/

void Smb4KScanner::processIPAddresses()
{
  if ( !m_bg_buffer.stripWhiteSpace().isEmpty() )
  {
    QString ip = m_bg_buffer.stripWhiteSpace().section( " ", 0, 0 ).stripWhiteSpace();
    QString host = m_bg_buffer.stripWhiteSpace().section( " ", 1, 1 ).section( "<00>", 0, 0 ).stripWhiteSpace();

    m_bg_buffer = QString::null;

    if ( !host.isEmpty() && !ip.isEmpty() )
    {
      Smb4KHostItem *item = getHost( host, sp.bgData->workgroup() );

      if ( item )
      {
        item->setIPAddress( ip );

        if ( item->isMaster() )
        {
          Smb4KWorkgroupItem *wg = getWorkgroup( sp.bgData->workgroup() );

          if ( wg )
          {
            wg->setMasterIPAddress( ip );
          }
        }

        emit ipAddress( item );
      }
    }
  }
}


/****************************************************************************
   Process the information about a host.
****************************************************************************/

void Smb4KScanner::processInfo()
{
  if ( m_main_proc->normalExit() )
  {
    QStringList list = QStringList::split( '\n', m_main_buffer, false );

    Smb4KHostItem *host = getHost( sp.mainData->host(), sp.mainData->workgroup() );

    if ( host )
    {
      host->setInfoChecked( true );

      for ( QStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
      {
        if ( (*it).stripWhiteSpace().startsWith( "Domain" ) || (*it).stripWhiteSpace().startsWith( "OS" ) )
        {
          // The OS string.
          host->setOSString( (*it).section( "OS=[", 1, 1 ).section( "]", 0, 0 ).stripWhiteSpace() );

          // The Server string.
          host->setServerString( (*it).section( "Server=[", 1, 1 ).section( "]", 0, 0 ).stripWhiteSpace() );

          break;
        }
        else if ( (*it).contains( "Connection to", true ) != 0 )
        {
          break;
        }
      }

      emit info( host );
    }
  }
}


/****************************************************************************
   Appends an item to the list of workgroups.
****************************************************************************/

void Smb4KScanner::appendWorkgroup( Smb4KWorkgroupItem *item )
{
  item->setPseudo();

  if ( getWorkgroup( item->workgroup() ) == 0 )
  {
    m_workgroup_list.append( item );
  }
}


/****************************************************************************
   Get a workgroup item out of the workgroup list.
****************************************************************************/

Smb4KWorkgroupItem *Smb4KScanner::getWorkgroup( const QString &workgroup )
{
  QValueListIterator<Smb4KWorkgroupItem *> it;

  for ( it = m_workgroup_list.begin(); it != m_workgroup_list.end(); ++it )
  {
    if ( QString::compare( (*it)->workgroup(), workgroup ) == 0 )
    {
      break;
    }
    else
    {
      continue;
    }
  }

  return it == m_workgroup_list.end() ? NULL : *it;
}


/****************************************************************************
   Get a workgroup item out of the workgroup list.
****************************************************************************/

Smb4KHostItem *Smb4KScanner::getHost( const QString &name, const QString &workgroup )
{
  QValueListIterator<Smb4KHostItem *> it;

  for ( it = m_hosts_list.begin(); it != m_hosts_list.end(); ++it )
  {
    if ( !workgroup.stripWhiteSpace().isEmpty() && QString::compare( (*it)->workgroup(), workgroup ) != 0 )
    {
      continue;
    }

    if ( QString::compare( (*it)->name(), name ) == 0 )
    {
      break;
    }
    else
    {
      continue;
    }
  }

  return it == m_hosts_list.end() ? NULL : *it;
}


/****************************************************************************
   Connect or disconnect the timer
****************************************************************************/

void Smb4KScanner::connect_timer( bool con )
{
  if ( con )
  {
    connect( timer(), SIGNAL( timeout() ), this, SLOT( start() ) );
  }
  else
  {
    disconnect( timer(), SIGNAL( timeout() ), this, SLOT( start() ) );
  }
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KScanner::slotReceivedMainProcessStdout( KProcess *, char *buf, int len )
{
  m_main_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KScanner::slotMainProcessExited( KProcess * )
{
  endMainProcess();
}


void Smb4KScanner::slotReceivedMainProcessStderr( KProcess *, char  *buf, int len )
{
  m_main_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KScanner::slotReceivedBackgroundProcessStdout( KProcess *, char *buf, int len )
{
  m_bg_buffer.append( QString::fromLocal8Bit( buf, len ) );

  switch ( m_bg_state )
  {
    case IPAddresses:
      processIPAddresses();
      break;
    default:
      break;
  }
}


void Smb4KScanner::slotReceivedBackgroundProcessStderr( KProcess *, char *buf, int len )
{
  m_bg_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KScanner::slotBackgroundProcessExited( KProcess * )
{
  endBackgroundProcess();
}


void Smb4KScanner::start()
{
  // Look for the thing to do (next).
  // At this point, the topmost item will not be
  // dequeued. This will be done below.
  int todo = Idle;
  QString *head = NULL;

  if ( (head = m_queue.head()) != 0 )
  {
    todo = head->section( ":", 0, 0 ).toInt();
  }

  if ( todo == IPAddresses || (!m_working && !m_queue.isEmpty()) )
  {
    // Start processing with dequeueing the item:
    QString *item = m_queue.dequeue();

    // Tell the program, that the scanner is running.
    if ( todo != IPAddresses )
    {
      m_working = true;
    }

    switch ( todo )
    {
      case Init:
      {
        emit state( SCANNER_INIT );
        init();
        break;
      }
      case Hosts:
      {
        emit state( SCANNER_OPENING_WORKGROUP );
        scanForWorkgroupMembers( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      }
      case Shares:
      {
        if ( !sp.retry )
        {
          emit state( SCANNER_OPENING_HOST );
        }
        else
        {
          emit state( SCANNER_RETRYING_OPENING_HOST );
          sp.retry = false;
        }
        scanForShares( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ) );
        break;
      }
      case Info:
      {
        emit state( SCANNER_RETRIEVING_INFO );
        scanForInfo( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      }
      case Preview:
      {
        emit state( SCANNER_RETRIEVING_PREVIEW );
        preview( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ), item->section( ":", 5, 5 ) );
        break;
      }
      case Search:
      {
        emit state( SCANNER_SEARCHING );
        searchForHost( item->section( ":", 1, 1 ) );
        break;
      }
      case IPAddresses:
      {
        if ( !m_bg_proc_working )
        {
          getIPAddresses();
        }
        break;
      }
      default:
        break;
    }

    delete item;
  }

  sp.timerTicks++;

  if ( sp.timerTicks * timerInterval() >= 250 /* msec */
       && m_queue.isEmpty() && !m_hosts_list.isEmpty() )
  {
    m_queue.enqueue( new QString( QString( "%1:" ).arg( IPAddresses ) ) );

    // Because it would cause too much network traffic, information about
    // a host will only be gathered on request and not automatically.

    sp.timerTicks = 0;
  }
}


#include "smb4kscanner.moc"
