/***************************************************************************
    smb4kbrowserwidget.cpp  -  Network browser widget class of Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qheader.h>
#include <qvaluelist.h>
#include <qlabel.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdialogbase.h>
#include <kglobalsettings.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kdebug.h>

// application specific includes
#include "smb4kbrowserwidget.h"
#include "smb4kpreviewdialog.h"
#include "smb4kprintdialog.h"
#include "smb4kmountdialog.h"
#include "../core/smb4kshare.h"
#include "../core/smb4kcore.h"
#include "../core/smb4kpasswordhandler.h"
#include "../core/smb4kglobal.h"
#include "../core/smb4kdefs.h"

using namespace Smb4KGlobal;


Smb4KBrowserWidget::Smb4KBrowserWidget( QWidget *parent, const char *name ) : KListView( parent, name )
{
  setFrameShape( KListView::PopupPanel );
  setFrameShadow( KListView::Sunken );
  setRootIsDecorated( true );
  setAllColumnsShowFocus( false );
  setMouseTracking( true );

  addColumn( i18n( "Network" ), -1 );
  addColumn( i18n( "Type" ), -1 );
  addColumn( i18n( "IP Address" ), -1 );
  addColumn( i18n( "Comment" ), -1 );

  m_collection = new KActionCollection( this, "BrowserWidget_ActionCollection", KGlobal::instance() );

  m_menu = new Smb4KBrowserActionMenu( Smb4KBrowserActionMenu::Full, m_collection, QString::null, QIconSet(), this, "BrowserWidget_ActionMenu" );

  m_tooltip = NULL;

  initActions();

  // Widget specific connections.
  connect( this, SIGNAL( executed( QListViewItem * ) ),
           this, SLOT( slotItemExecuted( QListViewItem * ) ) );
  connect( this, SIGNAL( expanded( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( collapsed( QListViewItem * ) ),
           this, SLOT( slotItemExpandedCollapsed( QListViewItem * ) ) );
  connect( this, SIGNAL( selectionChanged( QListViewItem * ) ),
           this, SLOT( slotSelectionChanged( QListViewItem * ) ) );
  connect( this, SIGNAL( pressed( QListViewItem * ) ),
           this, SLOT( slotMouseButtonPressed( QListViewItem * ) ) );
  connect( this, SIGNAL( rightButtonPressed( QListViewItem *, const QPoint &, int ) ),
           this, SLOT( slotRightButtonPressed( QListViewItem *, const QPoint &, int ) ) );

  // External connections.
  connect( smb4k_core->scanner(), SIGNAL( workgroups( const QValueList<Smb4KWorkgroupItem *> & ) ),
           this,                  SLOT( slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( members( const QValueList<Smb4KHostItem *> & ) ),
           this,                  SLOT( slotMembers( const QValueList<Smb4KHostItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( shares( const QValueList<Smb4KShareItem *> & ) ),
           this,                  SLOT( slotShares( const QValueList<Smb4KShareItem *> & ) ) );
  connect( smb4k_core->scanner(), SIGNAL( state( int ) ),
           this,                  SLOT( slotReceiveScannerState( int ) ) );
  connect( smb4k_core->scanner(), SIGNAL( ipAddress( Smb4KHostItem * ) ),
           this,                  SLOT( slotAddIPAddress( Smb4KHostItem * ) ) );
  connect( smb4k_core->scanner(), SIGNAL( info( Smb4KHostItem * ) ),
           this,                  SLOT( slotAddInfo( Smb4KHostItem * ) ) );
  connect( smb4k_core->mounter(), SIGNAL( updated() ),
           this,                  SLOT( slotMarkShares() ) );
}


Smb4KBrowserWidget::~Smb4KBrowserWidget()
{
  delete m_menu;
  delete m_tooltip;

  uint index = 0;

  while ( index < m_collection->count() )
  {
    delete m_collection->action( index++ );
  }

  m_collection->clear();
}


void Smb4KBrowserWidget::initActions()
{
  m_preview_action = m_menu->previewAction();
  m_preview_action->setGroup( "BrowserWidget" );
  m_preview_action->setEnabled( false );
  connect( m_preview_action, SIGNAL( activated() ), this, SLOT( slotPreview() ) );

  m_askpass_action = m_menu->askpassAction();
  m_askpass_action->setGroup( "BrowserWidget" );
  m_askpass_action->setEnabled( false );
  connect( m_askpass_action, SIGNAL( activated() ), this, SLOT( slotAskPass() ) );

  m_print_action = m_menu->printAction();
  m_print_action->setGroup( "BrowserWidget" );
  m_print_action->setEnabled( false );
  connect( m_print_action, SIGNAL( activated() ), this, SLOT( slotPrint() ) );

  m_mount_action = m_menu->mountAction();
  m_mount_action->setGroup( "BrowserWidget" );
  m_mount_action->setEnabled( false );
  connect( m_mount_action, SIGNAL( activated() ), this, SLOT( slotMountShare() ) );

  m_bookmark_action = m_menu->bookmarkAction();
  m_bookmark_action->setGroup( "BrowserWidget" );
  m_bookmark_action->setEnabled( false );
  connect( m_bookmark_action, SIGNAL( activated() ), this, SLOT( slotAddBookmark() ) );

  m_rescan_action = m_menu->rescanAction();
  m_rescan_action->setGroup( "BrowserWidget" );
  m_rescan_action->setEnabled( true );
  connect( m_rescan_action, SIGNAL( activated() ), this, SLOT( slotRescan() ) );

  m_abort_action = m_menu->abortAction();
  m_abort_action->setGroup( "BrowserWidget" );
  m_abort_action->setEnabled( false );
  connect( m_abort_action, SIGNAL( activated() ), this, SLOT( slotAbort() ) );

  m_search_action = m_menu->searchAction();
  m_search_action->setGroup( "BrowserWidget" );
  m_search_action->setEnabled( true );
  connect( m_search_action, SIGNAL( activated() ), this, SLOT( slotSearch() ) );

  m_manual_action = m_menu->manualMountAction();
  m_manual_action->setGroup( "BrowserWidget" );
  m_manual_action->setEnabled( true );
  connect( m_manual_action, SIGNAL( activated() ), this, SLOT( slotManualMount() ) );
}


void Smb4KBrowserWidget::setOpen( QListViewItem *item, bool open )
{
  if ( open )
  {
    switch ( item->depth() )
    {
      case 0:
      {
        Smb4KWorkgroupItem *i = smb4k_core->scanner()->getWorkgroup( item->text( Network ) );
        smb4k_core->scanner()->getWorkgroupMembers( item->text( Network ), i->master(), i->ip() );
        break;
      }
      case 1:
      {
        smb4k_core->scanner()->getShares( item->parent()->text( Network ), item->text( Network ), item->text( IP ) );
        break;
      }
      default:
        break;
    }
  }
  else
  {
    switch ( item->depth() )
    {
      case 1:
      {
        // Remove all children (shares).
        QListViewItem *shareItem = item->firstChild();

        while ( shareItem )
        {
          delete shareItem;
          shareItem = item->firstChild();
        }

        break;
      }
      default:
        break;
    }
  }

  KListView::setOpen( item, open );
}


void Smb4KBrowserWidget::insertItem( Smb4KHostItem *item )
{
  if ( smb4k_core->scanner()->getWorkgroup( item->workgroup() ) == 0 )
  {
    smb4k_core->scanner()->appendWorkgroup( new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
  }

  // Now put the host into the list.
  QListViewItem *workgroupItem = findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

  if ( workgroupItem )
  {
    QListViewItem *hostItem = findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( !hostItem )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );

      emit itemInserted( item );
    }
    else if ( hostItem && QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) != 0 )
    {
      Smb4KBrowserWidgetItem *newItem = new Smb4KBrowserWidgetItem( workgroupItem, item );
      newItem->setExpandable( true );

      emit itemInserted( item );
    }
    else
    {
      // Do not need to do anything here. The user is notified by the search dialog if
      // a host is already in the list.
    }
  }
  else
  {
    Smb4KBrowserWidgetItem *groupItem = new Smb4KBrowserWidgetItem( this, new Smb4KWorkgroupItem( item->workgroup(), item->name(), item->ip() ) );
    groupItem->setExpandable( true );

    Smb4KBrowserWidgetItem *hostItem = new Smb4KBrowserWidgetItem( groupItem, item );
    hostItem->hostItem()->setMaster( true );  // pseudo master
    hostItem->setExpandable( true );

    emit itemInserted( item );
  }
}


void Smb4KBrowserWidget::readOptions()
{
  // Read the configuration.
  config()->setGroup( "Appearance" );
  bool showType = config()->readBoolEntry( "Show Type", true );
  bool showComment = config()->readBoolEntry( "Show Comment", true );
  bool showIP = config()->readBoolEntry( "Show IP", true );

  // Now put everything back in according to the wishes of the user.
  if ( showIP )
  {
    setColumnWidth( IP, 10 );
    setColumnWidthMode( IP, QListView::Maximum );
  }
  else
  {
    setColumnWidth( IP, 0 );
    setColumnWidthMode( IP, QListView::Manual );
  }

  if ( showType )
  {
    setColumnWidth( Type, 10 );
    setColumnWidthMode( Type, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Type, 0 );
    setColumnWidthMode( Type, QListView::Manual );
  }

  if ( showComment )
  {
    setColumnWidth( Comment, 10 );
    setColumnWidthMode( Comment, QListView::Maximum );
  }
  else
  {
    setColumnWidth( Comment, 0 );
    setColumnWidthMode( Comment, QListView::Manual );
  }

  // Adjust the columns. Avoid those that have a width of 0.
  int index = 0;

  while ( index < this->columns() )
  {
    if ( this->columnWidth( index ) != 0 )
    {
      adjustColumn( index );
    }

    index++;
  }

  // Initialize the rest.
  m_hidden = config()->readBoolEntry( "Show Hidden", true );
  m_ipc = config()->readBoolEntry( "Show IPC", false );
  m_admin = config()->readBoolEntry( "Show ADMIN", false );
  m_printer = config()->readBoolEntry( "Show Printer", true );
  m_show_tooltip = config()->readBoolEntry( "Show Network Tooltip", true );
}


void Smb4KBrowserWidget::contentsMouseMoveEvent( QMouseEvent *e )
{
  m_pos = viewport()->mapFromGlobal( e->globalPos() );

  Smb4KBrowserWidgetItem *item = (Smb4KBrowserWidgetItem *)itemAt( m_pos );

  if ( item && m_show_tooltip )
  {
    // Check if we are on the root decoration.
    bool on_root = true;

    if ( m_pos.x() > header()->sectionPos( header()->mapToIndex( 0 ) ) +
         treeStepSize() * ( item->depth() + ( rootIsDecorated() ? 1 : 0 ) ) + itemMargin() ||
         m_pos.x() < header()->sectionPos( header()->mapToIndex( 0 ) ) )
    {
      on_root = false;
    }

    // Show the tool tip if there is an item, the list view has the mouse,
    // and the user wants to see it. Additionally, make sure that it is only shown,
    // if we are on the executable area (here: first column) and not on the root
    // decoration.
    if ( hasMouse() && isExecuteArea( m_pos ) /* first column */ && !on_root )
    {
      if ( !m_tooltip )
      {
        m_tooltip = new Smb4KNetworkItemTooltip( item, e->globalPos() );

        QTimer::singleShot( 2000, this, SLOT( slotShowTooltip() ) );
      }
      else if ( m_tooltip )
      {
        if ( m_tooltip->item() != item )
        {
          // If we get here, the user moved away from the previous
          // item. So, we will kill the process, that should retrieve
          // the additional info for it.
          if ( smb4k_core->scannerIsRunning() &&
               smb4k_core->scannerState() == SCANNER_RETRIEVING_INFO )
          {
            smb4k_core->scanner()->abort();
          }

          delete m_tooltip;
          m_tooltip = NULL;
        }
      }
    }
    else
    {
      if ( m_tooltip )
      {
        delete m_tooltip;
        m_tooltip = NULL;
      }
    }
  }
  else
  {
    if ( m_tooltip )
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }

  KListView::contentsMouseMoveEvent( e );
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KBrowserWidget::slotShowTooltip()
{
  if ( m_tooltip )
  {
    if ( m_tooltip->item()->depth() == 1 &&
         m_tooltip->item() == (Smb4KBrowserWidgetItem *)itemAt( m_pos ) )
    {
      // We do not need to do additional checks here, because
      // the scanner will only start a process, if this item wasn't
      // checked for additional information already.
      smb4k_core->scanner()->getInfo( m_tooltip->item()->hostItem()->workgroup(),
                                      m_tooltip->item()->hostItem()->name(),
                                      m_tooltip->item()->hostItem()->ip() );
    }

    if ( hasMouse() && isExecuteArea( m_pos ) )
    {
      m_tooltip->update();
      m_tooltip->showTip();
    }
    else
    {
      delete m_tooltip;
      m_tooltip = NULL;
    }
  }
}


void Smb4KBrowserWidget::slotItemExecuted( QListViewItem *item )
{
  if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( item )
  {
    if ( item->depth() == 2 )
    {
      if ( item->text( Type ).contains( "Printer" ) == 0 )
      {
        slotMountShare();
      }
      else
      {
        slotPrint();
      }
    }
    else
    {
      setOpen( item, !item->isOpen() );
    }
  }
}


void Smb4KBrowserWidget::slotItemExpandedCollapsed( QListViewItem *item )
{
  setSelected( item, true );
}


void Smb4KBrowserWidget::slotSelectionChanged( QListViewItem *item )
{
  if ( item && item->depth() == 2 )
  {
    m_preview_action->setEnabled( true );
    m_askpass_action->setEnabled( true );

    if ( item->text( Type ).contains( "Printer" ) == 0 )
    {
      m_bookmark_action->setEnabled( true );
      m_preview_action->setEnabled( true );
      m_mount_action->setEnabled( true );
      m_print_action->setEnabled( false );
    }
    else
    {
      m_bookmark_action->setEnabled( false );
      m_preview_action->setEnabled( false );
      m_mount_action->setEnabled( false );
      m_print_action->setEnabled( true );
    }
  }
  else if ( item && item->depth() == 1 )
  {
    m_bookmark_action->setEnabled( false );
    m_askpass_action->setEnabled( true );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
  }
  else
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotRightButtonPressed( QListViewItem *item, const QPoint &point, int )
{
  if ( !item )
  {
    m_menu->popupMenu()->changeTitle( 0, SmallIcon( "network" ), i18n( "Network" ) );
  }
  else
  {
    m_menu->popupMenu()->changeTitle( 0, *(item->pixmap( 0 )), item->text( 0 ) );

    if ( item->depth() == 0 )
    {
      m_rescan_action->setText( i18n( "Scan Wo&rkgroup" ) );
    }
    else
    {
      m_rescan_action->setText( i18n( "Scan Compute&r" ) );
    }
  }

  m_menu->popupMenu()->exec( point, 0 );

  m_rescan_action->setText( i18n( "Scan Netwo&rk" ) );
}


void Smb4KBrowserWidget::slotMouseButtonPressed( QListViewItem *item )
{
    if ( m_tooltip )
  {
    delete m_tooltip;
    m_tooltip = NULL;
  }

  if ( !item && !selectedItem() )
  {
    m_bookmark_action->setEnabled( false );
    m_preview_action->setEnabled( false );
    m_print_action->setEnabled( false );
    m_mount_action->setEnabled( false );
    m_askpass_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list )
{
  // Clear the list view:
  clear();

  // Adjust the columns.
  for ( int index = 0; index < columns(); index++ )
  {
    if ( this->columnWidth( index ) != 0 )
    {
      adjustColumn( index );
    }
  }

  // Put the workgroups into the list view.
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KWorkgroupItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      if ( findItem( (*it)->workgroup(), Network, CaseSensitive|ExactMatch ) == 0 )
      {
        Smb4KBrowserWidgetItem *workgroup = new Smb4KBrowserWidgetItem( this, *it );
        workgroup->setExpandable( true );
      }
      else
      {
        continue;
      }
    }
  }
}



void Smb4KBrowserWidget::slotMembers( const QValueList<Smb4KHostItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KHostItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      Smb4KBrowserWidgetItem *item = (Smb4KBrowserWidgetItem *)findItem( (*it)->name(), Network, CaseSensitive|ExactMatch );

      if ( item )
      {
        item->update( *it );

        continue;
      }
      else
      {
        QListViewItem *workgroup = findItem( (*it)->workgroup(), Network, ExactMatch|CaseSensitive );

        if ( workgroup )
        {
          Smb4KBrowserWidgetItem *host = new Smb4KBrowserWidgetItem( workgroup, *it );
          host->setExpandable( true );
        }
      }
    }
  }
}


void Smb4KBrowserWidget::slotShares( const QValueList<Smb4KShareItem *> &list )
{
  if ( !list.isEmpty() )
  {
    for ( QValueList<Smb4KShareItem *>::ConstIterator it = list.begin(); it != list.end(); ++it )
    {
      QListViewItem *host = findItem( (*it)->host(), Network, ExactMatch|CaseSensitive );

      if ( host != 0 )
      {
        if ( !m_hidden && (*it)->name().stripWhiteSpace().endsWith( "$" ) )
        {
          continue;
        }
        else
        {
          if ( !m_ipc && (*it)->name().contains( "IPC" ) != 0 )
          {
            continue;
          }

          if ( !m_admin && (*it)->name().contains( "ADMIN" ) != 0 )
          {
            continue;
          }
        }

        if ( !m_printer && QString::compare( (*it)->plainType(), "Printer" ) == 0 )
        {
          continue;
        }

        Smb4KBrowserWidgetItem *test = (Smb4KBrowserWidgetItem *)findItem( (*it)->name(), Network, ExactMatch|CaseSensitive );

        if ( !test || test->parent() != host )
        {
          Smb4KBrowserWidgetItem *share = new Smb4KBrowserWidgetItem( host, *it );
          share->setExpandable( false );

          continue;
        }
        else if ( test && test->parent() == host )
        {
          if ( QString::compare( test->text( Comment ), (*it)->comment() ) != 0 )
          {
            test->update( *it );
          }

          continue;
        }
      }
    }
  }
}


void Smb4KBrowserWidget::slotMountShare()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    smb4k_core->mounter()->mountShare( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ) );
  }
}


void Smb4KBrowserWidget::slotAskPass()
{
  if ( currentItem() && currentItem()->depth() == 1 )
  {
    QString workgroup = currentItem()->parent()->text( Network );
    QString host = currentItem()->text( Network );
    QString share = QString::null;

    smb4k_core->passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData );
  }
  else if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );

    smb4k_core->passwordHandler()->askpass( workgroup, host, share, Smb4KPasswordHandler::NewData );
  }
}


void Smb4KBrowserWidget::slotPreview()
{
  QString workgroup = currentItem()->parent()->parent()->text( Network );
  QString host = currentItem()->parent()->text( Network );
  QString ip = currentItem()->parent()->text( IP );
  QString share = currentItem()->text( Network );

  Smb4KPreviewDialog *preview = new Smb4KPreviewDialog( workgroup, host, ip, share, 0, 0, false );
  preview->show();

  // FIXME: Do we produce a memory leak here? We have Qt::WDestructiveClose set in
  // Smb4KPreviewDialog.
}


void Smb4KBrowserWidget::slotAddBookmark()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    QString host = currentItem()->parent()->text( Network );
    QString share = currentItem()->text( Network );
    QString workgroup = currentItem()->parent()->parent()->text( Network );
    QString ip = currentItem()->parent()->text( IP );
    QString type = currentItem()->text( Type );

    Smb4KBookmark *bookmark = new Smb4KBookmark( host, share, workgroup, ip, type );
    smb4k_core->bookmarkHandler()->addBookmark( bookmark );
  }
}


void Smb4KBrowserWidget::slotRescan()
{
  bool viewport_has_mouse;

  if ( m_pos.x() <= 0 || viewport()->width() <= m_pos.x() ||
       m_pos.y() <= 0 || viewport()->height() <= m_pos.y() )
  {
    viewport_has_mouse = false;
  }
  else
  {
    viewport_has_mouse = true;
  }

  if ( currentItem() && selectedItem() && viewport_has_mouse )
  {
    if ( currentItem()->depth() == 0 )
    {
      if ( !currentItem()->isOpen() )
      {
        setOpen( currentItem(), true );
      }
      else
      {
        Smb4KWorkgroupItem *item = ((Smb4KBrowserWidgetItem *)currentItem())->workgroupItem();
        smb4k_core->scanner()->getWorkgroupMembers( item->workgroup(), item->master(), item->ip() );
      }
    }
    else if ( currentItem()->depth() == 1 )
    {
      if ( !currentItem()->isOpen() )
      {
        setOpen( currentItem(), true );
      }
      else
      {
        Smb4KHostItem *item = ((Smb4KBrowserWidgetItem *)currentItem())->hostItem();
        smb4k_core->scanner()->getShares( item->workgroup(), item->name(), item->ip() );
      }
    }
    else
    {
      Smb4KHostItem *item = ((Smb4KBrowserWidgetItem *)currentItem()->parent())->hostItem();
      smb4k_core->scanner()->getShares( item->workgroup(), item->name(), item->ip() );
    }
  }
  else
  {
    smb4k_core->scanner()->rescan();
  }
}


void Smb4KBrowserWidget::slotAddIPAddress( Smb4KHostItem *item )
{
  if ( item )
  {
    Smb4KBrowserWidgetItem *workgroup = (Smb4KBrowserWidgetItem *)findItem( item->workgroup(), Network, ExactMatch|CaseSensitive );

    if ( workgroup && QString::compare( workgroup->text( Network ), item->workgroup() ) == 0 )
    {
      Smb4KWorkgroupItem *wg = smb4k_core->scanner()->getWorkgroup( item->workgroup() );

      if ( wg )
      {
        workgroup->update( wg );
      }
    }

    Smb4KBrowserWidgetItem *hostItem = (Smb4KBrowserWidgetItem *)findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( hostItem && hostItem->parent() && QString::compare( hostItem->parent()->text( Network ), item->workgroup() ) == 0 )
    {
      hostItem->update( item );

      if ( columnWidth( IP ) != 0 )
      {
        adjustColumn( IP );
      }
    }
  }
}


void Smb4KBrowserWidget::slotPrint()
{
  if ( currentItem() && currentItem()->depth() == 2 )
  {
    Smb4KPrintDialog *print = new Smb4KPrintDialog( currentItem()->parent()->parent()->text( Network ), currentItem()->parent()->text( Network ), currentItem()->parent()->text( IP ), currentItem()->text( Network ), currentItem()->text( Comment ), this, "PrintDialog" );
    print->exec();
  }
}


void Smb4KBrowserWidget::slotMarkShares()
{
  config()->setGroup( "Appearance" );
  bool show_all = config()->readBoolEntry( "Show All Shares", false );

  QListViewItemIterator it( this );
  QListViewItem *item;

  while ((item = it.current()) != 0 )
  {
    ++it;

    if ( item->depth() == 2 )
    {
      Smb4KShare *share = smb4k_core->mounter()->findShareByName( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) );

      if ( smb4k_core->mounter()->isMounted( QString( "//%1/%2" ).arg( item->parent()->text( Network ), item->text( Network ) ) ) &&
           (!share->isForeign() || show_all) )
      {
        if ( !((Smb4KBrowserWidgetItem *)item)->isMounted() )
        {
          ((Smb4KBrowserWidgetItem *)item)->setMounted( true );
        }

        continue;
      }
      else
      {
        ((Smb4KBrowserWidgetItem *)item)->setMounted( false );
        continue;
      }
    }
    else
    {
      continue;
    }
  }
}


void Smb4KBrowserWidget::slotSearch()
{
  emit searchRequest();
}


void Smb4KBrowserWidget::slotInsertItem( Smb4KHostItem *item )
{
  insertItem( item );
}


void Smb4KBrowserWidget::slotAbort()
{
  smb4k_core->scanner()->abort();
}


void Smb4KBrowserWidget::slotManualMount()
{
  Smb4KMountDialog *dlg = new Smb4KMountDialog( 0, "ManualMountDialog" );

  dlg->show();

  // The dialog will be closed distructively.
}


void Smb4KBrowserWidget::slotReceiveScannerState( int state )
{
  if ( state != SCANNER_STOP )
  {
    if ( state != SCANNER_SEARCHING  )
    {
      m_rescan_action->setEnabled( false );
    }

    m_abort_action->setEnabled( true );
  }
  else
  {
    m_rescan_action->setEnabled( true );
    m_abort_action->setEnabled( false );
  }
}


void Smb4KBrowserWidget::slotAddInfo( Smb4KHostItem *item )
{
  if ( item )
  {
    Smb4KBrowserWidgetItem *host = (Smb4KBrowserWidgetItem *)findItem( item->name(), Network, ExactMatch|CaseSensitive );

    if ( host && QString::compare( host->parent()->text( Network ), item->workgroup() ) == 0 )
    {
      host->update( item );
    }

    if ( m_tooltip && m_tooltip->item() == host )
    {
      m_tooltip->update();
    }
  }
}

#include "smb4kbrowserwidget.moc"
