/***************************************************************************
    smb4ksambaoptionshandler  -  This class handles the Samba options.
                             -------------------
    begin                : So Mai 14 2006
    copyright            : (C) 2006 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qfile.h>
#include <qtextstream.h>
#include <qdir.h>

// KDE includes
#include <kstandarddirs.h>
#include <kprocess.h>
#include <kdebug.h>

// system specific includes
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>

// application specific includes
#include "smb4ksambaoptionshandler.h"
#include "smb4kdefs.h"
#include "smb4kerror.h"



Smb4KSambaOptionsHandler::Smb4KSambaOptionsHandler( KConfig *config, QObject *parent, const char *name )
: QObject( parent, name ), m_config( config )
{
  if ( !m_config )
  {
    kdFatal() << "Smb4KSambaOptionsHandler: No KConfig object. Bailing out ..." << endl;
  }

  // We need the directory.
  KStandardDirs *stddir = new KStandardDirs();
  QString dir = locateLocal( "data", "smb4k", KGlobal::instance() );

  if ( !stddir->exists( dir ) )
  {
    stddir->makeDir( dir );
  }

  delete stddir;

  m_wins_server = QString::null;
}


Smb4KSambaOptionsHandler::~Smb4KSambaOptionsHandler()
{
  for ( QValueList<Smb4KSambaOptionsInfo *>::Iterator it = m_list.begin();
        it != m_list.end(); ++it )
  {
    delete *it;
  }

  m_list.clear();
}


const QValueList<Smb4KSambaOptionsInfo *> &Smb4KSambaOptionsHandler::customOptionsList()
{
  if ( m_list.isEmpty() )
  {
    read_options();
  }

  return m_list;
}


void Smb4KSambaOptionsHandler::read_options()
{
  // Clear the list before filling it (again)
  if ( !m_list.isEmpty() )
  {
    for ( QValueList<Smb4KSambaOptionsInfo *>::Iterator it = m_list.begin();
          it != m_list.end(); ++it )
    {
      delete *it;
    }

    m_list.clear();
  }

  QFile file( locateLocal( "data", "smb4k/custom_options", KGlobal::instance() ) );

  QStringList contents;

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
    ts.setEncoding( QTextStream::Locale );

    contents = QStringList::split( '\n', ts.read(), true );

    file.close();
  }
  else
  {
    if ( file.exists() )
    {
      Smb4KError::error( ERROR_READING_FILE, file.name() );
    }

    return;
  }

  if ( !contents.isEmpty() )
  {
    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
    {
      if ( (*it).startsWith( "[" ) )
      {
        Smb4KSambaOptionsInfo *info = new Smb4KSambaOptionsInfo( (*it).section( "[", 1, -1 ).section( "]", -2, 0 ) );

        for ( QStringList::ConstIterator i = ++it; i != contents.end(); ++i )
        {
          if ( (*i).startsWith( "remount=" ) )
          {
            bool remount = QString::compare( (*i).section( "=", 1, 1 ).stripWhiteSpace(), "true" ) == 0 ? true : false;

            info->setRemount( remount );

            continue;
          }
          else if ( (*i).startsWith( "port=" ) )
          {
            int port = (*i).section( "=", 1, 1 ).stripWhiteSpace().toInt();

            info->setPort( port );

            continue;
          }
#ifndef __FreeBSD__
          else if ( (*i).startsWith( "filesystem=" ) )
          {
            info->setFilesystem( (*i).section( "=", 1, 1 ).stripWhiteSpace() );

            continue;
          }
          else if ( (*i).startsWith( "read-write=" ) )
          {
            info->setReadWrite( QString::compare( (*i).section( "=", 1, 1 ).stripWhiteSpace(), "true" ) == 0 );

            continue;
          }
#endif
          else if ( (*i).startsWith( "protocol=" ) )
          {
            info->setProtocol( (*i).section( "=", 1, 1 ).stripWhiteSpace() );

            continue;
          }
          else if ( (*i).startsWith( "kerberos=" ) )
          {
            info->setKerberos( QString::compare( (*i).section( "=", 1, 1 ).stripWhiteSpace(), "true" ) == 0 );

            continue;
          }
          else if ( (*i).startsWith( "uid=" ) )
          {
            info->setUID( (*i).section( "=", 1, 1 ).stripWhiteSpace() );

            continue;
          }
          else if ( (*i).startsWith( "gid=" ) )
          {
            info->setGID( (*i).section( "=", 1, 1 ).stripWhiteSpace() );

            continue;
          }
          else if ( (*i).isEmpty() || (*i).stripWhiteSpace().startsWith( "[" ) )
          {
            it = i;

            break;
          }
          else
          {
            continue;
          }
        }

        m_list.append( info );
      }
      else
      {
        continue;
      }
    }
  }
}


void Smb4KSambaOptionsHandler::write_options()
{
  m_config->setGroup( "Samba" );
  int default_port = m_config->readNumEntry( "Port", 139 );
  QString default_protocol = m_config->readEntry( "Net Protocol", "auto" );
  bool default_kerberos = m_config->readBoolEntry( "Use Kerberos", false );
  QString default_uid = m_config->readEntry( "Mount UID", QString::null ); // config dlg: i18n( "default" )
  QString default_gid = m_config->readEntry( "Mount GID", QString::null ); // config dlg: i18n( "default" )
#ifndef __FreeBSD__
  QString default_filesystem = m_config->readEntry( "Mount Filesystem", "cifs" );
  bool default_readwrite = m_config->readBoolEntry( "Mount ReadWrite", true );
#endif

  QFile file( locateLocal( "data", "smb4k/custom_options", KGlobal::instance() ) );

  if ( !m_list.isEmpty() )
  {
    if ( file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &file );
      ts.setEncoding( QTextStream::Locale );

      for ( QValueList<Smb4KSambaOptionsInfo *>::ConstIterator it = m_list.begin(); it != m_list.end(); ++it )
      {
        switch ( (*it)->type() )
        {
          case Smb4KSambaOptionsInfo::Host:
          {
            // Check if we need to write anything:
            if ( (*it)->port() != -1 ||
                 !(*it)->protocol().isEmpty() ||
                 (*it)->kerberos() != default_kerberos )
            {
              ts << "[" << (*it)->itemName() << "]" << endl;
              ts << "port=" << ((*it)->port() != -1 ? (*it)->port() : default_port) << endl;
              ts << "kerberos=" << ((*it)->kerberos() ? "true" : "false") << endl;
              ts << "protocol=" << (!(*it)->protocol().stripWhiteSpace().isEmpty() ?
                                   (*it)->protocol() : default_protocol) << endl;
            }
            else
            {
              // Do nothing
            }

            break;
          }
          case Smb4KSambaOptionsInfo::Share:
          {
            if ( (*it)->port() != -1 ||
                 (*it)->remount() ||
#ifndef __FreeBSD__
                 !(*it)->filesystem().isEmpty() ||
                 (*it)->kerberos() != default_kerberos ||
                 (*it)->readwrite() != default_readwrite ||
#endif
                 !(*it)->uid().isEmpty() ||
                 !(*it)->gid().isEmpty() )
            {
              ts << "[" << (*it)->itemName() << "]" << endl;
              ts << "port=" << ((*it)->port() != -1 ? (*it)->port() : default_port) << endl;
              ts << "remount=" << ((*it)->remount() ? "true" : "false") << endl;
#ifndef __FreeBSD__
              ts << "kerberos=" << ((*it)->kerberos() ? "true" : "false") << endl;
              ts << "filesystem=" << (!(*it)->filesystem().stripWhiteSpace().isEmpty() ?
                                     (*it)->filesystem() : default_filesystem) << endl;
              ts << "read-write=" << ((*it)->readwrite() ? "true" : "false") << endl;
#endif
              // Since by default the default_uid string is empty (see
              // above), we check here if data needs to be written.
              if ( !(*it)->uid().isEmpty() || !default_uid.isEmpty() )
              {
                ts << "uid=" << (!(*it)->uid().stripWhiteSpace().isEmpty() ?
                                (*it)->uid() : default_uid) << endl;
              }

              // Since by default the default_gid string is empty (see
              // above), we check here if data needs to be written.
              if ( !(*it)->gid().isEmpty() || !default_gid.isEmpty() )
              {
                ts << "gid=" << (!(*it)->gid().stripWhiteSpace().isEmpty() ?
                                (*it)->gid() : default_gid) << endl;
              }
            }
            else
            {
              // Do nothing
            }

            break;
          }
          default:
          {
            break;
          }
        }

        ts << endl;
      }

      file.close();
    }
  }
  else
  {
    file.remove();
  }
}


void Smb4KSambaOptionsHandler::remount( Smb4KShare *share, bool yes )
{
  if ( share )
  {
    Smb4KSambaOptionsInfo *info = NULL;

    if ( (info = find_item( share->name() )) )
    {
      info->setRemount( yes );
    }
    else if ( !info && yes )
    {
      info = new Smb4KSambaOptionsInfo( share );
      info->setRemount( yes );

      m_list.append( info );
    }
  }
}


void Smb4KSambaOptionsHandler::sync()
{
  write_options();
}


Smb4KSambaOptionsInfo *Smb4KSambaOptionsHandler::find_item( const QString &item )
{
  // If the list is empty, we'll read the file.
  if ( m_list.isEmpty() )
  {
    read_options();
  }

  QString host = item.section( "/", 2, 2 ).stripWhiteSpace();

  Smb4KSambaOptionsInfo *info = NULL;

  if ( !item.stripWhiteSpace().isEmpty() )
  {
    for ( QValueList<Smb4KSambaOptionsInfo *>::ConstIterator it = m_list.begin();
          it != m_list.end(); ++it )
    {
      if ( QString::compare( item.lower(), (*it)->itemName().lower() ) == 0 )
      {
        info = *it;

        break;
      }
      else if ( QString::compare( host.lower(), (*it)->itemName().lower() ) == 0 )
      {
        if ( !info )
        {
          info = *it;
        }

        continue;
      }
      else
      {
        continue;
      }
    }
  }

  return info;
}


const QString Smb4KSambaOptionsHandler::smbclientOptions( const QString &share )
{
  // Get the global Samba options
  (void) globalSambaOptions();

  Smb4KSambaOptionsInfo *info = find_item( share );
  QString args = QString::null;

  // Read the options:
  m_config->setGroup( "Samba" );
  QString resolve_order = m_config->readEntry( "Client Resolve Order", m_samba_options["name resolve order"] );
  QString netbios_name = m_config->readEntry( "NetBIOS Name", m_samba_options["netbios name"] );
  QString netbios_scope = m_config->readEntry( "NetBIOS Scope", m_samba_options["netbios scope"] );
  QString socket_options = m_config->readEntry( "Socket Options", m_samba_options["socket options"] );
  QString buffer_size = m_config->readEntry( "Client Buffer Size", QString::null );
  bool kerberos = info ? info->kerberos() : m_config->readBoolEntry( "Use Kerberos", false );
  int port = info && info->port() != -1 ? info->port() : m_config->readNumEntry( "Port", 139 );

  // Options that aren't customizable:
  args.append( !resolve_order.isEmpty() ? QString( " -R %1" ).arg( KProcess::quote( resolve_order ) ) : "" );
  args.append( !buffer_size.isEmpty() ? QString( " -b %1" ).arg( buffer_size ) : "" );
  args.append( !netbios_name.isEmpty() ? QString( " -n %1" ).arg( KProcess::quote( netbios_name ) ) : "" );
  args.append( !netbios_scope.isEmpty() ? QString( " -i %1" ).arg( KProcess::quote( netbios_scope ) ) : "" );
  args.append( !socket_options.isEmpty() ? QString( " -O %1" ).arg( KProcess::quote( socket_options ) ) : "" );

  // Customizable options. The values are managed above.
  args.append( kerberos ? " -k" : "" );
  args.append( QString( " -p %1" ).arg( port ) );

  return args;
}


const QString Smb4KSambaOptionsHandler::nmblookupOptions( bool with_broadcast )
{
  // Get the global Samba options
  (void) globalSambaOptions();

  QString args = QString::null;

  m_config->setGroup( "Samba" );
  QString netbios_name = m_config->readEntry( "NetBIOS Name", m_samba_options["netbios name"] );
  QString netbios_scope = m_config->readEntry( "NetBIOS Scope", m_samba_options["netbios scope"] );
  QString socket_options = m_config->readEntry( "Socket Options", m_samba_options["socket options"] );
  QString domain = m_config->readEntry( "Domain", m_samba_options["workgroup"] );
  QString broadcast = m_config->readEntry( "NMB Broadcast", QString::null );
  bool port_137 = m_config->readBoolEntry( "NMB Port 137", false );

  // Options that aren't costumizable:

  if ( !netbios_name.isEmpty() )
  {
    args.append( QString( " -n %1" ).arg( KProcess::quote( netbios_name ) ) );
  }

  if ( !netbios_scope.isEmpty() )
  {
    args.append( " -i %1" ).arg( KProcess::quote( netbios_scope ) );
  }

  if ( !socket_options.isEmpty() )
  {
    args.append( QString( " -O %1" ).arg( KProcess::quote( socket_options ) ) );
  }

  if ( !domain.isEmpty() )
  {
    args.append( QString( " -W %1" ).arg( KProcess::quote( domain ) ) );
  }

  if ( !broadcast.isEmpty() && with_broadcast )
  {
    args.append( QString( " -B %1" ).arg( KProcess::quote( broadcast ) ) );
  }

  if ( port_137 )
  {
    args.append( " -r" );
  }

  return args;
}


const QString Smb4KSambaOptionsHandler::netOptions( int command, const QString &networkItem, const QString &protocol )
{
  QString args = QString::null;

  Smb4KSambaOptionsInfo *info = find_item( networkItem );

  m_config->setGroup( "Samba" );
  QString default_protocol = info && !info->protocol().isEmpty() ? info->protocol() : m_config->readEntry( "Net Protocol", "auto" );
  bool machine_account = m_config->readBoolEntry( "Net Machine Account", false );
  QString domain = m_config->readEntry( "Domain", m_samba_options["workgroup"] );
  QString netbios_name = m_config->readEntry( "NetBIOS Name", m_samba_options["netbios name"] );
  int port = info && info->port() != -1 ? info->port() : m_config->readNumEntry( "Port", 139 );

  switch ( command )
  {
    case Share:
    {
      if ( protocol.stripWhiteSpace().isEmpty() )
      {
        // The 'share' command only works with the RAP or RPC protocol:
        if ( QString::compare( default_protocol, "auto" ) != 0 && QString::compare( default_protocol, "ads" ) != 0 )
        {
          args.append( " "+default_protocol );
        }
      }
      else
      {
        args.append( " "+protocol );
      }

      args.append( " share -l" );

      break;
    }
    case ServerDomain:
    {
      args.append( " rap server" );

      // No network item is needed here.

      break;
    }
    case LookupHost:
    {
      args.append( " lookup host" );

      if ( networkItem.isEmpty() )
      {
        Smb4KError::error( ERROR_NET_COMMAND, args.stripWhiteSpace() );
        return QString::null;
      }

      args.append( " "+networkItem );

      break;
    }
    case LookupMaster:
    {
      args.append( " lookup master" );

      if ( networkItem.stripWhiteSpace().isEmpty() )
      {
        Smb4KError::error( ERROR_NET_COMMAND, args.stripWhiteSpace() );
        return QString::null;
      }

      args.append( " "+networkItem );

      break;
    }
    case Domain:
    {
      args.append( " rap domain" );

      break;
    }
    default:
    {
      return args;

      // Is this necessary? It won't be called, that's for sure.
      break;
    }
  };

  args.append( machine_account ? " -P" : "" );
  args.append( QString( " -W %1" ).arg( KProcess::quote( domain ) ) );
  args.append( QString( " -n %1" ).arg( KProcess::quote( netbios_name ) ) );
  args.append( QString( " -p %1" ).arg( port ) );

  return args;
}


const QString Smb4KSambaOptionsHandler::mountOptions( const QString &share )
{
  Smb4KSambaOptionsInfo *info = find_item( share );
  QString args;

  // Read the global Samba options from smb.conf:
  (void) globalSambaOptions();

  m_config->setGroup( "Super User Privileges" );
  bool run_suid = m_config->readBoolEntry( "Run SUID", false );

  m_config->setGroup( "Samba" );
  QString uid = info && !info->uid().isEmpty() ?
                info->uid() :
                m_config->readEntry( "Mount UID", QString::null );
  QString gid = info && !info->gid().isEmpty() ?
                info->gid() :
                m_config->readEntry( "Mount GID", QString::null );
  QString port = QString( "%1" ).arg( info && info->port() != -1 ?
                                      info->port() :
                                      m_config->readNumEntry( "Port", 139 ) );
  QString charset = m_config->readEntry( "Mount Charset", QString::null );
  QString codepage = m_config->readEntry( "Mount Codepage", QString::null );
  QString f_mask = m_config->readEntry( "Mount FMASK", QString::null );
  QString d_mask = m_config->readEntry( "Mount DMASK", QString::null );

#ifndef __FreeBSD__

  QString filesystem = info && !info->filesystem().isEmpty() ?
                       info->filesystem() :
                       m_config->readEntry( "Mount Filesystem", "cifs" );
  bool kerberos = info ?
                  info->kerberos() :
                  m_config->readBoolEntry( "Use Kerberos", false );
  bool read_write = info ?
                    info->readwrite() :
                    m_config->readBoolEntry( "Mount ReadWrite", true );

  QString netbios_name = m_config->readEntry( "NetBIOS Name", QString::null );
  QString socket_options = m_config->readEntry( "Socket Options", QString::null );
  QString netbios_scope = m_config->readEntry( "NetBIOS Scope", QString::null );
  QString mount_cache = m_config->readEntry( "Mount Cache", QString::null );
  bool unicode = m_config->readBoolEntry( "Mount Unicode", false );
  bool lfs = m_config->readBoolEntry( "Mount LFS", false );
  bool rsize = m_config->readBoolEntry( "Mount RSize", false );
  bool wsize = m_config->readBoolEntry( "Mount WSize", false );

  if ( QString::compare( filesystem.lower(), "smbfs" ) == 0 )
  {
    args.append( kerberos ? "krb," : "" );
    args.append( !socket_options.isEmpty() ? QString( "sockopt=\"%1\"," ).arg( socket_options ) : "" );
    args.append( !netbios_scope.isEmpty() ? QString( "scope=\"%1\"," ).arg( netbios_scope ) : "" );
    args.append( !codepage.isEmpty() ? QString( "codepage=%1," ).arg( codepage ) : "" );
    args.append( !mount_cache.isEmpty() ? QString( "ttl=%1," ).arg( mount_cache ) : "" );
    args.append( unicode ? "unicode," : "" );
    args.append( lfs ? "lfs," : "" );
    args.append( !f_mask.isEmpty() ? QString( "fmask=%1," ).arg( f_mask ) : "" );
    args.append( !d_mask.isEmpty() ? QString( "dmask=%1," ).arg( d_mask ) : "" );
  }
  else
  {
    // The file_mode and the dir_mode option _have_ to take the file mask
    // and the directory mask in octal, respectively!

    args.append( !f_mask.isEmpty() ? QString( "file_mode=%1," ).arg( f_mask ) : "" );
    args.append( !d_mask.isEmpty() ? QString( "dir_mode=%1," ).arg( d_mask ) : "" );
    args.append( rsize ? "rsize," : "" );
    args.append( wsize ? "wsize," : "" );
  }

  args.append( !netbios_name.isEmpty() ? QString( "netbiosname='%1'," ).arg( netbios_name ) : "" );
  args.append( !charset.isEmpty() ? QString( "iocharset=%1," ).arg( charset ) : "" );
  args.append( !uid.isEmpty() ? QString( "uid=%1," ).arg( uid ) :
               (run_suid ? QString( "uid=%1," ).arg( (int)getuid() ) : "") );
  args.append( !gid.isEmpty() ? QString( "gid=%1," ).arg( gid ) :
               (run_suid ? QString( "gid=%1," ).arg( (int)getgid() ) : "") );
  args.append( !port.isEmpty() ? QString( "port=%1," ).arg( port ) : "" );
  args.append( read_write ? "rw," : "ro," );

#else

  // FreeBSD: The port must be managed by the mounter itself!!!
  args.append( !uid.isEmpty() ? QString( " -u %1" ).arg( uid ) :
               (run_suid ? QString( " -u %1," ).arg( (int)getuid() ) : "") );
  args.append( !gid.isEmpty() ? QString( " -g %1" ).arg( gid ) :
               (run_suid ? QString( " -g %1," ).arg( (int)getgid() ) : "") );
  args.append( !f_mask.isEmpty() ? QString( " -f %1" ).arg( f_mask ) : "" );
  args.append( !d_mask.isEmpty() ? QString( " -d %1" ).arg( d_mask ) : "" );
  args.append( !charset.isEmpty() && !codepage.isEmpty() ? QString( " -E %1:%2" ).arg( charset, codepage ) : "" );

#endif

  return args;
}


void Smb4KSambaOptionsHandler::read_smb_conf()
{
  // Clear the options list before reading.
  m_samba_options.clear();

  QStringList paths;
  paths << "/etc";
  paths << "/etc/samba";
  paths << "/usr/local/etc";
  paths << "/usr/local/etc/samba";

  QFile f( "smb.conf" );

  QStringList contents;

  // Locate the file and read its contents:
  for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it )
  {
    QDir::setCurrent( *it );

    if ( f.exists() )
    {
      if ( f.open( IO_ReadOnly ) )
      {
        QTextStream ts( &f );
        ts.setEncoding( QTextStream::Locale );

        contents = QStringList::split( '\n', ts.read(), false );
      }

      f.close();

      break;
    }
    else
    {
      continue;
    }
  }

  // Process the file contents.
  for ( QStringList::Iterator it = contents.erase( contents.begin(), ++(contents.find( "[global]" )) ); it != contents.end(); ++it )
  {
    if ( (*it).stripWhiteSpace().startsWith( "#" ) || (*it).stripWhiteSpace().startsWith( ";" ) )
    {
      *it = QString::null;
    }
    else if ( (*it).stripWhiteSpace().startsWith( "include" ) )
    {
      // Put the contents of the included at this position.
      QString file = (*it).section( "=", 1, 1 ).stripWhiteSpace();
      *it = QString::null;
      f.setName( file );

      QStringList include;

      if ( f.exists() )
      {
        if ( f.open( IO_ReadOnly ) )
        {
          QTextStream ts( &f );
          ts.setEncoding( QTextStream::Locale );

          include = QStringList::split( '\n', ts.read(), false );
        }

        f.close();
      }

      for ( QStringList::Iterator i = include.begin(); i != include.end(); ++i )
      {
        if ( !(*i).stripWhiteSpace().isEmpty() )
        {
          contents.insert( it, *i );

          continue;
        }
        else
        {
          continue;
        }
      }

      continue;
    }
    else if ( (*it).startsWith( "[" ) )
    {
      contents.erase( it, contents.end() );

      break;
    }
    else
    {
      continue;
    }
  }

  contents.remove( QString::null );

  // Write all options into the map:
  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    QString key = (*it).section( "=", 0, 0 ).stripWhiteSpace().lower();
    m_samba_options[key] = QString( (*it).section( "=", 1, 1 ).stripWhiteSpace().upper() );
  }

  // Post-processing. Some values should be entered with their defaults, if they are
  // not already present.
  if ( !m_samba_options.contains( "netbios name" ) )
  {
    size_t hostnamelen = 255;
    char *hostname = new char[hostnamelen];

    if ( gethostname( hostname, hostnamelen ) == -1 )
    {
      int error = errno;
      Smb4KError::error( ERROR_GETTING_HOSTNAME, QString::null, strerror( error ) );
    }
    else
    {
      m_samba_options["netbios name"] = ( QString( "%1" ).arg( hostname ) ).upper();
    }

    delete [] hostname;
  }
}


const QMap<QString,QString> &Smb4KSambaOptionsHandler::globalSambaOptions()
{
  if ( m_samba_options.isEmpty() )
  {
    read_smb_conf();
  }

  return m_samba_options;
}


const QString &Smb4KSambaOptionsHandler::winsServer()
{
  if ( m_wins_server.isEmpty() )
  {
    (void) globalSambaOptions();

    if ( !m_samba_options["wins server"].isEmpty() )
    {
      m_wins_server = m_samba_options["wins server"];
    }
    else if ( !m_samba_options["wins support"].isEmpty() &&
              (QString::compare( m_samba_options["wins support"].lower(), "yes" ) == 0 ||
              QString::compare( m_samba_options["wins support"].lower(), "true" ) == 0) )
    {
      m_wins_server = "127.0.0.1";
    }
  }

  return m_wins_server;
}


void Smb4KSambaOptionsHandler::addItem( Smb4KSambaOptionsInfo *info, bool s )
{
  Smb4KSambaOptionsInfo *item = find_item( info->itemName() );

  if ( item && QString::compare( item->itemName().lower(), info->itemName().lower() ) == 0 )
  {
    item->setPort( info->port() );
#ifndef __FreeBSD__
    item->setFilesystem( info->filesystem() );
    item->setReadWrite( info->readwrite() );
#endif
    item->setRemount( info->remount() );
    item->setProtocol( info->protocol() );
    item->setKerberos( info->kerberos() );
    item->setUID( info->uid() );
    item->setGID( info->gid() );

    delete info;
  }
  else
  {
    m_list.append( info );
  }

  if ( s )
  {
    sync();
  }
}


void Smb4KSambaOptionsHandler::removeItem( const QString &name, bool s )
{
  Smb4KSambaOptionsInfo *item = find_item( name );

  if ( item && QString::compare( item->itemName().lower(), name.lower() ) == 0 )
  {
    m_list.remove( item );
    delete item;
  }

  if ( s )
  {
    sync();
  }
}

#include "smb4ksambaoptionshandler.moc"
