/***************************************************************************
    smb4knetworkitems  -  Network items used by the Smb4KScanner class
    to pass and store data.
                             -------------------
    begin                : Mi Jun 2 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KNETWORKITEMS_H
#define SMB4KNETWORKITEMS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstring.h>

/**
 * This class provides a container for a workgroup/domain found in the network
 * neighborhood.
 */

class Smb4KWorkgroupItem
{
  public:
    /**
     * The constructor.
     *
     * @param name        The name of the workgroup/domain.
     *
     * @param master      The master browser of the workgroup.
     *
     * @param masterIP    The IP address of the workgroup master browser.
     */
    Smb4KWorkgroupItem( const QString &name,
                        const QString &master,
                        const QString &masterIP = QString::null );

    /**
     * Empty constructor.
     */
    Smb4KWorkgroupItem() {}

    /**
     * The destructor.
     */
    ~Smb4KWorkgroupItem();

    /**
     * Returns the workgroup name.
     */
    const QString &name() const { return m_name; }

    /**
     * Returns the name of the master browser.
     */
    const QString &master() const { return m_master; }

    /**
     * Returns the IP address of the master browser. If it is not a valid
     * IPv4 or IPv6 address, QString::null is returned.
     *
     * @returns the valid IP v4 or v6 address of the workgroup master browser
     * or QString::null.
     */
    const QString &masterIP() const { return m_ip; }

    /**
     * You can mark the master as 'pseudo' with this function. That means that
     * this master was not reported as being the master browser of this
     * workgroup, but it's the only one found (e.g. by a custom search).
     */
    void setPseudoMaster();

    /**
     * Returns TRUE, if the master is a 'pseudo' master. @see setPseudoMaster()
     * for further information.
     */
    bool hasPseudoMaster() const { return m_pseudo; }

    /**
     * This function sets the IP address of the master browser.
     *
     * @param ip            The IP address of the master browser
     */
    void setMasterIP( const QString &ip );

    /**
     * This function sets the master browser.
     *
     * @param name          The name of the master browser
     *
     * @param ip            The IP address of the master browser
     *
     * @param pseudo        Determines if this is a real master browser or if
     *                      it is a faked one, i.e. a pseudo master.
     */
    void setMaster( const QString &name,
                    const QString &ip,
                    bool pseudo = false );

  private:
    /**
     * The name of the workgroup.
     */
    QString m_name;

    /**
     * The name of the workgroup master.
     */
    QString m_master;

    /**
     * The IP address of the master.
     */
    QString m_ip;

    /**
     * Determines whether the master is a 'pseudo'-master.
     */
    bool m_pseudo;

    /**
     * This function checks if the IP address is valid, i.e. the
     * IP address is either IP v4 or IP v6. It returns either TRUE
     * or FALSE.
     *
     * @param ip            The IP address that's going to be checked.
     *
     * @returns TRUE if the IP address is valid and FALSE otherwise.
     */
    bool ipIsValid( const QString &ip );
};


/**
 * This class provides a container for a host found in the network
 * neighborhood.
 */

class Smb4KHostItem
{
  public:
    /**
     * The default constructor.
     *
     * @param workgroup     The workgroup/domain of the host
     *
     * @param name          The name of the host
     *
     * @param comment       The comment that describes the host. May be empty.
     *
     * @param ip            The IP address of the host
     */
    Smb4KHostItem( const QString &workgroup,
                   const QString &name,
                   const QString &comment = QString::null,
                   const QString &ip = QString::null );

    /**
     * The copy constructor.
     *
     * @param host          A Smb4KHostItem representing a host.
     */
    Smb4KHostItem( const Smb4KHostItem &host );

    /**
     * The empty constructor.
     */
    Smb4KHostItem() {}

    /**
     * The destructor.
     */
    ~Smb4KHostItem();

    /**
     * Returns the workgroup the host is in.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Returns the name of the host.
     */
    const QString &name() const { return m_name; }

    /**
     * Returns the IP address of the host.
     */
    const QString &ip() const { return m_ip; }

    /**
     * Returns the comment for this host.
     */
    const QString &comment() const { return m_comment; }

    /**
     * Sets the Server string that is reported by the host.
     */
    void setServerString( const QString &server );

    /**
     * Returns the Server string.
     */
    const QString &serverString() const { return m_server_string; }

    /**
     * Sets the OS string that is reported by the host.
     */
    void setOSString( const QString &os );

    /**
     * Returns the OS string.
     */
    const QString &osString() const { return m_os_string; }

    /**
     * This functions determines whether this host should be
     * registered as a master browser.
     *
     * @param master        Set this to TRUE if the host is a master browser.
     */
    void setMaster( bool master );

    /**
     * This function tells you if the host is a master or not.
     *
     * @returns             TRUE if the host is a master browser.
     */
    const bool isMaster() const { return m_master; }

    /**
     * This function sets the IP address of a host.
     *
     * @param ip            The IP address of a host
     */
    void setIPAddress( const QString &ip );

    /**
     * This function sets the command for a host.
     *
     * @param comment       The comment
     */
    void setComment( const QString &comment );

    /**
     * This function is used to tell the host item, if a
     * check for the IP address has already been performed.
     *
     * @param yes           Should be set to TRUE if a check was performed.
     */
    void setIPAddressChecked( bool yes );

    /**
     * Tells if a check for the IP address has already been performed.
     *
     * @returns             TRUE if a check was performed, and FALSE otherwise.
     */
    const bool ipAddressChecked() const { return m_ip_checked; }

    /**
     * This function is used to tell the host item, if a
     * check for the information (OS and Server string ) has already been performed.
     *
     * @param yes           Should be set to TRUE if a check was performed.
     */
    void setInfoChecked( bool yes );

    /**
     * Tells if a check for the information (OS and Server string) has already
     * been performed.
     *
     * @returns             TRUE is the check was performed previously.
     */
    const bool infoChecked() const { return m_info_checked; }

  private:
    /**
     * The name of the workgroup.
     */
    QString m_workgroup;

    /**
     * The name of the host.
     */
    QString m_name;

    /**
     * The comment for this host.
     */
    QString m_comment;

    /**
     * The IP address of the host.
     */
    QString m_ip;

    /**
     * The Server string as reported by the host.
     */
    QString m_server_string;

    /**
     * The operating system string as reported by the host.
     */
    QString m_os_string;

    /**
     * This boolian determines if the host is a master browser
     * or not.
     */
    bool m_master;

    /**
     * This boolean tells if a check for the IP address was already
     * performed.
     */
    bool m_ip_checked;

    /**
     * This boolean tells if a check for the information (OS and server string)
     * was already performed.
     */
    bool m_info_checked;

    /**
     * This function checks if the IP address is valid, i.e. the
     * IP address is either IP v4 or IP v6. It returns either TRUE
     * or FALSE.
     *
     * @param ip            The IP address that's going to be checked.
     *
     * @returns TRUE if the IP address is valid and FALSE otherwise.
     */
    bool ipIsValid( const QString &ip );
};


/**
 * This class provides a container for a share found in the
 * network neighborhood.
 */

class Smb4KShareItem
{
  public:
    /**
     * The constructor.
     *
     * @param workgroup         The workgroup/domain name.
     *
     * @param host              The name of the host where the share is located.
     *
     * @param name              The name of the share.
     *
     * @param type              The type of the share as returned by the Samba programs, i.e.
     *                          "Disk", "Printer", "IPC$" or "ADMIN$".
     *
     * @param comment           The comment of the share.
     */
    Smb4KShareItem( const QString &workgroup,
                    const QString &host,
                    const QString &name,
                    const QString &type,
                    const QString &comment );

    /**
     * Empty constructor.
     */
    Smb4KShareItem() {}

    /**
     * The destructor.
     */
    ~Smb4KShareItem();

    /**
     * Returns the workgroup of the host where the share is located.
     */
    const QString &workgroup() const { return m_workgroup; }

    /**
     * Returns the name of the host where the share is located.
     */
    const QString &host() const { return m_host; }

    /**
     * Returns the name of the share.
     */
    const QString &name() const { return m_name; }

    /**
     * Returns the type of the share.
     */
    const QString &plainType() const { return m_type; }

    /**
     * Returns a translated version of the type of the share.
     *
     * @returns           The translated share type
     */
    const QString translatedType() const;

    /**
     * Returns the comment for this share.
     */
    const QString &comment() const { return m_comment; }

    /**
     * This function tells if the share is a hidden one.
     *
     * @returns           TRUE is the share is a hidden one and FALSE otherwise.
     */
    bool isHidden() const;

    /**
     * This function is TRUE if the share is a printer and
     * FALSE otherwise.
     *
     * @returns	          TRUE if the share is a printer
     */
    bool isPrinter() const;

    /**
     * This function returns TRUE if the share is an IPC$
     * share and FALSE otherwise.
     *
     * @returns           TRUE if the share is an IPC$ share
     */
    bool isIPC() const;

    /**
     * This function returns TRUE if the share is an ADMIN$
     * share and FALSE otherwise.
     *
     * @returns           TRUE if the share is an ADMIN$ share
     */
    bool isADMIN() const;

  private:
    /**
     * The workgroup object.
     */
    QString m_workgroup;

    /**
     * The host name.
     */
    QString m_host;

    /**
     * The name of the share.
     */
    QString m_name;

    /**
     * The type of the share.
     */
    QString m_type;

    /**
     * The comment for this share.
     */
    QString m_comment;
};

#endif
