/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* $Id: smtpguard.c,v 1.1 2005/11/09 03:09:12 tkitame Exp $ 
 *
 * Copyright (c) 2005 VA Linux Systems Japan, K.K. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

/*
 * smtpguard(8)
 *  AUTHOR: kitame@valinux.co.jp
 * 
 * smtpguard interface via postfix policy service.
 *
 * master.cf:
 *  smtpguard  unix  -       n       n       -       -       smtpguard
 *
 * main.cf:
 *  smtpd_recipient_restrictions = check_policy_service unix:private/smtpguard
 *  smtpguard_user = nobody
 *
 */

/* System library. */

#include <sys_defs.h>
#include <sys/wait.h>
#include <syslog.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <fcntl.h>

#ifdef STRCASECMP_IN_STRINGS_H
#include <strings.h>
#endif

/* Utility library. */
/* include/postfix */
#include <msg.h>
#include <argv.h>
#include <dict.h>
#include <mymalloc.h>
#include <split_at.h>
#include <timed_wait.h>
#include <set_eugid.h>
#include <htable.h>
#include <vstring.h>
#include <vstring_vstream.h>
#include <stringops.h>
#include <namadr_list.h>
#include <match_parent_style.h>

/* Global library */
/* include/postfix */
#include <mypwd.h>

/* Single server skeleton. */
/* include/postfix */
#include <mail_params.h>
#include <mail_server.h>
#include <mail_conf.h>
#include <resolve_local.h>

/* Application-specific. */
#include <smtpguard/smtpguard.h>


/* Extention */
#define VAR_SMTPGUARD_USER             "smtpguard_user"
#define DEF_SMTPGUARD_USER             "nobody"
char *var_smtpguard_user;

#define VAR_SMTPGUARD_FALLBACK_ACTION "smtpguard_fallback_action"
#define DEF_SMTPGUARD_FALLBACK_ACTION "defer_if_permit Server configuration error."
char *var_smtpguard_fallback_action;

static NAMADR_LIST *match_mynetworks;
HTABLE *instance_table;

/* return new allocated value */
static char *
smtpd_access_policy(HTABLE *htable)
{
	gint error;
	gchar * action = NULL; /* will be malloced */
	FGSmtpInfo *fsi = NULL;
	struct mypasswd *pwd;
	gchar *instance = NULL;
	GError *ge = NULL;
	gchar *ip = NULL, *host = NULL;
	gchar *rcptto = NULL;
	gchar *message = NULL;
	
	instance = mystrdup(htable_find(htable, "instance"));
	if (msg_verbose) msg_info ("request instance: %s", instance);
	
	if (htable_find (instance_table, instance)) {
		if (msg_verbose) msg_info ("found instance: %s", instance);
		fsi = fg_smtp_info_new (FALSE);
	} else {
		htable_enter (instance_table, instance, "yes");
		fsi = fg_smtp_info_new (TRUE);		
	}
	
	fg_smtp_info_set_pid (fsi, getpid());

	host = htable_find (htable, "client_name");
	ip = htable_find (htable, "client_address");
	/* check mynetworks */
	match_mynetworks = namadr_list_init(match_parent_style(VAR_MYNETWORKS),
					    var_mynetworks);
	if (namadr_list_match(match_mynetworks, host, ip))
		fg_smtp_info_set_addr (fsi, ip, TRUE);
	else
		fg_smtp_info_set_addr (fsi, ip, FALSE);        
	

	fg_smtp_info_set_mailfrom (fsi, htable_find(htable, "sender"));
	
	rcptto = htable_find(htable, "recipient");
	if (rcptto) { /* check mudestination */
		gchar *addr = rcptto;
		while (addr) if (*addr++ == '@') break;
		if (addr && resolve_local(addr)) {
			if (msg_verbose)
				msg_info ("resolve_local is : true");
			fg_smtp_info_set_rcptto (fsi, rcptto, TRUE);
		} else {
			if (msg_verbose)
				msg_info ("rcpt to domain: %s, is not in mydest", addr);
			fg_smtp_info_set_rcptto (fsi, rcptto, FALSE);            
		}
	}

	/* switch effective uid/gid */
	pwd = mypwnam (var_smtpguard_user);
	set_eugid(pwd->pw_uid, pwd->pw_gid);
	mypwfree(pwd);
	
	if (msg_verbose)
		msg_info("call fg_spam_update_db()");

	error =  fg_spam_update_db (fsi, &ge);
	
	set_eugid (var_owner_uid, var_owner_gid);
	
	if(ge != NULL) {
		if (fsi)
			fg_smtp_info_free (fsi);
		msg_error ("fg_spam_update_db(%d): %s", ge->code, ge->message);
		g_error_free (ge);
		if (var_smtpguard_fallback_action) {
			if (msg_verbose)
				msg_info ("fallback action: %s",
					  var_smtpguard_fallback_action);
			action = concatenate ("action=",
					      var_smtpguard_fallback_action,
					      NULL);
		} else {
			action = mystrdup ("action=dunno");
		}
		if (msg_verbose)
			msg_info ("policy action: %s", action);
		return action;
	} 

	fg_smtp_info_sleep_wait (fsi);
	message = fg_smtp_info_get_action_message (fsi);
	
	if (message) {
		if (msg_verbose)
			msg_info ("returned message: %s", message);
		action = concatenate ("action=", message , NULL);
	} else {
		action = mystrdup ("action=dunno");
	}
	
	if (msg_verbose)
		msg_info ("policy action: %s", action);
	
	return action;
}

static void
smtpguard_service (VSTREAM *client_stream, char *service, char **argv)
{
	char   *myname = "smtpguard_service";  
	HTABLE *htable;
	VSTRING *buf;
	char *action;
	
	if (msg_verbose)
		msg_info( "%s: service=%s", myname, service);
    
	buf = vstring_alloc (VSTREAM_BUFSIZE);
	htable = htable_create (0);
	
	while (vstring_fgets(buf, client_stream)) {
		char *val, *p;
		char *key;
		val = p = mystrdup(vstring_str(buf));
		if ((strchr (val, '=') != 0)) {
			if (val[strlen(val) - 1] == '\n')
				val[strlen(val) - 1] = '\0';
			key = mystrtok(&val, "=");
			if (msg_verbose)
				msg_info ("htable_dump: key=%s, val=%s", key, val);
			htable_enter (htable, key, val);
		} else if (val[0] == '\n') {
			char * tmp;
			if ((tmp = htable_find(htable, "request")) == NULL ||
			    strcmp (tmp, "smtpd_access_policy") != 0) {
				msg_fatal ("request does not includes \"request=smtpd_access_policy\"");
			}
			action = smtpd_access_policy (htable);
			vstream_fprintf (client_stream, "%s\n\n", action);
			vstream_fflush (client_stream);
			myfree (action);
			htable_free (htable, (void (*) (char *)) 0);
			htable = htable_create(0);
		} else {
			msg_warn ("ignoring garbage: %.100s", val);
		}
	}
	
	htable_free (htable, (void (*) (char *)) 0);
	vstring_free (buf);
}

/* pre_accept - see if tables have changed */
static void pre_accept (char *unused_name, char **unused_argv)
{
    const char *table;
    
    if ((table = dict_changed_name()) != 0) {
	    msg_info("table %s has changed -- restarting", table);
	    exit(0);
    }
}

/* drop_privileges - drop privileges most of the time */
static void drop_privileges (char *unused_name, char **unused_argv)
{
    set_eugid(var_owner_uid, var_owner_gid);
}


int
main(int argc, char **argv)
{
	
	static CONFIG_STR_TABLE str_table[] = {
		{ VAR_SMTPGUARD_FALLBACK_ACTION,  DEF_SMTPGUARD_FALLBACK_ACTION, &var_smtpguard_fallback_action, 1, 0 },
		{ VAR_SMTPGUARD_USER,  DEF_SMTPGUARD_USER, &var_smtpguard_user, 1, 0 },
		{ NULL }
	};
	
	g_type_init ();
	
	instance_table = htable_create (0);
	single_server_main(argc, argv, smtpguard_service,
			   MAIL_SERVER_STR_TABLE, str_table,
			   MAIL_SERVER_POST_INIT, drop_privileges,
			   MAIL_SERVER_PRE_ACCEPT, pre_accept,
			   0);
	htable_free (instance_table, (void (*) (char *)) 0);    
}
