/*==================================================================
 * sfdump.c - sound font dump routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <glib.h>
#include "sfont.h"
#include "sffile.h"


static void get_genvalstr (SFGen * gen, gchar * buf, gboolean absval);

/* sf file chunk IDs */
char *info_lbls[] = { "ISNG", "INAM", "IROM", "IVER", "ICRD", "IENG", "IPRD",
  "ICOP", "ICMT", "ISFT"
};

/* generator types */
char *gen_lbls[] = { "StartAddrOfs", "EndAddrOfs", "StartLoopAddrOfs",
  "EndLoopAddrOfs", "StartAddrCoarseOfs", "ModLFO2Pitch",
  "VibLFO2Pitch", "ModEnv2Pitch", "FilterFc", "FilterQ",
  "ModLFO2FilterFc", "ModEnv2FilterFc", "EndAddrCoarseOfs",
  "ModLFO2Vol", "Unused1", "ChorusSend", "ReverbSend", "Pan",
  "Unused2", "Unused3", "Unused4",
  "ModLFODelay", "ModLFOFreq", "VibLFODelay", "VibLFOFreq",
  "ModEnvDelay", "ModEnvAttack", "ModEnvHold", "ModEnvDecay",
  "ModEnvSustain", "ModEnvRelease", "Key2ModEnvHold",
  "Key2ModEnvDecay", "VolEnvDelay", "VolEnvAttack",
  "VolEnvHold", "VolEnvDecay", "VolEnvSustain", "VolEnvRelease",
  "Key2VolEnvHold", "Key2VolEnvDecay", "Instrument",
  "Reserved1", "KeyRange", "VelRange",
  "StartLoopAddrCoarseOfs", "Keynum", "Velocity",
  "Attenuation", "Reserved2", "EndLoopAddrCoarseOfs",
  "CoarseTune", "FineTune", "SampleId", "SampleModes",
  "Reserved3", "ScaleTune", "ExclusiveClass", "OverrideRootKey"
};


void
sfont_dump (SFData * sf, FILE * fd)
{
  GSList *p, *p2, *p3;
  SFPreset *pr;
  SFZone *zo;
  SFGen *ge;
  SFInst *in;
  SFSample *sa;
  guint8 infid;
  gchar scratch[32];
  gchar *infname;
  gint warn = 0;
  gint i;

  fprintf (fd, "Smurf Sound Font Dump ...\n");
  fprintf (fd, "------------------- Sound Font Info --------------------\n");
  fprintf (fd, "Version: %d.%d\n", sf->version.major, sf->version.minor);
  fprintf (fd, "Up2date: %s\n", sf->up2date ? "TRUE" : "FALSE");

  p = sf->info;
  while (p)
    {
      infid = *(guint8 *) (p->data);
      if ((infid < ISNG_ID) || (infid > ISFT_ID))
	{
	  fprintf (fd, "***** Warning ***** Unknown info id\n");
	  warn++;
	  sprintf (scratch, "(0x%02X)", infid);
	  infname = scratch;
	}
      infname = info_lbls[infid - ISNG_ID];
      fprintf (fd, "INFO: %s = \"%s\"\n", infname, (char *) (p->data) + 1);
      p = g_slist_next (p);
    }

//    fprintf(fd, "Sample data: Pos = 0x%0lX, Size = %d bytes\n", sf->samplepos,
//      sf->samplesize);

  fprintf (fd, "------------------- Preset Info -----------------------\n");

  p = sf->preset;
  while (p)
    {				/* Traverse presets */
      pr = (SFPreset *) (p->data);
      fprintf (fd, "Preset %03d-%03d \"%s\"\n", pr->bank, pr->prenum,
	pr->name);
      p2 = pr->zone;
      if (!p2)
	{
	  fprintf (fd, "\t***** Warning ***** No zones!\n");
	  warn++;
	}
      while (p2)
	{			/* Traverse zones */
	  zo = (SFZone *) (p2->data);
	  if (!zo->instsamp && !zo->gen && !zo->mod)
	    {
	      fprintf (fd, "\t***** Warning ***** Empty zone!\n");
	      warn++;
	      p2 = g_slist_next (p2);
	      continue;
	    }
	  if (!zo->instsamp)
	    fprintf (fd, "\tGlobal Zone\n");
	  else
	    fprintf (fd, "\tInstrument \"%s\"\n",
	      ((SFInst *) (zo->instsamp->data))->name);
	  p3 = zo->gen;
	  while (p3)
	    {			/* Traverse generators */
	      ge = (SFGen *) (p3->data);
	      if (!gen_validp (ge->id))
		{
		  fprintf (fd,
		    "\t\t***** Warning ***** Invalid gen id \"%d\"\n",
		    ge->id);
		  warn++;
		  p3 = g_slist_next (p3);
		  continue;
		}
	      get_genvalstr (ge, scratch, FALSE);

	      fprintf (fd, "\t\tGen \"%s\" (%d) = 0x%04X (%s)\n",
		gen_lbls[ge->id], ge->id, ge->amount.uword, scratch);

	      p3 = g_slist_next (p3);
	    }

	  p3 = zo->mod;
	  while (p3)
	    {			/* Traverse modulators */
	      fprintf (fd, "\t\tModulator (Not implimented)\n");
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  fprintf (fd, "------------------- Instrument Info -------------------\n");

  p = sf->inst;
  while (p)
    {				/* Traverse instruments */
      in = (SFInst *) (p->data);
      fprintf (fd, "Instrument \"%s\"\n", in->name);
      p2 = in->zone;
      if (!p2)
	{
	  fprintf (fd, "\t***** Warning ***** No zones!\n");
	  warn++;
	}
      while (p2)
	{
	  zo = (SFZone *) (p2->data);
	  if (!zo->instsamp && !zo->gen && !zo->mod)
	    {
	      fprintf (fd, "\t***** Warning ***** Empty zone!\n");
	      warn++;
	      p2 = g_slist_next (p2);
	      continue;
	    }
	  if (!zo->instsamp)
	    fprintf (fd, "\tGlobal Zone\n");
	  else
	    fprintf (fd, "\tSample \"%s\"\n",
	      ((SFSample *) (zo->instsamp->data))->name);
	  p3 = zo->gen;
	  while (p3)
	    {
	      ge = (SFGen *) (p3->data);
	      if (!gen_valid (ge->id))
		{
		  fprintf (fd,
		    "\t\t***** Warning ***** Invalid gen id \"%d\"\n",
		    ge->id);
		  warn++;
		  p3 = g_slist_next (p3);
		  continue;
		}
	      get_genvalstr (ge, scratch, TRUE);

	      fprintf (fd, "\t\tGen \"%s\" (%d) = 0x%04X (%s)\n",
		gen_lbls[ge->id], ge->id, ge->amount.uword, scratch);

	      p3 = g_slist_next (p3);
	    }

	  p3 = zo->mod;
	  while (p3)
	    {			/* Traverse modulators */
	      fprintf (fd, "\t\tModulator (Not implimented)\n");
	      p3 = g_slist_next (p3);
	    }
	  p2 = g_slist_next (p2);
	}
      p = g_slist_next (p);
    }

  fprintf (fd, "------------------- Sample Info -----------------------\n");

  i = 0;			/* current sample index */
  p = sf->sample;
  while (p)
    {
      sa = (SFSample *) (p->data);
      fprintf (fd, "Sample \"%s\" (%d)\n", sa->name, i);
      fprintf (fd, "\tSample in ");
      if (sa->datainfo->samfile)
	fprintf (fd, "BUFFER\n");
      else
	fprintf (fd, "SFFILE\n");
      fprintf (fd, "\tStart = 0x%0X\n\tEnd = %d\n", sa->datainfo->start,
	       sa->end);
      fprintf (fd, "\tLoop: Start = %d -> End = %d\n", sa->loopstart,
	sa->loopend);
      fprintf (fd, "\tSamplerate = %d\n", sa->samplerate);
      fprintf (fd, "\tPitch: Root = %d, Adj = %d cents\n", sa->origpitch,
	sa->pitchadj);
      fprintf (fd, "\tType: 0x%0X\n", sa->sampletype);
      p = g_slist_next (p);
      i++;			/* increment sample index */
    }
}

static void
get_genvalstr (SFGen * gen, gchar * buf, gboolean absval)
{
  GenConv *gc;
  float val;
  gchar *txt;

  if ((gen->id == Gen_KeyRange) || (gen->id == Gen_VelRange))
    {
      sprintf (buf, "%d-%d +range", gen->amount.range.lo,
	gen->amount.range.hi);
      return;
    }

  gc = &genconv[genparms[gen->id].unit];

  if (absval)
    {
      if (!gc->sf2user)
	{
	  sprintf (buf, "No conversion");
	  return;
	}
      val = (*gc->sf2user) (gen->amount);
      txt = gc->unittxt;
    }
  else
    {
      if (!gc->sf2ofs)
	{
	  sprintf (buf, "No conversion");
	  return;
	}
      val = (*gc->sf2ofs) (gen->amount);
      txt = gc->ofstxt;
    }

  sprintf (buf, "%.*f %s", gc->digits, val, txt);
}
