/*==================================================================
 * sfont.c - Sound font routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <glib.h>
#include "sfont.h"
#include "sample.h"
#include "util.h"
#include "i18n.h"
#include "uif_sftree.h"

/* optimum chunk area sizes (could be more optimum) */
#define PRESET_CHUNK_OPTIMUM_AREA	256
#define INST_CHUNK_OPTIMUM_AREA		256
#define SAMPLE_CHUNK_OPTIMUM_AREA	256
#define SAMDATAINFO_CHUNK_OPTIMUM_AREA  256
#define ZONE_CHUNK_OPTIMUM_AREA		256
#define MOD_CHUNK_OPTIMUM_AREA		256
#define GEN_CHUNK_OPTIMUM_AREA		256

/* Memory chunk pointers */
static GMemChunk *chunk_preset;
static GMemChunk *chunk_inst;
static GMemChunk *chunk_sample;
static GMemChunk *chunk_samdatainfo;
static GMemChunk *chunk_zone;
static GMemChunk *chunk_mod;
static GMemChunk *chunk_gen;

static gboolean sfont_chunks_inited = FALSE; /* Have chunks been initialized? */

/*
    Balanced binary tree of all sound font item GtkCTreeNodes
    so references can be easily managed (not pointers but unique IDs)
    which means an item can be removed and references don't need
    updating only the GTree does which is managed on add/remove
    of items, no segfaults from old references :)
*/
GTree *sfont_itemids = NULL;

/* sf item unique ID counter (increments) */
static gint sfont_itemid_counter = 1;

/* Default range value stored in 2 byte form with correct host byte order */
#define DEFRANGE	GINT16_FROM_LE(0x7F00)

guint16 badgen[] = { Gen_Unused1, Gen_Unused2, Gen_Unused3, Gen_Unused4,
  Gen_Reserved1, Gen_Reserved2, Gen_Reserved3, 0
};

guint16 badpgen[] = { Gen_StartAddrOfs, Gen_EndAddrOfs, Gen_StartLoopAddrOfs,
  Gen_EndLoopAddrOfs, Gen_StartAddrCoarseOfs, Gen_EndAddrCoarseOfs,
  Gen_StartLoopAddrCoarseOfs, Gen_Keynum, Gen_Velocity,
  Gen_EndLoopAddrCoarseOfs, Gen_SampleModes, Gen_ExclusiveClass,
  Gen_OverrideRootKey, 0
};

/* SF -> User unit type conversion table, comments are for NLS translators */
GenConv genconv[] = {
  {NULL, NULL, NULL, NULL, 0, NULL, NULL},	// None
/* Short abbreviation for "samples" */
  {float2int, int2float, float2int, int2float, 0, N_("smpls"), N_("smpls")},	// Unit_Smpls
/* Short abbreviation for 32k (i.e. 32 thousand) "samples" */
  {float2int, int2float, float2int, int2float, 0, N_("32ksmpls"), N_("32ksmpls")},	// Unit_32kSmpls
/* Cents (100th of a semitone) */
  {float2int, int2float, float2int, int2float, 0, N_("cents"), N_("+cents")},	// Unit_Cent
/* Hertz abbreviation (cycles per second) */
  {hz2cents, cents2hz, sec2tcent, tcent2sec, 2, N_("Hz"), "X"},	// Unit_HzCent
/* Seconds abbreviation (second, as in the measurement of time) */
  {sec2tcent, tcent2sec, sec2tcent, tcent2sec, 3, N_("sec"), "X"},	// Unit_TCent
/* Decibel abbreviation */
  {db2cb, cb2db, db2cb, cb2db, 2, N_("dB"), N_("+dB")},	// Unit_cB
  {perc2tperc, tperc2perc, perc2tperc, tperc2perc, 1, "%", "+%"},	// Unit_Percent
/* semitone abbreviation (1/12th of an octave) */
  {float2int, int2float, float2int, int2float, 0, N_("semi"), N_("+semi")},	// Unit_Semitone
  {NULL, NULL, NULL, NULL, 0, N_("range"), N_("+range")}	// Unit_Range
};

#define MAXNEG -32768
#define MAXPOS 32767

GenParms genparms[] = {
  {0, MAXPOS, 0, Unit_Smpls, NULL},	// Gen_StartAddrOfs
  {MAXNEG, 0, 0, Unit_Smpls, NULL},	// Gen_EndAddrOfs
  {MAXNEG, MAXPOS, 0, Unit_Smpls, NULL},	// Gen_StartLoopAddrOfs
  {MAXNEG, MAXPOS, 0, Unit_Smpls, NULL},	// Gen_EndLoopAddrOfs
  {0, MAXPOS, 0, Unit_32kSmpls, NULL},	// Gen_StartAddrCoarseOfs
  {-12000, 12000, 0, Unit_Cent, N_("To Pitch")},	// Gen_ModLFO2Pitch
  {-12000, 12000, 0, Unit_Cent, N_("To Pitch")},	// Gen_VibLFO2Pitch
  {-12000, 12000, 0, Unit_Cent, N_("To Pitch")},	// Gen_ModEnv2Pitch
  {1500, 13500, 13500, Unit_HzCent, N_("Filter Cutoff")},	// Gen_FilterFc
  {0, 960, 0, Unit_cB, N_("Filter Q")},	// Gen_FilterQ
  {-12000, 12000, 0, Unit_Cent, N_("To Filter Cutoff")}, // Gen_ModLFO2FilterFc
  {-12000, 12000, 0, Unit_Cent, N_("To Filter Cutoff")}, // Gen_ModEnv2FilterFc
  {MAXNEG, 0, 0, Unit_32kSmpls, NULL},	// Gen_EndAddrCoarseOfs
  {-960, 960, 0, Unit_cB, N_("To Volume")},	// Gen_ModLFO2Vol
  {0, 0, 0, None, NULL},	// Gen_Unused1
  {0, 1000, 0, Unit_Percent, N_("Chorus")},	// Gen_ChorusSend
  {0, 1000, 0, Unit_Percent, N_("Reverb")},	// Gen_ReverbSend
  {-500, 500, 0, Unit_Percent, N_("Pan")},	// Gen_Pan
  {0, 0, 0, None, NULL},	// Gen_Unused2
  {0, 0, 0, None, NULL},	// Gen_Unused3
  {0, 0, 0, None, NULL},	// Gen_Unused4
  {-12000, 5000, -12000, Unit_TCent, N_("Delay")},	// Gen_ModLFODelay
  {-16000, 4500, 0, Unit_HzCent, N_("Frequency")},	// Gen_ModLFOFreq
  {-12000, 5000, -12000, Unit_TCent, N_("Delay")},	// Gen_VibLFODelay
  {-16000, 4500, 0, Unit_HzCent, N_("Frequency")},	// Gen_VibLFOFreq
  {-12000, 5000, -12000, Unit_TCent, N_("Delay")},	// Gen_ModEnvDelay
  {-12000, 8000, -12000, Unit_TCent, N_("Attack")},	// Gen_ModEnvAttack
  {-12000, 5000, -12000, Unit_TCent, N_("Hold")},	// Gen_ModEnvHold
  {-12000, 8000, -12000, Unit_TCent, N_("Decay")},	// Gen_ModEnvDecay
  {0, 1000, 0, Unit_Percent, N_("Sustain")},	// Gen_ModEnvSustain
  {-12000, 8000, -12000, Unit_TCent, N_("Release")},	// Gen_ModEnvRelease
  {-1200, 1200, 0, Unit_Cent, N_("Key to Hold")},	// Gen_Key2ModEnvHold
  {-1200, 1200, 0, Unit_Cent, N_("Key to Decay")},	// Gen_Key2ModEnvDecay
  {-12000, 5000, -12000, Unit_TCent, N_("Delay")},	// Gen_VolEnvDelay
  {-12000, 8000, -12000, Unit_TCent, N_("Attack")},	// Gen_VolEnvAttack
  {-12000, 5000, -12000, Unit_TCent, N_("Hold")},	// Gen_VolEnvHold
  {-12000, 8000, -12000, Unit_TCent, N_("Decay")},	// Gen_VolEnvDecay
  {0, 1440, 0, Unit_cB, N_("Sustain")},	// Gen_VolEnvSustain
  {-12000, 8000, -12000, Unit_TCent, N_("Release")},	// Gen_VolEnvRelease
  {-1200, 1200, 0, Unit_Cent, N_("Key to Hold")},	// Gen_Key2VolEnvHold
  {-1200, 1200, 0, Unit_Cent, N_("Key to Decay")},	// Gen_Key2VolEnvDecay
  {0, MAXPOS, 0, None, NULL},	// Gen_Instrument
  {0, 0, 0, None, NULL},	// Gen_Reserved1
  {0, 127, DEFRANGE, Unit_Range, NULL},	// Gen_KeyRange
  {0, 127, DEFRANGE, Unit_Range, NULL},	// Gen_VelRange
  {MAXNEG, MAXPOS, 0, Unit_32kSmpls, NULL},	// Gen_StartLoopAddrCoarseOfs
  {-1, 127, -1, None, NULL},	// Gen_Keynum
  {-1, 127, -1, None, NULL},	// Gen_Velocity
  {0, 1440, 0, Unit_cB, N_("Attenuation")},	// Gen_InitAttenuation
  {0, 0, 0, None, NULL},	// Gen_Reserved2
  {MAXNEG, MAXPOS, 0, Unit_32kSmpls, NULL},	// Gen_EndLoopAddrCoarseOfs
  {-120, 120, 0, Unit_Semitone, N_("Coarse Tune")},	// Gen_CourseTune
  {-99, 99, 0, Unit_Cent, N_("Fine Tune")},	// Gen_FineTune
  {0, MAXPOS, 0, None, NULL},	// Gen_sampleId
  {MAXNEG, MAXPOS, 0, None, NULL},	// Gen_SampleModes
  {0, 0, 0, None, NULL},	// Gen_Reserved3
  {0, 1200, 100, Unit_Cent, N_("Scale Tune")},	// Gen_ScaleTuning
  {0, 127, 0, None, NULL},	// Gen_ExclusiveClass
  {-1, 127, -1, None, NULL}	// Gen_OverrideRootKey
};

static gint sfont_itemid_compare_func (gconstpointer a, gconstpointer b);

/*----- General sfont routines -----*/

/* initialize sound font mem chunks */
void
sfont_init_chunks (void)
{
  if (sfont_chunks_inited) return;
  chunk_preset =
    g_mem_chunk_create (SFPreset, PRESET_CHUNK_OPTIMUM_AREA,
    G_ALLOC_AND_FREE);
  chunk_inst =
    g_mem_chunk_create (SFInst, INST_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_sample =
    g_mem_chunk_create (SFSample, SAMPLE_CHUNK_OPTIMUM_AREA,
    G_ALLOC_AND_FREE);
  chunk_samdatainfo =
    g_mem_chunk_create (SFSamDataInfo, SAMPLE_CHUNK_OPTIMUM_AREA,
    G_ALLOC_AND_FREE);
  chunk_zone =
    g_mem_chunk_create (SFZone, ZONE_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_mod =
    g_mem_chunk_create (SFMod, MOD_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);
  chunk_gen =
    g_mem_chunk_create (SFGen, GEN_CHUNK_OPTIMUM_AREA, G_ALLOC_AND_FREE);

  /* initialize sfont_itemids balanced binary tree */
  sfont_itemids = g_tree_new (sfont_itemid_compare_func);

  sfont_chunks_inited = TRUE;
}

/* generates the next available item ID */
SFItemID
sfont_next_itemid (void)
{
  return (sfont_itemid_counter++);
}

/* adds an item ID as a key to val */
void
sfont_register_itemid (SFItemID itemid, gpointer val)
{
  g_tree_insert (sfont_itemids, GINT_TO_POINTER (itemid), val);
}

/* looks up an item ID and returns the associated pointer value */
gpointer
sfont_lookup_itemid (SFItemID itemid)
{
  return (g_tree_lookup (sfont_itemids, GINT_TO_POINTER (itemid)));
}

/* removes a sound font item ID */
void
sfont_remove_itemid (SFItemID itemid)
{
  g_tree_remove (sfont_itemids, GINT_TO_POINTER (itemid));
}

static gint
sfont_itemid_compare_func (gconstpointer a, gconstpointer b)
{
  return (GPOINTER_TO_INT (a) - GPOINTER_TO_INT (b));
}

/* ------------------------------------- */
/* ---- SFData Sound Font functions ---- */
/* ------------------------------------- */

/* create a new sound font */
SFData *
sfont_new (gint titlenum)
{
  SFData *sf;
  gchar *s;

  sf = sfont_sfdata_alloc ();

  sf->version.major = 2;
  sf->version.minor = 0;

  /* file name to give new "untitled" sound fonts */
  sf->fname = g_strdup_printf (_("untitled%d.sf2"), titlenum);

  sfont_set_info (sf, ISNG_ID, "EMU8000");

  /* name to give to new "untitled" sound fonts */
  s = g_strdup_printf (_("Untitled %d"), titlenum);
  sfont_set_info (sf, INAM_ID, s);
  g_free (s);

  s = g_strconcat (SFONT_ISFT_NAME VERSION, ":", NULL);
  sfont_set_info (sf, ISFT_ID, s);
  g_free (s);

  sf->up2date = TRUE;	/* do this last, SF functions set it to FALSE */
  sf->beensaved = FALSE;	/* has not been saved yet */

  return (sf);
}

/* close a sound font and destroy all its data */
void
sfont_close (SFData * sf)
{
  sfont_destroy (sf);

  if (sf->sffd)
    fclose (sf->sffd);
}

/* recursively delete a sound font (all presets, samples, instruments etc) */
void
sfont_destroy (SFData * sf)
{
  GSList *p, *p2;

  if (sf->fname)
    g_free (sf->fname);

  p = sf->info;
  while (p)
    {
      g_free (p->data);
      p = g_slist_next (p);
    }
  g_slist_free (sf->info);
  sf->info = NULL;

  p = sf->preset;
  while (p)
    {				/* loop over presets */
      p2 = ((SFPreset *) (p->data))->zone;
      while (p2)
	{			/* loop over preset's zones */
	  sfont_zone_destroy (p2->data);
	  p2 = g_slist_next (p2);
	}			/* free preset's zone list */
      g_slist_free (((SFPreset *) (p->data))->zone);
      g_mem_chunk_free (chunk_preset, p->data);	/* free preset chunk */
      p = g_slist_next (p);
    }
  g_slist_free (sf->preset);
  sf->preset = NULL;

  p = sf->inst;
  while (p)
    {				/* loop over instruments */
      p2 = ((SFInst *) (p->data))->zone;
      while (p2)
	{			/* loop over inst's zones */
	  sfont_zone_destroy (p2->data);
	  p2 = g_slist_next (p2);
	}			/* free inst's zone list */
      g_slist_free (((SFInst *) (p->data))->zone);
      g_mem_chunk_free (chunk_inst, p->data);	/* free inst chunk */
      p = g_slist_next (p);
    }
  g_slist_free (sf->inst);
  sf->inst = NULL;

  p = sf->sample;
  while (p)
    {
      p2 = p;
      p = g_slist_next (p);

      /* better use the routine since it moves sample data if needed */
      sfont_remove_sample (sf, (SFSample *) (p2->data));
    }
  sf->sample = NULL;
}

/* allocate and initialize an SFData structure to dead state */
SFData *
sfont_sfdata_alloc (void)
{
  SFData *sf;
  sf = g_new0 (SFData, 1);
  return (sf);
}

/* free a sound font data structure */
void
sfont_sfdata_free (SFData *sf)
{
  g_free (sf);
}

/* create a new preset in a sound font */
SFPreset *
sfont_new_preset (SFData * sf, gchar * name, guint16 bank, guint16 prenum)
{
  SFPreset *pset;

  pset = sfont_preset_alloc ();

  strcpy (pset->name, name);
  pset->prenum = prenum;
  pset->bank = bank;
  sf->preset = g_slist_insert_sorted (sf->preset, pset,
    (GCompareFunc) sfont_preset_compare_func);

  sf->up2date = FALSE;

  return (pset);
}

/* add an existing preset into a sound font (sorted) */
void
sfont_add_preset (SFData * sf, SFPreset * pset)
{
  sf->preset = g_slist_insert_sorted (sf->preset, pset,
    (GCompareFunc) sfont_preset_compare_func);

  sf->up2date = FALSE;
}

/* destroy and remove a preset from a sound font */
void
sfont_remove_preset (SFData * sf, SFPreset * pset)
{
  sfont_preset_destroy (pset);
  sf->preset = g_slist_remove (sf->preset, pset);

  sf->up2date = FALSE;
}

/* preset sort function, first by bank, then by preset # */
gint
sfont_preset_compare_func (gconstpointer a, gconstpointer b)
{
  gint32 aval, bval;

  aval = (gint32) (((SFPreset *) a)->bank) << 16 | ((SFPreset *) a)->prenum;
  bval = (gint32) (((SFPreset *) b)->bank) << 16 | ((SFPreset *) b)->prenum;

  return (aval - bval);
}

/* create a new instrument in a sound font */
SFInst *
sfont_new_inst (SFData * sf, gchar * name)
{
  SFInst *inst;

  inst = sfont_inst_alloc ();
  sfont_add_inst (sf, inst);
  strcpy (inst->name, name);

  sf->up2date = FALSE;

  return (inst);
}

/* add (append) an instrument to a sound font */
void
sfont_add_inst (SFData * sf, SFInst * inst)
{
  sf->inst = g_slist_append (sf->inst, inst);
  sf->up2date = FALSE;
}

/* insert an instrument into a sound font */
void
sfont_insert_inst (SFData * sf, SFInst * inst, gint pos)
{
  sf->inst = g_slist_insert (sf->inst, inst, pos);
  sf->up2date = FALSE;
}

/* destroy an instrument and remove it from a sound font */
void
sfont_remove_inst (SFData * sf, SFInst * inst)
{
  sfont_inst_destroy (inst);
  sf->inst = g_slist_remove (sf->inst, inst);

  sf->up2date = FALSE;
}

/* add a sample record to a sfdata structure */
void
sfont_add_sample (SFData * sf, SFSample * sam)
{
  sf->sample = g_slist_append (sf->sample, sam);
  sf->up2date = FALSE;
}

/* insert a sample into a sound font */
void
sfont_insert_sample (SFData * sf, SFSample * sam, gint pos)
{
  sf->sample = g_slist_insert (sf->sample, sam, pos);
  sf->up2date = FALSE;
}

/* delete a sample from a sound font */
void
sfont_remove_sample (SFData * sf, SFSample * sam)
{
  if (SAM_DATAINFO_REFCOUNT (sam->datainfo) <= 1)
    sam_datainfo_destroy (sam->datainfo);
  else if (sam->datainfo->sf == sf) /* -if data resides in sf and other refs */
    sam_datainfo_move2sambuf (sam, sf);	/* -move data to sambuf */

  sfont_sample_destroy (sam, TRUE);
  sf->sample = g_slist_remove (sf->sample, sam);

  sf->up2date = FALSE;
}

/* create a new zone for a preset */
SFZone *
sfont_new_preset_zone (SFData * sf, SFPreset * pset, GSList * inst)
{
  SFZone *zone;

  /* request to create global zone? Does one already exist? */
  if (!inst && pset->zone && !((SFZone *) (pset->zone->data))->instsamp)
    return (NULL);

  zone = sfont_zone_alloc ();
  zone->instsamp = inst;

  if (inst)
    pset->zone = g_slist_append (pset->zone, zone);
  else  /* global zone should be first zone */
    pset->zone = g_slist_prepend (pset->zone, zone);

  sf->up2date = FALSE;

  return (zone);
}

/* add an already existing zone to a preset */
gint
sfont_add_preset_zone (SFData * sf, SFPreset * pset, SFZone * zone)
{
  /* refuse to link in a global zone if one already exists */
  if (!zone->instsamp && pset->zone
    && !((SFZone *) (pset->zone->data))->instsamp)
    return (FAIL);

  if (zone->instsamp)
    pset->zone = g_slist_append (pset->zone, zone);
  else  /* global zone should be first zone */
    pset->zone = g_slist_prepend (pset->zone, zone);

  sf->up2date = FALSE;
  return (OK);
}

/* create a new zone for an instrument */
SFZone *
sfont_new_inst_zone (SFData * sf, SFInst * inst, GSList * sam)
{
  SFZone *zone;

  /* request to create global zone? Does one already exist? */
  if (!sam && inst->zone && !((SFZone *) (inst->zone->data))->instsamp)
    return (NULL);

  zone = sfont_zone_alloc ();
  zone->instsamp = sam;

  if (sam)
    inst->zone = g_slist_append (inst->zone, zone);
  else  /* global zone should be first zone */
    inst->zone = g_slist_prepend (inst->zone, zone);

  sf->up2date = FALSE;

  return (zone);
}

/* add an already existing zone to an instrument */
gint
sfont_add_inst_zone (SFData * sf, SFInst * inst, SFZone * zone)
{
  /* refuse to link in a global zone if one already exists */
  if (!zone->instsamp && inst->zone
    && !((SFZone *) (inst->zone->data))->instsamp)
    return (FAIL);

  if (zone->instsamp)
    inst->zone = g_slist_append (inst->zone, zone);
  else  /* global zone should be first zone */
    inst->zone = g_slist_prepend (inst->zone, zone);

  sf->up2date = FALSE;
  return (OK);
}

/* remove zone from zone list */
void
sfont_remove_zone (SFData * sf, GSList ** zlist, SFZone * zone)
{
  *zlist = g_slist_remove (*zlist, (gpointer) zone);
  sfont_zone_destroy (zone);

  sf->up2date = FALSE;
}

/* Find first unused preset bank preset# in the melodic or percussion branch */
void
sfont_find_free_preset (SFData *sf, gint *bank, gint *prenum, gboolean melodic)
{
  SFPreset *pset;
  GSList *p;
  gint b, n;			/* Stores current bank and preset number */

  if (melodic) b = 0;
  else b = 128;
  n = 0;

  p = sf->preset;
  while (p)
    {
      pset = (SFPreset *)(p->data);
      if (pset->bank > b || (pset->bank == b && pset->prenum > n)) break;
      if (melodic || pset->bank == 128)
	{
	  if (++n > 127)
	    {
	      n = 0;
	      b++;
	    }
	}
      p = g_slist_next (p);
    }
  *bank = b;
  *prenum = n;
}

/* find preset by name or bank/preset #, match name if !NULL and match
   bank/prenum if bank <= 128, match name || bank/prenum if both specified,
   excl specifies a preset to exclude from the search or NULL */
GSList *
sfont_find_preset (SFData * sf, gchar * name, guint bank, guint prenum,
  SFPreset * excl)
{
  GSList *p;
  SFPreset *pset;

  p = sf->preset;
  while (p)
    {
      pset = (SFPreset *) (p->data);
      if (pset != excl		/* if excl is NULL it will never == pset */
	&& ((pset->bank <= 128 && pset->bank == bank
	    && pset->prenum == prenum)
	  || (name && strcmp (pset->name, name) == 0)))
	return (p);
      p = g_slist_next (p);
    }
  return (NULL);
}

/* find instrument by name, excl is an instrument to exclude from find or NULL */
GSList *
sfont_find_inst (SFData * sf, gchar * name, SFInst * excl)
{
  GSList *p;
  SFInst *inst;

  p = sf->inst;
  while (p)
    {
      inst = (SFInst *) (p->data);
      if (inst != excl && strcmp (inst->name, name) == 0)
	return (p);
      p = g_slist_next (p);
    }
  return (NULL);
}

/* find sample by name */
GSList *
sfont_find_sample (SFData * sf, gchar * name, SFSample * excl)
{
  GSList *p;
  SFSample *sam;

  p = sf->sample;
  while (p)
    {
      sam = (SFSample *) (p->data);
      if (sam != excl && strcmp (sam->name, name) == 0)
	return (p);
      p = g_slist_next (p);
    }
  return (NULL);
}

/* set a sound font file name */
void
sfont_set_fname (SFData * sf, gchar * fname)
{
  gchar *temp;

  temp = g_strdup (fname);	/* just in case fname == sf->fname */
  if (sf->fname)
    g_free (sf->fname);
  sf->fname = temp;
}

/* Get an info string by id */
gchar *
sfont_get_info (SFData * sf, guint8 id)
{
  GSList *p;

  p = sf->info;
  while (p)
    {
      if (*(guint8 *) (p->data) == id)
	return ((gchar *) ((guint8 *) (p->data) + 1));
      p = g_slist_next (p);
    }
  return ("");
}

/* Set an info string by id */
void
sfont_set_info (SFData * sf, guint8 id, gchar * str)
{
  GSList *p;
  gchar *newstr;
  guint length;

  p = sf->info;
  while (p)
    {
      if (*(guint8 *) (p->data) == id)
	break;
      p = g_slist_next (p);
    }

  /* make sure length of info string is even */
  length = strlen (str);
  length++;			/* 1 more for terminator */
  newstr = g_malloc (length + length % 2 + 1);

  newstr[0] = id;
  strcpy (&newstr[1], str);
  if (length % 2)
    newstr[length + 1] = '\0';	/* even terminator = '0' */

  if (p)
    {
      g_free (p->data);
      p->data = newstr;
    }
  else
    sf->info = g_slist_append (sf->info, newstr);

  sf->up2date = FALSE;
}

/* Set namestr */
void
sfont_set_namestr (SFData * sf, gchar * namestr, gchar * name)
{
  sfont_set_namestr_nosf (namestr, name);
  sf->up2date = FALSE;
}

/* set a namestr, without updating sound font up2date variable */
void
sfont_set_namestr_nosf (gchar * namestr, gchar * name)
{
  if (strlen (name) > 20)
    {
      strncpy (namestr, name, 20);
      namestr[20] = '\0';
    }
  else strcpy (namestr, name);
}

/* ---------------------------- */
/* ---- SFPreset functions ---- */
/* ---------------------------- */

/* duplicate a preset and its zones, see sfont_zone_dup for details on mode */
SFPreset *
sfont_preset_dup (SFPreset * pset, SFDupEnum mode)
{
  SFPreset *dup;
  SFZone *dupz;
  GSList *p;

  dup = sfont_preset_alloc ();  /* allocate a new preset */

  strcpy (dup->name, pset->name);
  dup->prenum = pset->prenum;
  dup->bank = pset->bank;
  dup->libr = pset->libr;
  dup->genre = pset->genre;
  dup->morph = pset->morph;

  /* if not SFDUP_NORMAL mode then duplicate itemid */
  if (mode != SFDUP_NORMAL) dup->itemid = pset->itemid;

  p = pset->zone;
  while (p)
    {				/* loop over pset zones and copy them */
      /* duplicate the zone (with same mode) */
      dupz = sfont_zone_dup ((SFZone *)(p->data), mode);
      dup->zone = g_slist_append (dup->zone, dupz);
      p = g_slist_next (p);
    }

  return (dup);
}

/* destroy a preset and its zones */
void
sfont_preset_destroy (SFPreset * pset)
{
  GSList *p;

  p = pset->zone;
  while (p)
    {				/* loop over preset's zones */
      sfont_zone_destroy (p->data);
      p = g_slist_next (p);
    }
  g_slist_free (pset->zone);	/* free preset's zone list */

  sfont_preset_free (pset);
}

/* allocate a preset structure and initialize it to dead state */
SFPreset *
sfont_preset_alloc (void)
{
  return (g_chunk_new0 (SFPreset, chunk_preset));
}

/* free a preset structure */
void
sfont_preset_free (SFPreset *preset)
{
  g_mem_chunk_free (chunk_preset, preset);	/* free preset chunk */
}

/* -------------------------- */
/* ---- SFInst functions ---- */
/* -------------------------- */

/* duplicate instrument and its zones, see sfont_zone_dup for info on mode */
SFInst *
sfont_inst_dup (SFInst * inst, SFDupEnum mode)
{
  SFInst *dup;
  SFZone *dupz;
  GSList *p;

  dup = sfont_inst_alloc ();  /* allocate a new instrument */

  strcpy (dup->name, inst->name);

  /* if not SFDUP_NORMAL mode then duplicate itemid */
  if (mode != SFDUP_NORMAL) dup->itemid = inst->itemid;

  p = inst->zone;
  while (p)
    {				/* loop over inst zones and copy them */
      /* duplicate the zone (with same mode) */
      dupz = sfont_zone_dup ((SFZone *)(p->data), mode);
      dup->zone = g_slist_append (dup->zone, dupz);
      p = g_slist_next (p);
    }

  return (dup);
}

/* destroy an instrument and its zones */
void
sfont_inst_destroy (SFInst * inst)
{
  GSList *p;

  p = inst->zone;
  while (p)
    {				/* loop over instrument's zones */
      sfont_zone_destroy (p->data);
      p = g_slist_next (p);
    }
  g_slist_free (inst->zone);	/* free instrument's zone list */

  sfont_inst_free (inst);
}

/* allocate and initialize an SFInst structure */
SFInst *
sfont_inst_alloc (void)
{
  return (g_chunk_new0 (SFInst, chunk_inst));
}

/* free an SFInst structure */
void
sfont_inst_free (SFInst *inst)
{
  g_mem_chunk_free (chunk_inst, inst);	/* free inst chunk */
}

/* ---------------------------- */
/* ---- SFSample functions ---- */
/* ---------------------------- */

/* duplicate a SFSample structure, if mode != SFDUP_NORMAL then copy itemid
   otherwise clear it, same SFSamDataInfo structure is referenced and
   refcount/dorefcount is incremented depending on "refcount" == TRUE/FALSE */
SFSample *
sfont_sample_dup (SFSample * sam, SFDupEnum mode, gboolean refcount)
{
  SFSample *dup;

  dup = sfont_sample_alloc ();
  memcpy (dup, sam, sizeof (SFSample));

  /* if normal duplicate mode then clear itemid */
  if (mode == SFDUP_NORMAL) dup->itemid = 0;

  if (refcount)
    dup->datainfo->refcount++;
  else dup->datainfo->dorefcount++;

  return (dup);
}

/* destroy a sample and decrement refcount/dorefcount of the SamDataInfo
   structure, depending on the value of "refcount" (TRUE/FALSE) */
void
sfont_sample_destroy (SFSample *sam, gboolean refcount)
{
  if (refcount)
    sam->datainfo->refcount--;
  else sam->datainfo->dorefcount--;

  sfont_sample_free (sam);
}

/* allocate and initialize a sample structure to dead state */
SFSample *
sfont_sample_alloc (void)
{
  return (g_chunk_new0 (SFSample, chunk_sample));
}

/* free a sample structure */
void
sfont_sample_free (SFSample *sam)
{
  g_mem_chunk_free (chunk_sample, sam);  /* free sample chunk */
}

/* allocate a SFSamDataInfo structure */
SFSamDataInfo *
sfont_samdatainfo_alloc (void)
{
  SFSamDataInfo *datainfo;
  datainfo = g_chunk_new0 (SFSamDataInfo, chunk_samdatainfo);

  return (datainfo);
}

/* free a SFSamDataInfo structure */
void
sfont_samdatainfo_free (SFSamDataInfo *datainfo)
{
  g_mem_chunk_free (chunk_samdatainfo, datainfo);
}

/* -------------------------- */
/* ---- SFZone functions ---- */
/* -------------------------- */

/* free all elements of a zone (Preset or Instrument) */
void
sfont_zone_destroy (SFZone * zone)
{
  GSList *p;

  if (!zone)
    return;

  p = zone->gen;
  while (p)
    {				/* Free gen chunks for this zone */
      if (p->data)
	g_mem_chunk_free (chunk_gen, p->data);
      p = g_slist_next (p);
    }
  g_slist_free (zone->gen);	/* free genlist */

  p = zone->mod;
  while (p)
    {				/* Free mod chunks for this zone */
      if (p->data)
	g_mem_chunk_free (chunk_mod, p->data);
      p = g_slist_next (p);
    }
  g_slist_free (zone->mod);	/* free modlist */

  sfont_zone_free (zone);
}

/* duplicate a zone (preset or instrument), mode can be one of 3 states
   SFDUP_NORMAL: Normal duplicate, itemids are cleared instsamp GSList * dup'd
   SFDUP_ARCHIVE: Archival mode, zone itemid dup'd, GSList * -> itemid
   SFDUP_RESTORE: Restore from archival, zone item ID dup'd, itemid -> GSList *
*/
SFZone *
sfont_zone_dup (SFZone * zone, SFDupEnum mode)
{
  GSList *p;
  SFZone *dup;
  SFGen *gen, *dupgen;
  SFMod *mod, *dupmod;
  GtkCTreeNode *node;

  dup = sfont_zone_alloc ();
  if (mode != SFDUP_NORMAL) dup->itemid = zone->itemid;
  dup->gen = NULL;
  dup->mod = NULL;

  if ((dup->instsamp = zone->instsamp))
    {
      if (mode == SFDUP_ARCHIVE) /* if archive, GSList * -> itemid */
	dup->instsamp = GINT_TO_POINTER (*((SFItemID *)(dup->instsamp->data)));
      else if (mode == SFDUP_RESTORE)
	{			/* if restore, itemid -> GSList * */
	  node = SFTREE_LOOKUP_ITEMID (GPOINTER_TO_INT (dup->instsamp));
	  if (node)
	    dup->instsamp = (GSList *)(SFTREE_NODE_REF (node)->dptr);
	  else dup->instsamp = NULL;
	}
    }

  p = zone->gen;
  while (p)
    {
      gen = (SFGen *) (p->data);
      dupgen = sfont_gen_alloc ();
      dupgen->id = gen->id;
      dupgen->amount.sword = gen->amount.sword;

      dup->gen = g_slist_append (dup->gen, dupgen);
      p = g_slist_next (p);
    }

  p = zone->mod;
  while (p)
    {
      mod = (SFMod *) (p->data);
      dupmod = sfont_mod_alloc ();
      dupmod->src = mod->src;
      dupmod->dest = mod->dest;
      dupmod->amount = mod->amount;
      dupmod->amtsrc = mod->amtsrc;
      dupmod->trans = mod->trans;

      dup->mod = g_slist_append (dup->mod, dupmod);
      p = g_slist_next (p);
    }

  return (dup);
}

SFZone *
sfont_zone_alloc (void)
{
  return (g_chunk_new0 (SFZone, chunk_zone));
}

void
sfont_zone_free (SFZone *zone)
{
  g_mem_chunk_free (chunk_zone, zone);
}

/* ------------------------- */
/* ---- SFMod functions ---- */
/* ------------------------- */

SFMod *
sfont_mod_alloc (void)
{
  return (g_chunk_new0 (SFMod, chunk_mod));
}

void
sfont_mod_free (SFMod * mod)
{
  g_mem_chunk_free (chunk_mod, mod);
}

/* ------------------------- */
/* ---- SFGen functions ---- */
/* ------------------------- */

SFGen *
sfont_gen_alloc (void)
{
  return (g_chunk_new0 (SFGen, chunk_gen));
}

void
sfont_gen_free (SFGen * gen)
{
  g_mem_chunk_free (chunk_gen, gen);
}

/* Convert SF generator amount to user units */
float
gen_sf2userval (guint16 gen, SFGenAmount amt, gboolean absval)
{
  guint8 unit;

  unit = genparms[gen].unit;	/* get the unit enum for this gen */
  if (!unit)
    return (0.0);		/* if no unit specified, return */

  /* convert absolute or offset gen value? */
  if (absval)
    return ((*genconv[unit].sf2user) (amt));	/* convert abs gen */
  else
    return ((*genconv[unit].sf2ofs) (amt));	/* convert ofs gen */
}

/* gen_sf2userstr ------------------------------------------------------- *
 * Desc: Convert generator value from sound font units to user units	  *
 * Inputs: generator id, GenAmount, char *buf to store printable string   *
 * Outputs: Floating user unit value, see char *buf in inputs		  *
 * ---------------------------------------------------------------------- */
float
gen_sf2userstr (guint16 gen, SFGenAmount amt, gchar * buf, gboolean absval)
{
  float val;
  guint8 unit;

  unit = genparms[gen].unit;	/* get the unit type for this gen */
  if (!unit)
    return (0.0);		/* if no unit specified, return */

  /* convert absolute or offset gen value? */
  if (absval)
    val = (*genconv[unit].sf2user) (amt);	/* convert abs gen */
  else
    val = (*genconv[unit].sf2ofs) (amt);	/* convert ofs gen */

  sprintf (buf, "%.*f", (int) genconv[unit].digits, val);

  return (val);
}

gint16
gen_userstr2sf (guint16 gen, gchar * val, gboolean absval, gboolean * err)
{
  gchar *errptr = NULL;		/* if error occurs during ascii conv */
  float uval;			/* user float value, after ascii to float conversion */
  guint8 unit;
  gint sfval;
  gint ofsrange;		/* used only for offset gens, range of value */

  uval = strtod (val, &errptr);	/* convert user string to float */

  if (errptr == val)
    {				/* conversion error occured? */
      *err = TRUE;
      return (0);
    }

  *err = FALSE;			/* no more chance of error */

  unit = genparms[gen].unit;
  if (!unit)
    return (0);			/* if not convertable, bad programmer, bad! */

  /* calculate value in sound font units, and range check */
  if (absval)
    {
      sfval = (*genconv[unit].user2sf) (uval);
      if (sfval < genparms[gen].min)
	sfval = genparms[gen].min;
      else if (sfval > genparms[gen].max)
	sfval = genparms[gen].max;
    }
  else
    {
      sfval = (*genconv[unit].ofs2sf) (uval);
      ofsrange = genparms[gen].max - genparms[gen].min;
      if (sfval < -ofsrange)
	sfval = -ofsrange;
      else if (sfval > ofsrange)
	sfval = ofsrange;
    }

  return ((gint16) sfval);
}

/* Set inst gen value (deletes if value is default, creates if no exist) */
void
sfont_gen_set (SFData * sf, SFZone * zone, guint16 gen, SFGenAmount amt,
  gboolean preset)
{
  GSList *p;
  SFGen *genp;
  gboolean defval = FALSE;

  /* Check if requested value is the default for this gen */
  if (preset)
    {
      if ((gen == Gen_KeyRange || gen == Gen_VelRange)
	&& amt.uword == DEFRANGE)
	defval = TRUE;
      else if (amt.uword == 0)
	defval = TRUE;
    }
  else
    {
      if (genparms[gen].def == amt.sword)
	defval = TRUE;
    }

  p = zone->gen;
  while (p)
    {				/* loop through zone's generators */
      genp = (SFGen *) (p->data);
      if (genp->id == gen)
	{			/* Is this gen the requested one? */
	  if (defval)
	    {			/* gen found, if default val then remove */
	      g_mem_chunk_free (chunk_gen, genp);
	      zone->gen = g_slist_remove (zone->gen, genp);
	    }
	  else
	    genp->amount.sword = amt.sword;
	  sf->up2date = FALSE;	/* sfont has changed */
	  return;
	}
      p = g_slist_next (p);
    }
  if (defval)
    return;			/* if gen not found and default val, return */

  /* Generator not found, and amt isn't the default */
  genp = g_chunk_new (SFGen, chunk_gen);
  genp->id = gen;
  genp->amount.sword = amt.sword;
  if (gen == Gen_KeyRange)
    {				/* Keyranges must be first */
      zone->gen = g_slist_prepend (zone->gen, genp);
    }
  else if (gen == Gen_VelRange)
    {				/* Velranges preceded only by keyranges */
      gint i = 0;
      if (zone->gen && ((SFGen *) (zone->gen->data))->id == Gen_KeyRange)
	i++;
      zone->gen = g_slist_insert (zone->gen, genp, i);
    }
  else
    {				/* All other gens can just be appended */
      zone->gen = g_slist_append (zone->gen, genp);
    }
  sf->up2date = FALSE;		/* sfont has changed */
}

/* removes a generator from a zone, thereby falling back to its default val */
void
sfont_gen_unset (SFData * sf, SFZone * zone, guint16 gen)
{
  GSList *p;
  SFGen *genp;

  p = zone->gen;
  while (p)
    {
      genp = (SFGen *) (p->data);
      if (genp->id == gen)
	{
	  g_mem_chunk_free (chunk_gen, genp);
	  zone->gen = g_slist_remove (zone->gen, genp);
	  sf->up2date = FALSE;
	  return;
	}
      p = g_slist_next (p);
    }
}

/* retrieve a generator value */
SFGenAmount *
sfont_gen_get (SFZone * zone, guint16 gen)
{
  GSList *p;
  SFGen *genp;

  p = zone->gen;
  while (p)
    {				/* loop through zone's generators */
      genp = (SFGen *) (p->data);
      if (genp->id == gen)	/* Is this gen the requested one? */
	return (&genp->amount);
      p = g_slist_next (p);
    }
  return (NULL);
}

/* init a generator array to default values */
void
gen_initdef (SFGenAmount * garr)
{
  gint i;

  for (i = 0; i <= SFGen_MaxValid; i++)
    {				// for each generator item
      garr[i].uword = genparms[i].def;	// init to default val
    }
}

/* init an offset generator array (presets) to ZERO values */
void
gen_initofs (SFGenAmount * garr)
{
  gint i;

  for (i = 0; i <= SFGen_MaxValid; i++)
    {
      garr[i].uword = 0;
    }

  garr[Gen_KeyRange].uword = DEFRANGE;
  garr[Gen_VelRange].uword = DEFRANGE;

  return;
}

/* Insert generator values from zone into generator array */
void
gen_process_zone (SFZone * zone, SFGenAmount * garr)
{
  GSList *p;
  SFGen *gen;

  p = zone->gen;		// First generator
  while (p)
    {				// while generators
      gen = (SFGen *) (p->data);
      garr[gen->id].uword = gen->amount.uword;	// assign to array
      p = g_slist_next (p);
    }
}

/* Offset an array of absolute generators with offset generators */
void
gen_offset_zone (SFGenAmount * abs, SFGenAmount * ofs)
{
  gint i;
  gint32 temp;

  for (i = 0; i < Gen_KeyRange; i++)
    {
      temp = (gint32) (abs[i].sword) + (gint32) (ofs[i].sword);
      if (temp < (gint32) genparms[i].min)
	temp = genparms[i].min;
      else if (temp > (gint32) genparms[i].max)
	temp = genparms[i].max;
      abs[i].sword = (gint16) temp;
    }

  abs[Gen_KeyRange] = range_intersect (abs[Gen_KeyRange], ofs[Gen_KeyRange]);
  abs[Gen_VelRange] = range_intersect (abs[Gen_VelRange], ofs[Gen_VelRange]);

  for (i = Gen_VelRange + 1; i < SFGen_MaxValid; i++)
    {
      temp = (gint32) (abs[i].sword) + (gint32) (ofs[i].sword);
      if (temp < (gint32) genparms[i].min)
	temp = genparms[i].min;
      else if (temp > (gint32) genparms[i].max)
	temp = genparms[i].max;
      abs[i].sword = (gint16) temp;
    }

  return;
}

/* Find intersection of two generator ranges */
SFGenAmount
range_intersect (SFGenAmount a, SFGenAmount b)
{
  gint i = 0;
  guint8 p[2] = { 0, 0 };
  SFGenAmount val;

  if ((a.range.lo >= b.range.lo) && (a.range.lo <= b.range.hi))
    p[i++] = a.range.lo;

  if ((a.range.hi >= b.range.lo) && (a.range.hi <= b.range.hi))
    p[i++] = a.range.hi;

  if ((b.range.lo > a.range.lo) && (b.range.lo < a.range.hi))
    p[i++] = b.range.lo;

  if ((b.range.hi > a.range.lo) && (b.range.hi < a.range.hi))
    p[i++] = b.range.hi;

  if (p[0] < p[1])
    {
      val.range.lo = p[0];
      val.range.hi = p[1];
    }
  else
    {
      val.range.lo = p[1];
      val.range.hi = p[0];
    }

  return (val);
}

/* Find generator in gen list */
GSList *
gen_inlist (gint gen, GSList * genlist)
{				/* is generator in gen list? */
  GSList *p;

  p = genlist;
  while (p)
    {
      if (p->data == NULL)
	return (NULL);
      if (gen == ((SFGen *) p->data)->id)
	break;
      p = g_slist_next (p);
    }
  return (p);
}

/* check validity of instrument generator */
gint
gen_valid (gint gen)
{				/* is generator id valid? */
  gint i = 0;

  if (gen > SFGen_MaxValid)
    return (FALSE);
  while (badgen[i] && badgen[i] != gen)
    i++;
  return (badgen[i] == 0);
}

/* check validity of preset generator */
gint
gen_validp (gint gen)
{				/* is preset generator valid? */
  gint i = 0;

  if (!gen_valid (gen))
    return (FALSE);
  while (badpgen[i] && badpgen[i] != (guint16) gen)
    i++;
  return (badpgen[i] == 0);
}

/* conversion functions */

gint
float2int (float f)
{
  return ((gint) f);
}

float
int2float (SFGenAmount val)
{
  return ((float) val.sword);
}

gint
hz2cents (float hz)
{
  return ((gint) (log (hz / 8.176) / log (2) * 1200));
}

float
cents2hz (SFGenAmount cents)
{
  return (8.176 * pow (2.0, ((double) cents.sword) / 1200.0));
}

gint
db2cb (float db)
{
  return ((gint) (db * 10.0));
}

float
cb2db (SFGenAmount cb)
{
  return ((float) cb.sword / 10.0);
}

gint
sec2tcent (float sec)
{
  return ((gint) (log (sec) / log (2) * 1200));
}

float
tcent2sec (SFGenAmount tcent)
{
  return (pow (2.0, ((double) tcent.sword) / 1200.0));
}

gint
perc2tperc (float perc)
{
  return ((gint) (perc * 10));
}

float
tperc2perc (SFGenAmount tperc)
{
  return (((float) tperc.sword) / 10);
}
