/*==================================================================
 * uiface.c - GTK user interface routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2001 Josh Green
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/
#include <stdio.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include "uiface.h"
#include "uif_sfont.h"
#include "uif_sfgen.h"
#include "uif_menutbar.h"
#include "uif_pref.h"
#include "uif_sftree.h"
#include "uif_treemenu.h"
#include "uif_pianospan.h"
#include "uif_piano.h"
#include "uif_help.h"
#include "splash.h"
#include "glade_interface.h"
#include "widgets/keyspan.h"		/* for KEYSPAN_DEFAULT_SIZEX */
#include "widgets/popdog.h"
#include "midi.h"
#include "sequencer.h"
#include "wavetable.h"
#include "smurfcfg.h"
#include "util.h"
#include "i18n.h"

/* define size for minimum negotiated sound font tree window size */
#define TREEWIN_MIN_WIDTH	260
#define TREEWIN_MIN_HEIGHT	160

/* used if frame sizes are negotiated the max added vertical height of all
   elements in the smurf window except the sound font tree */
#define VERTWIN_JUNK_HEIGHT	300

/* Data */

GtkWidget *ui_main_window;
GtkWidget *ui_sftree_window;

/* Prototypes */

static void ui_real_quit (void);
static gint cb_main_win_delete_event (GtkWidget * widget, GdkEvent * event,
				      gpointer data);
static void ui_update_smurfcfg_state (void);

/* Functions */

void
ui_init (void)
{				/* initialize user interface */
  GtkWidget *win_main;
  gint width, height;
  GtkWidget *widg, *vpane, *hpane;

  win_main = create_mainwin ();
  ui_main_window = win_main;

  tbar_create ();		/* create the toolbar */

  vpane = gtk_object_get_data (GTK_OBJECT (win_main), "VPANE");
  hpane = gtk_object_get_data (GTK_OBJECT (win_main), "HPANE");

  /* restore window's size from config */
  gtk_widget_set_usize (win_main,
    smurfcfg_get_val (SMURFCFG_WIN_WIDTH)->v_int,
    smurfcfg_get_val (SMURFCFG_WIN_HEIGHT)->v_int);

  /* if configuration says to set window position, then do so */
  if (smurfcfg_get_val (SMURFCFG_SET_WIN_POS)->v_int)
    gtk_widget_set_uposition (win_main,
      smurfcfg_get_val (SMURFCFG_WIN_XPOS)->v_int,
      smurfcfg_get_val (SMURFCFG_WIN_YPOS)->v_int);

  gtk_signal_connect (GTK_OBJECT (win_main), "delete_event",
    GTK_SIGNAL_FUNC (cb_main_win_delete_event), NULL);

  /* we want key release events for piano */
  gtk_widget_add_events (win_main, GDK_KEY_RELEASE_MASK);
  gtk_signal_connect (GTK_OBJECT (win_main), "key-press-event",
    GTK_SIGNAL_FUNC (uipiano_cb_key_event), GINT_TO_POINTER (TRUE));
  gtk_signal_connect (GTK_OBJECT (win_main), "key-release-event",
    GTK_SIGNAL_FUNC (uipiano_cb_key_event), GINT_TO_POINTER (FALSE));

  /* does config say to set sound font tree pane size? */
  ui_sftree_window = sftree_create ();
  if (smurfcfg_get_val (SMURFCFG_SET_PANE_SIZE)->v_int)
    {
      width = smurfcfg_get_val (SMURFCFG_TREEWIN_WIDTH)->v_int;
      height = smurfcfg_get_val (SMURFCFG_TREEWIN_HEIGHT)->v_int;
    }
  else
    {				/* ?: No, force auto sizing */
      width = 0;
      height = 0;
    }
  if (!width)
    {
      width = smurfcfg_get_val (SMURFCFG_WIN_WIDTH)->v_int
        - KEYSPAN_DEFAULT_SIZEX - 12;
      if (width < TREEWIN_MIN_WIDTH)
	width = TREEWIN_MIN_WIDTH;
    }
  if (!height)
    {
      height = smurfcfg_get_val (SMURFCFG_WIN_HEIGHT)->v_int
        - VERTWIN_JUNK_HEIGHT;
      if (height < TREEWIN_MIN_HEIGHT)
	height = TREEWIN_MIN_HEIGHT;
    }

  /* set pane separator positions */
  gtk_paned_set_position (GTK_PANED (hpane), width);
  gtk_paned_set_position (GTK_PANED (vpane), height);

  gtk_paned_pack1 (GTK_PANED (hpane), ui_sftree_window, TRUE, TRUE);

  widg = pianospan_create ();
  gtk_paned_pack2 (GTK_PANED (hpane), widg, TRUE, TRUE);

  widg = sfgen_win_create ();
  gtk_paned_pack2 (GTK_PANED (vpane), widg, TRUE, TRUE);

  /* set initial view menu radio items as Glade seems to unexpectedly alter
     the initial menu item from time to time */
  sfmenu_set_lowpane_radio_item (SFGEN_VIEW);
  sfmenu_set_piano_mode_radio_item (PIANOSPAN_PIANO);

  sfmenu_update_control_buttons ();

  uisf_init ();			/* init stuff for sound font user interface */
  treemenu_init ();		/* initialize right click menus */

  gtk_widget_show (win_main);

  uisf_deactivate_selection ();	/* make UI tree selection inactive */

  /* pop up smurfy tip window if enabled */
  if (smurfcfg_get_val (SMURFCFG_TIPS_ENABLED)->v_int)
    uihlp_smurfytips_create ();

  /* display splash only if not disabled */
  if (!smurfcfg_get_val (SMURFCFG_DISABLE_SPLASH)->v_int)
    splash_display (TRUE);	/* Display splash with timeout */
}

void
ui_quit (void)
{
  GtkWidget *popup;
  gchar *s;
  GList *p;
  gboolean unsaved = FALSE;
  GTokenValue *val;
  gint retval;

  if (!smurfcfg_up2date)
    {
      popup =
	util_quick_popup (_("Unsaved changes to preferences"),
			  _("Save"), smurfcfg_save, NULL,
			  _("Discard"), NULL, NULL,
			  _("Cancel"), NULL, NULL, NULL);

      retval = GPOINTER_TO_INT (util_waitfor_widget_action (popup));

      if (retval == 1)		/* save btn pressed, need to destroy popup */
	gtk_widget_destroy (popup);
      else if (retval == 3 || retval < 0)
	return;			/* Cancel, dialog killed, or gtk_main_quit */
    }

  p = uisf_sfonts;
  while (p)
    {
      if (!((UISFont *) (p->data))->sf->up2date)
	{
	  unsaved = TRUE;
	  break;
	}
      p = g_list_next (p);
    }

  val = smurfcfg_get_val (SMURFCFG_QUIT_CONFIRM);

  if (val->v_int == SMURFCFG_QUIT_CONFIRM_NEVER
    || (val->v_int == SMURFCFG_QUIT_CONFIRM_UNSAVED && !unsaved))
    {
      ui_real_quit ();
      return;
    }

  if (unsaved)
    s = _("Unsaved sound fonts, and you want to quit?");
  else
    s = _("Are you sure you want to quit?");

  popup = util_quick_popup (s, _("Quit"), ui_real_quit, NULL,
    _("Cancel"), NULL, NULL, NULL);
}

static void
ui_real_quit (void)
{
  ui_update_smurfcfg_state (); 	/* update variables before saving them */
  smurfcfg_save_state ();	/* save ~/.smurf/smurf_state.cfg */

  midi_close ();
  seq_close ();
  wtbl_close ();

  gtk_main_quit ();
}

static gint
cb_main_win_delete_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  ui_quit ();
  return (TRUE);
}

static void
ui_update_smurfcfg_state (void)
{
  /* see if window geometry should be updated */
  if (smurfcfg_get_int (SMURFCFG_SAVE_WIN_GEOM))
    ui_save_window_geometry ();
}

/* save main window geometry to smurfcfg state variables */
void
ui_save_window_geometry (void)
{
  GTokenValue val;
  gint x, y;

  /* make sure window exists */
  if (ui_main_window->window)
    {
      gdk_window_get_root_origin (ui_main_window->window, &x, &y);
      val.v_int = x;
      smurfcfg_set_val (SMURFCFG_WIN_XPOS, &val);
      val.v_int = y;
      smurfcfg_set_val (SMURFCFG_WIN_YPOS, &val);

      val.v_int = ui_main_window->allocation.width;
      smurfcfg_set_val (SMURFCFG_WIN_WIDTH, &val);
      val.v_int = ui_main_window->allocation.height;
      smurfcfg_set_val (SMURFCFG_WIN_HEIGHT, &val);

      val.v_int = ui_sftree_window->allocation.width;
      smurfcfg_set_val (SMURFCFG_TREEWIN_WIDTH, &val);
      val.v_int = ui_sftree_window->allocation.height;
      smurfcfg_set_val (SMURFCFG_TREEWIN_HEIGHT, &val);
    }
}
