# -*- Mode:Python; indent-tabs-mode:nil; tab-width:4 -*-
#
# Copyright (C) 2017 Canonical Ltd
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import click
import os

import snapcraft
from snapcraft.internal import errors, lxd
from ._options import get_project_options
from . import env


@click.group()
def containerscli():
    pass


@containerscli.command()
@click.option('--debug', is_flag=True,
              help='Shells into the environment if the build fails.')
def refresh(debug, **kwargs):
    """Refresh an existing LXD container.

    \b
    Examples:
        snapcraft refresh

    This will take care of updating the apt package cache, upgrading packages
    as needed as well as refreshing snaps.
    """

    container_config = env.get_container_config()
    if not container_config.use_container:
        raise errors.SnapcraftEnvironmentError(
            "The 'refresh' command only applies to LXD containers but "
            "SNAPCRAFT_CONTAINER_BUILDS is not set or 0.\n"
            "Maybe you meant to update the parts cache instead? "
            "You can do that with the following command:\n\n"
            "snapcraft update")

    project_options = get_project_options(**kwargs, debug=debug)
    config = snapcraft.internal.load_config(project_options)
    lxd.Project(project_options=project_options,
                remote=container_config.remote,
                output=None, source=os.path.curdir,
                metadata=config.get_metadata()).refresh()
