#include <pwd.h>
#include <errno.h>
#include <nss.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <stdio.h>

#include <pthread.h>

#include "../lib/lib.h"

static int copy_passwd(struct passwd *, char *, int, char *);

/* Global state */
static unsigned int userarray_idx;
static unsigned int userarray_count;

static char **userarray_list;

static pthread_mutex_t userarray_mutex = PTHREAD_MUTEX_INITIALIZER;

enum nss_status _nss_snui_setpwent_r (void)
{
	FILE *sock;
	int rv, numresp, i;
	char response[1024];
	char **new_list;
	
	pthread_mutex_lock(&userarray_mutex);
	
	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		pthread_mutex_unlock(&userarray_mutex);
		return NSS_STATUS_UNAVAIL;
	}
	
	fprintf(sock, "passwd list\n");
	
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		pthread_mutex_unlock(&userarray_mutex);
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (numresp != 300) {
		syslog(LOG_ERR, "Error while asking for list: %s", response);
		pthread_mutex_unlock(&userarray_mutex);
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	/* OK, now we can read lines until the cows come home */
	userarray_count = userarray_idx = 0;	
	fgets(response, 1024, sock);
	while (strlen(response) != 0) {
		userarray_count++;
		new_list = realloc(userarray_list, userarray_count);
		if (!new_list) {
			/* Memory limit reached */
			for (i = 0; i < userarray_count - 1; i++) {
				free(userarray_list[i]);
			}
			free(userarray_list);
			userarray_list = NULL;
			syslog(LOG_CRIT, "Out of memory error");
			errno = ENOMEM;
			pthread_mutex_unlock(&userarray_mutex);
			fclose(sock);
			return NSS_STATUS_TRYAGAIN;
		}
		userarray_list = new_list;
		userarray_list[userarray_count-1] = malloc(strlen(response) + 1);
		if (!userarray_list[userarray_count-1]) {
			/* Memory limit again */
			for (i = 0; i < userarray_count - 1; i++) {
				free(userarray_list[i]);
			}
			free(userarray_list);
			userarray_list = NULL;
			syslog(LOG_CRIT, "Out of memory error");
			errno = ENOMEM;
			pthread_mutex_unlock(&userarray_mutex);
			fclose(sock);
			return NSS_STATUS_TRYAGAIN;
		}
		
		strcpy(userarray_list[userarray_count-1], response);
	}

	pthread_mutex_unlock(&userarray_mutex);
	fclose(sock);
	
	return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_snui_endpwent_r(void)
{
	int i;
	
	pthread_mutex_lock(&userarray_mutex);
	
	for (i = 0; i < userarray_count; i++) {
		free(userarray_list[i]);
	}
	free(userarray_list);
	userarray_list = 0;
	userarray_count = 0;
	userarray_idx = 0;

	pthread_mutex_unlock(&userarray_mutex);

	return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_snui_getpwent_r(struct passwd *result_buf, char *buffer, int buflen, int *errnop)
{
	int rv;
	enum nss_status retval;
	
	if (userarray_list == NULL) {
		return NSS_STATUS_UNAVAIL;
	}
	
	pthread_mutex_lock(&userarray_mutex);
	
	if (userarray_idx >= userarray_count) {
		retval = NSS_STATUS_NOTFOUND;
	} else {
		rv = copy_passwd(result_buf, buffer, buflen, userarray_list[userarray_idx]);
		if (!rv) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else if (rv == -1) {
			retval = NSS_STATUS_UNAVAIL;
		} else {
			userarray_idx++;
			retval = NSS_STATUS_SUCCESS;
		}
	}
	
	pthread_mutex_unlock(&userarray_mutex);
	
	return retval;
}

enum nss_status _nss_snui_getpwuid_r (
			uid_t uid,
			struct passwd *result_buf,
			char *data_buf,
			size_t buflen,
			int *errnop)
{
	FILE *sock;
	int rv, numresp;
	char response[1024];
	enum nss_status retval;
	
	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		return NSS_STATUS_UNAVAIL;
	}
	
	fprintf(sock, "passwd id %i\n", uid);
	
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (numresp == 500) {
		syslog(LOG_ERR, "Error while searching for UID %i: %s", uid, response);
		retval = NSS_STATUS_UNAVAIL;
	} else if (numresp == 400) {
		retval = NSS_STATUS_NOTFOUND;
	} else if (numresp == 200) {
		rv = copy_passwd(result_buf, data_buf, buflen, response);
		if (rv < 0) {
			retval = NSS_STATUS_UNAVAIL;
		} else if (rv == 0) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else {
			retval = NSS_STATUS_SUCCESS;
		}
	} else {
		syslog(LOG_WARNING, "Unknown response from server: %i %s", numresp, response);
		retval = NSS_STATUS_UNAVAIL;
	}
	
	fclose(sock);
	
	return retval;
}

enum nss_status _nss_snui_getpwnam_r(
			char *name,
			struct passwd *result_buf,
			char *data_buf,
			size_t buflen,
			int *errnop)
{
	FILE *sock;
	int rv, numresp;
	char response[1024];
	enum nss_status retval;
	
	syslog(LOG_DEBUG, "Looking for a user named [%s]", name);
	if ((rv = net_client_init("nss", &sock))) {
		syslog(LOG_CRIT, "Failed to open network connection: %s - aborting", strerror(rv));
		return NSS_STATUS_UNAVAIL;
	}
	
	if (strlen(name) > 1000) {
		syslog(LOG_ERR, "Name given to search for is *way* too long (%i chars)", strlen(name));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
	fprintf(sock, "passwd name %s\n", name);
		
	if ((rv = net_read_response(sock, &numresp, response, 1024))) {
		syslog(LOG_ERR, "Error while reading command response: %s", strerror(rv));
		fclose(sock);
		return NSS_STATUS_UNAVAIL;
	}
		
	if (numresp == 500) {
		syslog(LOG_ERR, "Error while searching for name %s: %s", name, response);
		retval = NSS_STATUS_UNAVAIL;
	} else if (numresp == 400) {
		retval = NSS_STATUS_NOTFOUND;
	} else if (numresp == 200) {
		rv = copy_passwd(result_buf, data_buf, buflen, response);
		if (rv < 0) {
			retval = NSS_STATUS_UNAVAIL;
		} else if (rv == 0) {
			*errnop = ERANGE;
			retval = NSS_STATUS_TRYAGAIN;
		} else {
			retval = NSS_STATUS_SUCCESS;
		}
	} else {
		syslog(LOG_WARNING, "Unknown response from server: %i %s", numresp, response);
		retval = NSS_STATUS_UNAVAIL;
	}
	
	fclose(sock);
	
	return retval;


}

/* Parse a passwd line and put it into the password struct.  Return 0 on
 * insufficient space, -1 on parse error, and 1 on successful completion.
 */
static int copy_passwd(struct passwd *dest, char *destbuf, int destbuflen, char *src)
{
	char *p, *q;
	
	if (strlen(src) + 1 > destbuflen) {
		return 0;
	}
	strcpy(destbuf, src);
	dest->pw_name = destbuf;
			
	p = index(destbuf, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	
	p++;
	dest->pw_passwd = p;

	p = index(p, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	p++;
	q = index(p, ':');
	if (!q) {
		return -1;
	}
	*q = '\0';
	dest->pw_uid = atoi(p);

	q++;
	p = index(q, ':');
	if (!p) {
		return -1;
	}
	dest->pw_gid = atoi(q);
		
	p++;
	dest->pw_gecos = p;

	p = index(p, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	p++;
	dest->pw_dir = p;

	p = index(p, ':');
	if (!p) {
		return -1;
	}
	*p = '\0';
	p++;
	dest->pw_shell = p;
	
	return 1;
}
