/******************************************************************************
*       SOFA, Simulation Open-Framework Architecture, version 1.0 beta 3      *
*                (c) 2006-2008 MGH, INRIA, USTL, UJF, CNRS                    *
*                                                                             *
* This library is free software; you can redistribute it and/or modify it     *
* under the terms of the GNU Lesser General Public License as published by    *
* the Free Software Foundation; either version 2.1 of the License, or (at     *
* your option) any later version.                                             *
*                                                                             *
* This library is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License *
* for more details.                                                           *
*                                                                             *
* You should have received a copy of the GNU Lesser General Public License    *
* along with this library; if not, write to the Free Software Foundation,     *
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301 USA.          *
*******************************************************************************
*                               SOFA :: Modules                               *
*                                                                             *
* Authors: The SOFA Team and external contributors (see Authors.txt)          *
*                                                                             *
* Contact information: contact@sofa-framework.org                             *
******************************************************************************/
#ifndef SOFA_COMPONENT_TOPOLOGY_MANIFOLDEDGESETTOPOLOGYMODIFIER_H
#define SOFA_COMPONENT_TOPOLOGY_MANIFOLDEDGESETTOPOLOGYMODIFIER_H

#include <sofa/component/topology/EdgeSetTopologyModifier.h>

namespace sofa
{

namespace component
{

namespace topology
{
	class ManifoldEdgeSetTopologyContainer;
    
	using core::componentmodel::topology::BaseMeshTopology;
	typedef BaseMeshTopology::EdgeID EdgeID;
	typedef BaseMeshTopology::Edge Edge;
	typedef BaseMeshTopology::SeqEdges SeqEdges;
	typedef BaseMeshTopology::VertexEdges VertexEdges;		
	
	/** 
	* A class that can apply basic transformations on a set of points.
	*/
	class ManifoldEdgeSetTopologyModifier : public EdgeSetTopologyModifier
	{	
	public:
		ManifoldEdgeSetTopologyModifier() 
		: EdgeSetTopologyModifier()
		{}

		virtual ~ManifoldEdgeSetTopologyModifier() {}

		virtual void init();

		/** \brief Add an edge.
		*/
		void addEdgeProcess(Edge e);

		/** \brief Add some edges to this topology.
		* 
		* \sa addEdgesWarning
		*/
		virtual void addEdgesProcess(const sofa::helper::vector< Edge > &edges);

		/** \brief Effectively Remove a subset of edges. Eventually remove isolated vertices
		*
		* Elements corresponding to these edges are removed form the mechanical object's state vectors.
		*
		* Important : some structures might need to be warned BEFORE the edges are actually deleted, so always use method removeEdgesWarning before calling removeEdgesProcess.
		* \sa removeEdgesWarning
		*
		* Important : parameter indices is not const because it is actually sorted from the highest index to the lowest one.
		*
		* @param removeIsolatedItems if true isolated vertices are also removed
		*/
		virtual void removeEdgesProcess(const sofa::helper::vector<unsigned int> &indices, const bool removeIsolatedItems = false);
		
		/** \brief Remove a set  of edges
		@param edges an array of edge indices to be removed (note that the array is not const since it needs to be sorted)
		*
		*/
		// side effect: edges are sorted in removeEdgesWarning
		virtual void removeEdges(/*const*/ sofa::helper::vector< unsigned int >& edges, 
								const bool removeIsolatedPoints = true);

		/** \brief Generic method to remove a list of items.
		 */
		virtual void removeItems(/*const*/ sofa::helper::vector< unsigned int >& items);

		/** \brief Generic method for points renumbering
		*/
		virtual void renumberPoints( const sofa::helper::vector<unsigned int> & index,
									 const sofa::helper::vector<unsigned int> & inv_index);

		/** \brief add a set  of edges
		@param edges an array of pair of vertex indices describing the edge to be created		
		*
		*/
		virtual void addEdges(const sofa::helper::vector< Edge >& edges) ;

		/** \brief add a set  of edges
		@param edges an array of pair of vertex indices describing the edge to be created
		@param ancestors for each edge to be created provides an array of edge ancestors (optional)
		@param baryCoefs for each edge provides the barycentric coordinates (sum to 1) associated with each ancestor (optional)
		*
		*/
		virtual void addEdges(const sofa::helper::vector< Edge >& edges,
			const sofa::helper::vector< sofa::helper::vector< unsigned int > > & ancestors,
			const sofa::helper::vector< sofa::helper::vector< double > >& baryCoefs) ;

		/** \brief Swap a list of pair edges, replacing each edge pair ((p11, p12), (p21, p22)) by the edge pair ((p11, p21), (p12, p22))
		*
		*/
		virtual void swapEdges(const sofa::helper::vector< sofa::helper::vector< unsigned int > >& edgesPairs);

		/** \brief Fuse a list of pair edges, replacing each edge pair ((p11, p12), (p21, p22)) by one edge (p11, p22)
		*
		* @param removeIsolatedPoints if true isolated vertices are also removed
		*/
		virtual void fuseEdges(const sofa::helper::vector< sofa::helper::vector< unsigned int > >& edgesPairs, const bool removeIsolatedPoints = true);

		/** \brief Split an array of edges, replacing each edge (p1, p2) by two edges (p1, p3) and (p3, p2) where p3 is the new vertex
		* On each edge, a vertex is created based on its barycentric coordinates 
		*
		* @param removeIsolatedPoints if true isolated vertices are also removed
		*/
		virtual void splitEdges( sofa::helper::vector<unsigned int> &indices,
			const bool removeIsolatedPoints = true);

		/** \brief Split an array of edges, replacing each edge (p1, p2) by two edges (p1, p3) and (p3, p2) where p3 is the new vertex
		* On each edge, a vertex is created based on its barycentric coordinates 
		*
		* @param removeIsolatedPoints if true isolated vertices are also removed
		*/
		virtual void splitEdges( sofa::helper::vector<unsigned int> &indices,
			const sofa::helper::vector< sofa::helper::vector< double > >& baryCoefs,
			const bool removeIsolatedPoints = true);

	private:
		ManifoldEdgeSetTopologyContainer* 	m_container;		
	};

} // namespace topology

} // namespace component

} // namespace sofa

#endif
