/*
 * -----------------------------------------------------------------------
 * Emulation of Freescale iMX21 USB OTG module 
 *
 * (C) 2006 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: nothing works
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * -----------------------------------------------------------------------
 */

#include <errno.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <bus.h>
#include <fio.h>
#include <signode.h>
#include <imx21_otg.h>
#include <configfile.h>
#include <i2c.h>
#include <isp1301.h>
#include <usbproto.h>
#include <cycletimer.h>

/* otg is at  0x10024000 and 0x10025000 */

#define OTG_HWMODE(base) 	((base) + 0x00)
#define		HWMODE_FUNCREV_MASK	(0xff<<24)
#define		HWMODE_FUNCREF_SHIFT	(24)
#define		HWMODE_HSTREF_MASK	(0xff<<16)
#define		HWMODE_HSTREF_SHIFT	(16)
#define		HWMODE_ANASDBEN		(1<<14)
#define		HWMODE_OTGXVCR_MASK	(3<<6)
#define		HWMODE_OTGXVCR_SHIFT	(6)
#define		HWMODE_HOSTXVCR_MASK	(3<<4)
#define		HWMODE_HOSTXVCR_SHIFT	(4)
#define		HWMODE_CRECFG_MASK	(3<<0)
#define		HWMODE_CRECFG_SHIFT	(0)
#define		HWMODE_CRECFG_HOST_ONLY	(1<<0)
#define		HWMODE_CRECFG_FUNC_ONLY	(2<<0)
#define		HWMODE_CRECFG_HNP	(3<<0)

#define	OTG_CINT_STAT(base)	((base) + 0x04)
#define		CINT_STAT_ASHNPINT	(1<<5)
#define		CINT_STAT_ASFCINT	(1<<4)
#define		CINT_STAT_ASHCINT	(1<<3)
#define		CINT_STAT_HNPINT	(1<<2)
#define		CINT_STAT_FCINT		(1<<1)
#define		CINT_STAT_HCINT		(1<<0)

#define	OTG_CINT_STEN(base)	((base) + 0x08)
#define		CINT_STEN_ASHNPINTEN	(1<<5)
#define		CINT_STEN_ASFCINTEN	(1<<4)
#define		CINT_STEN_ASHCINTEN	(1<<3)
#define		CINT_STEN_HNPINTEN	(1<<2)
#define		CINT_STEN_FCINTEN	(1<<1)
#define		CINT_STEN_HCINTEN	(1<<0)

#define OTG_CLK_CTRL(base)	((base) + 0x0c)
#define		CLK_CTRL_FUNCCLK	(1<<2)
#define		CLK_CTRL_HSTCLK		(1<<1)
#define		CLK_CTRL_MAINCLK	(1<<0)

#define OTG_RST_CTRL(base)	((base) + 0x10)
#define		RST_CTRL_RSTI2C		(1<<15)
#define		RST_CTRL_RSTCTRL	(1<<5)
#define		RST_CTRL_RSTFC		(1<<4)
#define		RST_CTRL_RSTFSKE	(1<<3)
#define		RST_CTRL_RSTRH		(1<<2)
#define		RST_CTRL_RSTHSIE	(1<<1)
#define		RST_CTRL_RSTHC		(1<<0)

#define OTG_FRM_INVTL(base)	((base) + 0x14)
#define		FRM_INVTL_FRMINTPER_MASK	(0x3fff<<16)
#define		FRM_INVTL_FRMINTPER_SHIFT	(16)
#define		FRM_INVTL_RSTFRM		(1<<15)
#define		FRM_INVTL_FRMINT_MASK		(0x3fff)
#define		FRM_INVTL_FRMINT_SHIFT		(0)

#define OTG_FRM_REMAIN(base)	((base) + 0x18)
#define		FRM_REMAIN_FRMREMN_MASK		(0x3fff<16)
#define		FRM_REMAIN_FRMREMN_SHIFT	(16)

#define OTG_HNP_CTRL(base)	((base) + 0x1c)
#define		HNP_CTRL_HNPDAT		(1<<30)
#define		HNP_CTRL_VBUSBSE	(1<<29)
#define		HNP_CTRL_VBUSABSV	(1<<28)
#define		HNP_CTRL_VBUSGTAVV	(1<<27)
#define		HNP_CTRL_SLAVE		(1<<22)
#define		HNP_CTRL_MASTER		(1<<21)
#define		HNP_CTRL_BGEN		(1<<20)
#define		HNP_CTRL_CMPEN		(1<<19)
#define		HNP_CTRL_ISBDEV		(1<<18)
#define		HNP_CTRL_ISADEV		(1<<17)
#define		HNP_CTRL_SWVBUSPUL	(1<<15)
#define		HNP_CTRL_SWAUTORST	(1<<12)
#define		HNP_CTRL_SWPUDP		(1<<11)
#define		HNP_CTRL_SWPDDM		(1<<9)
#define		HNP_CTRL_CLRERROR	(1<<3)
#define		HNP_CTRL_ADROPBUS	(1<<2)
#define		HNP_CTRL_ABBUSREQ	(1<<1)

#define	OTG_HNP_INT_STAT(base)  ((base) + 0x2c)
#define		HNP_INT_STAT_I2COTGINT	(1<<15)
#define		HNP_INT_STAT_SRPINT	(1<<5)
#define		HNP_INT_STAT_ABSESVALID	(1<<3)
#define		HNP_INT_STAT_AVBUSVALID	(1<<2)
#define		HNP_INT_STAT_IDCHANGE	(1<<0)

#define	OTG_HNP_INT_EN(base) 	((base) + 0x30)
#define		HNP_INT_EN_I2COTGINTEN	(1<<15)
#define		HNP_INT_EN_SRPINTEN	(1<<5)
#define		HNP_INT_EN_ABSESVALIDEN (1<<3)
#define		HNP_INT_EN_AVBUSVALIDEN	(1<<2)
#define		HNP_INT_EN_IDCHANGEEN	(1<<0)

#define OTG_USBCTRL(base)	((base) + 0x600)
#define		USBCTRL_I2CWUINTSTAT	(1<<27)
#define		USBCTRL_OTGWUINTSTAT	(1<<26)
#define		USBCTRL_HOSTWUINTSTAT	(1<<25)
#define		USBCTRL_FNTWUINTSTAT	(1<<24)
#define		USBCTRL_I2CWUINTEN	(1<<19)
#define		USBCTRL_OTGWUINTEN	(1<<18)
#define		USBCTRL_HOSTWUINTEN	(1<<17)
#define		USBCTRL_FNTWUINTEN	(1<<16)
#define		USBCTRL_OTGRCVRXDP	(1<<15)
#define		USBCTRL_HOST1BYPTLL	(1<<14)
#define		USBCTRL_OTGBYPVAL_MASK		(3<<10)
#define		USBCTRL_OTGBYPVAL_SHIFT		(10)
#define		USBCTRL_HOST1BYPVAL_MASK 	(3<<8)
#define		USBCTRL_HOST1BYPVAL_SHIFT 	(8)
#define		USBCTRL_OTGPWRMASK		(1<<6)
#define		USBCTRL_HOST1PWRMASK		(1<<5)
#define		USBCTRL_HOST2PWRMASK		(1<<4)
#define		USBCTRL_USBBYP			(1<<2)
#define		USBCTRL_HOST1TXENOE		(1<<1)

#define FUNC_COMSTAT(base)	((base) + 0x40)
#define		COMSTAT_SOFTRESET	(1<<7)
#define		COMSTAT_BADISOAP	(1<<3)
#define		COMSTAT_SUSPDET		(1<<2)
#define		COMSTAT_RSMINPROC	(1<<1)
#define		COMSTAT_RESETDET	(1<<0)
#define	FUNC_DEVADDR(base)	((base) + 0x44)
#define		DEVADDR_MASK	(0x7f)

#define	FUNC_SYSINTSTAT(base)	((base) + 0x48)
#define		SYSINTSTAT_SOFDETINT		(1<<4)
#define		SYSINTSTAT_DONEREGINT		(1<<3)
#define		SYSINTSTAT_SUSPDETINT		(1<<2)
#define		SYSINTSTAT_RSMFININT		(1<<1)
#define		SYSINTSTAT_RESETINT		(1<<0)

#define	FUNC_SYSINTEN(base)	((base) + 0x4c)
#define		SYSINTEN_SOFDETIEN		(1<<4)
#define		SYSINTEN_DONEREGIEN		(1<<3)
#define		SYSINTEN_SUSPDETIEN		(1<<2)
#define		SYSINTEN_RSMFINIEN		(1<<1)
#define		SYSINTEN_RESETIEN		(1<<0)

#define FUNC_XBUFINTSTAT(base)	((base) + 0x50)
#define FUNC_YBUFINTSTAT(base)  ((base) + 0x54)
#define FUNC_XYINTEN(base)	((base) + 0x58)
#define FUNC_XFILLSTAT(base)	((base) + 0x5c)
#define FUNC_YFILLSTAT(base)	((base) + 0x60)
#define	FUNC_ENDPNTEN(base)	((base) + 0x64)
#define FUNC_ENDPNRDY(base)		((base) + 0x68)
#define FUNC_IMEDINT(base)	((base) + 0x6c)
#define FUNC_EPNTDONESTAT(base)	((base) + 0x70)
#define FUNC_EPNTDONEEN(base)	((base) + 0x74)
#define FUNC_EPNTTOGBITS(base)	((base) + 0x78)
#define FUNC_FNEPRDYCLR(base)	((base) + 0x7c)

#define HOST_CTRL(base)		((base) + 0x80)
#define		CTRL_HCRESET		(1<<31)
#define		CTRL_SCHEDOVR_MASK	(3<<16)
#define		CTRL_SCHEDOVR_SHIFT	(16)
#define		CTRL_RMTWUEN		(1<<4)
#define		CTRL_HCUSBSTE_MASK	(3<<2)
#define		CTRL_HCUSBSTE_SHIFT	(2)
#define         CTRL_HCUSBSTE_RESET  (0<<2)
#define         CTRL_HCUSBSTE_RESUME (1<<2)
#define         CTRL_HCUSBSTE_OPER      (2<<2)
#define         CTRL_HCUSBSTE_SUSP      (3<<2)
#define		CTRL_CTLBLKSR_MASK	(3)
#define		CTRL_CTLBLKSR_SHIFT	(0)


#define HOST_SYSISR(base)	((base) + 0x88)
#define		SYSISR_PSCINT		(1<<6)
#define		SYSISR_FMOFINT		(1<<5)
#define		SYSISR_HERRINT		(1<<4)
#define		SYSISR_RESDETINT	(1<<3)
#define		SYSISR_SOFINT		(1<<2)	
#define		SYSISR_DONEINT		(1<<1)
#define		SYSISR_SORINT		(1<<0)

#define HOST_SYSIEN(base)	((base) + 0x8c)
#define		SYSIEN_PSCINT		(1<<6)
#define		SYSIEN_FMOFINT		(1<<5)
#define		SYSIEN_HERRINT		(1<<4)
#define		SYSIEN_RESDETINT	(1<<3)
#define		SYSIEN_SOFINT		(1<<2)	
#define		SYSIEN_DONEINT		(1<<1)
#define		SYSIEN_SORINT		(1<<0)

#define	HOST_XBUFSTAT(base)	((base) + 0x98)
#define HOST_YBUFSTAT(base)	((base) + 0x9c)
#define HOST_XYINTEN(base) 	((base) + 0xa0)
#define HOST_XFILLSTAT(base)	((base) + 0xa8)
#define HOST_YFILLSTAT(base)	((base) + 0xac)
#define HOST_ETDENSET(base)	((base) + 0xc0)
#define HOST_ETDENCLR(base) 	((base) + 0xc4)
#define HOST_IMEDINT(base)	((base) + 0xcc)
#define	HOST_ETDDONESTAT(base)	((base) + 0xd0)
#define HOST_ETDDONEN(base)	((base) + 0xd4)
#define HOST_FRMNUMB(base)	((base) + 0xe0)
#define	HOST_LSTHRESH(base)	((base) + 0xe4)
#define	HOST_ROOTHUBA(base)	((base) + 0xe8)
#define		ROOTHUBA_PWRTOGOOD_MASK		(0xff<<24)
#define		ROOTHUBA_PWRTOGOOD_SHIFT	(24)
#define		ROOTHUBA_NOOVRCURP		(1<<12)
#define		ROOTHUBA_OVRCURPM		(1<<11)
#define		ROOTHUBA_DEVTYPE		(1<<10)
#define		ROOTHUBA_PWRSWTMD		(1<<9)
#define		ROOTHUBA_NOPWRSWT		(1<<8)
#define		ROOTHUBA_NDNSTMPRT_MASK		(0xff)
#define		ROOTHUBA_NDNSTMPRT_SHIFT	(0)
#define HOST_ROOTHUBB(base)	((base) + 0xec)
#define		ROOTHUBB_PRTPWRCM_MASK		(0xff<<16)
#define		ROOTHUBB_PRTPWRCM_SHIFT		(16)
#define		ROOTHUBB_DEVREMOVE_MASK		(0xff)
#define		ROOTHUBB_DEVREMOVE_SHIFT	(0)

#define HOST_ROOTSTAT(base)	((base) + 0xf0)
#define		ROOTSTAT_CLRRMTWUE	(1<<31)
#define		ROOTSTAT_OVRCURCHG	(1<<17)
#define		ROOTSTAT_DEVCONWUE	(1<<15)
#define		ROOTSTAT_OVRCURI	(1<<1)
#define		ROOTSTAT_LOCPWRS	(1<<0)

#define HOST_PORTSTAT1(base)	((base) + 0xf4)
#define HOST_PORTSTAT2(base)	((base) + 0xf8)
#define HOST_PORTSTAT3(base)	((base) + 0xfc)
#define		PORTSTAT_PRTRSTSC	(1<<20)
#define		PORTSTAT_OVRCURIC	(1<<19)
#define		PORTSTAT_PRTSTATSC	(1<<18)
#define		PORTSTAT_PRTENBLSC	(1<<17)
#define		PORTSTAT_CONNECTSC	(1<<16)
#define		PORTSTAT_LSDEVCON	(1<<9)
#define		PORTSTAT_PRTPWRST	(1<<8)
#define		PORTSTAT_PRTRSTST	(1<<4)
#define		PORTSTAT_PRTOVRCURI	(1<<3)
#define		PORTSTAT_PRTSUSPST	(1<<2)
#define		PORTSTAT_PRTENABST	(1<<1)
#define		PORTSTAT_CURCONST	(1<<0)
/* Write port status */
#define		PORTSTAT_CLRPWRST	(1<<9)
#define		PORTSTAT_SETPWRST	(1<<8)
#define		PORTSTAT_SETPRTRST	(1<<4)
#define		PORTSTAT_CLRSUSP	(1<<3)
#define		PORTSTAT_SETSUSP	(1<<2)
#define		PORTSTAT_SETPRTENAB	(1<<1)
#define		PORTSTAT_CLRPRTENAB	(1<<0)

/* 
 * --------------------------------------------------------------------
 *   I2C Block: All registers are 8 Bit wide only 
 *   The manual seems to be totally wrong , better use
 *   Philips ISP 1301 documentation 
 * --------------------------------------------------------------------
 */
#define TRANS_VENODR_ID_LOW(base)		((base) + 0x100)
#define TRANS_VENDOR_ID_HIGH(base)		((base) + 0x101)
#define TRANS_PRODUCT_ID_LOW(base)		((base) + 0x102)
#define	TRANS_PRODUCT_ID_HIGH(base)		((base) + 0x103)
#define	TRANS_MODE_CONTROL_SET(base)		((base) + 0x104)
#define TRANS_MODE_CONTROL_CLR(base)		((base) + 0x105)
#define	TRANS_OTG_CONTROL_SET(base)		((base) + 0x106)
#define	TRANS_OTG_CONTROL_CLR(base)		((base) + 0x107)
#define TRANS_INTERRUPT_SRC(base)		((base) + 0x108)
#define	TRANS_INTLAT_SET(base)			((base) + 0x10a)
#define	TRANS_INTLAT_CLR(base)			((base) + 0x10b)
#define	TRANS_INTMSK_FALSE_SET(base)		((base) + 0x10c)
#define TRANS_INTMSK_FALSE_CLR(base)		((base) + 0x10d)
#define TRANS_INTMSK_TRUE_SET(base)		((base) + 0x10e)
#define TRANS_INTMSK_TRUE_CLR(base)		((base) + 0x10f)
#define	TRANS_INTLATCH_SET(base)		((base) + 0x112)
#define	TRANS_INTLATCH_CLR(base)		((base) + 0x113)


#define I2C_OTGDEVADDR(base)			((base) + 0x118)
#define I2C_NUMSEQOPS(base)			((base) + 0x119)
#define	I2C_SEQREADSTRT(base)			((base) + 0x11a)
#define I2C_OPCTRL(base)			((base) + 0x11b)
#define		OPCTRL_I2CBUSY		(1<<7)
#define		OPCTRL_HWSWMODE		(1<<1)
#define		OPCTRL_I2COE		(1<<0)
#define I2C_DIVFACTOR(base)			((base) + 0x11e)
#define	I2C_INTANDCTRL(base)			((base) + 0x11f)

#define ETD_BASE(base)	((base) + 0x200)
#define ETD_ADDR(base,n)	((base) + 0x200 + (n) * 16)

#define EP_BASE(base)	((base) + 0x400)

/* The AHB DMA block */
#define DMA_REV(base)		((base) + 0x800)
#define DMA_INTSTAT(base)	((base) + 0x804)
#define		DMAINTSTAT_EPERR	(1<<1)
#define		DMAINTSTAT_ETDERR	(1<<0)
#define DMA_INTEN(base)		((base) + 0x808)
#define		DMAINTEN_EPERRINTEN	(1<<1)
#define		DMAINTEN_ETDERRINTEN	(1<<0)
#define	DMA_ETDDMAERSTAT(base) 	((base) + 0x80c)
#define DMA_EPDMAERSTAT(base)	((base) + 0x810)
#define DMA_ETDDMAEN(base)	((base) + 0x820)
#define DMA_EPDMAEN(base)	((base) + 0x824)
#define DMA_ETDDMAXTEN(base)	((base) + 0x828)
#define DMA_EPDMAXTEN(base)	((base) + 0x82c)
#define DMA_ETDDMAENXYT(base)	((base) + 0x830)
#define	DMA_EPDMAENXYT(base)	((base) + 0x834)
#define DMA_ETCDMABST4EN(base)  ((base) + 0x838)
#define DMA_EPDMABST4EN(base)	((base) + 0x83c)
#define	DMA_MISCCONTROL(base)	((base) + 0x840)
#define		MISCCONTROL_ISOPREVFRM		(1<<3)
#define		MISCCONTROL_SKPRTRY		(1<<2)
#define		MISCCONTROL_ARBMODE		(1<<1)
#define		MISCCONTROL_FILTCC		(1<<0)
#define DMA_ETDDMACHANCLR(base)	((base) + 0x848)
#define DMA_EPDMACHANCLR(base)	((base) + 0x84c)
#define DMA_ETDSMSA(base,n)		((base) + 0x900 + ((n)<<2))
#define DMA_EPSMSA(base,n)		((base) + 0x980 + ((n)<<2))
#define DMA_ETDDMABUFPTR(base,n) 	((base) + 0xa00 + ((n)<<2))
#define	DMA_EPDMABUFPTR(base,n)		((base) + 0xa80 + ((n)<<2))

#define DATA_MEM(base)		((base) + 0x1000)

#define HWC_ADDRESS_MASK		(0x7f)
#define HWC_ADDRESS_SHIFT		(0)
#define HWC_ENDPOINT_NUMBER_MASK	(0xf<<4)
#define HWC_ENDPOINT_NUMBER_SHIFT	(4)
#define HWC_DIRECT_MASK			(0x3<<11)	
#define HWC_DIRECT_SHIFT		(11)	
#define HWC_SPEED			(1<<13)
#define	HWC_FORMAT_MASK			(3<<14)
#define HWC_FORMAT_SHIFT		(14)
#define	HWC_MAXPKTSIZ_MASK		(0x3ff<16)
#define HWC_MAXPKTSIZ_SHIFT		(16)
#define	HWC_HALTED			(1<<27)
#define HWC_TOGCRY			(1<<28)
#define HWC_SNDNAK			(1<<30)

/* 
 * -----------------------------------------------------------------
 * Completion codes from reference manual table 36-2
 * -----------------------------------------------------------------
 */

#define CC_NOERROR     (0)
#define CC_CRC          (1)
#define CC_BITSTUFF     (2)
#define CC_TGLMISMATCH  (3)
#define CC_STALL        (4)
#define CC_DEVNOTRESP   (5)
#define CC_PIDFAIL      (6)
#define CC_RESERVED1    (7)
#define CC_DATAOVER     (8)
#define CC_DATAUNDER    (9)
#define CC_ACK          (10)
#define CC_NACK         (11)
#define CC_BUFOVER      (12)
#define CC_BUFUNDER     (13)
#define CC_SCHEDOVER    (14)
#define CC_NOTACC       (15) /* This is set by software before submission */

/* Control/Bulk Transfer descriptor */

typedef struct CBTd {
	uint32_t hwControl;
	uint32_t bufsrtad; 
	uint32_t bufstat;	
	uint32_t size;
} CBTd;

typedef struct IntTd {
	uint32_t hwControl;
	uint32_t bufsrtad;
	uint32_t bufstat;
	uint32_t size;
} IntTd;

typedef struct IsoTd {
	uint32_t hwControl;
	uint32_t bufsrtad;
	uint32_t bufstat;
	uint32_t pktstat0_1;
} IsoTd;

/* The following fields are the same for Control/Bulk ISO and Interrupt Transfer descriptors */
#define HETD_SNDNAK 		(1<<30)
#define HETD_TOGCRY		(1<<28)
#define	HETD_HALTED		(1<<27)
#define HETD_MAXPKTSIZ_MASK	(0x3ff << 16)
#define HETD_MAXPKTSIZ_SHIFT	(16)
#define HETD_FORMAT_MASK	(3<<14)
#define	HETD_FORMAT_SHIFT	(14)
#define         HETD_FORMAT_CONTROL     (0<<14)
#define         HETD_FORMAT_ISO         (1<<14)
#define         HETD_FORMAT_BULK        (2<<14)
#define         HETD_FORMAT_INTERRUPT   (3<<14)
#define HETD_SPEED		(1<<13)
#define         HETD_SPEED_LOW          (1<<13)
#define         HETD_SPEED_FULL         (0<<13)
#define HETD_DIRECT_MASK	(3<<11)
#define HETD_DIRECT_SHIFT	(11)
#define         HETD_DIRECT_FROM_TD     (0<<11)
#define         HETD_DIRECT_OUT         (1<<11)
#define         HETD_DIRECT_IN          (2<<11)
#define         HETD_DIRECT_FROM_TD_2   (3<<11)
#define HETD_ENDPNT_MASK	(0xf<<8)
#define HETD_ENDPNT_SHIFT	(8)
#define HETD_ADDRESS_MASK	(0x7f << 0)
#define HETD_ADDRESS_SHIFT	(0)

#define CBTD_YBUFSRTAD_MASK	(0xffff<<16)
#define CBTD_YBUFSRTAD_SHIFT	(16)
#define CBTD_XBUFSRTAD_MASK	(0xffff)
#define CBTD_XBUFSRTAB_SHIFT	(0)

#define CBTD_COMPCODE_MASK	(0xf << 28)
#define CBTD_COMPCODE_SHIFT	(28)
#define CBTD_ERRORCNT_MASK	(0xf << 24)
#define CBTD_ERRORCNT_SHIFT	(24)
#define CBTD_DATAOG_MASK	(3<<22)
#define CBTD_DATAOG_SHIFT	(22)
#define CBTD_DELAYINT_MASK	(7<<19)
#define CBTD_DELAYINT_SHIFT	(19)
#define CBTD_BUFROUND		(1<<18)
#define CBTD_DIRPID_MASK	(3<<16)
#define CBTD_DIRPID_SHIFT	(16)
#define CBTD_RTRYDELAY_MASK	(0xff)
#define	CBTD_RTRYDELAY_SHIFT	(0)

#define CBTD_BUFSIZE_MASK	(0x7ff<<21)
#define	CBTD_BUFSIZE_SHIFT	(21)
#define CBTD_TOTBYECNT_MASK	(0x1fffff)
#define	CBTD_TOTBYECNT_SHIFT	(0)

/* Interrupt transfer descriptor */
#define ITD_YBUFSRTAD_MASK	(0xffff<<16)
#define ITD_YBUFSRTAD_SHIFT	(16)
#define ITD_XBUFSRTAD_MASK	(0xffff)
#define ITD_XBUFSRTAB_SHIFT	(0)

#define ITD_COMPCODE_MASK	(0xf << 28)
#define ITD_COMPCODE_SHIFT	(28)
#define ITD_ERRORCNT_MASK	(0xf << 24)
#define ITD_ERRORCNT_SHIFT	(24)
#define ITD_DATAOG_MASK		(3<<22)
#define ITD_DATAOG_SHIFT	(22)
#define ITD_DELAYINT_MASK	(7<<19)
#define ITD_DELAYINT_SHIFT	(19)
#define ITD_BUFROUND		(1<<18)
#define ITD_DIRPID_MASK		(3<<16)
#define ITD_DIRPID_SHIFT	(16)
#define ITD_RELPOLPOS_MASK	(0xff<<8)
#define	ITD_RELPOLPOS_SHIFT	(8)
#define ITD_POLINTERV_MASK	(0xff)
#define ITD_POLINTERV_SHIFT	(0)

#define ITD_BUFSIZE_MASK	(0x7ff<<21)
#define	ITD_BUFSIZE_SHIFT	(21)
#define ITD_TOTBYECNT_MASK	(0x1fffff)
#define	ITD_TOTBYECNT_SHIFT	(0)

/* DWORD 1 */
#define ISOTD_YBUFSRTAD_MASK	(0xffff<<16)
#define ISOTD_YBUFSRTAD_SHIFT	(16)
#define ISOTD_XBUFSRTAD_MASK	(0xffff)
#define ISOTD_XBUFSRTAB_SHIFT	(0)

/* DWORD 2 */
#define ISOTD_COMPCODE_MASK	(0xf << 28)
#define ISOTD_COMPCODE_SHIFT	(28)
#define ISOTD_FRAMECNT		(1<<24)
#define ISOTD_DELAYINT_MASK	(7<<19)
#define ISOTD_DELAYINT_SHIFT	(19)
#define ISOTD_STARTFRM_MASK	(0xffff)
#define	ISOTD_STARTFRM_SHIFT	(0)

#define ISOTD_COMPCODE1_MASK	(0xf << 28)
#define ISOTD_COMPCODE1_SHIFT	(28)
#define	ISOTD_PKTLEN1_MASK	(0x3ff<<16)
#define	ISOTD_PKTLEN1_SHIFT	(16)
#define ISOTD_COMPCODE0_MASK	(0xf << 12)
#define ISOTD_COMPCODE0_SHIFT	(28)
#define	ISOTD_PKTLEN0_MASK	(0x3ff<<0)
#define	ISOTD_PKTLEN0_SHIFT	(0)



typedef struct IMXOtg {
	BusDevice bdev;
	ISP1301 *isp1301;
	CycleTimer frametimer;
	/* level sensitive interrupts */
	SigNode *intUsbWkup;
	SigNode *intUsbMnp;
	SigNode *intUsbDma;
	SigNode *intUsbFunc;
	SigNode *intUsbHost;
	SigNode *intUsbCtrl;

	uint32_t otg_hwmode;
	uint32_t otg_cint_stat;
	uint32_t otg_cint_sten;
	uint32_t otg_clk_ctrl;
	uint32_t otg_rst_ctrl;
	uint32_t otg_frm_invtl;
	uint32_t otg_frm_remain;
	uint32_t otg_hnp_ctrl;
	uint32_t otg_hnp_int_stat;
	uint32_t otg_hnp_int_en;
	uint32_t otg_usbctrl;

	uint32_t func_comstat;
	uint32_t func_devaddr;
	uint32_t func_sysintstat;
	uint32_t func_sysinten;
	uint32_t func_xbufintstat;
	uint32_t func_ybufintstat;
	uint32_t func_xyinten;
	uint32_t func_xfillstat;
	uint32_t func_yfillstat;
	uint32_t func_endpnten;
	uint32_t func_endpnrdy;
	uint32_t func_imedint;
	uint32_t func_epntdonestat;
	uint32_t func_epntdoneen;
	uint32_t func_epnttogbits; 
	uint32_t func_fneprdyclr;

	CycleTimer host_frame_timer;
	uint32_t host_ctrl;
	uint32_t host_sysisr;
	uint32_t host_sysien;
	uint32_t host_xbufstat;
	uint32_t host_ybufstat;
	uint32_t host_xyinten;
	uint32_t host_xfillstat;
	uint32_t host_yfillstat;
	uint32_t host_etden;
	uint32_t host_imedint;
	uint32_t host_etddonestat;
	uint32_t host_etddonen;
	uint32_t host_frmnumb;
	uint32_t host_lsthresh;
	uint32_t host_roothuba;
	uint32_t host_roothubb;
	uint32_t host_rootstat;
	uint32_t host_portstat[3];

	uint32_t dma_rev;
	uint32_t dma_intstat;
	uint32_t dma_inten;
	uint32_t dma_etddmaerstat;
	uint32_t dma_epdmaerstat;
	uint32_t dma_etddmaen;
	uint32_t dma_epdmaen;
	uint32_t dma_etddmaxten;
	uint32_t dma_epdmaxten;
	uint32_t dma_etddmaenxyt;
	uint32_t dma_epdmaenxyt;
	uint32_t dma_etcdmabst4en;
	uint32_t dma_epdmabst4en;
	uint32_t dma_misccontrol;
	uint32_t dma_etddmachanclr;
	uint32_t dma_epdmachanclr;
	uint32_t dma_etdsmsa;
	uint32_t dma_epsmsa;
	uint32_t dma_etddmabufptr;
	uint32_t dma_epdmabufptr;

	uint8_t i2c_otgdevaddr;
	uint8_t i2c_numseqops;
	uint8_t i2c_seqreadstrt;
	uint8_t i2c_opctrl;
	uint8_t i2c_divfactor;
	uint8_t i2c_intandctrl;
	uint8_t etd_mem[512];
	uint8_t ep_mem[512];
	uint8_t data_mem[4096];
} IMXOtg;

static uint8_t etdmem_initval[] = {
0x7a,0xb6,0x35,0x6d,0x80,0xe6,0x85,0x28,0x3f,0x4a,0x5f,0x3a,0x57,0x72,0xe8,0xd6,
0x77,0x4e,0x17,0xaa,0x7a,0x86,0x56,0x27,0x3a,0x63,0xca,0xb7,0x84,0x0a,0xbb,0xde,
0x86,0xa5,0x86,0x71,0xb5,0xf8,0x2c,0x78,0x92,0x4f,0xc3,0x72,0x39,0x93,0xb4,0x01,
0xac,0x1a,0x72,0xc8,0x7e,0x9c,0x99,0x1b,0x09,0xbd,0x5a,0xd9,0xee,0x63,0xb7,0xd5,
0x37,0x16,0xe1,0x2d,0x98,0xdb,0x15,0x7a,0x88,0xa8,0x5a,0x8f,0xc0,0xb4,0x1f,0xe3,
0x02,0xfa,0x5f,0x42,0x6e,0xd7,0x2a,0x69,0xb7,0x6c,0xa4,0x2e,0x60,0xa7,0xb7,0x47,
0xdc,0xc9,0x6c,0x17,0x03,0x84,0xfe,0x52,0x77,0xeb,0xcb,0x1a,0xe4,0xdb,0x04,0xac,
0x10,0x81,0x06,0xaa,0xc5,0x55,0x77,0x4d,0xcf,0x5d,0x27,0x2a,0x7c,0x62,0x96,0xe3,
0x37,0xfd,0x85,0x12,0xc7,0x32,0xed,0xd7,0x71,0x9c,0x5c,0xaa,0x66,0x36,0x79,0x6b,
0x0b,0xc0,0x9c,0xe5,0x0e,0xb6,0x1d,0x84,0x66,0x43,0x11,0x30,0x9d,0x3b,0x2a,0xd7,
0xe5,0xb9,0x81,0xe5,0x1a,0x7e,0x97,0x08,0xd4,0x93,0x43,0x27,0xcb,0xbe,0xa4,0xc5,
0x3f,0x6f,0x7e,0xec,0x88,0xcc,0xdf,0x12,0xa5,0xd0,0x70,0x13,0x87,0x36,0x99,0x44,
0x85,0x13,0x0c,0x4e,0xa0,0x32,0xf9,0xbf,0xa3,0x51,0x3c,0xf7,0x12,0x00,0x15,0x9a,
0x1c,0xa6,0x6c,0x02,0x5a,0xf8,0xf3,0xc6,0x11,0x27,0x83,0xa8,0xd9,0xee,0xe1,0x0d,
0xd3,0xe3,0xbf,0x1d,0x20,0x84,0xad,0xf8,0xf4,0x3d,0x7e,0xe5,0xde,0x6b,0x65,0xbf,
0x5a,0x8f,0x82,0x0f,0x3a,0x66,0x03,0x15,0x8f,0x42,0x60,0x54,0x4f,0x64,0xd4,0xa1,
0x3d,0x09,0xb6,0xbe,0xc5,0x32,0x88,0x53,0x48,0x4a,0x72,0x56,0x2b,0x11,0xd0,0x87,
0x3c,0x48,0xb1,0x86,0xec,0xd6,0xf6,0xd0,0x75,0xba,0x14,0xf4,0xc6,0xb4,0xbc,0x65,
0x2e,0xcf,0x24,0x6b,0x41,0xcc,0x7f,0xa9,0x5b,0xb0,0x99,0xb6,0x46,0x80,0xfe,0x3f,
0xd4,0xf8,0x4c,0x4e,0x35,0xfe,0x79,0x60,0xa1,0xd3,0xaf,0x45,0x89,0x21,0x71,0xc5,
0xdc,0xac,0x76,0x4c,0x23,0xc6,0x8b,0x09,0x57,0x81,0xba,0x55,0xf1,0x5d,0x48,0x8a,
0x5e,0x00,0x1a,0xf8,0x25,0x3d,0x97,0xd2,0x75,0x87,0x1c,0x0e,0xdf,0xf4,0xb2,0xce,
0x23,0xdc,0xa5,0x87,0x1e,0x93,0x9e,0xab,0x97,0x1a,0xfc,0x17,0x63,0x8c,0xe2,0x28,
0x44,0x49,0xe0,0x6b,0x2a,0x7f,0x6a,0xc6,0xb5,0x71,0xe4,0x75,0x77,0x57,0x1f,0x00,
0xa9,0xe9,0x27,0x8d,0xf6,0xb6,0x59,0x33,0x55,0x17,0x59,0xec,0x95,0xac,0x9f,0xc8,
0xfd,0x26,0x26,0x52,0xb6,0x6a,0x20,0x12,0x5b,0x0e,0x0a,0x1d,0xd6,0x9d,0x85,0xa5,
0x3f,0x4a,0xfb,0x7e,0x40,0x9c,0x43,0xee,0xdb,0xd6,0xea,0x0d,0xa4,0xc6,0x22,0x20,
0x3d,0x25,0x7e,0x32,0x69,0x5c,0x4c,0xca,0x50,0x46,0xe2,0x0c,0xd2,0xdc,0xa1,0x8e,
0x53,0x21,0xa1,0x42,0xeb,0x32,0xc7,0x23,0x1c,0x69,0x50,0x01,0x4e,0xd9,0x38,0x3e,
0x0d,0x9e,0xbf,0xd5,0x87,0xe1,0x2e,0x08,0x6f,0x21,0xcc,0xf4,0x77,0xc9,0xea,0xcf,
0x2d,0x1b,0xd0,0xc8,0xe6,0xee,0xbd,0x34,0x96,0x26,0x0f,0x17,0xa6,0xe9,0x9e,0x64,
0x21,0x8f,0xc9,0x98,0x89,0xaf,0x1b,0x44,0xa6,0x70,0x96,0xb5,0x67,0x73,0x44,0xb3
};

static uint8_t epmem_initval[] = {
0xc2,0xb7,0x32,0x3d,0x7a,0x11,0x0c,0x84,0xde,0x4c,0xbd,0x0c,0x57,0xb1,0xbb,0x36,
0x40,0x91,0x40,0x20,0xa8,0x6b,0xfc,0x95,0x5c,0x54,0x1b,0x7c,0x9f,0xcf,0x4b,0xf1,
0xe9,0x1c,0xb3,0xcb,0x0d,0x59,0x5a,0x30,0xb5,0xa6,0xca,0xff,0x85,0x52,0x96,0xf9,
0xbd,0xc4,0xf8,0x20,0x72,0xc9,0x63,0x9d,0x03,0xa7,0x34,0x33,0xed,0x3b,0x5b,0xee,
0xdd,0x2e,0x69,0xf1,0x14,0xe6,0xfc,0x61,0xaa,0xf9,0xdc,0x92,0x40,0x85,0x55,0xd4,
0xb2,0x0f,0xd9,0x09,0xc5,0xae,0x98,0xd0,0x37,0x9b,0xed,0x72,0xfa,0x42,0x9a,0x7d,
0x7e,0x2a,0x4e,0x52,0x2c,0xc9,0x89,0x6f,0x99,0xf3,0x32,0x3d,0xef,0x13,0x7c,0x7e,
0x9a,0x9d,0x86,0xd2,0x7c,0xcc,0xec,0x35,0x31,0x38,0x16,0x8b,0x3d,0xfa,0x72,0xab,
0x03,0x9d,0x43,0x00,0x40,0x82,0x24,0xd5,0x95,0x65,0x54,0x29,0x50,0xec,0x97,0xee,
0x02,0xe1,0xc4,0x3d,0x20,0x7e,0x1a,0x39,0xa7,0xc5,0xbf,0xf8,0x5e,0x25,0x6d,0xf2,
0x9f,0x4f,0xd6,0xd8,0x9e,0xea,0x0d,0x77,0x1d,0xe7,0x6e,0xa1,0x04,0x94,0x20,0x37,
0x96,0xb2,0x0e,0xe6,0x73,0xf5,0x2e,0xc4,0xab,0x8a,0x3a,0x5f,0xa2,0x16,0xe8,0xbd,
0x3e,0xa3,0xd2,0x2f,0x94,0x0a,0x19,0x32,0xf7,0x2b,0x3a,0x72,0x24,0x51,0x7d,0x38,
0x9a,0x60,0x6a,0x4f,0x2c,0x95,0x04,0xc1,0xc6,0x6c,0x8b,0xf0,0xf1,0x77,0xd6,0x31,
0x1c,0xfb,0x6c,0x50,0x4a,0x9c,0xa2,0x52,0x38,0xc8,0xfe,0x9e,0x39,0xdb,0xf8,0x0d,
0x3c,0x16,0x7c,0xe0,0x2f,0x30,0x39,0x97,0x16,0x00,0xfe,0x6b,0x3d,0xf2,0xfd,0xfc,
0xae,0x34,0x54,0xdb,0xfb,0x6e,0x41,0x28,0xb1,0xab,0xe4,0xdf,0x45,0x4d,0xbb,0xae,
0xa4,0xb1,0x1e,0xe6,0xf8,0xd3,0xc4,0x64,0xe8,0xf0,0xf0,0xd9,0x0a,0x84,0x70,0xaa,
0xe2,0xd5,0x36,0x3d,0xdb,0xb4,0x27,0x52,0xce,0x43,0x1c,0xf8,0xf5,0xab,0xc2,0xc6,
0x51,0xc5,0xb4,0x20,0x9a,0x7a,0x4a,0xb5,0xed,0x95,0xd4,0x5f,0xf0,0xf6,0xfb,0x4d,
0x2d,0x59,0xa3,0xa8,0xb6,0x93,0xd3,0x1f,0x47,0x75,0xde,0x8c,0x1c,0x20,0x8d,0x13,
0x94,0xd7,0xe1,0x3c,0x40,0x05,0x77,0x9a,0x22,0x70,0x9e,0x44,0x7f,0x2f,0x5a,0x7e,
0xea,0x5a,0x04,0x4f,0xfd,0x1f,0x13,0x42,0xa0,0x9d,0xa4,0x78,0x78,0x90,0x9c,0x0a,
0x38,0xb5,0x86,0xf3,0x4d,0x21,0x6d,0x6f,0x12,0x0b,0x93,0x4e,0x25,0xa9,0xac,0xb9,
0x25,0x6e,0x67,0x2b,0x59,0x4b,0x81,0xb5,0x7a,0x36,0x5e,0x6d,0xd6,0xa2,0x95,0xa9,
0xf5,0x32,0x84,0x0b,0xd5,0x79,0xd9,0x6e,0x66,0x4c,0x85,0xa4,0xf3,0xbc,0x7b,0x1d,
0xe0,0x2b,0x6f,0x4c,0x4e,0xa9,0xdc,0x0c,0xea,0xfe,0x85,0xa7,0x89,0x04,0x9a,0x77,
0x8a,0xd8,0x66,0x0c,0xd1,0x35,0x59,0x1c,0x6b,0x5b,0xad,0x23,0x6c,0xb6,0xa1,0x0f,
0x6b,0x42,0xac,0xee,0x70,0x8c,0x95,0x56,0xad,0x22,0xf3,0x9b,0x86,0xf4,0x24,0x59,
0xcb,0x56,0x80,0xef,0xd8,0x3e,0xdf,0x77,0xf4,0x64,0xc2,0x29,0x36,0x0f,0x74,0xe0,
0x31,0x7b,0x78,0xd9,0x73,0x61,0xe9,0x7a,0x69,0xe1,0x77,0x5e,0x57,0xe5,0x33,0x57,
0xf6,0x10,0x34,0x6e,0x3d,0xec,0xc2,0x00,0x56,0xe5,0xac,0x69,0x94,0x3f,0x2c,0xa6,
};


static void
update_host_interrupt(IMXOtg *otg) 
{
	if(otg->host_etddonestat & otg->host_etddonen) {
		otg->host_sysisr |= SYSISR_DONEINT;
	}
	if((otg->otg_cint_sten & (CINT_STAT_HCINT | CINT_STAT_ASHCINT))
		&& (otg->host_sysisr & otg->host_sysien)) 
	{
		SigNode_Set(otg->intUsbHost,SIG_LOW);
		if(!(otg->otg_clk_ctrl & CLK_CTRL_HSTCLK)) {
			otg->otg_cint_stat |= CINT_STAT_ASHCINT;
		} else {
			otg->otg_cint_stat &=~CINT_STAT_ASHCINT;
			otg->otg_cint_stat |= CINT_STAT_HCINT;
		}
	} else {
		SigNode_Set(otg->intUsbHost,SIG_HIGH);
	}
}


static void
update_func_interrupt(IMXOtg *otg) 
{
	if((otg->otg_cint_sten & (CINT_STAT_FCINT | CINT_STAT_ASFCINT))
	 && (otg->func_sysintstat & otg->func_sysinten)) 
	{
		SigNode_Set(otg->intUsbFunc,SIG_LOW);
		if(!(otg->otg_clk_ctrl & CLK_CTRL_FUNCCLK)) {
			otg->otg_cint_stat |= CINT_STAT_ASFCINT;
		} else {
			otg->otg_cint_stat &=~CINT_STAT_ASFCINT;
			otg->otg_cint_stat |= CINT_STAT_FCINT;
		}
	} else {
		SigNode_Set(otg->intUsbFunc,SIG_HIGH);
	}
}

static void
update_hnp_interrupt(IMXOtg *otg) 
{
	if((otg->otg_cint_sten & (CINT_STAT_HNPINT | CINT_STAT_ASHNPINT))
	 && (otg->otg_hnp_int_stat & otg->otg_hnp_int_en)) {
		SigNode_Set(otg->intUsbMnp,SIG_LOW);
		if(!(otg->otg_clk_ctrl & CLK_CTRL_MAINCLK)) {
			otg->otg_cint_stat |= CINT_STAT_ASHNPINT;
		} else {
			otg->otg_cint_stat &=~CINT_STAT_ASHNPINT;
			otg->otg_cint_stat |= CINT_STAT_HNPINT;
		}
	} else {
		SigNode_Set(otg->intUsbMnp,SIG_HIGH);
	}
}

static void
update_dma_interrupt(IMXOtg *otg) 
{
	if(otg->dma_intstat & otg->dma_inten) {
		SigNode_Set(otg->intUsbDma,SIG_LOW);
	} else {
		SigNode_Set(otg->intUsbDma,SIG_HIGH);
	}
}

static void
reset_i2c(IMXOtg *otg) 
{
	
	otg->i2c_otgdevaddr = 0xac;
	otg->i2c_numseqops = 0;
	otg->i2c_seqreadstrt = 8;
	otg->i2c_opctrl = 0;
	otg->i2c_divfactor = 0x78; /* 48 MHz / 0x78 = 400 kHz */
	otg->i2c_intandctrl = 0;
}

/*
 * --------------------------------------------------------------------------
 * reset_control
 *	reset the control block. The reset values are verified with
 *	real device
 * --------------------------------------------------------------------------
 */
static void
reset_control(IMXOtg *otg) 
{
	otg->otg_hwmode = 0x202000a3;
	otg->otg_cint_stat = 0; 
	otg->otg_cint_sten = 0;	
	otg->otg_clk_ctrl = 0;
	otg->otg_frm_invtl = 0x2a2f2edf;
	otg->otg_frm_remain = 0x21c40000; /* Manual seems to be wrong here */
	otg->otg_hnp_ctrl = 0x20040200;
	otg->otg_hnp_int_stat = 0;
	otg->otg_hnp_int_en = 0;
	otg->otg_usbctrl = 0x000f1000; /* Manual seems to be wrong */
	update_hnp_interrupt(otg);
	// update_clocks
}

/*
 * -------------------------------------------------------------------------
 * USB Function reset
 *	All reset values are 0 except first register 
 * 	(verified with real device 0x40-0x7f)
 * -------------------------------------------------------------------------
 */
static void
reset_func(IMXOtg *otg) 
{
	otg->func_comstat = 1;
	otg->func_devaddr = 0; 	
	otg->func_sysintstat = 0;
 	otg->func_sysinten = 0;
	otg->func_xbufintstat = 0;
	otg->func_ybufintstat = 0;
	otg->func_xyinten = 0;
	otg->func_xfillstat = 0;
	otg->func_yfillstat = 0;
	otg->func_endpnten = 0;
	otg->func_endpnrdy = 0;
	otg->func_imedint = 0;
	otg->func_epntdonestat = 0;
	otg->func_epntdoneen = 0;
	otg->func_epnttogbits = 0;
	otg->func_fneprdyclr = 0;
	update_func_interrupt(otg);
}

static void
reset_funcser(IMXOtg *otg) 
{
	fprintf(stderr,"IMX21 OTG: Function serial reset not implemented\n");
}

static void
reset_roothub(IMXOtg *otg) 
{
	fprintf(stderr,"IMX21 OTG: roothub reset not implemented\n");
}

static void
reset_hostser(IMXOtg *otg) 
{
	fprintf(stderr,"IMX21 OTG: hostser reset not implemented\n");
}

static void
reset_hc(IMXOtg *otg) 
{
	otg->host_ctrl = 0;
	otg->host_sysisr = 0;
	otg->host_sysien = 0;
	otg->host_xbufstat = 0;
	otg->host_ybufstat = 0;
	otg->host_xyinten = 0;
	otg->host_xfillstat = 0;
	otg->host_yfillstat = 0;
	otg->host_etden = 0;
	otg->host_imedint = 0;
	otg->host_etddonestat = 0;
	otg->host_etddonen = 0;
	otg->host_frmnumb = 0;
	otg->host_lsthresh = 0x628;
	otg->host_roothuba = 0x01000103;	
	otg->host_roothubb = 0x00070000;	
	otg->host_rootstat = 0x00070000;
	otg->host_portstat[0] = otg->host_portstat[1] = otg->host_portstat[2] = 0;
	update_host_interrupt(otg);
}

static void
reset_dma(IMXOtg *otg) 
{
	otg->dma_rev = 0x10;
	otg->dma_intstat = 0;
	otg->dma_inten = 0;
	otg->dma_etddmaerstat = 0;
	otg->dma_epdmaerstat = 0;
	otg->dma_etddmaen = 0;	
	otg->dma_epdmaen = 0;
	otg->dma_etddmaxten = 0;
	otg->dma_epdmaxten = 0;
	otg->dma_etddmaenxyt = 0;
	otg->dma_epdmaenxyt = 0;
	otg->dma_etcdmabst4en = 0xffffffff; /* ???? contradictions ! doku writer was drunken */
	otg->dma_epdmabst4en = 0xffffffff;  /* ???? contradictions ! doku writer was drunken */
	otg->dma_misccontrol = 0;
	otg->dma_etddmachanclr = 0;
	otg->dma_epdmachanclr = 0; /* doku: cut copy paste bug from etddmachanclr */
	otg->dma_etdsmsa = 0;
	otg->dma_epsmsa = 0;
	otg->dma_etddmabufptr = 0;
	otg->dma_epdmabufptr = 0;
	otg->dma_epdmabufptr = 0;
	update_dma_interrupt(otg);
}

/*
 * ----------------------------------------------------------------------------------------------
 * HWMODE
 *
 *  Bit 24-31 FUNCREV Function revision. fixed to 0x20
 *  Bit 16-23 HSTREF  Host revision. fixed to 0x20
 *  Bit 14    ANASDBEN Analog signal short debounce enable
 *  Bit 6     OTGXVCR  OTG transceiver properties
 *		00: differential TX/RX
 *		01: single ended TX, differential RX
 *		10: differential TX, single ended RX
 *		11: single ended TX/RX
 *
 *  Bit 4-5   HOSTXVCR 	Host transceiver properties, values like OTGXVCR
 *  Bit 0-1   CRECFG	USB OTG module configuration	
 *		00: reserved
 *		01: host only operation
 *		10: function operation (no HNP)
 *		11: Software HNP
 *
 * ----------------------------------------------------------------------------------------------
 */

static uint32_t
otg_hwmode_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->otg_hwmode;
}


static void
otg_hwmode_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG HWMODE: no effect implemented\n");
	otg->otg_hwmode = (otg->otg_hwmode & ~0xf3) | (value & 0xf3);
        return;
}

/*
 * ------------------------------------------------------------------------------
 * OTG_CINT_STAT USB Interrupt status register
 *
 *   Bit 5: ASHNPINT	Asynchronous HNP Interrupt
 *   Bit 4: ASFCINT	Asynchronous Function Interrupt
 *   Bit 3: ASHCINT	Asynchronous Host Interrupt
 *   Bit 2: HNPINT	HNP Interrupt
 *   Bit 1: FCINT	Function Interrupt
 *   Bit 0: HCINT	Host Interrupt
 * ------------------------------------------------------------------------------
 */

static uint32_t
otg_cint_stat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->otg_cint_stat;
}


static void
otg_cint_stat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	//fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------
 * OTG_CINT_STEN USB Interrupt enable registers
 *
 *  Bit 5: ASHNPINTEN	Asynchronous HNP interrupt enable
 *  Bit 4: ASFCINTEN	Asynchronous Function interrupt enable
 *  Bit 3: ASHCINTEN 	Asynchronous Host interrupt enable
 *  Bit 2: HNPINTEN 	HNP interrupt enable
 *  Bit 1: FCINTEN	Function interrupt enable
 *  Bit 0: HCINTEN	Host interrupt enable
 * ------------------------------------------------------------------
 */

static uint32_t
otg_cint_sten_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->otg_cint_sten;
}


static void
otg_cint_sten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->otg_cint_sten = value;
	update_host_interrupt(otg);
	update_func_interrupt(otg);
	update_hnp_interrupt(otg);
        return;
}

/*
 * ----------------------------------------------------------------------------------------
 *
 * OTG_CLK_CTRL
 *
 *  Bit 2: FUNCCLK 	Function clock
 *  Bit 1: HSTCLK	Host clock	
 *  Bit 0: MAINCLK	Main clock. Can not be cleared when func or host clock is enabled
 *
 * -----------------------------------------------------------------------------------------
 */

static uint32_t
otg_clk_ctrl_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->otg_clk_ctrl;
}


static void
otg_clk_ctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	if(value & CLK_CTRL_FUNCCLK) {
		otg->otg_cint_stat &= ~CINT_STAT_ASFCINT;	
	} 
	if(value & CLK_CTRL_HSTCLK) {
		otg->otg_cint_stat &= ~CINT_STAT_ASHCINT;	
	}
	if(value & CLK_CTRL_MAINCLK) {
		otg->otg_cint_stat &= ~CINT_STAT_ASHNPINT;	
	}
	otg->otg_clk_ctrl = value; 
	fprintf(stderr,"IMX21 OTG: CLK control not fully implemented\n");
        return;
}

/*
 * --------------------------------------------------------------------------------
 * OTG_RST_CTRL Reset control register
 *
 *  Bit 15: RSTI2C 	Reset the I2C controller 
 *  Bit 5:  RSTCTRL	Reset the control logic (selfclr)
 *  Bit 4:  RSTFC	Reset the function conroller (selfclr)
 *  Bit 3:  RSTFSKE	Reset the function serial engine (selfclr)
 *  Bit 2:  RSTRH	Reset roothub (selfclr)
 *  Bit 1:  RSTHSIE	Reset host serial engine
 *  Bit 0:  RSTHC	Reset host controller
 *
 * --------------------------------------------------------------------------------
 */

static uint32_t
otg_rst_ctrl_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->otg_rst_ctrl;
}


static void
otg_rst_ctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	if(value & RST_CTRL_RSTI2C) {
		reset_i2c(otg);
	}
	if(value & RST_CTRL_RSTCTRL) {
		reset_control(otg);
	}
	if(value & RST_CTRL_RSTFC) {
		reset_func(otg);
	}
	if(value & RST_CTRL_RSTFSKE) {
		reset_funcser(otg);
	}
	if(value & RST_CTRL_RSTRH) {
		reset_roothub(otg);
	}
	if(value & RST_CTRL_RSTHSIE) {
		reset_hostser(otg);
	}
	if(value & RST_CTRL_RSTHC) {
		reset_hc(otg);	
	}
	otg->otg_rst_ctrl = 0; /* should be delayed by some microseconds */
        return;
}
/*
 * -----------------------------------------------------------------------------------
 * FRM_INVTL	Frame Interval register 
 *
 *   Bit 16-29:	FRMINTPER Frame interval periodic
 *   Bit 15:	RSTFRM	  Reset frame
 *   Bit 0-13:  FRMINT	  frame interval (SOF spacing)
 * -----------------------------------------------------------------------------------
 */
static uint32_t
otg_frm_invtl_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_frm_invtl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------------
 * OTG FRM_REMAIN
 *	Bits 16-29: FRM_REMAIN Bits remaining to next frame (SOF)
 * ------------------------------------------------------------------------
 */

static uint32_t
otg_frm_remain_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_frm_remain_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * -----------------------------------------------------------------------------------------
 * OTG_HNP_CTRL	HNP control/status register
 *
 *  Bit 30: HNPDAT	HNP data toggle
 *  Bit 29: VBUSBSE	V bus greater than B session end
 *  Bit 28: VBUSABSV	V bus A B session valid
 *  Bit 27: VBUSGTAVV	V bus greater than A V bus valid
 *  Bit 22: SLAVE	HNP slave state: OTG working as function
 *  Bit 21: MASTER	HNP master state: OTG working as host
 *  Bit 20: BGEN	Enables charge pump band gap
 *  Bit 19: CMPEN	Enables the charge pump comparator
 *  Bit 18: ISBDEV	device is a B device (ID pin)
 *  Bit 17: ISADEV	device is a A device (ID pin)
 *  Bit 15: SWVBUSPUL	Software V Bus pulse enable
 *  Bit 12: SWAUTORST	Software automatic reset (selfclearing on USB reset)
 *  Bit 11: SWPUDP	Software pull up 
 *  Bit  9: SWPDDM	Software pull down
 *  Bit  3: CLRERROR 	HNP clear error state (selfclr ??????)	
 *  Bit  2: ADROPBUS	A Drop V Bus: A device wants to drop V-Bus
 *  Bit  1: ABBUSREQ	A B bus request: Software requests to be an a or an b master
 *
 * -----------------------------------------------------------------------------------------
 */

static uint32_t
otg_hnp_ctrl_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_hnp_ctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ----------------------------------------------------------------------
 * OTG_HNP_INT_STAT 
 *   Bit 15: I2COTGINT 		Interrupt from I2C OTG transceiver
 *   Bit  5: SRPINT		Session request detect interrupt
 *   Bit  3: ABSESVALID		A B session valid change interrupt
 *   Bit  2: AVBUSVALID	 	AV Bus valid change interrupt
 *   Bit  0: IDCHANGE		State of ID pin has changed
 * ----------------------------------------------------------------------
 */
static uint32_t
otg_hnp_int_stat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_hnp_int_stat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ----------------------------------------------------------------------
 * OTG_HNP_INT_EN 
 *   Bit 15: I2COTGINTEN	enable interrupt from I2C OTG transceiver
 *   Bit  5: SRPINTEN		enable session request detect interrupt
 *   Bit  3: ABSESVALIDEN	enable A B session valid change interrupt
 *   Bit  2: AVBUSVALIDEN	enable AV Bus valid change interrupt
 *   Bit  0: IDCHANGEEN		enable id-change interrupt
 * ----------------------------------------------------------------------
 */
static uint32_t
otg_hnp_int_en_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_hnp_int_en_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ------------------------------------------------------------------------------
 * OTG_USBCTRL	control the wakeup interrupts and mux logic
 *
 *  Bit 27:	I2CWUINTSTAT	I2C wake up interrupt status
 *  Bit 26:	OTGWUINTSTAT	OTG wake up interrupt status
 *  Bit 25:	HOSTWUINTSTAT	Host wake up interrupt status 
 *  Bit 24: 	FNTWUINTSTAT	Function wake up interrupt status
 *  Bit 19: 	I2CWUINTEN	I2C wake up interrupt enable
 *  Bit 18: 	OTGWUINTEN	OTG wake up interrupt enable
 *  Bit 17: 	HOSTWUINTEN	Host wake up interrupt enable
 *  Bit 16: 	FNTWUINTEN	Function wake up interrupt enable
 *  Bit 15: 	OTGRCVRXDP	OTG differential receiver connected to rxdp
 *  Bit 14: 	HOST1BYPTLL	Host 1 bypass transceiver less logic
 *  Bit 10-11:	OTGBYPVAL	OTG Bypass valid
 *  Bit 8-9: 	HOST1BYPVAL	Host 1 Bypass valid
 *  Bit 6: 	OTGPWRMASK	OTG power output pin mask
 *  Bit 5: 	HOST1PWRMASK	Host 1 power output pin mask
 *  Bit 4: 	HOST2PWRMASK	Host 2 power output pin mask
 *  Bit 2: 	USBBYP		USB Bypass enable
 *  Bit 1: 	HOST1TXENOE	Host 1 transmit enable output enable
 * ------------------------------------------------------------------------------
 */

static uint32_t
otg_usbctrl_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
otg_usbctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------
 * COMSTAT
 *
 *  Bit 7: SOFTRESET 	Trigger a software reset when set, read 0
 *  Bit 3: BADISOAP  	Accept bad isochronous packets
 *  Bit 2: SUSPDET   	Detect if USB is in suspended state
 *  Bit 1: RSMINPROC	Resume in progress
 *  Bit 0: RESETDET	USB Bus reset detected
 * --------------------------------------------------------------------
 */

static uint32_t
func_comstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_comstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ----------------------------------------------------------------------
 * DEVADDR
 *	Device address
 * ----------------------------------------------------------------------
 */

static uint32_t
func_devaddr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_devaddr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * SYSINTSTAT
 *
 *  Bit  4: SOFDETINT	start of frame detect interrupt
 *  Bit  3: DONEREGINT	done register interrupt (Endpoint status register) 
 *  Bit  2: SUSPDETINT	suspend detect interrupt
 *  Bit  1: RSMFININT   resume finished interrupt
 *  Bit  0: RESETINT	reset detected interrupt
 * ---------------------------------------------------------------------------
 */

static uint32_t
func_sysintstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_sysintstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------------------
 * SYSINTEN
 *
 *  Bit  4: SOFDETIEN 	start of frame detect interrupt enable
 *  Bit  3: DONEREGIEN	done register interrupt enable
 *  Bit  2: SUSPDETIEN	suspend detect interrupt enable
 *  Bit  1: RSMFINIEN	resume finished interrupt enable
 *  Bit  0: RESETIEN	reset detect interrupt enable
 * ------------------------------------------------------------------------------
 */

static uint32_t
func_sysinten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_sysinten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------------------
 * XBUFINTSTAT X-Buffer Interrupt status
 * 	All odd bits:	XBUFININT  X-Buffer input interrupt
 *	All even bits:  XBUFOUTINT X-Buffer output interrupt
 *	
 * ------------------------------------------------------------------------------
 */

static uint32_t
func_xbufintstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_xbufintstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------------------
 *
 * YBUFINTSTAT Y-Buffer Interrupt status
 *
 * 	All odd bits:	YBUFININT  Y-Buffer input interrupt
 *	All even bits:  YBUFOUTINT Y-Buffer output interrupt
 *	
 * ------------------------------------------------------------------------------
 */


static uint32_t
func_ybufintstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_ybufintstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------------
 * XYINTEN
 *	XY Interrupt enable register
 *  All odd bits: XYBUFININT	
 *  All even bits: XYBUFOUTINT
 * --------------------------------------------------------------------------------
 */
static uint32_t
func_xyinten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_xyinten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}
 
/*
 * --------------------------------------------------------------------------------
 * XFILLSTAT X-Filled status
 *	XFILLIN	 
 *	XFILLOUT
 * --------------------------------------------------------------------------------
 */

static uint32_t
func_xfillstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_xfillstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------------
 * YFILLSTAT Y-Filled status
 *	YFILLIN	 
 *	YFILLOUT
 * --------------------------------------------------------------------------------
 */

static uint32_t
func_yfillstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}

static void
func_yfillstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------------
 * ENDPNTEN
 *	Endpoint enable register
 *
 * all odd bits:  EPINEN enable input endpoint
 * all even bits: EPODEN enable output endpoint
 * --------------------------------------------------------------------------------
 */

static uint32_t
func_endpnten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_endpnten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------------
 * ENDPNRDY Endpoint ready set register
 *  all odd bits:  ENDPINRDY set in endpoint to ready
 *  all even bits: ENDPOUTRDY set out endpoint to ready
 * ---------------------------------------------------------------------------------
 */
static uint32_t
func_endpnrdy_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}

static void
func_endpnrdy_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ---------------------------------------------------------------------------------
 * IMEDINT	
 *	Immediate interrupt register
 *  All odd bits: IMININT
 *  All even bits: IMOUTINT
 * ---------------------------------------------------------------------------------
 */

static uint32_t
func_imedint_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_imedint_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * -----------------------------------------------------------------------------------
 * EPNTDONESTAT
 *	Endpoint done status register	
 *
 *  All odd bits: 	EPINDONE input endpoint done
 *  All even bits: 	EPOUTDONE output endpoint done
 * -----------------------------------------------------------------------------------
 */
static uint32_t
func_epntdonestat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_epntdonestat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------
 * EPNDONEEN
 *	Endpoint done interrupt enable
 *	Enables interupt on SOF or immediately
 * All odd bits: EPINDEN
 * All even bits: EPOUTDEN
 * ---------------------------------------------------------------------
 */
static uint32_t
func_epntdoneen_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_epntdoneen_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * -------------------------------------------------------------------------
 * EPNTTOGBITS 
 *	Endpoint toggle bit register
 * -------------------------------------------------------------------------
 */
static uint32_t
func_epnttogbits_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_epnttogbits_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}
/*
 * ----------------------------------------------------------------------------
 * Frame pointer and endpoint ready clear register
 *	read: return framenumber of last received SOF	
 * 	write: clean endpoint ready 
 * ----------------------------------------------------------------------------
 */

static uint32_t
func_fneprdyclr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
func_fneprdyclr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

static inline void
process_etd(IMXOtg *otg,int etdnum) 
{
	
}
/* 
 * ---------------------------------------------------------------
 * The 1 ms Frame intervall timer calls this
 * Maybe this handler should be disabled when no ETD is pending
 * and SOF interrupt is disabled
 * ---------------------------------------------------------------
 */
static void
host_do_frame(void *clientData)
{
	IMXOtg *otg = (IMXOtg *) clientData;
	uint16_t frm_interval = otg->otg_frm_invtl & 0x3fff;
	int nanoseconds;
	otg->host_frmnumb = (otg->host_frmnumb + 1) & 0xffff;

	if(otg->host_sysien & SYSIEN_SOFINT) {
		otg->host_sysisr |= SYSISR_SOFINT;
		update_host_interrupt(otg);
	}
		
	nanoseconds = frm_interval * 83; // 83.33
	if(!nanoseconds) {
		nanoseconds = 1000;
	}
	CycleTimer_Mod(&otg->host_frame_timer,NanosecondsToCycles(nanoseconds));
	if(otg->host_etden) {
#if 0
		for(i=0;i<32;i++) {
			if(otg->host_etden & (1<<i)) {
				process_etd(otg,i);
			}
		}
#endif
	}
}

/*
 * --------------------------------------------------------------------------
 * HOST_CTRL Host controller register
 *
 *  Bit 31:	HCRESET	 Initiate a software controlled hardware reset
 *  Bit 16-17: 	SCHEDOVR Scheduler overrun counter
 *  Bit 4:	RMTWUEN	 Remote wake up enable
 *  Bit 2-3	HCUSBSTE Hub controller USB state
 *		00: USB reset
 *		01: USB resume
 *		10: USB operational
 *		11: USB suspend
 *  Bit 0-1	CTLBLKSR Control/Bulk service ratio
 * --------------------------------------------------------------------------
 */

static uint32_t
host_ctrl_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_ctrl;
}


static void
host_ctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	int  status;
	if(value & CTRL_HCRESET) {
		reset_hc(otg);
		return;
	} 
	otg->host_ctrl = value;
	status = value & CTRL_HCUSBSTE_MASK;
	switch(status) {
		case CTRL_HCUSBSTE_RESET:
			CycleTimer_Remove(&otg->frametimer);
			break;
		case CTRL_HCUSBSTE_RESUME:
			if(!CycleTimer_IsActive(&otg->frametimer)) {
				CycleTimer_Mod(&otg->frametimer,MicrosecondsToCycles(1000));
			}
			break;
		case CTRL_HCUSBSTE_OPER:
			if(!CycleTimer_IsActive(&otg->frametimer)) {
				CycleTimer_Mod(&otg->frametimer,MicrosecondsToCycles(1000));
			}
			break;
		case CTRL_HCUSBSTE_SUSP:
			/* ???? */
			CycleTimer_Remove(&otg->frametimer);
			break;
	}
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ----------------------------------------------------------------------
 * HOST_SYSISR System interrupt status register 
 *  Bit 6:  PSCINT 	Port status change interrupt
 *  Bit 5:  FMOFINT	Frame number overflow interrupt
 *  Bit 4:  HERRINT	Host error interrupt
 *  Bit 3:  RESDETINT	Resume detect interrupt
 *  Bit 2:  SOFINT	Start of frame interrupt
 *  Bit 1:  DONEINT		Done register interrupt
 *  Bit 0:  SORINT		Scheduler overrun interrupt
 * ----------------------------------------------------------------------
 */
static uint32_t
host_sysisr_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_sysisr;
}


static void
host_sysisr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_sysisr &= ~(value & 0x3f);
	if(otg->host_etddonestat) {
		otg->host_sysisr |= SYSISR_DONEINT;
	}
	update_host_interrupt(otg);
        return;
}


/*
 * ----------------------------------------------------------------------
 * HOST_SYSIEN System interrupt enable register 
 *  Bit 6:  PSCINT 	Port status change interrupt enable
 *  Bit 5:  FMOFINT	Frame number overflow interrupt enable
 *  Bit 4:  HERRINT	Host error interrupt enable
 *  Bit 3:  RESDETINT	Resume detect interrupt enable
 *  Bit 2:  SOFINT	Start of frame interrupt enable
 *  Bit 1:  DONEINT	Done register interrupt enable
 *  Bit 0:  SORINT	Scheduler overrun interrupt enable
 * ----------------------------------------------------------------------
 */
static uint32_t
host_sysien_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_sysien;
}


static void
host_sysien_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_sysien = value;
	update_host_interrupt(otg);
        return;
}

/*
 * -----------------------------------------------------------------------
 * XBUFSTAT
 * 	X Buffer interrupt status bitfield for 32 ETD's
 * -----------------------------------------------------------------------
 */

static uint32_t
host_xbufstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_xbufstat;
}


static void
host_xbufstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_xbufstat &= ~value;
	// interrupt updates ??
        return;
}

/*
 * -------------------------------------------------------------------
 * YBUFSTAT
 *	Y Buffer interrupt status bitfield for 32 ETD's
 * -------------------------------------------------------------------
 */
static uint32_t
host_ybufstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_ybufstat;
}


static void
host_ybufstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_ybufstat &= ~value;
	/* Interrupt updates ?*/
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ------------------------------------------------------------------------------
 * XYINTEN
 * 	X/Y Buffer Interrupt enable bitfield
 * ------------------------------------------------------------------------------
 */
static uint32_t
host_xyinten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
host_xyinten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ----------------------------------------------------------
 * XFILLSTAT
 *	X Buffer filled status. One toggling bit per ETD
 * Out endpoint: application indicates that something sendable is in y
 * In endpoint: host indicates that something was received in y-buf
 * ----------------------------------------------------------
 */
static uint32_t
host_xfillstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_xfillstat;
}


static void
host_xfillstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_xfillstat ^= value;
	fprintf(stderr,"IMX21 OTG-host: XFILLSTAT has no functionality\n");
        return;
}

/*
 * --------------------------------------------------------------------
 * YFILLSTAT
 *	Y Buffer filled status
 * Out endpoint: application indicates that something sendable is in y
 * In endpoint: host indicates that something was received in y-buf
 * --------------------------------------------------------------------
 */
static uint32_t
host_yfillstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_yfillstat;
}


static void
host_yfillstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_yfillstat ^= value;
	fprintf(stderr,"OTG-Host: YFillstat has no funtionality\n");
        return;
}

/*
 * ------------------------------------------------------------------------
 * ETDENSET ETD enable set register
 *	0: no change
 *	1: enable etd
 * ------------------------------------------------------------------------
 */

static uint32_t
host_etdenset_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return otg->host_etden;
}


static void
host_etdenset_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_etden |= value;	
	/* action is taken automatically on next frame */
        return;
}


/*
 * ------------------------------------------------------------------------
 * ETDENCLR ETD enable clear register
 *	0: no change
 *	1: disable etd
 * ------------------------------------------------------------------------
 */

static uint32_t
host_etdenclr_read(void *clientData,uint32_t address,int rqlen)
{
	return 0;
}


static void
host_etdenclr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_etden &= ~value;
	/* action is taken automatically on next frame */
        return;
}

/*
 * ----------------------------------------------------------------------------------
 * IMEDINT 
 * 	Immediate interrupt enable bitfield. An interrupt is triggered 
 *	when ETD is done. Normaly interrupts are delayed until the following SOF 
 * ----------------------------------------------------------------------------------
 */

static uint32_t
host_imedint_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_imedint;
}

static void
host_imedint_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	/* Will take effect on next SOF */
	otg->host_imedint = value;
        return;
}

/*
 * ------------------------------------------------------------------------
 * ETDDONESTAT
 *	Endpointer descriptor done Bitfield, updated immediately but
 *	does not trigger interrupt before next SOF when immediate interrupt
 *	bit is not set
 * ------------------------------------------------------------------------
 */

static uint32_t
host_etddonestat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_etddonestat;
}


static void
host_etddonestat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_etddonestat &= ~value; /* ??? Write 1 to clear ??? */
        return;
}

/*
 * --------------------------------------------------------------------------
 * ETDDONEEN
 *	Endpointer done interrupt enable (immediately or on SOF)
 * --------------------------------------------------------------------------
 */
static uint32_t
host_etddonen_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	//otg->host_etddonen; 
	return otg->host_etddonen;
}


static void
host_etddonen_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_etddonen = value;
	update_host_interrupt(otg);
        return;
}

/*
 * --------------------------------------------------------------------------
 * FRMNUMB 
 *	Framenumber read. 
 * --------------------------------------------------------------------------
 */
static uint32_t
host_frmnumb_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_frmnumb;
}


static void
host_frmnumb_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_frmnumb = value & 0xffff;
        return;
}

/*
 * -------------------------------------------------------------------------------------
 * LSTRHRESH
 *	Low speed threshold. Bit times remaining in frame required to allow a low
 *	speed packet to be started
 * --------------------------------------------------------------------------------------
 */

static uint32_t
host_lsthresh_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_lsthresh;
}


static void
host_lsthresh_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->host_lsthresh = value & 0x7ff;
        return;
}


/*
 * ----------------------------------------------------------------------------------------
 * ROOTHUBA
 *	Root hub descriptor register A 
 *
 *  Bit 24-31:	PWRTOGOOD	Power on to power good time (unit ?) 
 *  Bit 12: 	NOOVRCURP	No overcurrent protection
 *  Bit 11:	OVRCURPM	Overcurrent protection mode
 *  Bit 10:	DEVTYPE		Device type (always 0)
 *  Bit  9:	PWRSWTMD	Power switching mode
 *  Bit  8:	NOPWRSWT	No power switching (always 1)
 *  Bit 0-7:    NDNSTMPRT	Number of downstream ports (always 3)
 * ----------------------------------------------------------------------------------------
 */
static uint32_t
host_roothuba_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_roothuba; 
}

static void
host_roothuba_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	fprintf(stderr,"IMX21 OTG: roothuba is not writable\n");
        return;
}

/*
 * --------------------------------------------------------------------------------------
 * ROOTHUBB
 * 	Root hub descriptor register B
 *
 *  Bit 16-23:	PRTPWRCM: 	Port power control mask 
 *  Bit 0-7:	DEVREMOVE:	Device removable bitfield
 * --------------------------------------------------------------------------------------
 */

static uint32_t
host_roothubb_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_roothubb;
}


static void
host_roothubb_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	fprintf(stderr,"IMX21 OTG: Roothubb is not writable\n");
        return;
}

/*
 * --------------------------------------------------------------------------------
 * ROOTSTAT
 *	Root Hub status register
 *
 *  Bit 31: CLRRMTWUE	Clear remote wake up enable
 *  Bit 17: OVRCURCHG	Overcurrent indicator change
 *  Bit 15: DEVCONWUE	Device connect wakeup enable
 *  Bit 1:  OVRCURI	Overcurrent indicator
 *  Bit 0:  LOCPWRS	Local power status (always 0)
 * --------------------------------------------------------------------------------
 */

static uint32_t
host_rootstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->host_rootstat;
}


/*
 * -----------------------
 * Currently ignored
 * -----------------------
 */
static void
host_rootstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	if(value & ROOTSTAT_OVRCURCHG) {
		otg->host_rootstat &= ~ROOTSTAT_OVRCURCHG;
	}
        return;
}


/*
 * --------------------------------------------------------------------------------
 * PORSTAT1-3 Port status register
 *
 *  Bit  20: PRTRSTSC	Port reset status change 
 *  Bit  19: OVRCURIC	Overcurrent indicator change
 *  Bit  18: PRTSTATSC	Port suspend status change
 *  Bit  17: PRTENBLSC	Port enable status change
 *  Bit  16: CONNECTSC	Connect status change
 *  Bit   9: LSDEVCON	Lowspeed device attachement / Clear port power
 *  Bit   8: PRTPWRST	Port power status
 *  Bit   4: PRTRSTST	Port reset status
 *  Bit   3: PRTOVRCURI	Port overcurrent indicator
 *  Bit   2: PRTSUSPST	Port suspend status
 *  Bit   1: PRTENABST	Port enable status
 *  Bit   0: CURCONST	Current connect status
 * --------------------------------------------------------------------------------
 */

static uint32_t
host_portstat_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	unsigned int index = ((address & 0xff) - 0xf4) >> 2; 
	if(index > 2)  {
		fprintf(stderr,"Illegal portstat index %d\n",index);
		return 0;
	}
	return otg->host_portstat[index];
}
static void
host_portstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	unsigned int index = ((address & 0xff) - 0xf4) >> 2; 
	uint32_t clearmask = ~(value & 0x001f0000);
	if(index > 2)  {
		fprintf(stderr,"Illegal portstat index %d\n",index);
		return;
	}
	otg->host_portstat[index] = otg->host_portstat[index] & clearmask;
	if(value & PORTSTAT_SETPRTRST) {
		if(otg->host_portstat[index] & PORTSTAT_CURCONST) {
			otg->host_portstat[index] |= PORTSTAT_PRTRSTST;
		} else {
			otg->host_portstat[index] |= PORTSTAT_CONNECTSC;
			// update_interrupts
		}
	}
	if(value & PORTSTAT_CLRSUSP) {
		otg->host_portstat[index] &= ~PORTSTAT_PRTSUSPST;
	}
	if(value & PORTSTAT_SETSUSP) {
		otg->host_portstat[index] |= PORTSTAT_PRTSUSPST;
	}
	if(value & PORTSTAT_SETPRTENAB) {
		if(otg->host_portstat[index] & PORTSTAT_CURCONST) {
			otg->host_portstat[index] |= PORTSTAT_PRTENABST;
		} else {
			otg->host_portstat[index] |= PORTSTAT_CONNECTSC;
			// update_interrupts();
		}
		printf("Port was enabled\n");
		exit(1);
	}
	if(value & PORTSTAT_CLRPWRST) {	
		otg->host_portstat[index] &= ~PORTSTAT_PRTPWRST;
	}
	if(value & PORTSTAT_SETPWRST) {
		otg->host_portstat[index] |= PORTSTAT_PRTPWRST;
	}
	if(value & PORTSTAT_CLRPRTENAB) {
		otg->host_portstat[index] &= ~PORTSTAT_PRTENABST;
	}
	fprintf(stderr,"Set portstat%d x%08x : result %08x not implemented\n",index,value,otg->host_portstat[index]);
        return;
}

static uint32_t
etd_mem_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t value=0;
	uint32_t index = address & 0x1ff;
	int i;
	for(i=0;i<rqlen;i++) {
		value = value | otg->etd_mem[(index+i)&0x1ff] << (i*8);
	}
	return value;

}
static void 
etd_mem_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t index = address & 0x1ff;
	int i;
	for(i=0;i<rqlen;i++) {
		otg->etd_mem[(index+i) & 0x1ff] = value;
		value = value >> 8;
	}
}
static uint32_t
ep_mem_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t value=0;
	uint32_t index = address & 0x1ff;
	int i;
	for(i=0;i<rqlen;i++) {
		value = value | otg->ep_mem[(index+i)&0x1ff] << (i*8);
	}
	return value;

}
static void 
ep_mem_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t index = address & 0x1ff;
	int i;
	for(i=0;i<rqlen;i++) {
		otg->ep_mem[(index+i) & 0x1ff] = value;
		value = value >> 8;
	}
}
/* The AHB DMA block */

/*
 * ------------------------------------------------------------------------------
 * DMA_REV_READ
 *	Revision code for DMAIP block design
 * ------------------------------------------------------------------------------
 */

static uint32_t
dma_rev_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->dma_rev;
}


static void
dma_rev_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ----------------------------------------------------------------------------
 * DMA_INTSTAT
 *   Bit 1:	EPERR   Endpoint error interrupt
 *   Bit 0: 	ETDERR	ETD error interrupt	
 * ----------------------------------------------------------------------------
 */
static uint32_t
dma_intstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_intstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ----------------------------------------------------------------------------
 * DMA_INTEN
 *   Bit 1:	EPERRINTEN	Endpoint error interrupt
 *   Bit 0: 	ETDERRINTEN	ETD error interrupt	
 * ----------------------------------------------------------------------------
 */

static uint32_t
dma_inten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_inten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------
 * DMA_ETDDMAERSTAT
 *	ETD dma error status bitfield
 * --------------------------------------------------------------------------
 */
static uint32_t
dma_etddmaerstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmaerstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------
 * DMA_EPDMAERSTAT
 *	EPD dma error status bitfield
 * --------------------------------------------------------------------------
 */

static uint32_t
dma_epdmaerstat_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmaerstat_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------
 * DMA_ETDDMAEN
 *	ETD dma enable bitfield 
 * --------------------------------------------------------------------------
 */

static uint32_t
dma_etddmaen_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmaen_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * --------------------------------------------------------------------------
 * DMA_EPDMAEN
 *	Endpoint dma enable register 
 * --------------------------------------------------------------------------
 */
static uint32_t
dma_epdmaen_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmaen_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------
 * DMA_ETDDRMAXTEN
 *	ETD dma enable X trigger request
 * --------------------------------------------------------------------------
 */

static uint32_t
dma_etddrmaxten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmaxten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------------
 * DMA_EPDDRMAXTEN
 *	Endpoint dma enable X trigger request
 * --------------------------------------------------------------------------
 */

static uint32_t
dma_epdmaxten_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmaxten_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * ETDDMAENXYT
 * 	ETD dma enable XY Trigger request
 * ---------------------------------------------------------------------------
 */

static uint32_t
dma_etddmaenxyt_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmaenxyt_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * EPDMAENXYT
 * 	EPD dma enable XY Trigger request
 * ---------------------------------------------------------------------------
 */

static uint32_t
dma_epdmaenxyt_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmaenxyt_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * ETCDMABST4EN 
 *	ETD DMA Burst4 enable
 * ---------------------------------------------------------------------------
 */
static uint32_t
dma_etcdmabst4en_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etcdmabst4en_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * EPDMABST4EN 
 *	EPD DMA Burst4 enable
 * ---------------------------------------------------------------------------
 */

static uint32_t
dma_epdmabst4en_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmabst4en_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * ------------------------------------------------------------------------
 * DMA_MISCCONTROL
 *  Bit 3: ISOPREVFRM	ISO out previous frame mode
 *  Bit 2: SKPRTRY	Skip on retry mode
 *  Bit 1: ARBMODE	Arbiter mode select
 *  Bit 0: FILTCC	Filter on completion code
 * ------------------------------------------------------------------------
 */
static uint32_t
dma_misccontrol_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_misccontrol_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * ETDDMACHANCLR
 *	ETD dma channel clear register: disable and abort dma
 * ---------------------------------------------------------------------------
 */

static uint32_t
dma_etddmachanclr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmachanclr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ---------------------------------------------------------------------------
 * EPDMACHANCLR
 *	EP dma channel clear register: disable and abort dma
 * ---------------------------------------------------------------------------
 */

static uint32_t
dma_epdmachanclr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmachanclr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * ----------------------------------------------------------------
 * ETDSMSA
 * 	Etd system memory start address register
 * ----------------------------------------------------------------
 */
static uint32_t
dma_etdsmsa_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etdsmsa_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * --------------------------------------------------------------------
 * EPSMSA
 *	Endpoint system memory start address
 * --------------------------------------------------------------------
 */

static uint32_t
dma_epsmsa_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epsmsa_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

/*
 * -------------------------------------------------------------------------
 * ETDDMABUFPTR
 *	etd dma buffer pointer. accessible only for debug purposes
 * -------------------------------------------------------------------------
 */
static uint32_t
dma_etddmabufptr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_etddmabufptr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}


/*
 * -------------------------------------------------------------------------
 * EPDMABUFPTR
 *	EP dma buffer pointer. accessible only for debug purposes
 * -------------------------------------------------------------------------
 */

static uint32_t
dma_epdmabufptr_read(void *clientData,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
	return 0;
}


static void
dma_epdmabufptr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        //IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: 0x%08x not implemented\n",address);
        return;
}

static uint32_t
isp1301_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	int reg = address & 0x1f;
	return ISP1301_Read(otg->isp1301,reg);
}

static void
isp1301_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	int reg = address & 0x1f;
	ISP1301_Write(otg->isp1301,value,reg);
        return;
}

static uint32_t
i2c_otgdevaddr_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->i2c_otgdevaddr;
}


static void
i2c_otgdevaddr_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->i2c_otgdevaddr = value;
        return;
}

static uint32_t
i2c_numseqops_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->i2c_numseqops;
}


static void
i2c_numseqops_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->i2c_numseqops = value;
        return;
}

static uint32_t
i2c_seqreadstrt_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->i2c_seqreadstrt;	
}

static void
i2c_seqreadstrt_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->i2c_seqreadstrt = value;
        return;
}

static uint32_t
i2c_opctrl_read(void *clientData,uint32_t address,int rqlen)
{
	IMXOtg *otg = (IMXOtg *) clientData;	
	/* Bit 7 is Busy for i2c transaction */
	return otg->i2c_opctrl;
}

static void
i2c_opctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	IMXOtg *otg = (IMXOtg *) clientData;	
	otg->i2c_opctrl = value;
        return;
}

static uint32_t
i2c_divfactor_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	return otg->i2c_divfactor;
}

static void
i2c_divfactor_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	
        IMXOtg *otg = (IMXOtg *) clientData;
	otg->i2c_divfactor = value;
        return;
}

static uint32_t
i2c_intandctrl_read(void *clientData,uint32_t address,int rqlen)
{
	IMXOtg *otg = (IMXOtg *) clientData;
	fprintf(stderr,"IMX21 OTG: intandctrl read\n");
	return otg->i2c_intandctrl;
}

static void
i2c_intandctrl_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	IMXOtg *otg = (IMXOtg *) clientData;
	uint8_t clearmask = ~((value & 6) | 0x70);
	otg->i2c_intandctrl = (otg->i2c_intandctrl & clearmask) | (value & 0x70); 
	fprintf(stderr,"IMX21 OTG: intandctrl write %02x\n",value);
        return;
}

static uint32_t
data_mem_read(void *clientData,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t value=0;
	uint32_t index = address & 0xfff;
	int i;
	for(i=0;i<rqlen;i++) {
		value = value | otg->data_mem[(index+i)&0xfff] << (i*8);
	}
	return value;
}

static void
data_mem_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        IMXOtg *otg = (IMXOtg *) clientData;
	uint32_t index = address & 0xfff;
	int i;
	for(i=0;i<rqlen;i++) {
		otg->data_mem[(index+i) & 0xfff] = value;
		value = value >> 8;
	}
        return;
}

static void
IMXOtg_Unmap(void *owner,uint32_t base,uint32_t mask)
{
	int i;
	IOH_Delete32(OTG_HWMODE(base));
	IOH_Delete32(OTG_CINT_STAT(base));
	IOH_Delete32(OTG_CINT_STEN(base));
	IOH_Delete32(OTG_CLK_CTRL(base));
	IOH_Delete32(OTG_RST_CTRL(base));
	IOH_Delete32(OTG_FRM_INVTL(base));
	IOH_Delete32(OTG_FRM_REMAIN(base));
	IOH_Delete32(OTG_HNP_CTRL(base));
	IOH_Delete32(OTG_HNP_INT_STAT(base));
	IOH_Delete32(OTG_HNP_INT_EN(base));
	IOH_Delete32(OTG_USBCTRL(base));	

	IOH_Delete32(FUNC_COMSTAT(base));
	IOH_Delete32(FUNC_DEVADDR(base));
	IOH_Delete32(FUNC_SYSINTSTAT(base));
	IOH_Delete32(FUNC_SYSINTEN(base));
	IOH_Delete32(FUNC_XBUFINTSTAT(base));
	IOH_Delete32(FUNC_YBUFINTSTAT(base));
	IOH_Delete32(FUNC_XYINTEN(base));
	IOH_Delete32(FUNC_XFILLSTAT(base));
	IOH_Delete32(FUNC_YFILLSTAT(base));
	IOH_Delete32(FUNC_ENDPNTEN(base));
	IOH_Delete32(FUNC_ENDPNRDY(base));
	IOH_Delete32(FUNC_IMEDINT(base));
	IOH_Delete32(FUNC_EPNTDONESTAT(base));
	IOH_Delete32(FUNC_EPNTDONEEN(base));
	IOH_Delete32(FUNC_EPNTTOGBITS(base));
	IOH_Delete32(FUNC_FNEPRDYCLR(base));

	IOH_Delete32(HOST_CTRL(base));
	IOH_Delete32(HOST_SYSISR(base));
	IOH_Delete32(HOST_SYSIEN(base));
	IOH_Delete32(HOST_XBUFSTAT(base));
	IOH_Delete32(HOST_YBUFSTAT(base));
	IOH_Delete32(HOST_XYINTEN(base));
	IOH_Delete32(HOST_XFILLSTAT(base));
	IOH_Delete32(HOST_YFILLSTAT(base));
	IOH_Delete32(HOST_ETDENSET(base));
	IOH_Delete32(HOST_ETDENCLR(base));
	IOH_Delete32(HOST_IMEDINT(base));
	IOH_Delete32(HOST_ETDDONESTAT(base));
	IOH_Delete32(HOST_ETDDONEN(base));
	IOH_Delete32(HOST_FRMNUMB(base));
	IOH_Delete32(HOST_LSTHRESH(base));
	IOH_Delete32(HOST_ROOTHUBA(base));
	IOH_Delete32(HOST_ROOTHUBB(base));
	IOH_Delete32(HOST_ROOTSTAT(base));
	IOH_Delete32(HOST_PORTSTAT1(base));
	IOH_Delete32(HOST_PORTSTAT2(base));
	IOH_Delete32(HOST_PORTSTAT3(base));
	IOH_DeleteRegion(ETD_BASE(base),0x200);
	IOH_DeleteRegion(EP_BASE(base),0x200);
/* The AHB DMA block */
	IOH_Delete32(DMA_REV(base));
	IOH_Delete32(DMA_INTSTAT(base));
	IOH_Delete32(DMA_INTEN(base));
	IOH_Delete32(DMA_ETDDMAERSTAT(base));
	IOH_Delete32(DMA_EPDMAERSTAT(base));
	IOH_Delete32(DMA_ETDDMAEN(base));
	IOH_Delete32(DMA_EPDMAEN(base));
	IOH_Delete32(DMA_ETDDMAXTEN(base));
	IOH_Delete32(DMA_EPDMAXTEN(base));
	IOH_Delete32(DMA_ETDDMAENXYT(base));
	IOH_Delete32(DMA_EPDMAENXYT(base));
	IOH_Delete32(DMA_ETCDMABST4EN(base));
	IOH_Delete32(DMA_EPDMABST4EN(base));
	IOH_Delete32(DMA_MISCCONTROL(base));
	IOH_Delete32(DMA_ETDDMACHANCLR(base));
	IOH_Delete32(DMA_EPDMACHANCLR(base));
	//loop
	for(i=0;i<0x20;i++) {
		IOH_Delete32(DMA_ETDSMSA(base,i));
		IOH_Delete32(DMA_EPSMSA(base,i));
		IOH_Delete32(DMA_ETDDMABUFPTR(base,i));
		IOH_Delete32(DMA_EPDMABUFPTR(base,i));
	}
	for(i=0;i<0x14;i++) {
		IOH_Delete8(TRANS_VENODR_ID_LOW(base)+i);
	}
	IOH_Delete8(I2C_OTGDEVADDR(base));
	IOH_Delete8(I2C_NUMSEQOPS(base));
	IOH_Delete8(I2C_OPCTRL(base));
	IOH_Delete8(I2C_SEQREADSTRT(base));
	IOH_Delete8(I2C_DIVFACTOR(base));
	IOH_Delete8(I2C_INTANDCTRL(base));
	IOH_DeleteRegion(DATA_MEM(base),0x1000);
}
static void
IMXOtg_Map(void *owner,uint32_t base,uint32_t mask,uint32_t mapflags)
{
	IMXOtg *otg = (IMXOtg *)owner;
	uint32_t flags;
	int i;
	IOH_New32(OTG_HWMODE(base),otg_hwmode_read,otg_hwmode_write,otg);
	IOH_New32(OTG_CINT_STAT(base),otg_cint_stat_read,otg_cint_stat_write,otg);
	IOH_New32(OTG_CINT_STEN(base),otg_cint_sten_read,otg_cint_sten_write,otg);
	IOH_New32(OTG_CLK_CTRL(base),otg_clk_ctrl_read,otg_clk_ctrl_write,otg);
	IOH_New32(OTG_RST_CTRL(base),otg_rst_ctrl_read,otg_rst_ctrl_write,otg);
	IOH_New32(OTG_FRM_INVTL(base),otg_frm_invtl_read,otg_frm_invtl_write,otg);
	IOH_New32(OTG_FRM_REMAIN(base),otg_frm_remain_read,otg_frm_remain_write,otg);
	IOH_New32(OTG_HNP_CTRL(base),otg_hnp_ctrl_read,otg_hnp_ctrl_write,otg);
	IOH_New32(OTG_HNP_INT_STAT(base),otg_hnp_int_stat_read,otg_hnp_int_stat_write,otg);
	IOH_New32(OTG_HNP_INT_EN(base),otg_hnp_int_en_read,otg_hnp_int_en_write,otg);
	IOH_New32(OTG_USBCTRL(base),otg_usbctrl_read,otg_usbctrl_write,otg);	

	IOH_New32(FUNC_COMSTAT(base),func_comstat_read,func_comstat_write,otg);
	IOH_New32(FUNC_DEVADDR(base),func_devaddr_read,func_devaddr_write,otg);
	IOH_New32(FUNC_SYSINTSTAT(base),func_sysintstat_read,func_sysintstat_write,otg);
	IOH_New32(FUNC_SYSINTEN(base),func_sysinten_read,func_sysinten_write,otg);
	IOH_New32(FUNC_XBUFINTSTAT(base),func_xbufintstat_read,func_xbufintstat_write,otg);
	IOH_New32(FUNC_YBUFINTSTAT(base),func_ybufintstat_read,func_ybufintstat_write,otg);
	IOH_New32(FUNC_XYINTEN(base),func_xyinten_read,func_xyinten_write,otg);
	IOH_New32(FUNC_XFILLSTAT(base),func_xfillstat_read,func_xfillstat_write,otg);
	IOH_New32(FUNC_YFILLSTAT(base),func_yfillstat_read,func_yfillstat_write,otg);
	IOH_New32(FUNC_ENDPNTEN(base),func_endpnten_read,func_endpnten_write,otg);
	IOH_New32(FUNC_ENDPNRDY(base),func_endpnrdy_read,func_endpnrdy_write,otg);
	IOH_New32(FUNC_IMEDINT(base),func_imedint_read,func_imedint_write,otg);
	IOH_New32(FUNC_EPNTDONESTAT(base),func_epntdonestat_read,func_epntdonestat_write,otg);
	IOH_New32(FUNC_EPNTDONEEN(base),func_epntdoneen_read,func_epntdoneen_write,otg);
	IOH_New32(FUNC_EPNTTOGBITS(base),func_epnttogbits_read,func_epnttogbits_write,otg);
	IOH_New32(FUNC_FNEPRDYCLR(base),func_fneprdyclr_read,func_fneprdyclr_write,otg);

	IOH_New32(HOST_CTRL(base),host_ctrl_read,host_ctrl_write,otg);
	IOH_New32(HOST_SYSISR(base),host_sysisr_read,host_sysisr_write,otg);
	IOH_New32(HOST_SYSIEN(base),host_sysien_read,host_sysien_write,otg);
	IOH_New32(HOST_XBUFSTAT(base),host_xbufstat_read,host_xbufstat_write,otg);
	IOH_New32(HOST_YBUFSTAT(base),host_ybufstat_read,host_ybufstat_write,otg);
	IOH_New32(HOST_XYINTEN(base),host_xyinten_read,host_xyinten_write,otg);
	IOH_New32(HOST_XFILLSTAT(base),host_xfillstat_read,host_xfillstat_write,otg);
	IOH_New32(HOST_YFILLSTAT(base),host_yfillstat_read,host_yfillstat_write,otg);
	IOH_New32(HOST_ETDENSET(base),host_etdenset_read,host_etdenset_write,otg);
	IOH_New32(HOST_ETDENCLR(base),host_etdenclr_read,host_etdenclr_write,otg);
	IOH_New32(HOST_IMEDINT(base),host_imedint_read,host_imedint_write,otg);
	IOH_New32(HOST_ETDDONESTAT(base),host_etddonestat_read,host_etddonestat_write,otg);
	IOH_New32(HOST_ETDDONEN(base),host_etddonen_read,host_etddonen_write,otg);
	IOH_New32(HOST_FRMNUMB(base),host_frmnumb_read,host_frmnumb_write,otg);
	IOH_New32(HOST_LSTHRESH(base),host_lsthresh_read,host_lsthresh_write,otg);
	IOH_New32(HOST_ROOTHUBA(base),host_roothuba_read,host_roothuba_write,otg);
	IOH_New32(HOST_ROOTHUBB(base),host_roothubb_read,host_roothubb_write,otg);
	IOH_New32(HOST_ROOTSTAT(base),host_rootstat_read,host_rootstat_write,otg);
	IOH_New32(HOST_PORTSTAT1(base),host_portstat_read,host_portstat_write,otg);
	IOH_New32(HOST_PORTSTAT2(base),host_portstat_read,host_portstat_write,otg);
	IOH_New32(HOST_PORTSTAT3(base),host_portstat_read,host_portstat_write,otg);
	
	IOH_NewRegion(ETD_BASE(base),0x200,etd_mem_read,etd_mem_write,IOH_FLG_LITTLE_ENDIAN,otg);
	IOH_NewRegion(EP_BASE(base),0x200,ep_mem_read,ep_mem_write,IOH_FLG_LITTLE_ENDIAN,otg);
/* The AHB DMA block */
	IOH_New32(DMA_REV(base),dma_rev_read,dma_rev_write,otg);
	IOH_New32(DMA_INTSTAT(base),dma_intstat_read,dma_intstat_write,otg);
	IOH_New32(DMA_INTEN(base),dma_inten_read,dma_inten_write,otg);
	IOH_New32(DMA_ETDDMAERSTAT(base),dma_etddmaerstat_read,dma_etddmaerstat_write,otg);
	IOH_New32(DMA_EPDMAERSTAT(base),dma_epdmaerstat_read,dma_epdmaerstat_write,otg);
	IOH_New32(DMA_ETDDMAEN(base),dma_etddmaen_read,dma_etddmaen_write,otg);
	IOH_New32(DMA_EPDMAEN(base),dma_epdmaen_read,dma_epdmaen_write,otg);
	IOH_New32(DMA_ETDDMAXTEN(base),dma_etddrmaxten_read,dma_etddmaxten_write,otg);
	IOH_New32(DMA_EPDMAXTEN(base),dma_epdmaxten_read,dma_epdmaxten_write,otg);
	IOH_New32(DMA_ETDDMAENXYT(base),dma_etddmaenxyt_read,dma_etddmaenxyt_write,otg);
	IOH_New32(DMA_EPDMAENXYT(base),dma_epdmaenxyt_read,dma_epdmaenxyt_write,otg);
	IOH_New32(DMA_ETCDMABST4EN(base),dma_etcdmabst4en_read,dma_etcdmabst4en_write,otg);
	IOH_New32(DMA_EPDMABST4EN(base),dma_epdmabst4en_read,dma_epdmabst4en_write,otg);
	IOH_New32(DMA_MISCCONTROL(base),dma_misccontrol_read,dma_misccontrol_write,otg);
	IOH_New32(DMA_ETDDMACHANCLR(base),dma_etddmachanclr_read,dma_etddmachanclr_write,otg);
	IOH_New32(DMA_EPDMACHANCLR(base),dma_epdmachanclr_read,dma_epdmachanclr_write,otg);
	//loop
	for(i=0;i<0x20;i++) {
		IOH_New32(DMA_ETDSMSA(base,i),dma_etdsmsa_read,dma_etdsmsa_write,otg);
		IOH_New32(DMA_EPSMSA(base,i),dma_epsmsa_read,dma_epsmsa_write,otg);
		IOH_New32(DMA_ETDDMABUFPTR(base,i),dma_etddmabufptr_read,dma_etddmabufptr_write,otg);
		IOH_New32(DMA_EPDMABUFPTR(base,i),dma_epdmabufptr_read,dma_epdmabufptr_write,otg);
	}
	flags = IOH_FLG_OSZR_NEXT | IOH_FLG_OSZW_NEXT;
	for(i=0;i<0x14;i++) {
		IOH_New8f(TRANS_VENODR_ID_LOW(base)+i,isp1301_read,isp1301_write,otg,flags);
	}
	flags = IOH_FLG_OSZR_NEXT | IOH_FLG_OSZW_NEXT;
	IOH_New8f(I2C_OTGDEVADDR(base),i2c_otgdevaddr_read,i2c_otgdevaddr_write,otg,flags);
	IOH_New8f(I2C_NUMSEQOPS(base),i2c_numseqops_read,i2c_numseqops_write,otg,flags);
	IOH_New8f(I2C_SEQREADSTRT(base),i2c_seqreadstrt_read,i2c_seqreadstrt_write,otg,flags);
	IOH_New8f(I2C_OPCTRL(base),i2c_opctrl_read,i2c_opctrl_write,otg,flags);
	IOH_New8f(I2C_DIVFACTOR(base),i2c_divfactor_read,i2c_divfactor_write,otg,flags);
	IOH_New8f(I2C_INTANDCTRL(base),i2c_intandctrl_read,i2c_intandctrl_write,otg,flags);

	IOH_NewRegion(DATA_MEM(base),0x1000,data_mem_read,data_mem_write,IOH_FLG_LITTLE_ENDIAN,otg);
}

BusDevice *
IMX21_OtgNew(const char *name)
{
        IMXOtg *otg = malloc(sizeof(IMXOtg));
	I2C_Slave *i2c_slave;
	char *trans_name = alloca(strlen(name) + 50);
        if(!otg) {
                fprintf(stderr,"Out of memory for IMX USB OTG\n");
                exit(1);
        }
        memset(otg,0,sizeof(IMXOtg));
	memcpy(otg->etd_mem,etdmem_initval,512);
	memcpy(otg->ep_mem,epmem_initval,512);
	otg->intUsbWkup =SigNode_New("%s.intWkup",name);
	otg->intUsbMnp = SigNode_New("%s.intMnp",name);
	otg->intUsbDma = SigNode_New("%s.intDma",name);
	otg->intUsbFunc = SigNode_New("%s.intFunc",name);
	otg->intUsbHost = SigNode_New("%s.intHost",name);
	otg->intUsbCtrl = SigNode_New("%s.intCtrl",name);
	if(!otg->intUsbWkup || !otg->intUsbMnp || !otg->intUsbDma || !otg->intUsbFunc 
		|| !otg->intUsbHost || !otg->intUsbCtrl) {

		fprintf(stderr,"Can not create interrupt nodes for IMX21 OTG\n");
		exit(7);
	}
        otg->bdev.first_mapping=NULL;
        otg->bdev.Map=IMXOtg_Map;
        otg->bdev.UnMap=IMXOtg_Unmap;
        otg->bdev.owner=otg;
        otg->bdev.hw_flags=MEM_FLAG_WRITABLE|MEM_FLAG_READABLE;
	CycleTimer_Init(&otg->frametimer,host_do_frame,otg);
	sprintf(trans_name,"%s.isp1301",name);
	i2c_slave = ISP1301_New(trans_name);
	otg->isp1301 = ISP1301_GetPtr(i2c_slave);
	CycleTimer_Init(&otg->host_frame_timer,host_do_frame,otg);
	reset_i2c(otg);
	reset_control(otg);
	reset_func(otg);
	reset_funcser(otg);
	reset_roothub(otg);
	reset_hostser(otg);
	reset_hc(otg);
	reset_dma(otg);
        fprintf(stderr,"IMX21 USB OTG controller \"%s\" created\n",name);
        return &otg->bdev;
}

