/*
 * ------------------------------------------------------------------------------
 *
 * Emulation of National Semiconductor LM75 I2C-Temperature Sensor 
 *
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 * State: Temperature reading is possible (fixed value). 
 * 	  Everything else doesn't work 
 *
 *	You can crash real LM75 by creating a stop/start condition during
 *	acknowledge of read operation. It will pull SDA to low forever.
 *	This bug is a missing feature of the emulator.
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * ------------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include "i2c.h"
#include "lm75.h"

#if 0
#define dprintf(x...) { fprintf(stderr,x); }
#else
#define dprintf(x...)
#endif

#define LM75_STATE_POINTER  	(0)
#define LM75_STATE_DATA1  	(1)
#define LM75_STATE_DATA2  	(2)
#define LM75_STATE_PAST_END  	(3)

#define LM75_REG_TEMP		(0)
#define LM75_REG_CONFIG		(1)
#define LM75_REG_THYST		(2)
#define LM75_REG_TOS		(3)

struct LM75 {
	I2C_Slave i2c_slave;
	uint8_t pointer_reg;
	int state;
	uint16_t data;
	uint8_t conf;
	int16_t temp;
	int16_t thyst;
	int16_t tos;
};

/*
 * ------------------------------------
 * LM75 Write state machine 
 * ------------------------------------
 */
static int 
lm75_write(void *dev,uint8_t data) {
	LM75 *lm = dev;
	if(lm->state==LM75_STATE_POINTER) {
		dprintf("LM75 Addr 0x%02x\n",data);
		lm->pointer_reg = data & 0x3;
		lm->state = LM75_STATE_DATA1;
	} else if(lm->state==LM75_STATE_DATA1) {
		dprintf("LM75 Write 0x%02x to %04x\n",data,lm->pointer_reg);
		if(lm->pointer_reg == LM75_REG_CONFIG) {
			lm->conf = data;
		} else {
			lm->data=data<<8;
			lm->state = LM75_STATE_DATA2;
		}
	} else if(lm->state==LM75_STATE_DATA2) {
		lm->data |= data;
		switch(lm->pointer_reg) {
			case LM75_REG_TEMP:
				return I2C_NACK;
			case LM75_REG_THYST:
				lm->thyst=data;
				break;
			case LM75_REG_TOS:
				lm->tos=data;
				break;
		}
	}
	return I2C_ACK;
};

static int 
lm75_read(void *dev,uint8_t *data) 
{
	LM75 *lm = dev;
	dprintf("LM75 read 0x%02x from %04x\n",*data,lm->pointer_reg);
	if(lm->state==LM75_STATE_DATA1) {
		switch(lm->pointer_reg) {
			case LM75_REG_CONFIG:
				*data=lm->conf;
				return I2C_DONE;
			case LM75_REG_TEMP:
				lm->data = lm->temp;
				break;
			case LM75_REG_THYST:
				lm->data = lm->thyst;
				break;
			case LM75_REG_TOS:
				lm->data = lm->tos;
				break;
		}
		lm->state = LM75_STATE_DATA2;
		*data = lm->data >> 8;
	} else if(lm->state==LM75_STATE_DATA2) {
		*data = lm->data; 
		lm->state = LM75_STATE_PAST_END;
	} else if(lm->state==LM75_STATE_PAST_END) {
		*data = 0xff;
	}
	return I2C_DONE;
};

static int
lm75_start(void *dev,int i2c_addr,int operation) {
	LM75 *lm = dev;
	dprintf("LM75 start\n");
	if(operation == I2C_READ) {
		lm->state = LM75_STATE_DATA1;
	} else if (operation == I2C_WRITE) {
		lm->state = LM75_STATE_POINTER;
	}
	return I2C_ACK;
}

static void 
lm75_stop(void *dev) {
	LM75 *lm = dev;
	dprintf("LM75 stop\n");
	lm->state =  LM75_STATE_POINTER; 
}


static I2C_SlaveOps lm75_ops = {
	.start = lm75_start,
	.stop =  lm75_stop,
	.read =  lm75_read,	
	.write = lm75_write	
};

I2C_Slave *
LM75_New(char *name) {
	LM75 *lm = malloc(sizeof(LM75)); 
	I2C_Slave *i2c_slave;
	if(!lm) {
		fprintf(stderr,"Out of memory\n");
		exit(8152);
	}
	memset(lm,0,sizeof(LM75));
	lm->temp = 20 << 8;
	i2c_slave = &lm->i2c_slave;
	i2c_slave->devops = &lm75_ops; 
	i2c_slave->dev = lm;
	i2c_slave->speed = I2C_SPEED_FAST;
	fprintf(stderr,"LM75 Temperature Sensor \"%s\" created\n",name);
	return i2c_slave;
}
