/*
 * -----------------------------------------------------------------------
 * M16C-Flash
 *      Emulation of the internal flash of Renesas M16C  series
 *
 *
 * (C) 2005 Jochen Karrer
 *    Author: Jochen Karrer
 *
 * Status: nothing is implemented 
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------------------------
 */

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

#include "m16c_flash.h"
#include "configfile.h"
#include "cycletimer.h"
#include "bus.h"
#include "signode.h"
#include "diskimage.h"
#include "sgstring.h"

#define FMR0	(0x1b7)
#define		FMR0_FMSTP	(1<<3)
#define FMR1	(0x1b5)

#define CMD_NONE 		(0)
#define CMD_READ_ARRAY 		(1)
#define CMD_READ_STATUS 	(2)
#define CMD_CLEAR_STATUS	(3)
#define CMD_PROGRAM		(4)
#define CMD_BLOCK_ERASE		(5)
#define CMD_ERASE_ALL_UNLOCKED	(6)
#define CMD_LOCK_BIT_PGM	(7)
#define CMD_READ_LOCK_BIT_STAT  (8)

#define MAP_READ_ARRAY  (0)
#define MAP_IO          (1)

typedef struct M16CFlash {
	uint8_t fmr0;
	uint8_t fmr1;

	/* State space from two variables */
	int cycle;
	int cmd;
	int mode; /* Read array or command mode */

        BusDevice bdev;
        DiskImage *disk_image;
        uint8_t *host_mem;
        int size;
} M16CFlash;

static void
switch_to_readarray(M16CFlash *mflash) {
        if(mflash->mode!=MAP_READ_ARRAY) {
                mflash->mode=MAP_READ_ARRAY;
                Mem_AreaUpdateMappings(&mflash->bdev);
        }
}

static void
switch_to_iomode(M16CFlash *mflash) {
        if(mflash->mode!=MAP_IO) {
                mflash->mode=MAP_IO;
                Mem_AreaUpdateMappings(&mflash->bdev);
        }
}

static void
flash_write_first(M16CFlash *mflash,uint32_t value,uint32_t mem_addr,int rqlen)
{
	switch(value & 0xff) {
		case 0xff:
			switch_to_readarray(mflash);
			break;

		case 0x70:
			/* Read status register */
			switch_to_iomode(mflash);
			mflash->cmd = CMD_READ_STATUS;
			mflash->cycle++;
			break;

		case 0x50:
			/* Clear status register  do immediately*/
			break;

		case 0x40:
			/* program  */
			switch_to_iomode(mflash); // ??????
			mflash->cmd = CMD_PROGRAM;	
			mflash->cycle++;
			break;

		case 0x20:
			/* block erase */
			switch_to_iomode(mflash); // ??????
			mflash->cmd = CMD_BLOCK_ERASE;
			mflash->cycle++;
			break;

		case 0xa7:
			/* erase all unlocked */
			switch_to_iomode(mflash); // ??????
			mflash->cmd = CMD_ERASE_ALL_UNLOCKED;
			mflash->cycle++;
			break;

		case 0x77:
			/* lock bit program */
			switch_to_iomode(mflash); // ??????
			mflash->cmd = CMD_LOCK_BIT_PGM;
			mflash->cycle++;
			break;

		case 0x71:
			/* read lock bit status */
			switch_to_iomode(mflash); // ??????
			mflash->cmd = CMD_READ_LOCK_BIT_STAT;
			mflash->cycle++;
			break;
		default:
			fprintf(stderr,"M16CFlash: Unknown command 0x%04x\n",value);	
			break;
	}
}

static void
flash_write_second(M16CFlash *mflash,uint32_t value,uint32_t mem_addr,int rqlen)
{
	switch(mflash->cmd) {
		case CMD_READ_STATUS: 
		case CMD_PROGRAM:
		case CMD_BLOCK_ERASE:
		case CMD_ERASE_ALL_UNLOCKED:
		case CMD_LOCK_BIT_PGM:
		case CMD_READ_LOCK_BIT_STAT:
			fprintf(stderr,"M16CFlash command %02x not implemented\n",mflash->cmd);
			break;
		default:
			fprintf(stderr,"M16CFlash illegal command state %02x\n",mflash->cmd);
			break;
	}
}

static void
flash_write(void *clientData,uint32_t value, uint32_t mem_addr,int rqlen)
{
	M16CFlash *mflash = (M16CFlash *) clientData;
	switch(mflash->cycle) {
		case 1:
			flash_write_first(mflash,value,mem_addr,rqlen);
			break;
		case 2:
			flash_write_second(mflash,value,mem_addr,rqlen);
			break;
		default:
			fprintf(stderr,"M16C flash emulator bug: cycle %d\n",mflash->cycle);
	}
}
/*
 * -----------------------------------------------------------------------------
 * In MAP_IO mode this flash_read is used, else direct access to mmaped file
 * -----------------------------------------------------------------------------
 */
static uint32_t 
flash_read(void *clientData,uint32_t mem_addr,int rqlen)
{
	fprintf(stderr,"M16CFlash: IO-mode read not implemented\n");
	return 0;
}

static void
Flash_Map(void *module_owner,uint32_t base,uint32_t mapsize,uint32_t flags)
{
        M16CFlash *mflash = module_owner;
        uint8_t *host_mem=mflash->host_mem;
	if(mflash->mode == MAP_READ_ARRAY) {
                flags &= MEM_FLAG_READABLE;
        	Mem_MapRange(base,host_mem,mflash->size,mapsize,flags);
        }

        IOH_NewRegion(base,mapsize,flash_read,flash_write,HOST_BYTEORDER,mflash);
}

static void
Flash_UnMap(void *module_owner,uint32_t base,uint32_t mapsize)
{
        Mem_UnMapRange(base,mapsize);
        IOH_DeleteRegion(base,mapsize);
}

uint32_t
fmr0_read(void *clientData,uint32_t address,int rqlen)
{
	M16CFlash *mflash = (M16CFlash *) clientData;	
	return mflash->fmr0;
}

static void
fmr0_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
	M16CFlash *mflash = (M16CFlash *) clientData;	
#if 0
	if(value & FMR0_CPU_REWRITE) {
		switch_to_iomode(mflash); // ?????
	}
#endif
	mflash->fmr0 = value;
        return;
}

uint32_t
fmr1_read(void *clientData,uint32_t address,int rqlen)
{
	M16CFlash *mflash = (M16CFlash *) clientData;	
	return mflash->fmr1;
}

static void
fmr1_write(void *clientData,uint32_t value,uint32_t address,int rqlen)
{
        return;
}

/*
 * ---------------------------------------------------------
 * The SFR part of the registers for flash is always at the 
 * same fixed location and will never be remapped
 * ---------------------------------------------------------
 */
static void
Flash_SFRMap(M16CFlash *mflash) 
{
	IOH_New8(FMR0,fmr0_read,fmr0_write,mflash);
	IOH_New8(FMR1,fmr1_read,fmr1_write,mflash);

}

BusDevice *
M16CFlash_New(const char *type,const char *flash_name)
{
        M16CFlash *mflash = sg_new(M16CFlash);
        char *imagedir;
        uint32_t flash_size;
        imagedir = Config_ReadVar("global","imagedir");
        flash_size = mflash->size = 192*1024;
        if(imagedir) {
                char *mapfile = alloca(strlen(imagedir) + strlen(flash_name)+20);
                sprintf(mapfile,"%s/%s.img",imagedir,flash_name);
                mflash->disk_image = DiskImage_Open(mapfile,mflash->size,DI_RDWR);
                if(!mflash->disk_image) {
                        fprintf(stderr,"Open disk image failed\n");
                        exit(42);
                }
                mflash->host_mem=DiskImage_Mmap(mflash->disk_image);
        } else {
                mflash->host_mem = sg_calloc(mflash->size);
                memset(mflash->host_mem,0xff,mflash->size);
        }
	mflash->cycle = 1;
        mflash->mode=MAP_READ_ARRAY;
	mflash->cmd = CMD_NONE;

        mflash->bdev.first_mapping=NULL;
        mflash->bdev.Map=Flash_Map;
        mflash->bdev.UnMap=Flash_UnMap;
        mflash->bdev.owner=mflash;
        mflash->bdev.hw_flags=MEM_FLAG_READABLE;
        fprintf(stderr,"Created M16C Flash with size %d\n",flash_size);
	Flash_SFRMap(mflash); /* ???? */
        return &mflash->bdev;
}

