/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#include "player.h"
#include "output.h"
#include "audio.h"
#include "ftime.h"
#include "other.h"
#include "ledbar.h"
#include "bad_list.h"

#ifdef ENABLE_GTK
#include "gtk/graphic.h"
#endif

#define OUTPUT_WAIT_TIMER 5000000.0

#define O_NULL	0
#define O_OPEN	1
#define O_CLOSE	2
#define O_TEST	3

int output_status = O_NULL;

int output_wait_return (void);
void output_create_file (char *f);

/* This is the controller of an output.
 * it controls the value of his status and run the correct functions. 
 * This functions are in his struct and so this controller doesn't know what
 * he realy does. */
void *
output_thread (void *arg)
{
  int ret;
  audio *ao = (audio *) arg;

  pthread_mutex_lock (&ao->mutex);

  while (1)
    {
      pthread_cond_wait (&ao->cond, &ao->mutex);
      ret = 0;

      if (ao->status == OUTPUT_EXIT)
	break;

      switch (ao->status)
	{
	case OUTPUT_OPEN:
	  ret =
	    ao->open (ao, play->ao_rate, play->ao_channels, play->ao_bitrate);
	  break;

	case OUTPUT_UPDATE:
	  if(ao->update)
  	    ao->update(ao);
	  
          ret=OUTPUT_DONE;

	  break;

	case OUTPUT_WRITE:

#ifdef ENABLE_GTK
	  if (!ao->stop)
#endif
	    if (ao->
		write (ao, play->ao_channels, play->ao_bitrate,
		       play->ao_buffer, play->ao_size) < 0)
	      ret = 1;

	  break;

	case OUTPUT_CLOSE:
	  ao->close (ao);
	  break;
	}

      if (ret)
	ao->status = OUTPUT_ERROR;
      else
	ao->status = OUTPUT_DONE;
    }

  pthread_mutex_unlock (&ao->mutex);

  pthread_exit (NULL);
}

/* This function controls the return values of the output controllers */
int
output_wait_return (void)
{
  audio *ao;
  int ret = 0;
  int realtime = 0;
  int norealtime = play->norealtime;
  struct timeval *t = NULL;
  double usec = 0;

  ao = play->output;

  if (play->ao_bitrate && play->ao_rate && play->ao_channels)
    {
      while (ao)
	{
	  if (ao->realtime)
	    realtime = 1;

	  ao = ao->next;
	}

       usec = play->ao_size / (play->ao_channels * (play->ao_bitrate / 8) * play->ao_rate);
    }

  if(norealtime) realtime=0;

  t = timer_start ();

  ao = play->output;

  while (ao)
    {

      if (ao->status == OUTPUT_DONE)
	{
	  pthread_mutex_unlock (&ao->mutex);
	  ao = ao->next;
	}

      else if (ao->status == OUTPUT_ERROR)
	{
	  audio *ao2;

	  pthread_mutex_unlock (&ao->mutex);
	  ao->status = OUTPUT_CLOSE;
	  pthread_cond_signal (&ao->cond);

	  while (realtime
		 && usec ? timer_time_u (t) <
		 (usec + 500000.0) : timer_time_u (t) < OUTPUT_WAIT_TIMER)
	    {
	      pthread_mutex_lock (&ao->mutex);

	      if (ao->status == OUTPUT_DONE || ao->status == OUTPUT_ERROR)
		{
		  pthread_mutex_unlock (&ao->mutex);
		  break;
		}
	      else
		pthread_mutex_unlock (&ao->mutex);
	    }

	  ret = 1;

	  ao2 = ao->next;

	  if ((ao->status == OUTPUT_DONE || ao->status == OUTPUT_ERROR)
	      && ao->audio_type != USE_MONITOR && play->badlist_max_retry)
	    output_bad_list (ao);
	  else
	    output_remove_nb (ao);

	  if (!play->badlist_max_retry && !play->output && !play->nooutput)
	    {
	      msg_error ("No output and no retry, so quit!");
	      events.quit = 1;
	    }

	  ao = ao2;

	}

      /* If realtime... the thread must work in time <= of audio sequence */
      else if (!norealtime && realtime && usec
	       && timer_time_u (t) > (usec + 500000.0))
	{
	  audio *ao2;

	  ret = 1;

	  ao2 = ao->next;
	  msg (_("Realtime problem... I must remove this output! %f %f"),
	       timer_time_u (t), usec);

	  pthread_mutex_unlock (&ao->mutex);

	  if (ao->audio_type != USE_MONITOR && play->badlist_max_retry)
	    output_bad_list_forced (ao);
	  else
	    output_remove_nb (ao);

	  ao = ao2;

	}

      /* No realtime... */
      else if (timer_time_u (t) > OUTPUT_WAIT_TIMER)
	{
	  audio *ao2;

	  ret = 1;

	  ao2 = ao->next;

	  pthread_mutex_unlock (&ao->mutex);

	  if (ao->audio_type != USE_MONITOR && play->badlist_max_retry)
	    output_bad_list_forced (ao);
	  else
	    output_remove_nb (ao);

	  ao = ao2;

	}
      else
	{
	  pthread_mutex_unlock (&ao->mutex);
	  usleep (500);
	}
    }

  if(usec && play->timeforced && timer_time_u(t)<usec)
     usleep(usec-timer_time_u(t));
  
  timer_clear (t);

  return ret;
}

/* The player thread runs this functions: 
 *   - output_open
 *   - output_write
 *   - output_update
 *   - output_check
 *   - output_close
 *   - output_info
 * This function set the parameter of the controllers of outputs .*/

void
output_open (long rate, int channels, int bitrate)
{
  audio *ao;

  if (output_status == O_TEST)
    {
      if (rate != play->ao_rate || channels != play->ao_channels
	  || bitrate != play->ao_bitrate)
	{
	  output_close ();
	}
      else
	{
	  output_status = O_OPEN;
	  return;
	}

    }

  output_status = O_OPEN;

  output_lock ();

  ao = play->output;
  play->ao_rate = rate;
  play->ao_channels = channels;
  play->ao_bitrate = bitrate;

  /* All outputs with status == OUTPUT_OPEN! */
  while (ao)
    {
      ao->status = OUTPUT_OPEN;

      pthread_cond_signal (&ao->cond);

      ao = ao->next;
    }

  /* If some ouput return error, i remove it. */
  if (output_wait_return ())
    {
      output_unlock ();
      msg_error (_("One output has some problems. I remove it."));

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	output_list_refresh ();
#endif

      return;
    }

  output_unlock ();

#ifdef ENABLE_GTK
  if (play->graphic && play->go_graphic)
    output_list_refresh ();
#endif

  output_update();
}

/* Output check */
int
output_check (void)
{
  audio *ao;

  output_lock ();

  ao = play->output;

  while (ao)
    {
      if (ao->check && !ao->check (ao))
	{
	  output_unlock ();

	  return 0;
	}

      ao = ao->next;
    }

  output_unlock ();

  return 1;
}

/* Output_update */
int
output_update(void)
{
  audio *ao;
  int k;

  output_lock ();

  ao = play->output;

  while (ao)
    {
      ao->status = OUTPUT_UPDATE;

      pthread_cond_signal (&ao->cond);

      ao = ao->next;
    }

  if ((k = output_wait_return ()))
    {
      output_unlock ();
      msg_error (_("One output has some problems. I remove it."));

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	output_list_refresh ();
#endif

      return k;
    }

  output_unlock ();

  return k;
}

/* Output_write */
int
output_write (void *buffer, size_t size)
{
  audio *ao;
  int k;

  if (play->trimming)
    {
      signed short *s = (signed short *) buffer;
      int i = 0;

      while (i < size / sizeof (signed short))
	{
	  if (s[i] != 0)
	    break;

	  i++;
	}

      if (i && i < size / sizeof (signed short))
	{
	  char *b = (char *) buffer;
	  int s = (int) size;

	  s -= i * sizeof (short);
	  b += i * sizeof (short);

	  size = (size_t) s;
	  buffer = (void *) b;
	}
      else if (i == size / sizeof (signed short))
	return 0;
    }

  output_lock ();

  ao = play->output;
  play->ao_buffer = buffer;
  play->ao_size = size;

  audio_write (play->ao_channels, buffer, size);

  while (ao)
    {
      ao->status = OUTPUT_WRITE;

      pthread_cond_signal (&ao->cond);

      ao = ao->next;
    }

  ledbar_check ();

  if ((k = output_wait_return ()))
    {
      output_unlock ();
      msg_error (_("One output has some problems. I remove it."));

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	output_list_refresh ();
#endif

      return k;
    }

  output_unlock ();

  return k;
}

/* output_check_close */
void
output_check_close (void)
{
  if (play->splitoutput)
    output_close ();
  else
    {

      if (output_status == O_OPEN || output_status == O_TEST)
	output_close ();
    }
}

/* output_forced_close */
void
output_forced_close (void)
{
  output_status = O_TEST;
  output_close ();
}

/* output_close */
void
output_close (void)
{
  audio *ao;
  int k;

  if (!play->splitoutput)
    {

      if (output_status == O_NULL)
	return;

      if (output_status == O_OPEN)
	{
	  if (!events.quit)
	    {
	      output_status = O_TEST;
	      return;
	    }
	}
    }

  output_status = O_CLOSE;

  output_lock ();

  ao = play->output;
  while (ao)
    {
      ao->status = OUTPUT_CLOSE;

      pthread_cond_signal (&ao->cond);

      ao = ao->next;
    }

  if ((k = output_wait_return ()))
    {
      output_unlock ();
      msg_error (_("One output has some problems. I remove it."));

#ifdef ENABLE_GTK
      if (play->graphic && play->go_graphic)
	output_list_refresh ();
#endif

      return;
    }

  output_unlock ();
}

/* output_info */
void
output_info (void)
{
  audio *ao;

  output_lock ();

  ao = play->output;
  while (ao)
    {
      ao->info (ao);
      ao = ao->next;
    }

  output_unlock ();
}

/* Close all outputs! */
void
output_exit (void)
{
  audio *ao, *tmp;

  output_lock ();

  ao = play->output;

  while (ao)
    {
      ao->status = OUTPUT_EXIT;
      pthread_cond_signal (&ao->cond);

      pthread_join (ao->th, NULL);

      tmp = ao;
      ao = ao->next;

      if (tmp->data)
	free (tmp->data);
      if (tmp->audio_dev)
	free (tmp->audio_dev);
      free (tmp);
    }

  output_unlock ();
}

/* Add output to the list of ouput */
audio *
output_add (int what)
{

  audio *output;

  if (!(output = (audio *) malloc (sizeof (audio))))
    fatal (_("Error: memory."));

  output->realtime = 0;

  output->next = NULL;
  output->data = NULL;
  output->open = NULL;
  output->info = NULL;
  output->write = NULL;
  output->close = NULL;
  output->check = NULL;

  output->audio_dev = NULL;

#ifdef ENABLE_GTK
  output->stop = 0;
#endif

  output->audio_type = what;
  if (!output->audio_type)
    msg (_("Searching your audio interface..."));

  output_lock ();

  output->next = play->output;
  play->output = output;

  output_unlock ();

  return output;
}

/* BadList */
void
output_bad_list (audio * ao)
{
  audio *tmp, *tmp2 = NULL;

  bad_list_insert (ao);

  tmp = play->output;
  if (!tmp)
    fatal (_("Internal error."));

  while (tmp)
    {
      if (tmp == ao)
	{
	  if (tmp2)
	    tmp2->next = tmp->next;
	  else
	    play->output = tmp->next;

	  tmp->status = OUTPUT_EXIT;
	  pthread_cond_signal (&tmp->cond);
	  pthread_join (tmp->th, NULL);

	  break;
	}

      tmp2 = tmp;
      tmp = tmp->next;
    }
}

/* Remove without lock */
void
output_remove_nb (audio * ao)
{
  audio *tmp, *tmp2 = NULL;

  tmp = play->output;
  if (!tmp)
    fatal (_("Internal error."));

  while (tmp)
    {
      if (tmp == ao)
	{
	  if (tmp2)
	    tmp2->next = tmp->next;
	  else
	    play->output = tmp->next;

	  while (tmp->status != OUTPUT_DONE && tmp->status != OUTPUT_ERROR)
	    sched_yield ();

	  tmp->status = OUTPUT_EXIT;
	  pthread_cond_signal (&tmp->cond);
	  pthread_join (tmp->th, NULL);

	  if ((0
#ifdef ENABLE_LAME
	       || tmp->audio_type == USE_LAME
#endif
#ifdef ENABLE_OGG
	       || tmp->audio_type == USE_OGG
#endif
#ifdef ENABLE_STREAMING
#ifdef ENABLE_ICECAST
	       || tmp->audio_type == USE_ICECAST
#endif
#ifdef ENABLE_ICECAST2
	       || tmp->audio_type == USE_ICECAST2
#endif
#ifdef ENABLE_SHOUTCAST
	       || tmp->audio_type == USE_SHOUTCAST
#endif
#endif
	      ) && tmp->data)
	    free (tmp->data);

	  if (tmp->audio_dev)
	    free (tmp->audio_dev);

	  free (tmp);

	  break;
	}

      tmp2 = tmp;
      tmp = tmp->next;
    }
}

/* Remove with lock */
void
output_remove (audio * ao)
{
  audio *tmp, *tmp2 = NULL;

  tmp = play->output;
  if (!tmp)
    fatal (_("Internal error."));

  output_lock ();

  while (tmp)
    {
      if (tmp == ao)
	{
	  if (tmp2)
	    tmp2->next = tmp->next;
	  else
	    play->output = tmp->next;

	  tmp->status = OUTPUT_EXIT;
	  pthread_cond_signal (&tmp->cond);
	  pthread_join (tmp->th, NULL);

	  if ((0
#ifdef ENABLE_LAME
	       || tmp->audio_type == USE_LAME
#endif
#ifdef ENABLE_OGG
	       || tmp->audio_type == USE_OGG
#endif
#ifdef ENABLE_STREAMING
#ifdef ENABLE_ICECAST
	       || tmp->audio_type == USE_ICECAST
#endif
#ifdef ENABLE_ICECAST2
	       || tmp->audio_type == USE_ICECAST2
#endif
#ifdef ENABLE_SHOUTCAST
	       || tmp->audio_type == USE_SHOUTCAST
#endif
#endif
	      ) && tmp->data)
	    free (tmp->data);

	  if (tmp->audio_dev)
	    free (tmp->audio_dev);

	  free (tmp);

	  break;
	}

      tmp2 = tmp;
      tmp = tmp->next;
    }

  output_unlock ();
}

/* Bad_List forced with lock */
void
output_bad_list_forced (audio * ao)
{
  audio *tmp, *tmp2 = NULL;

  bad_list_insert (ao);

  tmp = play->output;
  if (!tmp)
    fatal (_("Internal error."));

  while (tmp)
    {
      if (tmp == ao)
	{
	  if (tmp2)
	    tmp2->next = tmp->next;
	  else
	    play->output = tmp->next;

	  tmp->status = OUTPUT_EXIT;
	  pthread_cond_signal (&tmp->cond);
	  /* No waiting for return value */

	  break;
	}

      tmp2 = tmp;
      tmp = tmp->next;
    }
}

/* Many outputs write on a file (encode mp3/ogg, dump stream...). This func
 * returns the name of the parsed file. For examples if hello.mp3 exists, this
 * function returns hello01.mp3. */
char *
output_check_file (char *fl)
{
  struct stat st;
  char *t, *post = NULL, *pre = NULL;
  int a;

  /* I must use a mutex because the function can be exec by many outpus
   * in the same time. So... */
  static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

  pthread_mutex_lock (&mutex);


  if (lstat (fl, &st))
    {
      if (!(t = strdup (fl)))
	fatal ("Error: memory.");

      output_create_file (t);

      pthread_mutex_unlock (&mutex);
      return t;
    }

  for (a = strlen (fl); a > 0; a--)
    {
      if (*(fl + a) == '.')
	{
	  if (!(post = strdup (fl + a)))
	    fatal ("Error: memory.");
	  if (!(pre = strdup (fl)))
	    fatal ("Error: memory.");
	  *(pre + a) = 0;
	  break;
	}
    }

  if (!(t = (char *) malloc (sizeof (char) * (strlen (fl) + 3))))
    fatal ("Error: memory.");

  a = 0;
  while (a < 1000)
    {
      if (pre)
	sprintf (t, "%s%.3d%s", pre, a, post);
      else
	sprintf (t, "%s%.3d", fl, a);

      if (lstat (t, &st))
	{

	  output_create_file (t);

	  pthread_mutex_unlock (&mutex);
	  return t;
	}

      a++;
    }

  free (t);

  if (!(t = strdup (fl)))
    fatal ("Error: memory.");

  output_create_file (t);
  pthread_mutex_unlock (&mutex);

  return t;
}

void
output_create_file (char *f)
{
  FILE *fl;

  if (!(fl = fopen (f, "w")))
    return;
  fclose (fl);
}

/* NPTL problem for synchronize thread */
pthread_mutex_t output_sync = PTHREAD_MUTEX_INITIALIZER;

void
output_lock (void)
{
  pthread_mutex_lock (&output_sync);
  pthread_mutex_lock (&play->m_output);
  pthread_mutex_unlock (&output_sync);

}

void
output_unlock (void)
{
  pthread_mutex_unlock (&play->m_output);
}

/* EOF */
