<?php
/**
 * $Horde: forwards/lib/Driver/forwards.php,v 1.7.2.6 2003/02/05 01:57:14 ericr Exp $
 *
 * Copyright 2001-2003 Eric Rostetter <eric.rostetter@physics.utexas.edu>
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/bsdl.php.
 *
 * Forwards_Driver_forwards:: implements the Forwards_Driver API for ftp driven
 * dot-forward compliant mail servers.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @version $Revision: 1.7.2.6 $
 * @since   Forwards 2.1
 * @package forwards
 */

class Forwards_Driver_forwards extends Forwards_Driver {

    /** file pointer to the tmp file we create containing the forwarding
        address */
    var $fp;

    /** The error string returned to the user if an error occurs. */
    var $err_str;

    /** Hash containing configuration data. */
    var $params;

/**
 * Constructs a new ftp dot-forward Forwards_Driber object.
 *
 * @param array  $params    A hash containing connection parameters.
 */
function Forwards_Driver_forwards($params = array())
{
    $this->params = $params;
}

/**
 *
 * Create a temporary file and write some data to it.
 *
 * @param   $contents    What to write into the file.
 *
 * @return  mixed        False if file creation failed for any reason,
 *                       the file specification created on success.
 */

function _writeTempFile($contents)
{
        $conf = &$GLOBALS['conf'];

        $oldUmask = umask(0077);
        $myTempFile = tempnam(@$conf['tmpdir'], "forwards");

        if ($myTempFile == false) {
            $this->err_str = _("tempnam() failed!");
            return false;
        }
        if (!($fileHandle = fopen($myTempFile, "w"))) {
            $this->err_str = _("Could not create file");
            return false;
        }
        if (!(fputs($fileHandle, $contents))) {
            $this->err_str = _("Could not write to file");
            return false;
        }
        fclose($fileHandle);
        umask($oldUmask);
        return $myTempFile;
}

/**
 * Delete a file via ftp.
 *
 * @param   $user    The username to login as via ftp.
 * @param   $realm   The realm (domain) of the user, or "default" if none.
 * @param   $pass    The password to login with via ftp.
 * @param   $file    The file specification to delete on the remote machine.
 *
 * @return  boolean  True or False depending on the success of the delete.
 */

function _ftpDeleteFile($user, $realm, $pass, $file)
{
        if (!($ftpConnection = @ftp_connect($this->params[$realm]['host'],
                                            $this->params[$realm]['port']))) {
            $this->err_str = _("Server connection failed");
            return false;
        }

        if (!(@ftp_login($ftpConnection, $user, $pass))) {
            $this->err_str = _("Could not login - check password!");
            @ftp_quit($ftpConnection);
            return false;
        }
        // should probably do a cwd to their home, but we don't...
        //ftp_chdir($ftpConnection, "/home/eric/");
        if (!(@ftp_delete($ftpConnection, $file))) {
            $this->err_str = _("Error deleting $file file -- Maybe you didn't have a forward installed?");
            @ftp_quit($ftpConnection);
            return false;
        }
        @ftp_quit($ftpConnection);
        return true;
}

/**
 * Copy a file between two machines via ftp.
 *
 * @param   $user    The username to login as via ftp.
 * @param   $realm   The realm (domain) of the user, or "default" if none.
 * @param   $pass    The password to login with via ftp.
 * @param   $src     The source file specification to copy.
 * @param   $dst     The destination file specification to write.
 * @param   $mode    The ftp transfer mode (binary or text).
 *
 * @return  boolean  True or False depending on success of the copy.
 */

function _ftpCopyFile($user, $realm, $pass, $src, $dst, $mode=FTP_ASCII)
{
        if (!($ftpConnection = @ftp_connect($this->params[$realm]['host'],
                                            $this->params[$realm]['port']))) {
            $this->err_str = _("Server connection failed");
            return false;
        }

        if (!(@ftp_login($ftpConnection, $user, $pass))) {
            $this->err_str = _("Could not login - check password!");
            @ftp_quit($ftpConnection);
            return false;
        }
        // should probably do a cwd to their home, but we don't...
        //ftp_chdir($ftpConnection, "/home/eric/");
        // ftp_put won't overwrite existing file, so nuke it unconditionally
        @ftp_delete($ftpConnection, $dst);
        if (!(@ftp_put($ftpConnection, $dst, $src, $mode))) {
            $this->err_str = _("Could not set forwarding!");
            @ftp_quit($ftpConnection);
            return false;
        }
        @ftp_quit($ftpConnection);
        return true;
}

/**
 * Get a remote file via ftp.
 *
 * @param   $user    The username to login as via ftp.
 * @param   $realm   The realm (domain) of the user, or "default" if none.
 * @param   $pass    The password to login with via ftp.
 * @param   $src     The source file specification to copy.
 * @param   $dst     The destination file specification to write.
 * @param   $mode    The ftp transfer mode (binary or text).
 *
 * @return  boolean  True or False depending on success of the copy.
 */

function _ftpGetFile($user, $realm, $pass, $src, $dst, $mode=FTP_ASCII)
{
        if (!($ftpConnection = @ftp_connect($this->params[$realm]['host'],
                                            $this->params[$realm]['port']))) {
            $this->err_str = _("Server connection failed");
            return "";
        }

        if (!(@ftp_login($ftpConnection, $user, $pass))) {
            $this->err_str = _("Could not login - check password!");
            @ftp_quit($ftpConnection);
            return "";
        }
        // should probably do a cwd to their home, but we don't...
        //ftp_chdir($ftpConnection, "/home/eric/");
        if (!(@ftp_get($ftpConnection, $dst, $src, $mode))) {
            $this->err_str = _("Could not get forwarding!");
            @ftp_quit($ftpConnection);
            return "";
        }
        @ftp_quit($ftpConnection);
        return implode('',file($dst));
}

/**
 * Check if the realm has a specific configuration.  If not, try to fall
 * back on the default configuration.  If still not a valid configuration
 * then exit with an error.
 *
 * @param string    $realm      The realm of the user, or "default" if none.
 *                              Note: passed by reference so we can change
 *                              it's value!
 *
 */

function check_config(&$realm) {

    // If no realm passed in, or no host config for the realm passed in,
    // then we fall back to the default realm

    if ( empty($realm) || empty($this->params[$realm]['host']) ) {
       $realm = "default";
    }

    // If still no host/port, then we have a misconfigured module
    if (empty($this->params[$realm]['host']) ||
        empty($this->params[$realm]['port']) ) {
            $this->err_str = _("The module is not properly configured!");
            return false;
    }
    return true;
}

/**
 * Begins forwarding of mail for a user.
 *
 * @param string    $user       The username to enable forwarding for.
 * @param string    $realm      The realm of the user, or "default" if none.
 * @param string    $target     The email address that mail should be forwarded to.
 * @param optional boolean $keeplocal A flag that, when true, causes a copy of
 *                                    forwarded email to be kept in the local
 *                                    mailbox.
 *
 * @return boolean  Returns true on success, false on error.
 */

    function enableForwarding($user, $realm = 'default', $pass, $where,
                              $metoo = false) {

        // Make sure the configuration file is correct
        if ( !$this->check_config($realm) ) {
            return false;
        }

        // Create the forwarding information
        $address = $this->_make_email_address($where, $realm);
        if ($address === false) {
           return false;
        }

        // Append the user if they want to save a copy to themselves...
        if ($metoo == "on") {
            $address .= ", \\$user";
        }

        $myFile = $this->_writeTempFile("$address");
        if (!file_exists($myFile)) return false;
        $status = $this->_ftpCopyFile($user, $realm, $pass, $myFile, ".forward");
        unlink($myFile);
        if (! $status) return false;

        return true;
    }

/**
 * Stops forwarding of mail for a user.
 *
 * @param string    $user       The username of the user to disable forwarding.
 * @param string    $realm      The realm of the user.
 * @param string    $pass       The password of the user.
 *
 * @return boolean  Returns true on success, false on error.
 */

    function disableForwarding($user, $realm = 'default', $pass) {
        if ( !$this->check_config($realm) ) {
           return false;
        }

        if (! $this->_ftpDeleteFile($user, $realm, $pass, ".forward") ) {
             return false;
        }
        return true;
    }

/**
 * Retrieves current target of mail redirection
 *
 * @param string    $user       The username of the user to get forward of.
 * @param string    $realm      The realm of the user.
 *
 * @return mixed    A string of current forwarding mail address, or false.
 */

    function currentTarget($user, $realm = 'default', $pass) {

        // Make sure the configuration file is correct
        if ( !$this->check_config($realm) ) {
            return false;
        }

        $myFile = $this->_writeTempFile($user);
        if (!file_exists($myFile)) return "";
	@unlink($myFile);

        $status = $this->_ftpGetFile($user, $realm, $pass, ".forward", $myFile);
        @unlink($myFile);
        return $status;
    }

/**
 * Retrieves current state of mail redirection
 *
 * @param string    $user       The username to check forwarding for.
 * @param string    $realm      The realm of the user.
 * @return boolean  Returns true/false based on if forwarding is enabled.
 */

/* This function is implemented poorly, and should be rewritten */

    function isEnabledForwarding($user, $realm, $password) {
        return $this->currentTarget($user, $realm, $password) != false;
    }

}
?>
