<?php
/* $Horde: vacation/lib/Driver.php,v 1.2.2.2 2003/01/20 06:02:57 ericr Exp $
 *
 * Copyright 2001-2003 Eric Rostetter and Mike Cochrane
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/bsdl.php.
 * Vacation_Driver:: defines an API for implementing vacation backends
 * for the vacation module.
 *
 * Vacation_Driver:: defines an API for implementing vacation backends
 * for the vacation module.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @package Vacation
 */

class Vacation_Driver {
    /**
     * Attempts to return a concrete Vacation_Driver instance based on $driver.
     *
     * @param string    $driver     The type of concrete Vacation_Driver subclass
     *                              to return.  The is based on the vacation
     *                              driver ($driver).  The code is dynamically
     *                              included.
     *
     * @param array     $params     (optional) A hash containing any additional
     *                              configuration or connection parameters a
     *                              subclass might need.
     *
     * @return mixed    The newly created concrete Vacation_Driver instance, or
     *                  false on an error.
     */
    function &factory($driver, $params = array())
    {
        $driver = strtolower(basename($driver));
        include_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Vacation_Driver_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete Vacation_Driver instance
     * based on $driver.  It will only create a new instance if no
     * Vacation_Driver instance with the same parameters currently exists.
     *
     * This should be used if multiple storage sources are required.
     *
     * This method must be invoked as: $var = &Vacation_Driver::singleton()
     *
     * @param string    $driver     The type of concrete Vacation_Driver subclass
     *                              to return.  The is based on the vacation
     *                              driver ($driver).  The code is dynamically
     *                              included.
     *
     * @param array     $params     (optional) A hash containing any additional
     *                              configuration or connection parameters a
     *                              subclass might need.
     *
     * @return mixed    The created concrete Vacation_Driver instance, or false
     *                  on error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Vacation_Driver::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Setup vacation notices for a user.
     *
     * @param string    $user       The username to enable vacation notices for.
     * @param string    $realm      The realm of the user.
     * @param string    $pass       The password for the user.
     * @param string    $message    The text of the vacation notice.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function set_vacation($user, $realm = 'default', $pass = '', $message)
    {
        return false;
    }

    /**
     * Disables vacation notices for a user.
     *
     * @param string    $user       The user to disable vacation notices for.
     * @param string    $realm      The realm of the user.
     * @param string    $pass       The password of the user.
     *
     * @return boolean  Returns true on success, false on error.
     */
    function unset_vacation($user, $realm = 'default', $pass = '')
    {
        return false;
    }

    /**
     * Retrieves current state of vacation notices for a user.
     *
     * @param string    $user       The user to check vacation status of.
     * @param string    $realm      The realm of the user.
     * @param string    $pass       The password of the user.
     *
     * @return mixed    Returns true if vacation  is enabled for the user
     *                  or false if vacation is currently disabled.
     */
    function isEnabled($user, $realm = 'default', $password)
    {
        return false;
    }

    /**
     * Retrieves current vacation setup.
     *
     * @param string    $user       The user to fetch the setup from.
     * @param string    $realm      The realm of the user.
     * @param string    $password   The password of the user.
     *
     * @return string   Returns a string containing the current forwarding mail address or
     *                  false on error.
     */
    function currentMessage($user, $realm = 'default', $password)
    {
        return false;
    }

   /**
     * Parse an email address list and return it in a known standard form.
     * This will attempt to add the domain (realm) to unqualified addresses
     * if the realm is non-blank and not 'default'.
     *
     * @param   $user       The email address.
     * @param   $realm      The domain/realm to add if none is present.
     * @return  string      The email address(es) on success, false on error.
     */
    function _make_email_address($user, $realm)
    {
        $domain = ($realm != 'default') ? $realm : '';
        $email = '';

        require_once 'Mail/RFC822.php';
        $parsed_email = Mail_RFC822::parseAddressList($user, $domain, FALSE);
        if (is_array($parsed_email) && count($parsed_email) > 0) {
            for ($i=0; $i < count($parsed_email); $i++) {
               $email .= !empty($email) ? ',' : '';
               if (is_object($parsed_email[$i])) {
                 $email .= $parsed_email[$i]->mailbox;
                 $email .= !empty($parsed_email[$i]->host)
                        ? '@' . $parsed_email[$i]->host
                        : '';
              } else {
                 $email .= $parsed_email[$i];
              }
            }
        } else {
            $this->err_str = _("Can't parse your email address");
            $email = false;
        }

        return $email;
    }

}
