
#include "calibrate.h"
#include "config.h"
#include "tools.h"
#include "backend_plugins.h"
#include "replaygain_plugins.h"

#include <qfile.h>

#include <klocale.h>
#include <ktempfile.h>
#include <kmessagebox.h>


Calibrate::Calibrate()
{
    connect(&pPrepare,SIGNAL(processExited(KProcess*)),this,SLOT(prepareExit(KProcess*)));
    connect(&pPrepare,SIGNAL(receivedStdout(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));
    connect(&pPrepare,SIGNAL(receivedStderr(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));

    connect(&pDecode,SIGNAL(processExited(KProcess*)),this,SLOT(decodeExit(KProcess*)));
    connect(&pDecode,SIGNAL(receivedStdout(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));
    connect(&pDecode,SIGNAL(receivedStderr(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));

    connect(&pEncode,SIGNAL(processExited(KProcess*)),this,SLOT(encodeExit(KProcess*)));
    connect(&pEncode,SIGNAL(receivedStdout(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));
    connect(&pEncode,SIGNAL(receivedStderr(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));

    connect(&pReplayGain,SIGNAL(processExited(KProcess*)),this,SLOT(replayGainExit(KProcess*)));
    connect(&pReplayGain,SIGNAL(receivedStdout(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));
    connect(&pReplayGain,SIGNAL(receivedStderr(KProcess*,char*,int)),this,SLOT(processOutput(KProcess*,char*,int)));
}

Calibrate::~Calibrate()
{
    //cleanUp(); Doesn't effect anything or does it?
}

void Calibrate::start(QString fileName, QString fileFormat)
{
    format=fileFormat;
    file=fileName;

    kTempOriFile=new KTempFile(QString::null,".wav");
    if( kTempOriFile->status() != 0 ) {
        cleanUp();
        emit exited(1);
    }
    kTempOriFile->setAutoDelete(true);
    kTempOriFile->close();
    tempOriFile=kTempOriFile->name();

    kTempEncFile=new KTempFile(QString::null,".wav."+format);
    if( kTempEncFile->status() != 0 ) {
        cleanUp();
        emit exited(1);
    }
    kTempEncFile->setAutoDelete(true);
    kTempEncFile->close();
    tempEncFile=kTempEncFile->name();

    kTempDecFile=new KTempFile(QString::null,".wav."+format+".wav");
    if( kTempDecFile->status() != 0 ) {
        cleanUp();
        emit exited(1);
    }
    kTempDecFile->setAutoDelete(true);
    kTempDecFile->close();
    tempDecFile=kTempDecFile->name();

    QString sFormat=tools.fileExtension(file);
    length=tools.timeByFile(file);

    BackendPlugins::PluginStructure plugin=backendPlugins.plugin(prefs.fileFormat(sFormat).decoder);

    if( plugin.info.name == i18n("Unnamed") ) {
        for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
        {
            if( (*it).dec.synonymous_formats.findIndex(sFormat) != -1 ) {
                plugin=*it;
                break;
            }
        }
    }

    if( sFormat == "wav" )
    {
        reverse=false;

        pPrepare.clearArguments();

        pPrepare << "cp";
        pPrepare << file;
        pPrepare << tempOriFile;

        procParam="cp "+file+" "+tempOriFile;

        bCanceled=false;
        pPrepare.start(KProcess::NotifyOnExit,KProcess::AllOutput);
    }
    else if( sFormat == format )
    {
        reverse=true;

        decode();
    }
    else
    {
        reverse=false;

        QStringList params;
        QString param, paramSplinter;

        pPrepare.clearArguments();

        param=plugin.dec.bin;
        if( plugin.dec.param ) param.append(" "+plugin.dec.param);

        param.append(" "+plugin.dec.inOutFiles);

        params=QStringList::split(' ',param);
        procParam="";

        for( QStringList::Iterator it=params.begin(); it!=params.end(); ++it )
        {
            paramSplinter=*it;
            paramSplinter.replace("%i",file);
            paramSplinter.replace("%o",tempOriFile);
            procParam.append(paramSplinter+" ");
            pPrepare << paramSplinter;
        }

        bCanceled=false;
        pPrepare.start(KProcess::NotifyOnExit,KProcess::AllOutput);
    }
}

void Calibrate::encode()
{
    QString inFile, outFile;
    QString sStrength;
    char cs[32];
    int is;
    float fs;

    if(reverse) {
        inFile=tempDecFile;
        outFile=tempEncFile;
    }
    else {
        inFile=tempOriFile;
        outFile=tempEncFile;
    }

    QStringList params;
    QString param, paramSplinter;

    pEncode.clearArguments();

    BackendPlugins::PluginStructure plugin=backendPlugins.plugin(prefs.fileFormat(format).encoder);

    if( plugin.info.name == i18n("Unnamed") ) {
        cleanUp();
        emit exited(0);
        return;
    }

    param=QString::null;
    if( plugin.enc.param ) param.append(" "+plugin.enc.param);
    if( plugin.enc.silent_param ) param.append(" "+plugin.enc.silent_param);
    if( plugin.enc.overwrite ) param.append(" "+plugin.enc.overwrite);
    if( plugin.enc.replaygain.enabled && plugin.enc.replaygain.avoid ) param.append(" "+plugin.enc.replaygain.avoid);

    if( plugin.enc.strength.enabled ) {
        param.append(" "+plugin.enc.strength.param);
        if( plugin.enc.strength.profiles.empty() ) {
            if( plugin.enc.strength.step < 1 ) {
                if( plugin.enc.strength.range_max >= plugin.enc.strength.range_min )
                    fs = prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step;
                else
                    fs = plugin.enc.strength.range_min - prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step;
                //fs -= fs%plugin.enc.strength.step;
                sprintf( cs, "%.2f", fs );
            }
            else {
                if( plugin.enc.strength.range_max >= plugin.enc.strength.range_min )
                    is = (int)(prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step);
                else
                    is = (int)(plugin.enc.strength.range_min - prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step);
                //is -= is%plugin.enc.strength.step;
                sprintf( cs, "%i", is );
            }
            sStrength=QString(cs);
            if( plugin.enc.strength.seperator != '.' ) sStrength.replace(QChar('.'),plugin.enc.strength.seperator);
        }
        else {
            QStringList::Iterator it=plugin.enc.strength.profiles.at((int)prefs.fileFormat(format).compressionLevel);
            sStrength=*it;
        }
    }

    if( plugin.enc.inOutFiles.find("%p") != -1 ) {
        QString t_str=plugin.enc.inOutFiles;
        t_str.replace("%p",param);
        param=plugin.enc.bin+" "+t_str;
    }
    else {
        param=plugin.enc.bin+" "+param+" "+plugin.enc.inOutFiles;
    }

    params=QStringList::split(' ',param);
    procParam="";

    for( QStringList::Iterator it=params.begin(); it!=params.end(); ++it )
    {
        paramSplinter=*it;
        paramSplinter.replace("%i",inFile);
        paramSplinter.replace("%o",outFile);
        paramSplinter.replace("%c",sStrength);
        procParam.append(paramSplinter+" ");
        pEncode << paramSplinter;
    }

    bCanceled=false;
    time.start();
    pEncode.start(KProcess::NotifyOnExit,KProcess::AllOutput);
}

void Calibrate::decode()
{
    QString inFile, outFile;

    if(reverse) {
        inFile=file;
        outFile=tempDecFile;
    }
    else {
        inFile=tempEncFile;
        outFile=tempDecFile;
    }

    QStringList params;
    QString param, paramSplinter;

    pDecode.clearArguments();

    BackendPlugins::PluginStructure plugin=backendPlugins.plugin(prefs.fileFormat(format).decoder);

    if( plugin.info.name == i18n("Unnamed") ) {
        for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
        {
            if( (*it).dec.synonymous_formats.findIndex(format) != -1 ) {
                plugin=*it;
                break;
            }
        }
    }
    if( plugin.info.name == i18n("Unnamed") ) {
        cleanUp();
        emit exited(0);
        return;
    }

    param=QString::null;
    if( plugin.dec.param ) param.append(" "+plugin.dec.param);
    if( plugin.dec.silent_param ) param.append(" "+plugin.dec.silent_param);
    if( plugin.dec.overwrite ) param.append(" "+plugin.dec.overwrite);

    if( plugin.dec.inOutFiles.find("%p") != -1 ) {
        QString t_str=plugin.dec.inOutFiles;
        t_str.replace("%p",param);
        param=plugin.dec.bin+" "+t_str;
    }
    else {
        param=plugin.dec.bin+" "+param+" "+plugin.dec.inOutFiles;
    }

    params=QStringList::split(' ',param);
    procParam="";

    for( QStringList::Iterator it=params.begin(); it!=params.end(); ++it )
    {
        paramSplinter=*it;
        paramSplinter.replace("%i",inFile);
        paramSplinter.replace("%o",outFile);
        procParam.append(paramSplinter+" ");
        pDecode << paramSplinter;
    }

    bCanceled=false;
    time.start();
    pDecode.start(KProcess::NotifyOnExit,KProcess::AllOutput);
}

void Calibrate::replayGain()
{
    QString inFile, outFile;
    QString sStrength;
    char cs[32];
    int is;
    float fs;

    QStringList params;
    QString param, paramSplinter;

    pReplayGain.clearArguments();

    if( backendPlugins.plugin(prefs.fileFormat(format).encoder).enc.replaygain.enabled )
    {
        inFile=tempOriFile;
        outFile=tempEncFile;

        BackendPlugins::PluginStructure plugin=backendPlugins.plugin(prefs.fileFormat(format).encoder);

        param=plugin.enc.bin;
        if( plugin.enc.param ) param.append(" "+plugin.enc.param);
        if( plugin.enc.silent_param ) param.append(" "+plugin.enc.silent_param);
        if( plugin.enc.overwrite ) param.append(" "+plugin.enc.overwrite);
        if( plugin.enc.replaygain.use ) param.append(" "+plugin.enc.replaygain.use);

        if( plugin.enc.strength.enabled ) {
            param.append(" "+plugin.enc.strength.param);
            if( plugin.enc.strength.step < 1 ) {
                if( plugin.enc.strength.range_max >= plugin.enc.strength.range_min )
                    fs = prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step;
                else
                    fs = plugin.enc.strength.range_min - prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step;
                //fs -= fs%plugin.enc.strength.step;
                sprintf( cs, "%.2f", fs );
            }
            else {
                if( plugin.enc.strength.range_max >= plugin.enc.strength.range_min )
                    is = (int)(prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step);
                else
                    is = (int)(plugin.enc.strength.range_min - prefs.fileFormat(format).compressionLevel * plugin.enc.strength.step);
                //is -= is%plugin.enc.strength.step;
                sprintf( cs, "%i", is );
            }
            sStrength=QString(cs);
            if( plugin.enc.strength.seperator != '.' ) sStrength.replace(QChar('.'),plugin.enc.strength.seperator);
        }

        param.append(" "+plugin.enc.inOutFiles); // TODO verschieben von parametern nicht möglich
    }
    else if( replayGainPlugins.plugin(prefs.fileFormat(format).replaygain).info.name != i18n("Unnamed") )
    {
        inFile=tempEncFile;

        ReplayGainPlugins::PluginStructure plugin=replayGainPlugins.plugin(prefs.fileFormat(format).replaygain);

        param=plugin.replaygain.bin;
        if( plugin.replaygain.param ) param.append(" "+plugin.replaygain.param);
        if( plugin.replaygain.track ) param.append(" "+plugin.replaygain.track);
        if( plugin.replaygain.silent_param ) param.append(" "+plugin.replaygain.silent_param);

        param.append(" "+plugin.inFiles);
    }

    params=QStringList::split(' ',param);
    procParam="";

    for( QStringList::Iterator it=params.begin(); it!=params.end(); ++it )
    {
        paramSplinter=*it;
        paramSplinter.replace("%i",inFile);
        paramSplinter.replace("%o",outFile);
        paramSplinter.replace("%c",sStrength);
        procParam.append(paramSplinter+" ");
        pReplayGain << paramSplinter;
    }

    bCanceled=false;
    time.start();
    pReplayGain.start(KProcess::NotifyOnExit,KProcess::AllOutput);
}

void Calibrate::processOutput(KProcess* proc,char* data,int length)
{
    procOutput=data;
}

void Calibrate::prepareExit( KProcess *proc )
{
    if( bCanceled ) {
        cleanUp();
        emit exited(2);
        return;
    }
    else if( pPrepare.exitStatus() != 0 ) {
        KMessageBox::error(0,i18n("This is the last message, we recived from the backend:")+"\n"+procOutput+i18n("\nThe command was:")+"\n"+procParam,i18n("Error while calibrating (Preparing)"));
        cleanUp();
        emit exited(1);
        return;
    }

    encode();
}

void Calibrate::encodeExit( KProcess *proc )
{
    if( bCanceled ) {
        cleanUp();
        emit exited(2);
        return;
    }
    else if( pEncode.exitStatus() != 0 ) {
        KMessageBox::error(0,i18n("This is the last message, we recived from the backend:")+"\n"+procOutput+i18n("\nThe command was:")+"\n"+procParam,i18n("Error while calibrating (Encoding)"));
        cleanUp();
        emit exited(1);
        return;
    }

    for( FileFormatList::Iterator it=prefs.fileFormatList.begin(); it!=prefs.fileFormatList.end(); ++it )
    {
        if( (*it).extension == format ) {
            (*it).encTime=(float)(time.elapsed())/(float)(length*1000);
            break;
        }
    }

    BackendPlugins::PluginStructure plugin=backendPlugins.plugin(prefs.fileFormat(format).encoder);

    if( reverse ) {
        if( plugin.enc.replaygain.enabled || prefs.fileFormat(format).replaygain != "" ) {
            replayGain();
        }
        else {
            cleanUp();
            emit exited(0);
        }
    }
    else {
        if( plugin.dec.enabled ) {
            decode();
        }
        else if( plugin.enc.replaygain.enabled || prefs.fileFormat(format).replaygain != "" ) {
            replayGain();
        }
    }
}

void Calibrate::decodeExit( KProcess *proc )
{
    if( bCanceled ) {
        cleanUp();
        emit exited(2);
        return;
    }
    else if( pDecode.exitStatus() != 0 ) {
        KMessageBox::error(0,i18n("This is the last message, we recived from the backend:")+"\n"+procOutput+i18n("\nThe command was:")+"\n"+procParam,i18n("Error while calibrating (Decoding)"));
        cleanUp();
        emit exited(1);
        return;
    }

    for( FileFormatList::Iterator it=prefs.fileFormatList.begin(); it!=prefs.fileFormatList.end(); ++it )
    {
        if( (*it).extension == format ) {
            (*it).decTime=(float)(time.elapsed())/(float)(length*1000);
            break;
        }
    }

    if( reverse ) {
        encode();
    }
    else if( backendPlugins.plugin(prefs.fileFormat(format).encoder).enc.replaygain.enabled || prefs.fileFormat(format).replaygain != "" ) {
        replayGain();
    }
    else {
        cleanUp();
        emit exited(0);
    }
}

void Calibrate::replayGainExit( KProcess *proc )
{
    if( bCanceled || ( pReplayGain.exitStatus() != 0 && prefs.fileFormat(format).replaygain.find("mp3gain") == -1 ) ) {
        if( pReplayGain.exitStatus() != 0 ) KMessageBox::error(0,i18n("This is the last message, we recived from the backend:")+"\n"+procOutput+i18n("\nThe command was:")+"\n"+procParam,i18n("Error while calibrating (ReplayGain)"));
        cleanUp();
        emit exited(1);
        return;
    }

    for( FileFormatList::Iterator it=prefs.fileFormatList.begin(); it!=prefs.fileFormatList.end(); ++it )
    {
        if( (*it).extension == format ) {
                if( backendPlugins.plugin(prefs.fileFormat(format).encoder).enc.replaygain.enabled )
                    (*it).repTime=(float)(time.elapsed())/(float)(length*1000)-(*it).encTime;
                else if( prefs.fileFormat(format).replaygain != "" )
                    (*it).repTime=(float)(time.elapsed())/(float)(length*1000);
            break;
        }
    }

    cleanUp();
    emit exited(0);
}

void Calibrate::cleanUp()
{
    if( kTempOriFile != 0 ) {
        delete kTempOriFile;
        kTempOriFile = 0;
    }
    if( kTempEncFile != 0 ) {
        delete kTempEncFile;
        kTempEncFile = 0;
    }
    if( kTempDecFile != 0 ) {
        delete kTempDecFile;
        kTempDecFile = 0;
    }
}

void Calibrate::cancel()
{
    bCanceled=true;

    /*if( pPrepare.isRunning() ) pPrepare.kill();
    else if( pEncode.isRunning() ) pEncode.kill();
    else if( pDecode.isRunning() ) pDecode.kill();
    else if( pReplayGain.isRunning() ) pReplayGain.kill();*/

    pPrepare.kill();
    pEncode.kill();
    pDecode.kill();
    pReplayGain.kill();
    cleanUp();
    emit exited(1);
}

