/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * anim.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "coordsys.h"
#include "anim.h"


extern GLfloat user_matrix[19];


/*============*
 * ANIMATIONS *
 *============*/

void P3_coordsys_set_state (P3_coordsys* csys, P3_anim_coordsys* anim, float prev_time, float new_time) {
  P3_state_coordsys* s1;
  GLfloat f1; GLfloat f2;
  GLfloat q[4];

  if (new_time < anim->time_min) {

    GLfloat width; GLfloat height; GLfloat depth;
    /* interpolate between current state and anim 1rst state */
    f1 = (anim->time_min - new_time) / (anim->time_min - prev_time);
    f2 = 1.0 - f1;
    s1 = anim->states;
    P3_quaternion_from_matrix (q, csys->m);
    width  = f1 * csys->m[16] + f2 * s1->m[16];
    height = f1 * csys->m[17] + f2 * s1->m[17];
    depth  = f1 * csys->m[18] + f2 * s1->m[18];
    P3_quaternion_slerp (q, q, s1->quaternion, f2, f1);
    P3_matrix_from_quaternion (csys->m, q);
    csys->m[12] = f1 * csys->m[12] + f2 * s1->m[12];
    csys->m[13] = f1 * csys->m[13] + f2 * s1->m[13];
    csys->m[14] = f1 * csys->m[14] + f2 * s1->m[14];
    P3_matrix_scale (csys->m, width, height, depth);

  } else if (new_time > anim->time_max) {

    if (anim->cyclic_lap < 0.0) {

      /* set anim last state */
      s1 = anim->states + anim->nb_states - 1;
      P3_matrix_copy (csys->m, s1->m);

    } else {

      P3_state_coordsys* s2;

      /* assume new_time < anim->time_max + anim->cyclic_lap 
       * => interpolate between anim last state and 1rst state
       */
      f1 = (new_time - anim->time_max) / anim->cyclic_lap;
      f2 = 1.0 - f1;
      s1 = anim->states;
      s2 = anim->states + anim->nb_states - 1;
      P3_quaternion_slerp (q, s1->quaternion, s2->quaternion, f2, f1);
      P3_matrix_from_quaternion (csys->m, q);
      csys->m[12] = f1 * s1->m[12] + f2 * s2->m[12];
      csys->m[13] = f1 * s1->m[13] + f2 * s2->m[13];
      csys->m[14] = f1 * s1->m[14] + f2 * s2->m[14];
      P3_matrix_scale (csys->m, f1 * s1->m[16] + f2 * s2->m[16], 
                                f1 * s1->m[17] + f2 * s2->m[17], 
                                f1 * s1->m[18] + f2 * s2->m[18]);

    }

  } else {
    P3_state_coordsys* s2;
    int i_max;
    int i_min;
    int i;

    /* find previous and next state for new_time => dichotomic algorythm */
    i_min = 0;
    i_max = anim->nb_states;

    i = i_max >> 1;

    while (1) {
      s1 = anim->states + i;
      if (s1->time < new_time) {
        /* we must look after */
        i_min = i;
      } else if (s1->time > new_time) {
        /* we must look before */
        i_max = i;
      } else {
        /* found a state with time = new_time => no interpolation */
        P3_matrix_copy (csys->m, s1->m);
        P3_coordsys_lefthanded (csys);
        P3_object_invalid ((P3_any_object*) csys);
        return;
      }
      if (i_max - i_min <= 1) {
        break;
      }
      i = (i_max + i_min) >> 1;
    }
    s1 = anim->states + i_min;
    s2 = s1 + 1;

    if (s1->time == new_time) {
      P3_matrix_copy (csys->m, s1->m);
    } else if (s2->time == new_time) {
      P3_matrix_copy (csys->m, s2->m);
    } else {
      /* interpolate */
      f2 = (new_time - s1->time) / (s2->time - s1->time);
      f1 = 1.0 - f2;
      P3_quaternion_slerp (q, s1->quaternion, s2->quaternion, f2, f1);
      P3_matrix_from_quaternion (csys->m, q);
      csys->m[12] = f1 * s1->m[12] + f2 * s2->m[12];
      csys->m[13] = f1 * s1->m[13] + f2 * s2->m[13];
      csys->m[14] = f1 * s1->m[14] + f2 * s2->m[14];
      P3_matrix_scale (csys->m, f1 * s1->m[16] + f2 * s2->m[16], 
                                f1 * s1->m[17] + f2 * s2->m[17], 
                                f1 * s1->m[18] + f2 * s2->m[18]);
    }
  }

  P3_coordsys_lefthanded (csys);
  P3_object_invalid ((P3_any_object*) csys);
}

/*
float P3_anim_advance (P3_anim* anim, float cur_time, float gain_time) {
  float f;
  f = cur_time + gain_time;
  if (anim->cyclic_lap > 0.0) {
    float g;
    g = anim->cyclic_lap + anim->time_max;
    if (f > g) {
      f += anim->time_min - g;
    }
  }
  return f;
}
*/

/*
int P3_anim_get_states_for_time (P3_anim* anim, float time, P3_anim_state* before, P3_anim_state* after) {
  int i;
  int i_min;
  int i_max;
  float* state;
  // return P3_FALSE if we need an interpolation, else P3_TRUE
  if (time < anim->time_min) {
    before->state = NULL;
//    before->time  = 0.0;
    after->state  = anim->states;
    after->time   = anim->time_min;
    return P3_FALSE;
  }
  if (time > anim->time_max) {
    if (anim->cyclic_lap > 0.0) {
// TO DO WARNING assume: time < time_max + cyclic_lap
      before->state = anim->states + anim->nb_states * anim->size_state;
      before->time  = anim->time_max;
      after->state  = anim->states;
      after->time   = anim->time_max + anim->cyclic_lap;
      return P3_FALSE;
    } else {
      before->state = anim->states + anim->nb_states * anim->size_state;
      before->time = time;
//      after->state = before->state;
//      after->time = time;
      return P3_TRUE;
    }
  } else {
    i_min = 0;
    i_max = anim->nb_states;
    i = i_max >> 1;
    while (1) {
      if (i_max - i_min <= 1) {
        before->state = anim->states + i_min * anim->size_state;
        before->time  = *((float*) before->state);
        after->state  = anim->states + i_max * anim->size_state;
        after->time   = *((float*) after->state);
        if (fabs (before->time - time) < P3_EPSILON) {
          return P3_TRUE;
        } else {
          return P3_FALSE;
        }
      }
      state = (float*) anim->states + i * anim->size_state;
      if (fabs (time - *state) < P3_EPSILON) {
        before->state = state;
        before->time = time;
        return P3_TRUE;
      } else if (*state < time) {
        // we must look after state
        i_min = i;
      } else if (*state > time) {
        // we must look before state
        i_max = i;
      }
      i = (i_min + i_max) >> 1;
    };
  }
}
*/

/* Coordsys Animations */

P3_anim_coordsys* P3_anim_coordsys_new (P3_anim_coordsys* anim) {
  if (anim == NULL) {
    anim = (P3_anim_coordsys*) malloc (sizeof (P3_anim_coordsys));
  }
  anim->cyclic_lap = -1.0;
  anim->time_min = -1.0;
  anim->time_max = -1.0;
  anim->nb_states = 0;
  anim->states = NULL;
  return anim;
}

void P3_anim_coordsys_delete_state (P3_anim_coordsys* anim, int index) {
  P3_state_coordsys* s;
  s = anim->states + index;
  memmove (s, s + 1, (anim->nb_states - index) * sizeof (P3_state_coordsys));
  (anim->nb_states)--;
  anim->states = realloc (anim->states, anim->nb_states * sizeof (P3_state_coordsys));
}

void P3_anim_coordsys_remove_state (P3_anim_coordsys* anim, float time) {
  P3_state_coordsys* s;
  int i;
  for (i = 0; i < anim->nb_states; i++) {
    s = anim->states + i;
    if (s->time == time) {
      P3_anim_coordsys_delete_state (anim, i);
      return;
    }
  }
}

void P3_anim_coordsys_add_state (P3_anim_coordsys* anim, P3_coordsys* csys, float time) {
  P3_state_coordsys* s = NULL;
  GLfloat matrix[19];
  int broken;
  int n;
  int i;

  /* look for exactly the same time */
  broken = P3_FALSE;
  for (i = 0; i < anim->nb_states; i++) {
    s = anim->states + i;
    if (s->time == time) {
      broken = P3_TRUE;
      break;
    }
  }

  if (broken == P3_FALSE) {
    n = anim->nb_states;
    (anim->nb_states)++;
    anim->states = (P3_state_coordsys*) realloc (anim->states, anim->nb_states * sizeof (P3_state_coordsys));

    if (n == 0) {
      s = anim->states;
      anim->time_min = time;
      anim->time_max = time;
    } else {
      broken = P3_FALSE;
      for (i = 0; i < n; i++) {
        s = anim->states + i;
        if (s->time > time) {
          /* move following states */
          memmove (s + 1, s, (n - i) * sizeof (P3_state_coordsys));
          broken = P3_TRUE;
          break;
        }
      }
      if (broken == P3_FALSE) {
        s = anim->states + n;
      }
      if (time < anim->time_min) { anim->time_min = time; }
      if (time > anim->time_max) { anim->time_max = time; }
    }
  }

  s->time = time;
  P3_matrix_copy (s->m, csys->m);
  P3_matrix_copy (matrix, csys->m);
  P3_matrix_scale (matrix, 1.0 / csys->m[16], 1.0 / csys->m[17], 1.0 / csys->m[18]);
  P3_quaternion_from_matrix (s->quaternion, matrix);
}

/* *** WAS GOOD ***
void P3_coordsys_get_state (P3_state_coordsys* s, P3_coordsys* c) {
  P3_matrix_copy (s->m, c->m);
  P3_matrix_copy (user_matrix, c->m);
  P3_matrix_scale (user_matrix, 1.0 / c->m[16], 1.0 / c->m[17], 1.0 / c->m[18]);
  P3_quaternion_from_matrix (s->quaternion, user_matrix);
}
*/
/*
//void P3_anim_coordsys_state_base_set (P3_anim_coordsys_state_base* s, float time, P3_coordsys* c) {
  GLfloat m[19];
//  s->time = time;
  s->width  = c->m[16];
  s->height = c->m[17];
  s->depth  = c->m[18];
  P3_matrix_copy (s->m, c->m);
  P3_matrix_copy (m, c->m);
  P3_matrix_scale (m, 1.0 / c->m[16], 1.0 / c->m[17], 1.0 / c->m[18]);
  P3_quaternion_from_matrix (s->quaternion, m);
}
*/

/* *** WAS GOOD ***
void P3_state_coordsys_interpolate (P3_state_coordsys* s, P3_state_coordsys* s1, P3_state_coordsys* s2, float f2) {
  float f1;
  GLfloat width; GLfloat height; GLfloat depth;
  f1 = 1.0 - f2;
//  s->time = time;
//  s->width  = f1 * s1->width  + f2 * s2->width;
//  s->height = f1 * s1->height + f2 * s2->height;
//  s->depth  = f1 * s1->depth  + f2 * s2->depth;
  width  = f1 * s1->m[16] + f2 * s2->m[16];
  height = f1 * s1->m[17] + f2 * s2->m[17];
  depth  = f1 * s1->m[18] + f2 * s2->m[18];
  P3_quaternion_slerp (s->quaternion, s1->quaternion, s2->quaternion, f2);
  P3_matrix_from_quaternion (s->m, s->quaternion);
  s->m[12] = f1 * s1->m[12] + f2 * s2->m[12];
  s->m[13] = f1 * s1->m[13] + f2 * s2->m[13];
  s->m[14] = f1 * s1->m[14] + f2 * s2->m[14];
//  P3_matrix_translate (s->m, f1 * s1->m[12] + f2 * s2->m[12], f1 * s1->m[13] + f2 * s2->m[13], f1 * s1->m[14] + f2 * s2->m[14]);
  P3_matrix_scale (s->m, width, height, depth);
//  P3_matrix_scale (s->m, s->width, s->height, s->depth);
}

void P3_coordsys_set_state (P3_coordsys* c, P3_state_coordsys* s) {
  P3_matrix_copy (c->m, s->m);
// TO DO scale factors change while the matrix is not scaled ?!?
//  c->m[16] = s->width;
//  c->m[17] = s->height;
//  c->m[18] = s->depth;
  P3_coordsys_lefthanded (c);
  P3_object_invalid ((P3_any_object*) c);
}
*/

/*
void P3_coordsys_set_anim_state (P3_coordsys** coordsys, P3_anim_coordsys* anim, float time) {
  P3_anim_state before;
  P3_anim_state after;
  P3_anim_coordsys_state* s;
  int i;
  if (P3_anim_get_states_for_time ((P3_anim*) anim, time, &before, &after) == P3_TRUE) {
    s = (P3_anim_coordsys_state*) before.state;
    for (i = 0; i < anim->nb_elements; i++) {
      P3_coordsys_set_state_base (coordsys[anim->elements[i]], s->base_states + i);
    }
  } else {
    P3_anim_coordsys_state* s2;
    P3_anim_coordsys_state_base bs;
    float f2;
    // must interpolate
    s  = (P3_anim_coordsys_state*) before.state;
    s2 = (P3_anim_coordsys_state*)  after.state;
    if (s == NULL) {
      // s must be current time or last integer (not interpolated) time
// TO DO
//      s = ;
//      before->time = time;
    }
    f2 = (after.time - time) / (after.time - before.time);
    for (i = 0; i < anim->nb_elements; i++) {
      P3_anim_coordsys_state_base_interpolate (&bs, s->base_states + i, s2->base_states + i, f2);
      P3_coordsys_set_state_base (coordsys[anim->elements[i]], &bs);
    }
  }
}
*/

