/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*******************************************************
 * cal3d4p3.c : use Cal3D (http://cal3d.sourceforge.net)
 *              rocking animation lib with Soya
 * Copyright (C) 2003 Jean-Baptiste 'Jiba' LAMY
 *                    Bertrand 'blam' LAMY
 *******************************************************/

#ifdef HAVE_CAL3D

#include "p3_base.h"
#include "math3d.h"
#include "renderer.h"
#include "frustum.h"
#include "coordsys.h"
#include "material.h"
#include "util.h"
#include "mesh.h"
#include "light.h"
#include "cal3d4p3.h"

extern P3_renderer* renderer;
extern P3_material* light_shader;
extern P3_material* current_material;
extern GLfloat white[4];
extern int shadow_display_list;

int cal3d_nb_vertices = 0;
float* cal3d_coords_array    = NULL;
float* cal3d_normals_array   = NULL;
float* cal3d_texcoords_array = NULL;
float* cal3d_shades_array    = NULL;
int cal3d_nb_faces = 0;
CalIndex* cal3d_faces_array  = NULL;
float* cal3d_fnormals_array  = NULL;

P3_chunk* cal3d_data;



P3_class P3_class_cal3d_shape = { 
  P3_ID_CAL3D_SHAPE,
  (batch_func)     0,
  (render_func)    P3_cal3d_shape_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_class P3_class_cal3d_volume = { 
  P3_ID_CAL3D_VOLUME,
  (batch_func)     P3_cal3d_volume_batch,
  (render_func)    0,
  (shadow_func)    P3_cal3d_volume_shadow,
  (raypick_func)   0,
  (raypick_b_func) 0,
};


void P3_cal3d_init (void) {
  cal3d_data = P3_chunk_new ();
}

void P3_cal3d_quit (void) {
  free (cal3d_coords_array);
  free (cal3d_normals_array);
  free (cal3d_normals_array);
  free (cal3d_texcoords_array);
  free (cal3d_shades_array);
  free (cal3d_faces_array);
  free (cal3d_fnormals_array);

  P3_chunk_dealloc (cal3d_data);
}


/*=============*
 * Cal3d Shape *
 *=============*/

P3_cal3d_shape* P3_cal3d_shape_new (P3_cal3d_shape* shape) {
  if (shape == NULL) shape = (P3_cal3d_shape*) malloc (sizeof (P3_cal3d_shape));
  shape->class = &P3_class_cal3d_shape;
  shape->core_model = CalCoreModel_New();
  shape->nb_materials = 0;
  shape->materials = NULL;

  shape->nb_vertices = 0;
  shape->nb_faces = 0;
  shape->faces = NULL;
  shape->face_neighbors = NULL;

  if (!CalCoreModel_Create (shape->core_model, "")) {
    P3_error ("CalCoreModel_Create failed: %s", CalError_GetLastErrorDescription ());
  }
  return shape;
}

void P3_cal3d_shape_dealloc (P3_cal3d_shape* shape) {
  int i;
  if (shape->nb_materials > 0) {
    for (i = 0; i < shape->nb_materials; i++) {
      Py_DECREF (shape->materials[i]);
    }
    free (shape->materials);
  }

  if (shape->option & P3_CAL3D_CELL_SHADING) {
    P3_material_decref (((P3_mesh_cell_shading*) shape->xtra)->shader);
    free (shape->xtra);
  }

  free (shape->faces);
  free (shape->face_neighbors);

  CalCoreModel_Destroy (shape->core_model);
  CalCoreModel_Delete  (shape->core_model);
}

void P3_cal3d_shape_face_set_neighborhood (P3_cal3d_shape* shape, int index1, int index2, GLfloat* vertices) {
  GLfloat* coord1;
  GLfloat* coord2;
  GLfloat* coord1a;
  GLfloat* coord2a;
  int* face1 = shape->faces + index1 * 3;
  int* face2 = shape->faces + index2 * 3;
  int i, j;
  for (i = 0; i < 3; i++) {
    for (j = 0; j < 3; j++) {
      coord1 = vertices + face1[i] * 3;
      coord2 = vertices + face2[j] * 3;
      if (fabs (coord1[0] - coord2[0]) < P3_EPSILON &&
          fabs (coord1[1] - coord2[1]) < P3_EPSILON &&
          fabs (coord1[2] - coord2[2]) < P3_EPSILON) {
        /* the 2 faces share 1 coordinate */
        if (i == 2) coord1a = vertices + face1[0] * 3; else coord1a = vertices + face1[i + 1] * 3;
        if (j == 2) coord2a = vertices + face2[0] * 3; else coord2a = vertices + face2[j + 1] * 3;
        if (fabs (coord1a[0] - coord2a[0]) < P3_EPSILON &&
            fabs (coord1a[1] - coord2a[1]) < P3_EPSILON &&
            fabs (coord1a[2] - coord2a[2]) < P3_EPSILON) {
          /* point after is also shared -> neighbor */
          shape->face_neighbors[index1 * 3 + i] = index2;
          shape->face_neighbors[index2 * 3 + j] = index1;
          /* face can't normally have 2 times the same neighbor */
          return;
        }
        if (j == 0) coord2a = vertices + face2[2] * 3; else coord2a = vertices + face2[j - 1] * 3;
        if (fabs (coord1a[0] - coord2a[0]) < P3_EPSILON &&
            fabs (coord1a[1] - coord2a[1]) < P3_EPSILON &&
            fabs (coord1a[2] - coord2a[2]) < P3_EPSILON) {
          shape->face_neighbors[index1 * 3 + i] = index2;
          if (j == 0) { 
            shape->face_neighbors[index2 * 3] = index1;
          } else {
            shape->face_neighbors[index2 * 3 + j - 1] = index1;
          }
          /* face can't normally have 2 times the same neighbor */
          return;
        }
      }
    }
  }
}

void P3_cal3d_shape_init (P3_cal3d_shape* shape) {
  struct CalRenderer* cal_renderer;
  struct CalModel* cal_model;
  struct CalCoreMesh* cal_core_mesh;
  struct CalCoreSubmesh* cal_core_submesh;
  P3_cal3d_submesh* my_submesh;
  P3_chunk* vertices;
  P3_chunk* faces;
  int nb_mesh, nb_submesh;
  int i, j, k, n, q;

  cal_model = CalModel_New ();
  CalModel_Create (cal_model, shape->core_model);
  nb_mesh = CalCoreModel_GetCoreMeshCount (shape->core_model);
  for (i = 0; i < nb_mesh; i++) CalModel_AttachMesh (cal_model, i);
  CalModel_SetMaterialSet (cal_model, 0);

  cal_renderer = CalModel_GetRenderer (cal_model);
  if (!CalRenderer_BeginRendering (cal_renderer)) {
    P3_error ("CalRenderer_BeginRendering failed: %s", CalError_GetLastErrorDescription());
    return;
  }

  vertices = P3_chunk_new ();
  faces    = P3_chunk_new ();
  shape->nb_vertices = 0;
  shape->nb_faces = 0;
  shape->nb_submeshes = 0;
  shape->submeshes = NULL;

  for (i = 0; i < nb_mesh; i++) {
    cal_core_mesh = CalCoreModel_GetCoreMesh (shape->core_model, i);
    nb_submesh = CalCoreMesh_GetCoreSubmeshCount (cal_core_mesh);
    for (j = 0; j < nb_submesh; j++) {
      cal_core_submesh = CalCoreMesh_GetCoreSubmesh (cal_core_mesh, j);
      CalRenderer_SelectMeshSubmesh (cal_renderer, i, j);
      /* create my new submesh */
      (shape->nb_submeshes)++;
      shape->submeshes = (P3_cal3d_submesh*) realloc (shape->submeshes, shape->nb_submeshes * sizeof (P3_cal3d_submesh));
      my_submesh = shape->submeshes + shape->nb_submeshes - 1;
      my_submesh->mesh = i;
      my_submesh->submesh = j;
      my_submesh->first_face = shape->nb_faces * 3;
      my_submesh->first_vertex = shape->nb_vertices;
      my_submesh->material = shape->materials[(int) CalCoreSubmesh_GetCoreMaterialThreadId (CalCoreMesh_GetCoreSubmesh (cal_core_mesh, j))];
      if (my_submesh->material->option & P3_MATERIAL_ALPHA) shape->option |= P3_CAL3D_ALPHA;
      /* get all faces */
      k = CalCoreSubmesh_GetFaceCount (cal_core_submesh);
      my_submesh->nb_faces = k;
      shape->nb_faces += k;
      n = P3_chunk_register (faces, 3 * k * sizeof (int));
      CalRenderer_GetFaces (cal_renderer, faces->content + n);
      /* add offset to all vertex indices */
      for (q = 0; q < k * 3; q++) {
        ((int*) (faces->content + n))[q] += shape->nb_vertices;
      }
      /* get all vertices */
      k = CalCoreSubmesh_GetVertexCount (cal_core_submesh);
      my_submesh->nb_vertices = k;
      shape->nb_vertices += k;
      n = P3_chunk_register (vertices, 3 * k * sizeof (GLfloat));
      CalRenderer_GetVertices (cal_renderer, vertices->content + n);

      if (cal3d_nb_vertices < k) {
        cal3d_normals_array   = (float*) realloc (cal3d_normals_array,   k * 3 * sizeof (float));
        cal3d_texcoords_array = (float*) realloc (cal3d_texcoords_array, k * 2 * sizeof (float));
        if (shape->option & P3_CAL3D_CELL_SHADING)
          cal3d_shades_array = (float*) realloc (cal3d_shades_array, k * sizeof (float));
        cal3d_nb_vertices = k;
      }

    }
  }

  shape->faces = (int*) realloc (faces->content, shape->nb_faces * 3 * sizeof (int));

  /* now compute face neighbors if needed */
  if (shape->option & P3_CAL3D_NEIGHBORS) {
    shape->face_neighbors = (int*) malloc (shape->nb_faces * 3 * sizeof (int));
    for (i = 0; i < shape->nb_faces * 3; i++) shape->face_neighbors[i] = -1;
    /* take each couple of face */
    for (i = 0; i < shape->nb_faces; i++) {
      for (j = i + 1; j < shape->nb_faces; j++) {
        P3_cal3d_shape_face_set_neighborhood (shape, i, j, (GLfloat*) vertices->content);
      }
    }
  } else {
    shape->face_neighbors = NULL;
  }

  CalRenderer_EndRendering (cal_renderer);
  P3_chunk_dealloc (vertices);
  free (faces);
  CalModel_Destroy (cal_model);
  CalModel_Delete (cal_model);
  shape->option |= P3_CAL3D_INITED;
}

void P3_cal3d_shape_set_cell_shading (P3_cal3d_shape* shape, P3_material* shader, GLfloat* color, GLfloat line_width_factor) {
  P3_mesh_cell_shading* cshade;
  cshade = (P3_mesh_cell_shading*) malloc (sizeof (P3_mesh_cell_shading));
  cshade->shader = shader;
  P3_material_incref (shader);
  memcpy (cshade->line_color, color, 4 * sizeof (GLfloat));
  cshade->line_width_factor = line_width_factor;
  shape->option |= P3_CAL3D_NEIGHBORS;
  shape->option |= P3_CAL3D_CELL_SHADING;
  shape->xtra = cshade;
}

static void P3_cal3d_compute_cell_shades (int nb, GLfloat* coords, GLfloat* normals, P3_list* lights) {
  P3_light* light;
  GLfloat v[3];
  GLfloat tmp;
  int i, j;
  for (i = 0; i < lights->nb; i++) {
    light = (P3_light*) P3_list_get (lights, i);
    if (light->option & P3_LIGHT_DIRECTIONAL) {
      /* directional light */
      for (j = 0; j < nb; j++) {
        tmp = - P3_vector_dot_product (normals, light->data);
        if (tmp > 0.0) cal3d_shades_array[j] += tmp;
        normals += 3;
      }
    } else {
      /* positional light */
      for (j = 0; j < nb; j++) {
        v[0] = light->data[0] - coords[0];
        v[1] = light->data[1] - coords[1];
        v[2] = light->data[2] - coords[2];
        P3_vector_normalize (v);
        tmp = P3_vector_dot_product (normals, v);
        if (tmp > 0.0) cal3d_shades_array[j] += tmp;
        normals += 3;
        coords  += 3;
      }
    }
  }
}

static void P3_cal3d_volume_compute_face_normal (P3_cal3d_volume* volume) {
  P3_cal3d_submesh* submesh;
  GLfloat* ptrf = volume->face_normals;
  int* face;
  int i, j;
  /* compute face normals (planes in fact) */
  for (i = 0; i < volume->shape->nb_submeshes; i++) {
    submesh = volume->shape->submeshes + i;
    if (volume->submeshes[i] == 0) {
      ptrf += 4 * submesh->nb_faces;
      continue;
    }
    face = volume->shape->faces + submesh->first_face;
    for (j = 0; j < submesh->nb_faces; j++) {
      /* it's useless to normalize here */
      P3_face_plane (ptrf, volume->vertex_coords + face[0] * 3,
                           volume->vertex_coords + face[1] * 3,
                           volume->vertex_coords + face[2] * 3);
      ptrf += 4;
      face += 3;
    }
  }
}

void P3_cal3d_shape_render (P3_cal3d_shape* shape, P3_instance* csys) {
  P3_cal3d_volume* volume = (P3_cal3d_volume*) csys;
  struct CalRenderer* cal_renderer;
  struct CalModel* cal_model;
  struct CalCoreMesh* cal_core_mesh;
  struct CalCoreSubmesh* cal_core_submesh;
  P3_cal3d_submesh* submesh;
  P3_chunk* vertices;
  GLfloat* ptrf;
  int* face;
  int nb_mesh, nb_submesh;
  int i, j, k, n, q;
  int nb_vnormals;

  if (!(shape->option & P3_CAL3D_INITED)) P3_cal3d_shape_init (shape);

  cal_renderer = CalModel_GetRenderer (volume->model);
    
  if (!CalRenderer_BeginRendering (cal_renderer)) {
    P3_error ("CalRenderer_BeginRendering failed: %s", CalError_GetLastErrorDescription());
    return;
  }

  glDisable (GL_CULL_FACE);
  glEnableClientState (GL_VERTEX_ARRAY);
  glEnableClientState (GL_NORMAL_ARRAY);
  glEnableClientState (GL_TEXTURE_COORD_ARRAY);
  
  if (shape->option & P3_CAL3D_CELL_SHADING) light_shader = ((P3_mesh_cell_shading*) shape->xtra)->shader;

//  cal3d_data->nb = 0;
//  P3_chunk_register (cal3d_data, volume->nb_vertices * 3 * sizeof (GLfloat));
//  ptrf = (GLfloat*) cal3d_data->content;
  ptrf = volume->vertex_coords;

  for (i = 0; i < shape->nb_submeshes; i++) {

// TO DO sort alpha and opaque submesh ?

    submesh = shape->submeshes + i;
    if (volume->submeshes[i] == 0) {
      ptrf += submesh->nb_vertices * 3;
      continue;
    }

    CalRenderer_SelectMeshSubmesh (cal_renderer, submesh->mesh, submesh->submesh);

    /* get all vertices */
    CalRenderer_GetVertices (cal_renderer, ptrf);

    CalRenderer_GetNormals  (cal_renderer, cal3d_normals_array);
    CalRenderer_GetTextureCoordinates (cal_renderer, 0, cal3d_texcoords_array);

    glVertexPointer   (3, GL_FLOAT, 0, ptrf);
    glNormalPointer   (   GL_FLOAT, 0, cal3d_normals_array);
    glTexCoordPointer (2, GL_FLOAT, 0, cal3d_texcoords_array);

    P3_material_activate (submesh->material);

    if (shape->option & P3_CAL3D_CELL_SHADING) {
      GLfloat shade;
      P3_light_list_cast_into (renderer->top_lights, (P3_coordsys*) csys);
      P3_light_list_cast_into (renderer->c_context->lights, (P3_coordsys*) csys);
      glClientActiveTexture (GL_TEXTURE0);
      /* initialize shades */
      for (j = 0; j < submesh->nb_vertices; j++) cal3d_shades_array[j] = 0.0;
      /* compute shades */
      P3_cal3d_compute_cell_shades (submesh->nb_vertices, ptrf, cal3d_normals_array, renderer->top_lights);
      P3_cal3d_compute_cell_shades (submesh->nb_vertices, ptrf, cal3d_normals_array, renderer->c_context->lights);
      /* clip shades */
      for (j = 0; j < submesh->nb_vertices; j++)
        if (cal3d_shades_array[j] > 1.0) cal3d_shades_array[j] = 1.0;
      /* render */
      glDisable (GL_LIGHTING);
      glActiveTexture (GL_TEXTURE1);
      glEnable (GL_TEXTURE_2D);
      glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      if (light_shader->id == 0) P3_material_init (light_shader);
      glBindTexture (GL_TEXTURE_2D, light_shader->id);

      face = shape->faces + submesh->first_face;
      glBegin (GL_TRIANGLES);
      for (j = 0; j < submesh->nb_faces * 3; j++) {
        k = *face++ - submesh->first_vertex;
        shade = cal3d_shades_array[k];
        glMultiTexCoord2f (GL_TEXTURE1, shade, shade);
        glArrayElement (k);
      }
      glEnd ();
      glDisable (GL_TEXTURE_2D);
      glActiveTexture (GL_TEXTURE0);
      glEnable (GL_LIGHTING);

    } else {

      face = shape->faces + submesh->first_face;
      glBegin (GL_TRIANGLES);
      for (j = 0; j < submesh->nb_faces * 3; j++) {
        glArrayElement (*face++ - submesh->first_vertex);
      }
      glEnd ();

    }
    ptrf += submesh->nb_vertices * 3;
  }

  CalRenderer_EndRendering (cal_renderer);
  glDisableClientState (GL_NORMAL_ARRAY);
  glDisableClientState (GL_TEXTURE_COORD_ARRAY);

  /* draw cell-shading outline */
  if (shape->option & P3_CAL3D_CELL_SHADING) {
    P3_chunk* chunk;
    char* face_options;
    int* neighbors;
    GLfloat d;
    int neighbor;
    GLfloat* frustum_position;

    P3_cal3d_volume_compute_face_normal (volume);

    chunk = P3_get_chunk ();
    P3_chunk_register (chunk, shape->nb_faces * sizeof (char));
    face_options = (char*) chunk->content;

    frustum_position = (P3_renderer_get_frustum (csys))->position;

    /* determine if faces are front or back from the camera point of view */
    k = 0;
    for (i = 0; i < shape->nb_submeshes; i++) {
      submesh = shape->submeshes + i;
      if (volume->submeshes[i] == 0) {
        k += submesh->nb_faces;
        continue;
      }
      face = shape->faces + submesh->first_face;
      ptrf = volume->face_normals + k * 4;
      for (j = 0; j < submesh->nb_faces; j++) {
        if (ptrf[0] * frustum_position[0] + ptrf[1] * frustum_position[1] + ptrf[2] * frustum_position[2] + ptrf[3] > 0.0)
          /* front */
          face_options[k] = 1;
        else
          /* back */
          face_options[k] = 0;
        ptrf += 4;
        k++;
      }
    }

    /* render */
    /* compute distance between camera and mesh to adjust outline width */
    if (volume->sphere[3] > 0.0) {
      d = P3_sphere_distance_point (volume->sphere, frustum_position);
      if (d < 1.0) 
        d = ((P3_mesh_cell_shading*) shape->xtra)->line_width_factor;
      else
        d = ((P3_mesh_cell_shading*) shape->xtra)->line_width_factor / d;
    } else {
      glLineWidth (2.0);
    }
    glVertexPointer (3, GL_FLOAT, 0, volume->vertex_coords);
    glColor4fv (((P3_mesh_cell_shading*) shape->xtra)->line_color);
    glDisable (GL_LIGHTING);
    glDepthFunc (GL_LEQUAL);
    P3_material_inactivate (current_material);
    current_material = NULL;

    glBegin (GL_LINES);

    k = 0;
    for (i = 0; i < shape->nb_submeshes; i++) {
      submesh = shape->submeshes + i;
      if (volume->submeshes[i] == 0) {
        k += submesh->nb_faces;
        continue;
      }
      for (j = 0; j < submesh->nb_faces; j++) {
        if (face_options[k] == 1) {
          /* test if neighbors are back */
          neighbors = shape->face_neighbors + k * 3;
          for (n = 0; n < 3; n++) {
            neighbor = neighbors[n];
            if (neighbor == -1 || face_options[neighbor] == 0) {
              /* add edge k = vertices k and k + 1 */
              face = shape->faces + submesh->first_face + j * 3;
              glArrayElement (face[n]);
              if (n == 2) glArrayElement (face[0]);
              else        glArrayElement (face[n + 1]);
            }
          }
        }
        k++;
      }
    }
    glEnd ();

    P3_drop_chunk (chunk);

    glEnable (GL_LIGHTING);
    glDepthFunc (GL_LESS);
    glColor4fv (white);
  }
  


#if 0

  submesh = volume->submeshes;    
  for (i = 0; i < volume->nb_submeshes; i++) {

    if (volume->option & P3_CAL3D_ALPHA && 
        ((submesh->material->option & P3_MATERIAL_ALPHA && renderer->state == P3_RENDERER_STATE_OPAQUE) ||
         (!(submesh->material->option & P3_MATERIAL_ALPHA) && renderer->state == P3_RENDERER_STATE_ALPHA))) {
      submesh++;
      continue;
    }

    if (!CalRenderer_SelectMeshSubmesh (cal_renderer, submesh->mesh, submesh->submesh))
      P3_error ("CalRenderer_SelectMeshSubmesh failed: %s", CalError_GetLastErrorDescription());

    CalRenderer_GetVertices (cal_renderer, cal3d_coords_array);
    nb_vnormals = CalRenderer_GetNormals  (cal_renderer, cal3d_normals_array);
    CalRenderer_GetTextureCoordinates (cal_renderer, 0, cal3d_texcoords_array);
    nb_faces = CalRenderer_GetFaces (cal_renderer, cal3d_faces_array);

    glVertexPointer   (3, GL_FLOAT, 0, cal3d_coords_array);
    glNormalPointer   (   GL_FLOAT, 0, cal3d_normals_array);
    glTexCoordPointer (2, GL_FLOAT, 0, cal3d_texcoords_array);

    P3_material_activate (submesh->material);

    if (shape->option & P3_CAL3D_CELL_SHADING) {
      CalIndex* ptr;
      GLfloat shade;
      int j;
      glClientActiveTexture (GL_TEXTURE0);
      P3_light_list_cast_into (renderer->top_lights, (P3_coordsys*) csys);
      P3_light_list_cast_into (renderer->c_context->lights, (P3_coordsys*) csys);
      /* initialize shades */
      for (j = 0; j < nb_vnormals; j++) cal3d_shades_array[j] = 0.0;
      /* compute shades */
      P3_cal3d_compute_cell_shades (nb_vnormals, renderer->top_lights);
      P3_cal3d_compute_cell_shades (nb_vnormals, renderer->c_context->lights);
      /* clip shades */
      for (j = 0; j < nb_vnormals; j++)
        if (cal3d_shades_array[j] > 1.0) cal3d_shades_array[j] = 1.0;
      /* render */
      glDisable (GL_LIGHTING);
      glActiveTexture (GL_TEXTURE1);
      glEnable (GL_TEXTURE_2D);
      glTexEnvi (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      if (light_shader->id == 0) P3_material_init (light_shader);
      glBindTexture (GL_TEXTURE_2D, light_shader->id);
      ptr = cal3d_faces_array;
      glBegin (GL_TRIANGLES);
      for (j = 0; j < nb_faces * 3; j++) {
        shade = cal3d_shades_array[*ptr];
        glMultiTexCoord2f (GL_TEXTURE1, shade, shade);
        glArrayElement (*ptr);
        ptr++;
      }
      glEnd ();
      glDisable (GL_TEXTURE_2D);
      glActiveTexture (GL_TEXTURE0);
      glEnable (GL_LIGHTING);
    } else {
      glDrawElements (GL_TRIANGLES, nb_faces * 3, GL_UNSIGNED_INT, cal3d_faces_array);
    }

    submesh++;
  }
    
#endif

  glDisableClientState (GL_VERTEX_ARRAY);
  glEnable (GL_CULL_FACE);
}

int P3_cal3d_shape_shadow (P3_cal3d_shape* shape, P3_cal3d_volume* volume, P3_light* light) {
  P3_cal3d_submesh* submesh;
  P3_chunk* chunk;
  P3_frustum* frustum;
  GLfloat* ptrf;
  int* neighbors;
  int* face;
  char* face_options;
  GLfloat* coord_ptr;
  GLfloat coord[4];
  GLfloat cone[9];
  GLfloat b = renderer->c_camera->back;
  int neighbor;
  int i, j, k, n;

  if (!(shape->option & P3_CAL3D_SHADOW_CAST)) return P3_FALSE;
  
  if (!(shape->option & P3_CAL3D_CELL_SHADING)) P3_cal3d_volume_compute_face_normal (volume);

  chunk = P3_get_chunk ();
  P3_chunk_register (chunk, shape->nb_faces * sizeof (char));
  face_options = (char*) chunk->content;

  frustum = P3_renderer_get_frustum ((P3_instance*) volume);

  /* tag all faces front or back */
  P3_light_cast_into (light, (P3_coordsys*) volume);
  if (light->option & P3_LIGHT_DIRECTIONAL) {
    if (volume->sphere[3] > 0.0) P3_cone_from_sphere_and_vector (cone, volume->sphere, light->data, b);
    k = 0;
    for (i = 0; i < shape->nb_submeshes; i++) {
      submesh = shape->submeshes + i;
      if (volume->submeshes[i] == 0) {
        k += submesh->nb_faces;
        continue;
      }
      face = shape->faces + submesh->first_face;
      ptrf = volume->face_normals + k * 4;
      for (j = 0; j < submesh->nb_faces; j++) {
        if (P3_vector_dot_product (light->data, ptrf) >= 0.0) {
          /* back */
          face_options[k] = 0;
        } else {
          /* front */
          face_options[k] = 1;
        }
        ptrf += 4;
        k++;
      }
    }
  } else {
    if (volume->sphere[3] > 0.0) {
      if (P3_cone_from_sphere_and_origin (cone, volume->sphere, light->data, b) == P3_FALSE) return P3_FALSE;
    }
    k = 0;
    for (i = 0; i < shape->nb_submeshes; i++) {
      submesh = shape->submeshes + i;
      if (volume->submeshes[i] == 0) {
        k += submesh->nb_faces;
        continue;
      }
      face = shape->faces + submesh->first_face;
      ptrf = volume->face_normals + k * 4;
      for (j = 0; j < submesh->nb_faces; j++) {
        if (light->data[0] * ptrf[0] + light->data[1] * ptrf[1] + light->data[2] * ptrf[2] + ptrf[3] > 0.0) {
          /* front */
          face_options[k] = 1;
        } else {
          /* back */
          face_options[k] = 0;
        }
        ptrf += 4;
        k++;
      }
    }
  }

  /* draw shadow volume 1rst step */
  glStencilFunc (GL_ALWAYS, 1, 0xFFFFFFFF);
  glFrontFace (GL_CW);
  glStencilOp (GL_KEEP, GL_KEEP, GL_INCR);
  glLoadMatrixf (volume->render_matrix);
  glNewList (shadow_display_list, GL_COMPILE_AND_EXECUTE);
  glBegin (GL_QUADS);
  /* test if camera is inside the shadow */
  coord[0] = 0.5 * (frustum->points[0] + frustum->points[6]);
  coord[1] = 0.5 * (frustum->points[1] + frustum->points[7]);
  coord[2] = 0.5 * (frustum->points[2] + frustum->points[8]);
  coord[3] = P3_point_distance_to (coord, frustum->points);
  if (volume->sphere[3] > 0.0 && P3_sphere_is_in_cone (coord, cone) == P3_TRUE) {
    /* camera is inside the shadow => special case 
     * we must draw the intersection of the shadow volume with the camera front plane
     * by chance we already have functions to do such thing in the watercube ;)
     */

// TO DO

#if 0

    P3_watercube_underwater* underwater;
    P3_chunk* chunk = P3_get_chunk ();
    GLfloat* coord_ptr2;
    GLfloat* m1;
    GLfloat* m2;
    GLfloat coord2[3];
    GLfloat coord3[3];
    GLfloat coord4[3];
    m1 = P3_coordsys_get_root_matrix ((P3_coordsys*) inst);
    m2 = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
    /* find edges and draw shadow volume */
    for (i = 0; i < fl->nb_faces; i++) {
      face = fl->faces[i];
      if (face->option & P3_FACE_BACK) {
        /* test if neighbors are front */
        if (face->option & P3_FACE_QUAD) { 
          neighbors = (P3_xface**) (&(face->v4) + 1);
          nbv = 4;
        } else {
          neighbors = (P3_xface**) (&(face->v4));
          nbv = 3;
        }
        for (k = 0; k < nbv; k++) {
          if (neighbors[k] == NULL || neighbors[k]->option & P3_FACE_FRONT) {

// TO DO avoid pushing 2 vertices with the same coord

            /* add edge k = vertices k and k + 1 */
            coord_ptr = mesh->vertex_coords[(&(face->v1))[k]];
            glVertex3fv (coord_ptr);
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              coord[0] = coord_ptr[0] + b * light->data[0];
              coord[1] = coord_ptr[1] + b * light->data[1];
              coord[2] = coord_ptr[2] + b * light->data[2];
              glVertex3fv (coord);
            } else {
              coord[0] = coord_ptr[0] - light->data[0];
              coord[1] = coord_ptr[1] - light->data[1];
              coord[2] = coord_ptr[2] - light->data[2];
              P3_vector_normalize (coord);
              coord[0] = coord_ptr[0] + b * coord[0];
              coord[1] = coord_ptr[1] + b * coord[1];
              coord[2] = coord_ptr[2] + b * coord[2];
              glVertex3fv (coord);
            }
            if (k < nbv - 1) {
              coord_ptr2 = mesh->vertex_coords[(&(face->v1))[k + 1]];
            } else {
              coord_ptr2 = mesh->vertex_coords[face->v1];
            }
            /* push coord far away */
            if (light->option & P3_LIGHT_DIRECTIONAL) {
              coord2[0] = coord_ptr2[0] + b * light->data[0];
              coord2[1] = coord_ptr2[1] + b * light->data[1];
              coord2[2] = coord_ptr2[2] + b * light->data[2];
              glVertex3fv (coord2);
            } else {
              coord2[0] = coord_ptr2[0] - light->data[0];
              coord2[1] = coord_ptr2[1] - light->data[1];
              coord2[2] = coord_ptr2[2] - light->data[2];
              P3_vector_normalize (coord2);
              coord2[0] = coord_ptr2[0] + b * coord2[0];
              coord2[1] = coord_ptr2[1] + b * coord2[1];
              coord2[2] = coord_ptr2[2] + b * coord2[2];
              glVertex3fv (coord2);
            }
            glVertex3fv (coord_ptr2);
            /* convert point to camera coordsys */
            P3_point_by_matrix_copy (coord3, coord_ptr,  m1);
            P3_point_by_matrix_copy (coord4, coord_ptr2, m1);
            P3_point_by_matrix (coord,  m1);
            P3_point_by_matrix (coord2, m1);
            P3_point_by_matrix (coord,  m2);
            P3_point_by_matrix (coord2, m2);
            P3_point_by_matrix (coord3, m2);
            P3_point_by_matrix (coord4, m2);
            P3_watercube_underwater_intersect_face (chunk, coord3, coord, coord2);
            P3_watercube_underwater_intersect_face (chunk, coord2, coord4, coord3);
          }
        }
      }
    }
    glEnd ();
    glEndList ();
    glDisable (GL_CULL_FACE);
    glDisable (GL_DEPTH_TEST);
    glLoadIdentity ();
    if (chunk->nb == 0) {
      /* 2 cases: front plane entirely inside the shadow or entirely outside
       * use a raypicking to be sure
       */
      P3_raypick_data rdata;
      GLfloat f;
      rdata.option = P3_RAYPICK_HALF_LINE;
      rdata.raypicked = P3_get_list ();
      rdata.raypick_data = P3_get_chunk ();
      if (light->option & P3_LIGHT_DIRECTIONAL) {
        rdata.root_data[0] = renderer->r_frustum->position[0];
        rdata.root_data[1] = renderer->r_frustum->position[1];
        rdata.root_data[2] = renderer->r_frustum->position[2];
        coord_ptr = P3_coordsys_get_root_matrix ((P3_coordsys*) light);
        rdata.root_data[6] = renderer->c_camera->back;
        rdata.root_data[3] = rdata.root_data[0] + rdata.root_data[6] * coord_ptr[ 8];
        rdata.root_data[4] = rdata.root_data[1] + rdata.root_data[6] * coord_ptr[ 9];
        rdata.root_data[5] = rdata.root_data[2] + rdata.root_data[6] * coord_ptr[10];
      } else {
        coord_ptr = P3_coordsys_get_root_matrix ((P3_coordsys*) light);
        memcpy (rdata.root_data, coord_ptr + 12, 3 * sizeof (GLfloat));
        rdata.root_data[3] = renderer->r_frustum->position[0] - rdata.root_data[0];
        rdata.root_data[4] = renderer->r_frustum->position[1] - rdata.root_data[1];
        rdata.root_data[5] = renderer->r_frustum->position[2] - rdata.root_data[2];
        rdata.root_data[6] = P3_vector_length (rdata.root_data + 3);
        f = 1.0 / rdata.root_data[6];
        rdata.root_data[3] *= f;
        rdata.root_data[4] *= f;
        rdata.root_data[5] *= f;
      }
      if (P3_mesh_raypick_b (mesh, &rdata, (P3_raypickable*) inst) == P3_TRUE) {
        /* all the screen must be shadowed
         * that's cool I have the coordinate of the edge of the screen in the gl vertex array ;)
         */
        glDrawArrays (GL_QUADS, 0, 4);
      }
      for (i = 0; i < rdata.raypicked->nb; i++) {
        ((P3_raypickable*) P3_list_get (rdata.raypicked, i))->raypick_data = -1;
      }
      P3_drop_list  (rdata.raypicked);
      P3_drop_chunk (rdata.raypick_data);
    } else if (chunk->nb > 0) {
      underwater = P3_watercube_underwater_join_segments ((GLfloat*) chunk->content, chunk->nb / (3 * sizeof (GLfloat)));
      P3_watercube_underwater_draw_segments (underwater);
      free (underwater->points);
      free (underwater->seq_sizes);
      free (underwater);
    }
    P3_drop_chunk (chunk);
    glEnable (GL_CULL_FACE);
    glEnable (GL_DEPTH_TEST);
    /* draw shadow volume 2nd step */
    glLoadMatrixf (inst->render_matrix);
    glFrontFace (GL_CCW);
    glStencilFunc (GL_ALWAYS, 0, 0xFFFFFFFF);
    glStencilOp (GL_KEEP, GL_KEEP, GL_DECR);
    glCallList (shadow_display_list);

#endif

  } else {
    /* find edges and draw shadow volume */
    k = 0;
    for (i = 0; i < shape->nb_submeshes; i++) {
      submesh = shape->submeshes + i;
      if (volume->submeshes[i] == 0) {
        k += submesh->nb_faces;
        continue;
      }
      for (j = 0; j < submesh->nb_faces; j++) {
        if (face_options[k] == 0) {
          /* test if neighbors are front */
          neighbors = shape->face_neighbors + k * 3;
          for (n = 0; n < 3; n++) {
            neighbor = neighbors[n];
            if (neighbor == -1 || face_options[neighbor] == 0) {
              /* add edge n = vertices n and n + 1 */
              face = shape->faces + submesh->first_face + j * 3;
              coord_ptr = volume->vertex_coords + face[n] * 3;
              glVertex3fv (coord_ptr);
              /* push coord far away */
              if (light->option & P3_LIGHT_DIRECTIONAL) {
                glVertex3f (coord_ptr[0] + b * light->data[0], coord_ptr[1] + b * light->data[1], coord_ptr[2] + b * light->data[2]);
              } else {
                coord[0] = coord_ptr[0] - light->data[0];
                coord[1] = coord_ptr[1] - light->data[1];
                coord[2] = coord_ptr[2] - light->data[2];
                P3_vector_normalize (coord);
                glVertex3f (coord_ptr[0] + b * coord[0], coord_ptr[1] + b * coord[1], coord_ptr[2] + b * coord[2]);
              }
              if (n == 2) {
                coord_ptr = volume->vertex_coords + face[0] * 3;
              } else {
                coord_ptr = volume->vertex_coords + face[n + 1] * 3;
              }
              /* push coord far away */
              if (light->option & P3_LIGHT_DIRECTIONAL) {
                glVertex3f (coord_ptr[0] + b * light->data[0], coord_ptr[1] + b * light->data[1], coord_ptr[2] + b * light->data[2]);
              } else {
                coord[0] = coord_ptr[0] - light->data[0];
                coord[1] = coord_ptr[1] - light->data[1];
                coord[2] = coord_ptr[2] - light->data[2];
                P3_vector_normalize (coord);
                glVertex3f (coord_ptr[0] + b * coord[0], coord_ptr[1] + b * coord[1], coord_ptr[2] + b * coord[2]);
              }
              glVertex3fv (coord_ptr);
            }
          }
        }
      }
    }
    glEnd ();
    glEndList ();
    /* draw shadow volume 2nd step */
    glFrontFace (GL_CCW);
    glStencilFunc (GL_ALWAYS, 0, 0xFFFFFFFF);
    glStencilOp (GL_KEEP, GL_KEEP, GL_DECR);
    glCallList (shadow_display_list);
  }
  return P3_TRUE;
}


/*==============*
 * Cal3d Volume *
 *==============*/

P3_cal3d_volume* P3_cal3d_volume_new (P3_cal3d_volume* volume) {
  if (volume == NULL) {
    volume = (P3_cal3d_volume*) malloc (sizeof (P3_cal3d_volume));
  }
  P3_coordsys_initialize ((P3_coordsys*) volume);
  volume->class             = &P3_class_cal3d_volume;
  volume->shape             = NULL;
//  volume->nb_submeshes      = 0;
  volume->submeshes         = NULL;
  volume->delta_time        = 0.0f;
  volume->sphere[3]         = -1.0f;
  volume->attached_coordsys = PyList_New (0);
  
  return volume;
}

void P3_cal3d_volume_dealloc (P3_cal3d_volume* volume) {
  CalModel_Destroy (volume->model);
  CalModel_Delete  (volume->model);
  free (volume->attached_states);
  free (volume->submeshes);
}

void P3_cal3d_volume_set_attached (P3_cal3d_volume* volume, int mesh_id, int attach) {
  if ((volume->model != NULL) && (volume->attached_states[mesh_id] != attach)) {
    if (attach > 0) {
      CalModel_AttachMesh(volume->model, mesh_id);
      volume->attached_states[mesh_id] = 1;
    }
    else {
      CalModel_DetachMesh(volume->model, mesh_id);
      volume->attached_states[mesh_id] = 0;
    }
  }
// TO DO rebuld submeshes?
}

void P3_cal3d_volume_attach_all (P3_cal3d_volume* volume) {
  int i;
  if (volume->model != NULL) {
    for (i = 0; i < volume->nb_attached_states; i++) {
      if (volume->attached_states[i] == 0) {
        volume->attached_states[i] = 1;
        CalModel_AttachMesh(volume->model, i);
      }
    }
    P3_cal3d_volume_build_submeshes(volume);
  }
}

void P3_cal3d_volume_batch (P3_cal3d_volume* volume, P3_instance* csys) {
  int i;
  int nb;
  PyObject* attached_coordsys_bone_id;
  struct CalBone* bone;
  P3_instance* instance;
  float* trans;
  float* quat;
  
  volume->frustum_data = -1;

//  if (volume->nb_submeshes > 0) {
    CalModel_Update (volume->model, volume->delta_time); 
    volume->delta_time = 0.0f;
    
    /* Updates coordsys attached to a bone */
    nb = P3_children_size (volume->attached_coordsys);
    for (i = 0; i < nb; i++) {
#ifdef COMPILE_FOR_PYTHON
      attached_coordsys_bone_id = PySequence_Fast_GET_ITEM (volume->attached_coordsys, i);
      instance = (P3_instance*) PySequence_Fast_GET_ITEM (attached_coordsys_bone_id, 0);
      
      bone = CalSkeleton_GetBone (CalModel_GetSkeleton (volume->model), (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (attached_coordsys_bone_id, 1)));
#endif /* COMPILE_FOR_PYTHON */
      
      quat = CalQuaternion_Get (CalBone_GetRotationAbsolute(bone));
      quat[3] = -quat[3]; // Cal3D use indirect frame or what ???
      P3_matrix_from_quaternion (instance->m, quat);
      
      trans = CalVector_Get (CalBone_GetTranslationAbsolute(bone));
      instance->m[12] = (GLfloat) trans[0];
      instance->m[13] = (GLfloat) trans[1];
      instance->m[14] = (GLfloat) trans[2];
      P3_object_invalid ((P3_any_object*) instance);
    }
    
// USELESS !
/*
    if (volume->sphere[3] > 0.0 &&
        P3_sphere_in_frustum (P3_renderer_get_frustum ((P3_coordsys*) volume), volume->sphere) == P3_FALSE) {
      return;
    }
*/   
//  }
  if (!(volume->option & P3_OBJECT_HIDDEN)) {
    if (volume->sphere[3] > 0.0) {
      volume->frustum_data = -1;
      if (P3_sphere_in_frustum (P3_renderer_get_frustum ((P3_instance*) volume), volume->sphere) == P3_FALSE)
        return;
    }
    
    P3_multiply_matrix (volume->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) volume));
    
    P3_renderer_add (volume->shape, (P3_instance*) volume);
    if (volume->option & P3_CAL3D_ALPHA) P3_renderer_add_alpha (volume->shape, (P3_instance*) volume);
  }
}

int P3_cal3d_volume_shadow (P3_cal3d_volume* volume, P3_instance* inst, P3_light* light) {
  return P3_cal3d_shape_shadow (volume->shape, volume, light);
}

void P3_cal3d_volume_set_shape (P3_cal3d_volume* volume, P3_cal3d_shape* shape) {
  int i;
  if(volume->shape != NULL) {
    CalModel_Destroy(volume->model);
    CalModel_Delete (volume->model);
    free (volume->attached_states); volume->nb_attached_states = 0;
  }
  
  if (shape == NULL) {
    volume->shape = NULL;
  } else {
    volume->shape = shape;
    volume->model = CalModel_New();
    CalModel_Create(volume->model, shape->core_model);
    
    volume->nb_attached_states = CalCoreModel_GetCoreMeshCount(shape->core_model);
    volume->attached_states = malloc(volume->nb_attached_states * sizeof(int));
    for (i = 0; i < volume->nb_attached_states; i++) { volume->attached_states[i] = 0; }
    
  }
  P3_cal3d_volume_build_submeshes(volume);
}

void P3_cal3d_volume_build_submeshes (P3_cal3d_volume* volume) {
  struct CalCoreSubmesh* submesh;
  struct CalCoreMesh* mesh;
  P3_cal3d_submesh* p3submesh;
  P3_cal3d_submesh* ptr;
  char* ptrc;
  int i;
  int j;
  int l = 0;
  int k = 0;
  int nb_coremeshes;
  int nb_submeshes;
  int nb;

  if (volume->submeshes != NULL) {
// TO DO material decref?
    free (volume->submeshes);
    volume->submeshes = NULL;
  }
//  volume->nb_submeshes = 0;
//  volume->nb_faces = 0;
//  volume->nb_vertices = 0;
//  if (volume->model == NULL) return;
  if (!(volume->shape->option & P3_CAL3D_INITED)) P3_cal3d_shape_init (volume->shape);

  volume->submeshes = (char*) calloc (volume->shape->nb_submeshes, sizeof (char));
  ptrc = volume->submeshes;

  nb_coremeshes = CalCoreModel_GetCoreMeshCount(volume->shape->core_model);
  l = 0;
  for (i = 0; i < nb_coremeshes; i++) {
    mesh = CalCoreModel_GetCoreMesh (volume->shape->core_model, i);
    nb_submeshes = CalCoreMesh_GetCoreSubmeshCount (mesh);

    if (volume->attached_states[i] == 1) {

//      j = volume->nb_submeshes;
//      volume->nb_submeshes += nb_submeshes;
//      volume->submeshes = (P3_cal3d_submesh**) realloc (volume->submeshes, volume->nb_submeshes * sizeof (P3_cal3d_submesh*));

      for (j = 0; j < nb_submeshes; j++) {
        *ptrc++ = 1;
//        for (k = 0; k < volume->shape->nb_submeshes; k++) {
//          ptr = volume->shape->submeshes + k;
//          if (ptr->mesh == i && ptr->submesh == j) break;
//        }
//        volume->submeshes[l++] = ptr;
//        volume->nb_faces += ptr->nb_faces;
//        volume->nb_vertices += ptr->nb_vertices;
      }
    } else {
      ptrc += nb_submeshes;
    }
  }

  volume->vertex_coords = (GLfloat*) malloc (volume->shape->nb_vertices * 3 * sizeof (GLfloat));
  if (volume->shape->option & P3_CAL3D_CELL_SHADING || volume->shape->option & P3_CAL3D_SHADOW_CAST)
    volume->face_normals = (GLfloat*) malloc (volume->shape->nb_faces * 4 * sizeof (GLfloat));


#if 0  
  CalModel_SetMaterialSet (volume->model, 0);
  
  nb_coremeshes = CalCoreModel_GetCoreMeshCount(volume->shape->core_model);

  for (i = 0; i < nb_coremeshes; i++) {
    if (volume->attached_states[i] == 1) {
      volume->nb_submeshes += CalCoreMesh_GetCoreSubmeshCount (CalCoreModel_GetCoreMesh (volume->shape->core_model, i));
    }
  }
  if (volume->nb_submeshes == 0) { return; }
  volume->submeshes = (P3_cal3d_submesh*) malloc (volume->nb_submeshes * sizeof (P3_cal3d_submesh));
  
  for (i = 0; i < nb_coremeshes; i++) {
    if (volume->attached_states[i] == 1) {
      mesh = CalCoreModel_GetCoreMesh (volume->shape->core_model, i);
      nb_submeshes = CalCoreMesh_GetCoreSubmeshCount (mesh);

      for (j = 0; j < nb_submeshes; j++) {
        submesh = CalCoreMesh_GetCoreSubmesh (mesh, j);
        p3submesh = volume->submeshes + k;

        p3submesh->mesh     = l;
        p3submesh->submesh  = j;
        p3submesh->material = volume->shape->materials[(int) CalCoreSubmesh_GetCoreMaterialThreadId (CalCoreMesh_GetCoreSubmesh (mesh, j))];
        if (p3submesh->material->option & P3_MATERIAL_ALPHA) { volume->option |= P3_CAL3D_ALPHA; }

        nb = CalCoreSubmesh_GetFaceCount (submesh);
        if (cal3d_nb_faces < nb) {
          cal3d_faces_array = (CalIndex*) realloc (cal3d_faces_array, nb * 3 * sizeof (CalIndex));
          if (volume->shape->option & P3_CAL3D_CELL_SHADING)
            cal3d_fnormals_array = (float*) realloc (cal3d_fnormals_array, nb * 4 * sizeof (float));
          cal3d_nb_faces = nb;
        }

        nb = CalCoreSubmesh_GetVertexCount (submesh);
        if (cal3d_nb_vertices < nb) {
          cal3d_coords_array  = (float*) realloc (cal3d_coords_array,  nb * 3 * sizeof (float));
          cal3d_normals_array   = (float*) realloc (cal3d_normals_array,   nb * 3 * sizeof (float));
          cal3d_texcoords_array = (float*) realloc (cal3d_texcoords_array, nb * 2 * sizeof (float));
          if (volume->shape->option & P3_CAL3D_CELL_SHADING)
            cal3d_shades_array = (float*) realloc (cal3d_shades_array, nb * sizeof (float));
          cal3d_nb_vertices = nb;
        }

        k++;
      }
      l++;
    }
  }

#endif

}

void P3_cal3d_volume_get_data (P3_cal3d_volume* v, P3_chunk* chunk) {
/*
  P3_chunk_add (chunk, v->m, 19 * sizeof (GLfloat));
  P3_chunk_add_int (chunk, v->option);
  P3_chunk_add (chunk, v->sphere, 4 * sizeof (GLfloat));
*/
}

void P3_cal3d_volume_set_data (P3_cal3d_volume* v, P3_chunk* chunk) {
/*
  v->class = &P3_class_cal3d_volume;
  v->validity = P3_COORDSYS_INVALID;
  v->nb_submeshes = 0;
  v->submeshes = NULL;
  v->delta_time = 0.0;
  v->parent = NULL;
  P3_chunk_get(chunk, v->m, 19 * sizeof (GLfloat));
  v->option = P3_chunk_get_int (chunk);
  P3_chunk_get (chunk, v->sphere, 4 * sizeof (GLfloat));
*/
}

#endif /* HAVE_CAL3D */
