/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * land.c
 * land size must be like this: (2^n + 1)
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include <math.h>

#include "p3_base.h"
#include "gladd.h"
#include "math3d.h"
#include "util.h"
#include "coordsys.h"
#include "frustum.h"
#include "renderer.h"
#include "material.h"
#include "image.h"
#include "light.h"
#include "raypick.h"
#include "mesh.h"
#include "fx.h"
#include "land.h"

extern P3_renderer* renderer;
extern int engine_option;
extern GLfloat white[4];
extern GLfloat black[4];
extern GLfloat transparency[4];
extern P3_material* current_material;
extern P3_list* land_tri_recycler;

#define P3_land_tri_has_child(tri) \
  (tri->left_child != NULL)

#define P3_land_tri_diamond(tri) \
  (tri->base_neighbor == NULL || (tri->v2 == tri->base_neighbor->v3 && tri->v3 == tri->base_neighbor->v2))

#define JIBA_BUGGED_OPENGL 1


/*======+
 | LAND |
 +======*/

P3_class P3_class_land = { 
  P3_ID_LAND,
  (batch_func)     P3_land_batch,
  (render_func)    P3_land_render,
  (shadow_func)    0,
  (raypick_func)   P3_land_raypick,
  (raypick_b_func) P3_land_raypick_b,
};


void P3_land_check_size (P3_land* land) {
  int i;
  int d;
  i = 0;
  while (1) {
    d = (1 << i) + 1;
    if (land->nb_vertex_width < d) {
      P3_error ("WARNING Land size width must be (2^n) + 1");
      break;
    } else if (land->nb_vertex_width == d) { break; }
    i++;
  }
  i = 0;
  while (1) {
    d = (1 << i) + 1;
    if (land->nb_vertex_depth < d) {
      P3_error ("WARNING Land size depth must be (2^n) + 1");
      break;
    } else if (land->nb_vertex_depth == d) { break; }
    i++;
  }
}

P3_land* P3_land_new (P3_land* land, int width, int depth) {
  if (land == NULL) land = (P3_land*) malloc (sizeof (P3_land));
  land->class = &P3_class_land;
  land->nb_vertex_width = width;
  land->nb_vertex_depth = depth;
  land->patch_size = 8;
  land->nb_materials = 0;
  land->materials = NULL;
  land->texture_factor = 1.0;
  land->scale_factor = 1.5;
  land->split_factor = 2.0;
  land->nb_patchs = 0;
  land->patchs = NULL;
  land->nb_colors = 0;
  land->colors = NULL;
  land->vertex_options = NULL;
  land->vertex_colors  = NULL;
  land->vertex_warfogs = NULL;
  if (width == 0 || depth == 0) {
    land->vertices = NULL;
    land->normals = NULL;
  } else {
    P3_land_vertex* v;
    int i;
    int nb = land->nb_vertex_width * land->nb_vertex_depth;
    P3_land_check_size (land);
    land->vertices = (P3_land_vertex*) malloc (nb * sizeof (P3_land_vertex));
    for (i = 0; i < nb; i++) {
      v = land->vertices + i;
      v->coord[1] = 0;
      v->pack = P3_xpack_get (P3_FACE_TRIANGLE, NULL);
    }
    land->normals = (GLfloat*) malloc ((land->nb_vertex_width - 1) * (land->nb_vertex_depth - 1) * 6 * sizeof (GLfloat));
  }
  P3_land_compute_coords (land);
  return land;
}

void P3_land_tri_dealloc (P3_land_tri* tri) {
  if (P3_land_tri_has_child (tri)) {
    P3_land_tri_dealloc (tri->left_child);
    P3_land_tri_dealloc (tri->right_child);
  }
  free (tri);
}

void P3_land_free_patchs (P3_land* land) {
  P3_land_patch* patch;
  int i;
  for (i = 0; i < land->nb_patchs; i++) {
    patch = land->patchs + i;
    P3_land_tri_dealloc (patch->tri_top);
    P3_land_tri_dealloc (patch->tri_left);
    P3_land_tri_dealloc (patch->tri_right);
    P3_land_tri_dealloc (patch->tri_bottom);
  }
  free (land->patchs);
  land->patchs = NULL;
}

void P3_land_dealloc (P3_land* land) {
  P3_land_free_patchs (land);
  free (land->vertices);
  free (land->colors);
  free (land->normals);
  free (land->materials);
  free (land->vertex_options);
  free (land->vertex_colors);
  free (land->vertex_warfogs);
}

void P3_land_from_image (P3_land* land, P3_image* image) {
  GLubyte* ptr;
  int nb;
  int i;
  land->nb_vertex_width = image->width;
  land->nb_vertex_depth = image->height;
  P3_land_check_size (land);
  if (land->vertices != NULL) { free (land->vertices); }
  nb = land->nb_vertex_width * land->nb_vertex_depth;
  land->vertices = (P3_land_vertex*) malloc (nb * sizeof (P3_land_vertex));
  for (i = 0; i < nb; i++) { (land->vertices + i)->pack = P3_xpack_get (P3_FACE_TRIANGLE, NULL); }
  land->normals = (GLfloat*) malloc ((land->nb_vertex_width - 1) * (land->nb_vertex_depth - 1) * 6 * sizeof (GLfloat));
  switch (image->nb_color) {
  case 3:
    for (i = 0; i < nb; i++) {
      ptr = image->pixels + i * 3;
      (land->vertices + i)->coord[1] = (ptr[0] + ptr[1] + ptr[2]) / (3.0 * 255.0);
    }
    break;
  case 4:
    for (i = 0; i < nb; i++) {
      ptr = image->pixels + i * 4;
      (land->vertices + i)->coord[1] = (ptr[0] + ptr[1] + ptr[2] + ptr[3]) / (4.0 * 255.0);
    }
    break;
  case 1:
    for (i = 0; i < nb; i++) {
      (land->vertices + i)->coord[1] = image->pixels[i] / 255.0;
    }
    break;
  }
  land->option &= ~P3_LAND_INITED;
  P3_land_compute_coords (land);
}

void P3_land_add_material (P3_land* land, P3_material* material) {
  int i;
  for (i = 0; i < land->nb_materials; i++) {
    if (land->materials[i] == material) {
      return;
    }
  }
  i = land->nb_materials;
  (land->nb_materials)++;
  land->materials = (P3_material**) realloc (land->materials, land->nb_materials * sizeof (P3_material*));
  land->materials[i] = material;
  if (material != NULL) { P3_material_incref (material); }
  land->option &= ~P3_LAND_INITED;
}

void P3_land_set_material_layer (P3_land* land, P3_material* material, GLfloat start, GLfloat end) {
  int i; int j;
  P3_land_vertex* vertex;
  P3_xpack* pack;
  P3_land_add_material (land, material);
  pack = P3_xpack_get (P3_FACE_TRIANGLE, material);
  for (j = 0; j < land->nb_vertex_depth; j++) {
    for (i = 0; i < land->nb_vertex_width; i++) {
      vertex = P3_land_get_vertex (land, i, j);
      if (vertex->coord[1] >= start && vertex->coord[1] <= end) {
        vertex->pack = pack;
      }
    }
  }
  land->option &= ~P3_LAND_INITED;
}

void P3_land_set_material_layer_angle (P3_land* land, P3_material* material, GLfloat start, GLfloat end, GLfloat angle_min, GLfloat angle_max) {
  int i; int j;
  P3_land_vertex* vertex;
  P3_xpack* pack;
  GLfloat f;
  GLfloat v[3] = { 0.0, 1.0, 0.0 };
  /* we must have computed normals */
  if (!(land->option & P3_LAND_INITED)) { P3_land_init (land); }
  /* angle 0 is horizontal */
  P3_land_add_material (land, material);
  pack = P3_xpack_get (P3_FACE_TRIANGLE, material);
  for (j = 0; j < land->nb_vertex_depth; j++) {
    for (i = 0; i < land->nb_vertex_width; i++) {
      vertex = P3_land_get_vertex (land, i, j);
      f = fabs (P3_to_degrees (P3_vector_angle (vertex->normal, v)));
      if (vertex->coord[1] >= start && vertex->coord[1] <= end && f >= angle_min && f <= angle_max) {
        vertex->pack = pack;
      }
    }
  }
  land->option &= ~P3_LAND_INITED;
}

GLfloat P3_land_get_height (P3_land* land, int x, int z) {
  if (x < 0 || z < 0 || x >= land->nb_vertex_width || z >= land->nb_vertex_depth) { return 0.0; }
  return P3_land_get_vertex(land, x, z)->coord[1];
}

P3_material* P3_land_get_material (P3_land* land, int x, int z) {
  if (x < 0 || z < 0 || x >= land->nb_vertex_width || z >= land->nb_vertex_depth) { return NULL; }
  return P3_land_get_vertex(land, x, z)->pack->material;
}

void P3_land_set_height (P3_land* land, int x, int z, GLfloat value) {
  if (x >= 0 || x < land->nb_vertex_width || z >= 0 || z < land->nb_vertex_depth) {
    P3_land_get_vertex(land, x, z)->coord[1] = value;
  }
}

/*========================+
 | CREATION / COMPUTATION |
 +========================*/

static void P3_land_tri_sphere (GLfloat r[4], GLfloat* p1, GLfloat* p2, GLfloat* p3) {
  GLfloat x; GLfloat y; GLfloat z;
  GLfloat f;
  GLfloat d;
  /* fast function to compute a bounding sphere for 3 points 
   * less optimized than P3_sphere_from_points()
   */
  f = 1.0 / 3.0;
  r[0] = (p1[0] + p2[0] + p3[0]) * f;
  r[1] = (p1[1] + p2[1] + p3[1]) * f;
  r[2] = (p1[2] + p2[2] + p3[2]) * f;
  x = p1[0] - r[0];
  y = p1[1] - r[1];
  z = p1[2] - r[2];
  d = x * x + y * y + z * z;
  x = p2[0] - r[0];
  y = p2[1] - r[1];
  z = p2[2] - r[2];
  f = x * x + y * y + z * z;
  if (f > d) { d = f; }
  x = p3[0] - r[0];
  y = p3[1] - r[1];
  z = p3[2] - r[2];
  f = x * x + y * y + z * z;
  if (f > d) { d = f; }
  r[3] = sqrt (d);
}

static P3_land_tri* P3_land_get_tri (void) {
  if (land_tri_recycler->nb > 0) {
    (land_tri_recycler->nb)--;
    return (P3_land_tri*) land_tri_recycler->content[land_tri_recycler->nb];
  } else {
    return (P3_land_tri*) malloc (sizeof (P3_land_tri));
  }
}

static void P3_land_tri_drop (P3_land_tri* t) {
  if (land_tri_recycler->nb > 1000) {
    free (t);
  } else {
    P3_list_add (land_tri_recycler, t);
  }
}

static P3_land_tri* P3_land_create_tri (P3_land_vertex* v1, P3_land_vertex* v2, P3_land_vertex* v3, P3_land_patch* patch) {
  P3_land_tri* tri;
  tri = P3_land_get_tri ();
  tri->level = 0;
  tri->v1 = v1;
  tri->v2 = v2;
  tri->v3 = v3;
  P3_face_normal (tri->normal, tri->v1->coord, tri->v2->coord, tri->v3->coord);
  P3_vector_normalize (tri->normal);
  tri->parent = NULL;
  tri->left_child  = NULL;
  tri->right_child = NULL;
  tri->base_neighbor = NULL;
  tri->patch = patch;
  P3_land_tri_sphere (tri->sphere, tri->v1->coord, tri->v2->coord, tri->v3->coord);
  return tri;
}

static P3_land_tri* P3_land_create_child_tri (P3_land_vertex* v1, P3_land_vertex* v2, P3_land_vertex* v3, P3_land_tri* tparent) {
  P3_land_tri* tri;
  tri = P3_land_get_tri ();
  tri->level = tparent->level + 1;
  tri->parent = tparent;
  tri->v1 = v1;
  tri->v2 = v2;
  tri->v3 = v3;
  P3_face_normal (tri->normal, tri->v1->coord, tri->v2->coord, tri->v3->coord);
  P3_vector_normalize (tri->normal);
  tri->left_child  = NULL;
  tri->right_child = NULL;
  tri->patch = tparent->patch;
  P3_land_tri_sphere (tri->sphere, tri->v1->coord, tri->v2->coord, tri->v3->coord);
  return tri;
}


static void P3_land_compute_normal (P3_land* land, int x, int y) {
  GLfloat y0, y1, y2, y3, y5, y6, y7, y8;
  GLfloat a, b, c;
  P3_land_vertex* vertex;
  /* compute vertex normal */
  vertex = P3_land_get_vertex (land, x, y);
  y0 = P3_land_get_height (land, x - 1, y - 1);
  y1 = P3_land_get_height (land, x    , y - 1);
  y2 = P3_land_get_height (land, x + 1, y - 1);
  y3 = P3_land_get_height (land, x - 1, y    );
  y5 = P3_land_get_height (land, x + 1, y    );
  y6 = P3_land_get_height (land, x - 1, y + 1);
  y7 = P3_land_get_height (land, x    , y + 1);
  y8 = P3_land_get_height (land, x + 1, y + 1);
  a = (-y0 + y2 + 2.0 * y5 + y8 - y6 - 2.0 * y3) * 0.125;
  b = (-2.0 * y1 - y2 + y8 + 2.0 * y7 + y6 - y0) * 0.125;
  c = 1.0 / sqrt (a * a + 1.0f + b * b);
  vertex->normal[0] = - a * c;
  vertex->normal[1] =       c;
  vertex->normal[2] = - b * c;
}

void P3_land_compute_normals (P3_land* land) {
  int i; int j;
  GLfloat* ptr;
  P3_land_vertex* v1; P3_land_vertex* v2; P3_land_vertex* v3; P3_land_vertex* v4;
  /* compute points normals */
  for (j = 0; j < land->nb_vertex_depth; j++) {
    for (i = 0; i < land->nb_vertex_width; i++) {
      P3_land_compute_normal (land, i, j);
    }
  }
  /* compute triangles normals */
  ptr = land->normals;
  for (j = 0; j < land->nb_vertex_depth - 1; j++) {
    for (i = 0; i < land->nb_vertex_width - 1; i++) {
      v1 = P3_land_get_vertex (land, i,     j);
      v2 = P3_land_get_vertex (land, i + 1, j);
      v3 = P3_land_get_vertex (land, i + 1, j + 1);
      v4 = P3_land_get_vertex (land, i,     j + 1);
      if (((i & 1) && (j & 1)) || (!(i & 1) && !(j & 1))) {
        P3_face_normal (ptr,     v4->coord, v3->coord, v1->coord);
        P3_vector_normalize (ptr);
        P3_face_normal (ptr + 3, v2->coord, v1->coord, v3->coord);
        P3_vector_normalize (ptr + 3);
      } else {
        P3_face_normal (ptr,     v1->coord, v4->coord, v2->coord);
        P3_vector_normalize (ptr);
        P3_face_normal (ptr + 3, v3->coord, v2->coord, v4->coord);
        P3_vector_normalize (ptr + 3);
      }
      ptr += 6;
    }
  }
}

void P3_land_compute_coords (P3_land* land) {
  int i; int j; int k;
  P3_land_vertex* vertex;
  k = 0;
  for (i = 0; i < land->nb_vertex_depth; i++) {
    for (j = 0; j < land->nb_vertex_width; j++) {
      vertex = land->vertices + k;
      vertex->coord[0]    = ((GLfloat) j) * land->scale_factor;
      vertex->coord[2]    = ((GLfloat) i) * land->scale_factor;
      vertex->texcoord[0] = ((GLfloat) j) * land->texture_factor;
      vertex->texcoord[1] = ((GLfloat) i) * land->texture_factor;
      k++;
    }
  }
}

static void P3_land_create_patch (P3_land* land, P3_land_patch* patch, int x, int z, int patch_size) {
  int i; int j; int m;
  int nb;
  int x1; int z1; int x2; int z2;
  P3_land_vertex* vertex;
  P3_land_vertex* center;
  P3_land_vertex* v0;
  P3_land_vertex* v1;
  P3_land_vertex* v2;
  P3_land_vertex* v3;
  GLfloat* coords;
  patch->level = 0;
  x1 = x       * patch_size;
  x2 = (x + 1) * patch_size;
  z1 = z       * patch_size;
  z2 = (z + 1) * patch_size;
  /* compute bounding sphere */
  nb = (patch_size + 1) * (patch_size + 1);
  coords = (GLfloat*) malloc (nb * 3 * sizeof (GLfloat));
  m = 0;
  for (i = x1; i <= x2; i++) {
    for (j = z1; j <= z2; j++) {
      vertex = P3_land_get_vertex (land, i, j);
      memcpy (coords + m, vertex->coord, 3 * sizeof (GLfloat));
      m += 3;
    }
  }
  P3_sphere_from_points (patch->sphere, coords, nb);
  free (coords);
  /* create tris */
  v0 = P3_land_get_vertex(land, x1, z1);
  v1 = v0 + land->patch_size;
  v2 = v0 + land->patch_size * land->nb_vertex_width;
  v3 = v2 + land->patch_size;
  center = v0 + ((v3 - v0) >> 1);
  patch->tri_top    = P3_land_create_tri (center, v1, v0, patch);
  patch->tri_left   = P3_land_create_tri (center, v0, v2, patch);
  patch->tri_right  = P3_land_create_tri (center, v3, v1, patch);
  patch->tri_bottom = P3_land_create_tri (center, v2, v3, patch);
  patch->tri_top->left_neighbor     = patch->tri_right;
  patch->tri_top->right_neighbor    = patch->tri_left;
  patch->tri_top->base_neighbor     = NULL;
  patch->tri_left->left_neighbor    = patch->tri_top;
  patch->tri_left->right_neighbor   = patch->tri_bottom;
  patch->tri_left->base_neighbor    = NULL;
  patch->tri_right->left_neighbor   = patch->tri_bottom;
  patch->tri_right->right_neighbor  = patch->tri_top;
  patch->tri_right->base_neighbor   = NULL;
  patch->tri_bottom->left_neighbor  = patch->tri_left;
  patch->tri_bottom->right_neighbor = patch->tri_right;
  patch->tri_bottom->base_neighbor  = NULL;
}

void P3_land_create_patchs (P3_land* land) {
  int i; int j; int k;
  P3_land_patch* m;
  land->max_level = P3_exp_of_2 (land->patch_size) * 2 - 1;
  /* create patchs */
  land->nb_patch_width = (int) ((land->nb_vertex_width - 1) / land->patch_size);
  land->nb_patch_depth = (int) ((land->nb_vertex_depth - 1) / land->patch_size);
  land->nb_patchs = land->nb_patch_width * land->nb_patch_depth;
  if (land->patchs != NULL) {
    P3_land_free_patchs (land);
  }
  land->patchs = (P3_land_patch*) malloc (land->nb_patchs * sizeof (P3_land_patch));
  k = 0;
  for (j = 0; j < land->nb_patch_depth; j++) {
    for (i = 0; i < land->nb_patch_width; i++) {
      P3_land_create_patch (land, land->patchs + k, i, j, land->patch_size);
      k++;
    }
  }
  /* set neighbors */
  for (j = 0; j < land->nb_patch_depth; j++) {
    for (i = 0; i < land->nb_patch_width; i++) {
      m = land->patchs + i + j * land->nb_patch_width;
      if (i > 0)                        { m->tri_left->base_neighbor   = (land->patchs + (i - 1) + j       * land->nb_patch_width)->tri_right;  }
      if (j > 0)                        { m->tri_top->base_neighbor    = (land->patchs + i       + (j - 1) * land->nb_patch_width)->tri_bottom; }
      if (i < land->nb_patch_width - 1) { m->tri_right->base_neighbor  = (land->patchs + (i + 1) + j       * land->nb_patch_width)->tri_left;   }
      if (j < land->nb_patch_depth - 1) { m->tri_bottom->base_neighbor = (land->patchs + i       + (j + 1) * land->nb_patch_width)->tri_top;    }
    }
  }
}

GLfloat* P3_land_register_color (P3_land* land, GLfloat color[4]) {
  GLfloat* old;
  int i;
  GLfloat* c = land->colors;
  for (i = 0; i < land->nb_colors; i++) {
    if (fabs (color[0] - c[0]) < P3_EPSILON &&
        fabs (color[1] - c[1]) < P3_EPSILON &&
        fabs (color[2] - c[2]) < P3_EPSILON &&
        fabs (color[3] - c[3]) < P3_EPSILON) return c;
    c += 4;
  }
  i = land->nb_colors * 4;
  (land->nb_colors)++;
  old = land->colors;
  land->colors = (GLfloat*) realloc (land->colors, land->nb_colors * 4 * sizeof (GLfloat));
  if (land->colors != old) {
    int nb = land->nb_vertex_width * land->nb_vertex_depth;
    int j;
    GLfloat* max = old + (land->nb_colors - 1) * 4;
    if (land->option & P3_LAND_COLORED) {
      for (j = 0; j < nb; j++) {
        if (land->vertex_colors[j] >= old && land->vertex_colors[j] < max) {
          land->vertex_colors[j] = land->colors + (land->vertex_colors[j] - old);
        }
      }
    }
    if (land->option & P3_LAND_WARFOG) {
      for (j = 0; j < nb; j++) {
        if (land->vertex_warfogs[j] >= old && land->vertex_warfogs[j] < max) {
          land->vertex_warfogs[j] = land->colors + (land->vertex_warfogs[j] - old);
        }
      }
    }
  }
  c = land->colors + i;
  memcpy (c, color, 4 * sizeof (GLfloat));
  return c;
}

void P3_land_compute_shadow_color (P3_land* land, P3_light* light, GLfloat* shadow_color) {
  int i;
  int nb = land->nb_vertex_width * land->nb_vertex_depth;
  GLfloat* scolor;
  GLfloat* wcolor;
  GLfloat* ocolor;
  GLfloat* old_colors;
  GLfloat color[4];
  if (!(land->option & P3_LAND_INITED)) { P3_land_init (land); }

printf ("Compute Land shadow colors\n");

  /* initialize vertex colors if necessary */
  old_colors = land->colors;
  land->colors = NULL;
  land->nb_colors = 0;
  if (!(land->option & P3_LAND_COLORED)) {
    land->option |= P3_LAND_COLORED;
    land->vertex_colors = (GLfloat**) malloc (nb * sizeof (GLfloat*));
    for (i = 0; i < nb; i++) {
      land->vertex_colors[i] = NULL;
    }
    land->nb_colors = 2;
    land->colors = (GLfloat*) malloc (8 * sizeof (GLfloat));
    wcolor = land->colors;
    scolor = land->colors + 4;
    memcpy (wcolor, white, 4 * sizeof (GLfloat));
    memcpy (scolor, shadow_color, 4 * sizeof (GLfloat));
  }
  /* compute shadows */
  for (i = 0; i < nb; i++) {
    if (P3_light_get_shadow_at (light, (land->vertices + i)->coord) == P3_TRUE) { 
      if (land->vertex_colors[i] == NULL) {
        land->vertex_colors[i] = scolor;
      } else {
        ocolor = land->vertex_colors[i];
        color[0] = ocolor[0] * shadow_color[0];
        color[1] = ocolor[1] * shadow_color[1];
        color[2] = ocolor[2] * shadow_color[2];
        color[3] = ocolor[3] * shadow_color[3];
        land->vertex_colors[i] = P3_land_register_color (land, color);
      }
    } else {
      if (land->vertex_colors[i] == NULL) {
        land->vertex_colors[i] = wcolor;
      } else {
        land->vertex_colors[i] = P3_land_register_color (land, land->vertex_colors[i]);
      }
    }
  }
  free (old_colors);

printf ("  indexed %i colors\n", land->nb_colors);
printf ("  [DONE]\n");

}


/*===============+
 | SPLIT / MERGE |
 +===============*/

static void P3_land_tri_update_neighbor_after_split (P3_land_tri* tri) {
  tri->left_child->left_neighbor = tri->right_child;
  if (tri->left_neighbor != NULL && P3_land_tri_has_child (tri->left_neighbor)) {
    tri->left_child->base_neighbor = tri->left_neighbor->right_child;
  } else {
    tri->left_child->base_neighbor = tri->left_neighbor;
  }
  tri->right_child->right_neighbor = tri->left_child;
  if (tri->right_neighbor != NULL && P3_land_tri_has_child (tri->right_neighbor)) {
    tri->right_child->base_neighbor  = tri->right_neighbor->left_child;
  } else {
    tri->right_child->base_neighbor  = tri->right_neighbor;
  }
  if (tri->base_neighbor == NULL) {
    tri->left_child->right_neighbor = NULL;
    tri->right_child->left_neighbor = NULL;
  } else {
    tri->left_child->right_neighbor = tri->base_neighbor->right_child;
    tri->right_child->left_neighbor = tri->base_neighbor->left_child;
  }
  if (tri->left_neighbor != NULL) {
    if (tri->v1 == tri->left_neighbor->v1) {
      tri->left_neighbor->right_neighbor = tri->left_child;
    } else {
      tri->left_neighbor->base_neighbor = tri->left_child;
      if (tri->left_neighbor->parent != NULL) {
        tri->left_neighbor->parent->right_neighbor = tri->left_child;
      }
    }
  }
  if (tri->right_neighbor != NULL) {
    if (tri->v1 == tri->right_neighbor->v1) {
      tri->right_neighbor->left_neighbor = tri->right_child;
    } else {
      tri->right_neighbor->base_neighbor = tri->right_child;
      if (tri->right_neighbor->parent != NULL) {
        tri->right_neighbor->parent->left_neighbor = tri->right_child;
      }
    }
  }
}

static void P3_land_tri_force_split (P3_land_tri* tri, P3_land_vertex* apex) {
  tri->left_child  = P3_land_create_child_tri (apex, tri->v1, tri->v2, tri);
  tri->right_child = P3_land_create_child_tri (apex, tri->v3, tri->v1, tri);
  P3_land_tri_update_neighbor_after_split (tri);
}

static void P3_land_tri_split (P3_land* land, P3_land_tri* tri) {
  P3_land_vertex* v;
  if (!P3_land_tri_diamond (tri)) {
    P3_land_tri_split (land, tri->base_neighbor);
  }
  /* hum that's beautiful code to get the vertex that is the middle of v2-v3 */
  if (tri->v2 < tri->v3) {
    v = tri->v2 + ((tri->v3 - tri->v2) >> 1);
  } else {
    v = tri->v3 + ((tri->v2 - tri->v3) >> 1);
  }
  tri->left_child  = P3_land_create_child_tri (v, tri->v1, tri->v2, tri);
  tri->right_child = P3_land_create_child_tri (v, tri->v3, tri->v1, tri);
  if (tri->base_neighbor != NULL) {
    P3_land_tri_force_split (tri->base_neighbor, v);
  }
  P3_land_tri_update_neighbor_after_split (tri);
}

static void P3_land_tri_update_neighbor_after_merge (P3_land_tri* tri) {
  if (tri->left_neighbor != NULL) {
    if (tri->left_neighbor->v1 == tri->v1) {
      tri->left_neighbor->right_neighbor = tri;
      if (P3_land_tri_has_child(tri->left_neighbor)) {
        tri->left_neighbor->right_child->base_neighbor = tri;
      }
    } else {
      tri->left_neighbor->base_neighbor = tri;
      if (tri->left_neighbor->parent != NULL) {
        tri->left_neighbor->parent->right_neighbor = tri;
      }
    }
  }
  if (tri->right_neighbor != NULL) {
    if (tri->right_neighbor->v1 == tri->v1) {
      tri->right_neighbor->left_neighbor = tri;
      if (P3_land_tri_has_child(tri->right_neighbor)) {
        tri->right_neighbor->left_child->base_neighbor = tri;
      }
    } else {
      tri->right_neighbor->base_neighbor = tri;
      if (tri->right_neighbor->parent != NULL) {
        tri->right_neighbor->parent->left_neighbor = tri;
      }
    }
  }
}

static int P3_land_tri_merge_child (P3_land* land, P3_land_tri* tri) {
  P3_land_tri* base;
  /* return P3_FALSE if merge can't be done, else P3_TRUE if the merge has been done */
  base = tri->base_neighbor;
  if (tri->left_child->level <= tri->left_child->patch->level || 
      (base != NULL && base->left_child->level <= base->left_child->patch->level)) {
    return P3_FALSE;
  }

#define P3_land_tri_recurse_merge(land, aaa) \
  if (P3_land_tri_has_child(aaa->left_child)) { \
    if (P3_land_tri_merge_child (land, aaa->left_child) == P3_FALSE) { return P3_FALSE; } \
  } \
  if (P3_land_tri_has_child(aaa->right_child)) { \
    if (P3_land_tri_merge_child (land, aaa->right_child) == P3_FALSE) { return P3_FALSE; } \
  }

  P3_land_tri_recurse_merge (land, tri);
  if (base != NULL) {
    P3_land_tri_recurse_merge (land, base);
  }

#undef P3_land_tri_recurse_merge

  if (land->option & P3_LAND_VERTEX_OPTIONS &&
      land->vertex_options[tri->left_child->v1 - land->vertices] & P3_LAND_VERTEX_FORCE_PRESENCE) {
    return P3_FALSE;
  }
  P3_land_tri_update_neighbor_after_merge (tri);
  if (base != NULL) {
    P3_land_tri_update_neighbor_after_merge (base);
    P3_land_tri_drop (base->left_child);
    P3_land_tri_drop (base->right_child);
    base->left_child  = NULL;
    base->right_child = NULL;
  }
  P3_land_tri_drop (tri->left_child);
  P3_land_tri_drop (tri->right_child);
  tri->left_child  = NULL;
  tri->right_child = NULL;
  return P3_TRUE;
}

static void P3_land_tri_set_level (P3_land* land, P3_land_tri* tri, char level) {
  if (P3_land_tri_has_child (tri)) {
    P3_land_tri_set_level (land, tri->left_child,  level);
    if (tri->left_child == NULL) {
      /* this means we have merged the children */
      P3_land_tri_set_level (land, tri, level);
    } else {
      P3_land_tri_set_level (land, tri->right_child, level);
    }
  } else {
    if (tri->level > level && tri->parent != NULL) {
        P3_land_tri_merge_child (land, tri->parent);
    } else if (tri->level < level) {
      P3_land_tri_split (land, tri);
      P3_land_tri_set_level (land, tri->left_child,  level);
      P3_land_tri_set_level (land, tri->right_child, level);
    }
  }
}

static void P3_land_patch_set_level (P3_land* land, P3_land_patch* patch, char level) {
  if (patch->level != level) {
    patch->level = level;
    P3_land_tri_set_level (land, patch->tri_top,    level);
    P3_land_tri_set_level (land, patch->tri_left,   level);
    P3_land_tri_set_level (land, patch->tri_right,  level);
    P3_land_tri_set_level (land, patch->tri_bottom, level);
  }
}

static void P3_land_patch_update (P3_land* land, P3_land_patch* patch, char* visibility, 
                                  P3_frustum* frustum, GLfloat* frustum_box) {
  GLfloat* v1;
  GLfloat* v2;
  GLfloat d; GLfloat r;
  char level;
  /* update patch LOD */
  v1 = patch->tri_top->v3->coord;
  v2 = patch->tri_bottom->v3->coord;
  if (frustum_box[0] > v2[0] || frustum_box[2] < v1[0] ||
      frustum_box[1] > v2[2] || frustum_box[3] < v1[2] ||
      P3_sphere_in_frustum (frustum, patch->sphere) == P3_FALSE) {
    *visibility = P3_FALSE;
    P3_land_patch_set_level (land, patch, 0);
  } else {
    *visibility = P3_TRUE;
    d = P3_point_distance_to (patch->sphere, frustum->position);
    r = patch->sphere[3] * land->split_factor;
    if (d <= r) {
      P3_land_patch_set_level (land, patch, land->max_level);
    } else {
      level = (char) (land->max_level + 1 - (int) (d / r));
      if (level < 0) { level = 0; }
      P3_land_patch_set_level (land, patch, level);
    }
  }
}

static void P3_land_tri_force_presence (P3_land* land, P3_land_tri* tri, P3_land_vertex* v) {
  if (v == tri->v1 || v == tri->v2 || v == tri->v3) { return; }
  if (P3_land_tri_has_child (tri)) {
    P3_land_tri_force_presence (land, tri->left_child,  v);
    P3_land_tri_force_presence (land, tri->right_child, v);
  } else {
    GLfloat u[2];
    GLfloat w[2];
    GLfloat k; GLfloat m; GLfloat f;
    GLfloat x; GLfloat z;
    u[0] = tri->v2->coord[0] - tri->v1->coord[0];
    u[1] = tri->v2->coord[2] - tri->v1->coord[2];
    w[0] = tri->v3->coord[0] - tri->v1->coord[0];
    w[1] = tri->v3->coord[2] - tri->v1->coord[2];
    x = v->coord[0] - tri->v1->coord[0];
    z = v->coord[2] - tri->v1->coord[2];
    f = 1.0 / (u[0] * w[1] - u[1] * w[0]);
    m = (u[0] * z - u[1] * x) * f;
    k = (w[1] * x - w[0] * z) * f;
    if (m >= 0.0 && m <= 1.0 && k >= 0.0 && k <= 1.0 && k + m <= 1.0) {
      P3_land_tri_split (land, tri);
      P3_land_tri_force_presence (land, tri->left_child,  v);
      P3_land_tri_force_presence (land, tri->right_child, v);
    }
  }
}

static void P3_land_force_presence (P3_land* land) {
  P3_land_vertex* v;
  P3_land_patch* patch;
  int a; int b;
  int i; int j;
  for (j = 0; j < land->nb_vertex_depth; j++) {
    for (i = 0; i < land->nb_vertex_width; i++) {
      if (land->vertex_options[i + j * land->nb_vertex_width] & P3_LAND_VERTEX_FORCE_PRESENCE) {
        a = (int) (((GLfloat) i) / land->patch_size);
        b = (int) (((GLfloat) j) / land->patch_size);
        if (a >= land->nb_patch_width) { a = land->nb_patch_width - 1; }
        if (b >= land->nb_patch_depth) { b = land->nb_patch_depth - 1; }
        patch = land->patchs + a + b * land->nb_patch_width;
        v = P3_land_get_vertex (land, i, j);
        P3_land_tri_force_presence (land, patch->tri_top,    v);
        P3_land_tri_force_presence (land, patch->tri_left,   v);
        P3_land_tri_force_presence (land, patch->tri_right,  v);
        P3_land_tri_force_presence (land, patch->tri_bottom, v);
      }
    }
  }
}


/*===========+
 | RENDERING |
 +===========*/

static void P3_land_tri_batch (P3_land* land, P3_land_tri* tri, P3_frustum* frustum) {
  P3_xpack* p1;
  P3_xpack* p2;
  P3_xpack* p3;
  P3_xpack* p4;
  if (P3_sphere_in_frustum (frustum, tri->sphere) == P3_TRUE) {
    if (P3_land_tri_has_child (tri)) {
      /* recurse to children */
      P3_land_tri_batch (land, tri->left_child,  frustum);
      P3_land_tri_batch (land, tri->right_child, frustum);
    } else {
      if (land->option & P3_LAND_VERTEX_OPTIONS) {
        int o1 = land->vertex_options[tri->v1 - land->vertices];
        int o2 = land->vertex_options[tri->v2 - land->vertices];
        int o3 = land->vertex_options[tri->v3 - land->vertices];
        if (o1 & (P3_LAND_VERTEX_HIDDEN | P3_LAND_VERTEX_INVISIBLE) && 
            o2 & (P3_LAND_VERTEX_HIDDEN | P3_LAND_VERTEX_INVISIBLE) && 
            o3 & (P3_LAND_VERTEX_HIDDEN | P3_LAND_VERTEX_INVISIBLE)) return;
        if (o1 & P3_LAND_VERTEX_ALPHA || o2 & P3_LAND_VERTEX_ALPHA || o3 & P3_LAND_VERTEX_ALPHA) {
          P3_xpack* p;

          /* all is drawed in second pass */
          p1 = P3_xpack_get_secondpass (tri->v1->pack);
          p2 = P3_xpack_get_secondpass (tri->v2->pack);
          p3 = P3_xpack_get_secondpass (tri->v3->pack);

// TO DO

          if (p1 < p2) p = p1; else p = p2;
          if (p3 < p)  p = p3;
          P3_xpack_batch_face (land, P3_xpack_get_secondpass (p), tri);
          if (p1 != p) P3_xpack_batch_face (land, p1, tri);
          if (p2 != p && p2 != p1) P3_xpack_batch_face (land, p2, tri);
          if (p3 != p && p3 != p1 && p3 != p2) P3_xpack_batch_face (land, p3, tri);


/*
          if (o1 & P3_LAND_VERTEX_ALPHA) {
            P3_xpack_batch_face (P3_xpack_get_secondpass (p1), tri);
            if (p2 != p1) P3_xpack_batch_face (p2, tri);
            if (p3 != p1 && p3 != p2) P3_xpack_batch_face (p3, tri);
            return;
          } else if (o2 & P3_LAND_VERTEX_ALPHA) {
            P3_xpack_batch_face (P3_xpack_get_secondpass (p2), tri);
            if (p1 != p2) P3_xpack_batch_face (p1, tri);
            if (p3 != p1 && p3 != p2) P3_xpack_batch_face (p3, tri);
            return;
          } else if (o3 & P3_LAND_VERTEX_ALPHA) {
            P3_xpack_batch_face (P3_xpack_get_secondpass (p3), tri);
            if (p1 != p3) P3_xpack_batch_face (p1, tri);
            if (p2 != p1 && p2 != p3) P3_xpack_batch_face (p2, tri);
            return;
          }
*/
/*
          if (nbalpha == P3_LAND_VERTEX_ALPHA) {

            if (o1 & P3_LAND_VERTEX_ALPHA && p2 != p3) {
              P3_xpack_batch_face (P3_xpack_get_secondpass (p1), tri);
              if (p2 != p1) P3_xpack_batch_face (p2, tri);
              if (p3 != p1 && p3 != p2) P3_xpack_batch_face (p3, tri);
              return;
            } else if (o2 & P3_LAND_VERTEX_ALPHA && p1 != p3) {
              P3_xpack_batch_face (P3_xpack_get_secondpass (p2), tri);
              if (p1 != p2) P3_xpack_batch_face (p1, tri);
              if (p3 != p1 && p3 != p2) P3_xpack_batch_face (p3, tri);
              return;
            } else if (o3 & P3_LAND_VERTEX_ALPHA && p1 != p2) {
              P3_xpack_batch_face (P3_xpack_get_secondpass (p3), tri);
              if (p1 != p3) P3_xpack_batch_face (p1, tri);
              if (p2 != p1 && p2 != p3) P3_xpack_batch_face (p2, tri);
              return;
            }

          }
*/

// ???
//          P3_xpack_batch_face (land, p1, tri);
//          if (p2 != p1) P3_xpack_batch_face (land, p2, tri);
//          if (p3 != p1 && p3 != p2) P3_xpack_batch_face (land, p3, tri);
          return;
        }
      }
      p1 = tri->v1->pack;
      p2 = tri->v2->pack;
      p3 = tri->v3->pack;
      /* we must render without alpha the pack that is the most represented 
       * this constraint is needed for special texturing smoothing
       */
      if (p1 == p2) {
        P3_xpack_batch_face (land, p1, tri);
        if (p3 != p1) P3_xpack_batch_face (land, P3_xpack_get_secondpass (p3), tri);
      } else if (p1 == p3) {
        P3_xpack_batch_face (land, p1, tri);
        if (p2 != p1) P3_xpack_batch_face (land, P3_xpack_get_secondpass (p2), tri);
      } else if (p2 == p3) {
        P3_xpack_batch_face (land, p2, tri);
        if (p1 != p2) P3_xpack_batch_face (land, P3_xpack_get_secondpass (p1), tri);
      } else {
        /* take in account the base neighbor if it's a diamond */
        if (tri->base_neighbor != NULL && (tri->v2 == tri->base_neighbor->v3 && tri->v3 == tri->base_neighbor->v2)) {
          p4 = tri->base_neighbor->v1->pack;
          if (p2 == p4) {
            P3_xpack_batch_face (land, p2, tri);
            P3_xpack_batch_face (land, P3_xpack_get_secondpass (p1), tri);
            P3_xpack_batch_face (land, P3_xpack_get_secondpass (p3), tri);
            return;
          } else if (p3 == p4) {
            P3_xpack_batch_face (land, p3, tri);
            P3_xpack_batch_face (land, P3_xpack_get_secondpass (p1), tri);
            P3_xpack_batch_face (land, P3_xpack_get_secondpass (p2), tri);
            return;
          }
        }
        P3_xpack_batch_face (land, p1, tri);
        P3_xpack_batch_face (land, P3_xpack_get_secondpass (p2), tri);
        P3_xpack_batch_face (land, P3_xpack_get_secondpass (p3), tri);
      }
    }
  }
}

static void P3_land_patch_batch (P3_land* land, P3_land_patch* patch, P3_frustum* frustum) {
  if (P3_sphere_in_frustum (frustum, patch->sphere) == P3_TRUE) {
    /* batch tri */
    P3_land_tri_batch (land, patch->tri_top,    frustum);
    P3_land_tri_batch (land, patch->tri_left,   frustum);
    P3_land_tri_batch (land, patch->tri_right,  frustum);
    P3_land_tri_batch (land, patch->tri_bottom, frustum);
  }
}

void P3_land_batch (P3_land* land, P3_instance* inst) {
  P3_chunk* chk;
  P3_land_patch* patch;
  P3_frustum* frustum;
  GLfloat* ptr;
  char* patch_visibility;
  GLfloat frustum_box[4];
  int i;
  if (!(land->option & P3_LAND_INITED)) { 
    P3_land_init (land);
    /* nb_patchs have changed */
    P3_land_batch (land, inst);
    return;
  }
  /* quadtree visibility computation */
  frustum = P3_renderer_get_frustum (inst);
  ptr = frustum->points;
  frustum_box[0] = ptr[0];
  frustum_box[1] = ptr[2];
  frustum_box[2] = ptr[0];
  frustum_box[3] = ptr[2];
  for (i = 3; i < 24; i++) {
    if (ptr[i] < frustum_box[0]) { frustum_box[0] = ptr[i]; }
    if (ptr[i] > frustum_box[2]) { frustum_box[2] = ptr[i]; }
    i += 2;
    if (ptr[i] < frustum_box[1]) { frustum_box[1] = ptr[i]; }
    if (ptr[i] > frustum_box[3]) { frustum_box[3] = ptr[i]; }
  }
  /* update all the land patches and compute their visibility */
  chk = P3_get_chunk ();
  P3_chunk_register (chk, land->nb_patchs * sizeof (char));
  patch_visibility = (char*) chk->content;
  patch = land->patchs;
  for (i = 0; i < land->nb_patchs; i++) {
    P3_land_patch_update (land, patch, patch_visibility + i, frustum, frustum_box);
    patch++;
  }
  /* batch */
  P3_xmesh_batch_start (inst);
  patch = land->patchs;
  for (i = 0; i < land->nb_patchs; i++) {
    if (patch_visibility[i] == P3_TRUE) { 
      P3_land_patch_batch (land, patch, frustum); 
    }
    patch++;
  }
  P3_drop_chunk (chk);
  P3_xmesh_batch_end ();
}

static void P3_land_vertex_render_opaque (P3_land* land, P3_land_vertex* vertex) {
  int index = (vertex - land->vertices);
#if JIBA_BUGGED_OPENGL 
  if (renderer->colors != NULL) glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, renderer->colors[index]);
#else
  if (renderer->colors != NULL) glColor4fv (renderer->colors[index]);
#endif
  glArrayElement (index);
}

#if JIBA_BUGGED_OPENGL 
#define SET_COLOR_OPAQUE \
  if (renderer->colors == NULL) {  \
    if (current_material == NULL) { \
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, white); \
    } else { \
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, current_material->diffuse); \
    } \
  } else { \
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, renderer->colors[index]); \
  }

#define SET_COLOR_ALPHA \
  if (renderer->colors == NULL) {  \
    if (current_material == NULL) { \
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, transparency); \
    } else { \
      color = current_material->diffuse; \
      ccc[0] = color[0]; \
      ccc[1] = color[1]; \
      ccc[2] = color[2]; \
      ccc[3] = 0.0; \
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, ccc); \
    } \
  } else { \
    color = renderer->colors[index]; \
    ccc[0] = color[0]; \
    ccc[1] = color[1]; \
    ccc[2] = color[2]; \
    ccc[3] = 0.0; \
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, ccc); \
  }
#else

#define SET_COLOR_OPAQUE \
  if (renderer->colors == NULL) {  \
    if (current_material == NULL) { \
      glColor4fv (white); \
    } else { \
      glColor4fv (current_material->diffuse); \
    } \
  } else { \
    glColor4fv (renderer->colors[index]); \
  }

#define SET_COLOR_ALPHA \
  if (renderer->colors == NULL) {  \
    if (current_material == NULL) { \
      glColor4fv (transparency); \
    } else { \
      color = current_material->diffuse; \
      glColor4f (color[0], color[1], color[2], 0.0); \
    } \
  } else { \
    color = renderer->colors[index]; \
    glColor4f (color[0], color[1], color[2], 0.0); \
  }

#endif

static void P3_land_vertex_render_secondpass (P3_land* land, P3_land_vertex* vertex) {
#if JIBA_BUGGED_OPENGL 
  GLfloat ccc[4];
#endif
  GLfloat* color;
  int index = vertex - land->vertices;
  if (vertex->pack->material == current_material) {
    SET_COLOR_OPAQUE
  } else {
    SET_COLOR_ALPHA
  }
  glArrayElement (index);
}

static void P3_land_tri_render_middle (P3_land* land, P3_land_tri* tri) {
#if JIBA_BUGGED_OPENGL 
  GLfloat ccc[4];
#endif
  P3_land_vertex* v;
  GLfloat* color;
  GLfloat* color2;
  if (renderer->colors == NULL) {
    if (current_material == NULL) {
#if JIBA_BUGGED_OPENGL 
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, transparency);
#else
      glColor4fv (transparency);
#endif
    } else {
      color = current_material->diffuse;
#if JIBA_BUGGED_OPENGL 
      ccc[0] = color[0];
      ccc[1] = color[1];
      ccc[2] = color[2];
      ccc[3] = 0.0;
      glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, ccc);
#else
      glColor4f (color[0], color[1], color[2], 0.0);
#endif
    }
  } else {
    color  = renderer->colors[tri->v2 - land->vertices];
    color2 = renderer->colors[tri->v3 - land->vertices];
#if JIBA_BUGGED_OPENGL 
    ccc[0] = (color[0] + color2[0]) * 0.5;
    ccc[1] = (color[1] + color2[1]) * 0.5;
    ccc[2] = (color[2] + color2[2]) * 0.5;
    ccc[3] = 0.0;
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, ccc);
#else
    glColor4f ((color[0] + color2[0]) * 0.5,
               (color[1] + color2[1]) * 0.5,
               (color[2] + color2[2]) * 0.5, 0.0);
#endif
  }
  if (tri->level == land->max_level) {
    glTexCoord2f ((tri->v2->texcoord[0] + tri->v3->texcoord[0]) * 0.5,
                  (tri->v2->texcoord[1] + tri->v3->texcoord[1]) * 0.5);
    glNormal3f ((tri->v2->normal[0] + tri->v3->normal[0]) * 0.5,
                (tri->v2->normal[1] + tri->v3->normal[1]) * 0.5,
                (tri->v2->normal[2] + tri->v3->normal[2]) * 0.5);
    glVertex3f ((tri->v2->coord[0] + tri->v3->coord[0]) * 0.5,
//               (tri->v2->coord[1] + tri->v3->coord[1]) * 0.5 + 0.1,
                (tri->v2->coord[1] + tri->v3->coord[1]) * 0.5,
                (tri->v2->coord[2] + tri->v3->coord[2]) * 0.5);
  } else {
    if (tri->v2 < tri->v3) {
      v = tri->v2 + ((tri->v3 - tri->v2) >> 1);
    } else {
      v = tri->v3 + ((tri->v2 - tri->v3) >> 1);
    }
    glArrayElement (v - land->vertices);
  }
}

static void P3_land_tri_render_secondpass (P3_land* land, P3_land_tri* tri) {
#if JIBA_BUGGED_OPENGL 
  GLfloat ccc[4];
#endif
  GLfloat* color;
  int index;

// TO DO reject alpha ?

  if (

      (!(land->option & P3_LAND_VERTEX_OPTIONS) || 
       !((land->vertex_options[tri->v1 - land->vertices] & P3_LAND_VERTEX_ALPHA) ||
         (land->vertex_options[tri->v2 - land->vertices] & P3_LAND_VERTEX_ALPHA) ||
         (land->vertex_options[tri->v3 - land->vertices] & P3_LAND_VERTEX_ALPHA))) &&

      (tri->base_neighbor != NULL) && (tri->v2 == tri->base_neighbor->v3 && tri->v3 == tri->base_neighbor->v2)) {
    /* diamond base neighbor without alpha => special smooth texturing */
    if (tri->v2->pack->material == current_material && 
        tri->v2->pack != tri->v1->pack && 
        tri->v2->pack != tri->v3->pack &&
        tri->v2->pack != tri->base_neighbor->v1->pack) {
      index = tri->v1 - land->vertices;
      SET_COLOR_ALPHA
      glArrayElement (index);
      index = tri->v2 - land->vertices;
      SET_COLOR_OPAQUE
      glArrayElement (index);
      P3_land_tri_render_middle (land, tri);
      return;
    } else if (tri->v3->pack->material == current_material && 
               tri->v3->pack != tri->v1->pack && 
               tri->v3->pack != tri->v2->pack &&
               tri->v3->pack != tri->base_neighbor->v1->pack) {
      index = tri->v1 - land->vertices;
      SET_COLOR_ALPHA
      glArrayElement (index);
      P3_land_tri_render_middle (land, tri);
      index = tri->v3 - land->vertices;
      SET_COLOR_OPAQUE
      glArrayElement (index);
      return;
    }
  }
  P3_land_vertex_render_secondpass (land, tri->v1);
  P3_land_vertex_render_secondpass (land, tri->v2);
  P3_land_vertex_render_secondpass (land, tri->v3);
}

// TO DO

static void P3_land_vertex_render_special (P3_land* land, P3_land_vertex* vertex) {
  int index = (vertex - land->vertices);
  GLfloat* color = renderer->colors[index];
#if JIBA_BUGGED_OPENGL
  GLfloat ccc[4];
  if (land->vertex_options[index] & P3_LAND_VERTEX_ALPHA && vertex->pack->material != current_material) {
    ccc[0] = color[0];
    ccc[1] = color[1];
    ccc[2] = color[2];
    ccc[3] = 0.0;
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, ccc);
  } else {
    glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);
  }
//  if (renderer->colors != NULL) glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, renderer->colors[index]);
#else
  if (land->vertex_options[index] & P3_LAND_VERTEX_ALPHA && vertex->pack->material != current_material) {
    glColor4f (color[0], color[1], color[2], 0.0);
  } else {
    glColor4fv (color);
  }
#endif
  glArrayElement (index);
}


#undef SET_COLOR_OPAQUE
#undef SET_COLOR_ALPHA

void P3_land_render (P3_land* land, P3_instance* inst) {
  P3_land_tri* tri;
  P3_double_chain* chain;
  P3_chain* face;
  P3_xpack* pack;
  if (land->option & P3_LAND_WARFOG) {
    renderer->colors = land->vertex_warfogs;
  } else if (land->option & P3_LAND_COLORED) {
    renderer->colors = land->vertex_colors;
  } else {
    renderer->colors = NULL;
  }
#if JIBA_BUGGED_OPENGL 
  glColor4fv (white);
  glDisable (GL_COLOR_MATERIAL);
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, white);
#endif
  glInterleavedArrays (GL_T2F_N3F_V3F, sizeof (P3_land_vertex), land->vertices);
  chain = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
  if (renderer->state == P3_RENDERER_STATE_OPAQUE) {
    while (1) {
      pack = (P3_xpack*) chain->data;
      face = (P3_chain*) (renderer->faces->content + chain->link);
      P3_material_activate (pack->material);
      glBegin (GL_TRIANGLES);
      while (1) {
        tri = (P3_land_tri*) face->data;
        P3_land_vertex_render_opaque (land, tri->v1);
        P3_land_vertex_render_opaque (land, tri->v2);
        P3_land_vertex_render_opaque (land, tri->v3);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
      glEnd ();
      if (chain->next == -1) break;
      chain = (P3_double_chain*) (renderer->data->content + chain->next);
    }
  } else if (renderer->state == P3_RENDERER_STATE_SECONDPASS) {

// TO DO

    glEnable (GL_BLEND);
    /* draw the special */
    while (1) {
      pack = (P3_xpack*) chain->data;
      if (pack->option & P3_PACK_SPECIAL) {
        face = (P3_chain*) (renderer->faces->content + chain->link);
        P3_material_activate (pack->material);
        glBegin (GL_TRIANGLES);
        while (1) {
//          P3_land_tri_render_secondpass (land, (P3_land_tri*) face->data, colors);
          tri = (P3_land_tri*) face->data;
          P3_land_vertex_render_special (land, tri->v1);
          P3_land_vertex_render_special (land, tri->v2);
          P3_land_vertex_render_special (land, tri->v3);
          if (face->next == -1) break;
          face = (P3_chain*) (renderer->faces->content + face->next);
        }
        glEnd ();
      }
      if (chain->next == -1) break;
      chain = (P3_double_chain*) (renderer->data->content + chain->next);
    }
    /* draw the secondpass */
    glDepthFunc (GL_LEQUAL);
    glPolygonOffset (-1.0, 0.0);
    chain = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
    while (1) {
      pack = (P3_xpack*) chain->data;
      if (!(pack->option & P3_PACK_SPECIAL)) {
        face = (P3_chain*) (renderer->faces->content + chain->link);
        P3_material_activate (pack->material);
        /* I do know why to work I need to enable and disable offset between each glBegin/glEnd */
        glEnable (GL_POLYGON_OFFSET_FILL);
        glBegin (GL_TRIANGLES);
        while (1) {
          P3_land_tri_render_secondpass (land, (P3_land_tri*) face->data);
          if (face->next == -1) break;
          face = (P3_chain*) (renderer->faces->content + face->next);
        }
        glEnd ();
        glDisable (GL_POLYGON_OFFSET_FILL);
      }
      if (chain->next == -1) break;
      chain = (P3_double_chain*) (renderer->data->content + chain->next);
    }

#if 0
    /* draw the secondpass */
    glEnable (GL_BLEND);
    glDepthFunc (GL_LEQUAL);
    glPolygonOffset (-1.0, 0.0);
    chain = (P3_double_chain*) (renderer->data->content + renderer->data->nb);
    while (1) {
      pack = (P3_xpack*) chain->data;
      face = (P3_chain*) (renderer->faces->content + chain->link);
      P3_material_activate (pack->material);
      /* I do know why to work I need to enable and disable offset between each glBegin/glEnd */
      glEnable (GL_POLYGON_OFFSET_FILL);
      glBegin (GL_TRIANGLES);
      while (1) {
        P3_land_tri_render_secondpass (land, (P3_land_tri*) face->data, colors);
        if (face->next == -1) break;
        face = (P3_chain*) (renderer->faces->content + face->next);
      }
      glEnd ();
      glDisable (GL_POLYGON_OFFSET_FILL);
      if (chain->next == -1) break;
      chain = (P3_double_chain*) (renderer->data->content + chain->next);
    }
#endif

    glDisable (GL_BLEND);
    glDepthFunc (GL_LESS);
    glDisable (GL_POLYGON_OFFSET_FILL);
  }
#if JIBA_BUGGED_OPENGL 
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, white);
  glEnable (GL_COLOR_MATERIAL);
#else
  glColor4fv (white);
#endif
  glDisableClientState (GL_VERTEX_ARRAY);
  glDisableClientState (GL_NORMAL_ARRAY);
  glDisableClientState (GL_TEXTURE_COORD_ARRAY);
}


/*============+
 | RAYPICKING |
 +============*/

static void P3_land_tri_raypick (P3_land* land, P3_land_tri* tri, GLfloat* raydata, P3_raypick_data* data, P3_raypickable* parent) {
  GLfloat r;
  if (P3_land_tri_has_child (tri)) {
    P3_land_tri_raypick (land, tri->left_child,  raydata, data, parent);
    P3_land_tri_raypick (land, tri->right_child, raydata, data, parent);
  } else {
    if (land->option & P3_LAND_VERTEX_OPTIONS && 
        land->vertex_options[tri->v1 - land->vertices] & P3_LAND_VERTEX_NON_SOLID &&
        land->vertex_options[tri->v2 - land->vertices] & P3_LAND_VERTEX_NON_SOLID &&
        land->vertex_options[tri->v3 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) {
      return;
    }
    if (P3_triangle_raypick (raydata, tri->v1->coord, tri->v2->coord, tri->v3->coord, tri->normal, data->option, &r) != P3_FALSE) {
      if (r < data->result || data->ret_csys == NULL) {
        data->result = r;
        data->ret_csys = parent;
        memcpy (data->normal, tri->normal, 3 * sizeof (GLfloat));
      }
    }
  }
}

static void P3_land_full_raypick (P3_land* land, P3_land_vertex* v1, P3_land_vertex* v2, P3_land_vertex* v3, GLfloat* normal,
                                  GLfloat* raydata, P3_raypick_data* data, P3_raypickable* parent) {
  GLfloat* coord1 = v1->coord;
  GLfloat* coord2 = v2->coord;
  GLfloat* coord3 = v3->coord;
  GLfloat a, b, c;
  if (!(land->option & P3_LAND_VERTEX_OPTIONS) ||
      !(land->vertex_options[v1 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) ||
      !(land->vertex_options[v2 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) ||
      !(land->vertex_options[v3 - land->vertices] & P3_LAND_VERTEX_NON_SOLID)) {

// TO DO ? 

    /* compute distance sphere-line */
    a = - raydata[5] * (coord1[0] - raydata[0]) + raydata[3] * (coord1[2] - raydata[2]);
    if (a < 2 * land->scale_factor) {
      b = - raydata[5] * (coord2[0] - raydata[0]) + raydata[3] * (coord2[2] - raydata[2]);
      c = - raydata[5] * (coord3[0] - raydata[0]) + raydata[3] * (coord3[2] - raydata[2]);
      if (!(a > 0.0 && b > 0.0 && c > 0.0) && !(a < 0.0 && b < 0.0 && c < 0.0)) {
        if (P3_triangle_raypick (raydata, coord1, coord2, coord3, normal, data->option, &a) != P3_FALSE) {
          if (a < data->result || data->ret_csys == NULL) {
            data->result = a;
            data->ret_csys = parent;
            memcpy (data->normal, normal, 3 * sizeof (GLfloat));
          }
        }
      }
    }
  }
}

static void P3_land_full_raypick_rect (P3_land* land, int x1, int z1, int x2, int z2, GLfloat* raydata,
                                       P3_raypick_data* data, P3_raypickable* parent) {
  int i; int j;
  P3_land_vertex* p;
  P3_land_vertex* p0;
  GLfloat* normal;
  p0 = P3_land_get_vertex (land, x1, z1);
  for (j = z1; j < z2; j++) {
    p = p0;
    normal = land->normals + 6 * (x1 + j * (land->nb_vertex_width - 1));
    for (i = x1; i < x2; i++) {
      if (((i & 1) && (j & 1)) || (!(i & 1) && !(j & 1))) {
        P3_land_full_raypick (land, p + land->nb_vertex_width, p + land->nb_vertex_width + 1, p, normal, raydata, data, parent);
        normal += 3;
        P3_land_full_raypick (land, p + 1, p, p + land->nb_vertex_width + 1, normal, raydata, data, parent);
        normal += 3;
      } else {
        P3_land_full_raypick (land, p, p + land->nb_vertex_width, p + 1, normal, raydata, data, parent);
        normal += 3;
        P3_land_full_raypick (land, p + land->nb_vertex_width + 1, p + 1, p + land->nb_vertex_width, normal, raydata, data, parent);
        normal += 3;
      }
      p++;
    }
    p0 += land->nb_vertex_width;
  }
}

void P3_land_raypick (P3_land* land, P3_raypick_data* rdata, P3_raypickable* parent) {
  int i; 
  P3_land_patch* patch;
  GLfloat* data;
  data = P3_raypickable_get_raypick_data (parent, rdata);
  if (!(land->option & P3_LAND_INITED)) P3_land_init (land);
  if (land->option & P3_LAND_REAL_LOD_RAYPICK) {
    for (i = 0; i < land->nb_patchs; i++) {
      patch = land->patchs + i;
      if (P3_sphere_raypick (data, patch->sphere) == P3_FALSE) { continue; }
      /* raypick on tris */
      P3_land_tri_raypick (land, patch->tri_top,    data, rdata, parent);
      P3_land_tri_raypick (land, patch->tri_left,   data, rdata, parent);
      P3_land_tri_raypick (land, patch->tri_right,  data, rdata, parent);
      P3_land_tri_raypick (land, patch->tri_bottom, data, rdata, parent);
    }
  } else {
    if (data[3] == 0.0 && data[5] == 0.0) {
      int x1 = (int) (data[0] / land->scale_factor);
      int z1 = (int) (data[2] / land->scale_factor);
      if (x1 >= 0.0 && z1 >= 0.0 && x1 < land->nb_vertex_width && z1 < land->nb_vertex_depth) {
        P3_land_full_raypick_rect (land, x1, z1, x1 + 1, z1 + 1, data, rdata, parent);
      }
    } else {
      if (data[6] < 0.0) {
        for (i = 0; i < land->nb_patchs; i++) {
          patch = land->patchs + i;
          if (P3_sphere_raypick (data, patch->sphere) == P3_FALSE) { continue; }
          /* raypick on tris with full accuracy */
          P3_land_full_raypick_rect (land,
                                     (int) (patch->tri_top->v3->coord[0] / land->scale_factor),
                                     (int) (patch->tri_top->v3->coord[2] / land->scale_factor),
                                     (int) (patch->tri_bottom->v3->coord[0] / land->scale_factor),
                                     (int) (patch->tri_bottom->v3->coord[2] / land->scale_factor),
                                     data, rdata, parent);
        }
      } else {
        GLfloat x1; GLfloat z1; GLfloat x2; GLfloat z2;
        GLfloat f;
        if (rdata->option & P3_RAYPICK_HALF_LINE) {
          x1 = data[0];
          z1 = data[2];
        } else {
          x1 = data[0] - data[6] * data[3];
          z1 = data[2] - data[6] * data[5];
        }
        x2 = data[0] + data[6] * data[3];
        z2 = data[2] + data[6] * data[5];
        if (x1 > x2) {
          f = x1;
          x1 = x2;
          x2 = f;
        }
        if (z1 > z2) {
          f = z1;
          z1 = z2;
          z2 = f;
        }
        x1 /= land->scale_factor;
        z1 /= land->scale_factor;
        x2 /= land->scale_factor;
        z2 /= land->scale_factor;
        if (x2 < 0.0) return;
        if (z2 < 0.0) return;
        if (x1 >= (GLfloat) land->nb_vertex_width) return;
        if (z1 >= (GLfloat) land->nb_vertex_depth) return;
        if (x1 < 0.0) x1 = 0.0;
        if (z1 < 0.0) z1 = 0.0;
        x2++; z2++;
        if (x2 >= (GLfloat) land->nb_vertex_width) x2 = (GLfloat) land->nb_vertex_width - 1;
        if (z2 >= (GLfloat) land->nb_vertex_depth) z2 = (GLfloat) land->nb_vertex_depth - 1;
        P3_land_full_raypick_rect (land, (int) x1, (int) z1, (int) x2, (int) z2, data, rdata, parent);
      }
    }
  }
}

static int P3_land_tri_raypick_b (P3_land* land, P3_land_tri* tri, GLfloat* raydata, int option, P3_raypickable* parent) {
  GLfloat r;
  if (P3_land_tri_has_child (tri)) {
    if (P3_land_tri_raypick_b (land, tri->left_child,  raydata, option, parent) == P3_TRUE) return P3_TRUE;
    if (P3_land_tri_raypick_b (land, tri->right_child, raydata, option, parent) == P3_TRUE) return P3_TRUE;
  } else {
    if (land->option & P3_LAND_VERTEX_OPTIONS) {
      if (land->vertex_options[tri->v1 - land->vertices] & P3_LAND_VERTEX_NON_SOLID &&
          land->vertex_options[tri->v2 - land->vertices] & P3_LAND_VERTEX_NON_SOLID &&
          land->vertex_options[tri->v3 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) {
        return P3_FALSE;
      }
    }
    if (P3_triangle_raypick (raydata, tri->v1->coord, tri->v2->coord, tri->v3->coord, tri->normal, option, &r) != P3_FALSE) {
      return P3_TRUE;
    }
  }
  return P3_FALSE;
}

static int P3_land_full_raypick_b (P3_land* land, P3_land_vertex* v1, P3_land_vertex* v2, P3_land_vertex* v3, 
                                    GLfloat* normal, GLfloat* raydata, int option) {
  GLfloat* coord1 = v1->coord;
  GLfloat* coord2 = v2->coord;
  GLfloat* coord3 = v3->coord;
  GLfloat a, b, c;
  if (!(land->option & P3_LAND_VERTEX_OPTIONS) ||
      !(land->vertex_options[v1 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) ||
      !(land->vertex_options[v2 - land->vertices] & P3_LAND_VERTEX_NON_SOLID) ||
      !(land->vertex_options[v3 - land->vertices] & P3_LAND_VERTEX_NON_SOLID)) {

// TO DO ? 

    /* compute distance sphere-line */
    a = - raydata[5] * (coord1[0] - raydata[0]) + raydata[3] * (coord1[2] - raydata[2]);
    if (a < 2 * land->scale_factor) {
      b = - raydata[5] * (coord2[0] - raydata[0]) + raydata[3] * (coord2[2] - raydata[2]);
      c = - raydata[5] * (coord3[0] - raydata[0]) + raydata[3] * (coord3[2] - raydata[2]);
      if (!(a > 0.0 && b > 0.0 && c > 0.0) && !(a < 0.0 && b < 0.0 && c < 0.0)) {
        if (P3_triangle_raypick (raydata, coord1, coord2, coord3, normal, option, &a) != P3_FALSE) {
          return P3_TRUE;
        }
      }
    }
  }
  return P3_FALSE;
}

static int P3_land_full_raypick_rect_b (P3_land* land, int x1, int z1, int x2, int z2, GLfloat* raydata, int option) {
  int i; int j;
  P3_land_vertex* p;
  P3_land_vertex* p0;
  GLfloat* normal;
  p0 = P3_land_get_vertex (land, x1, z1);
  for (j = z1; j < z2; j++) {
    p = p0;
    normal = land->normals + 6 * (x1 + j * (land->nb_vertex_width - 1));
    for (i = x1; i < x2; i++) {
      if (((i & 1) && (j & 1)) || (!(i & 1) && !(j & 1))) {
        if (P3_land_full_raypick_b (land, p + land->nb_vertex_width, p + land->nb_vertex_width + 1, p, normal, raydata, option) == P3_TRUE) return P3_TRUE;
        normal += 3;
        if (P3_land_full_raypick_b (land, p + 1, p, p + land->nb_vertex_width + 1, normal, raydata, option) == P3_TRUE) return P3_TRUE;
        normal += 3;
      } else {
        if (P3_land_full_raypick_b (land, p, p + land->nb_vertex_width, p + 1, normal, raydata, option) == P3_TRUE) return P3_TRUE;
        normal += 3;
        if (P3_land_full_raypick_b (land, p + land->nb_vertex_width + 1, p + 1, p + land->nb_vertex_width, normal, raydata, option) == P3_TRUE) return P3_TRUE;
        normal += 3;
      }
      p++;
    }
    p0 += land->nb_vertex_width;
  }
  return P3_FALSE;
}

int P3_land_raypick_b (P3_land* land, P3_raypick_data* rdata, P3_raypickable* parent) {
  int i; 
  P3_land_patch* patch;
  GLfloat* data;
  if (!(land->option & P3_LAND_INITED)) P3_land_init (land);
  data = P3_raypickable_get_raypick_data (parent, rdata);
  if (land->option & P3_LAND_REAL_LOD_RAYPICK) {
    for (i = 0; i < land->nb_patchs; i++) {
      patch = land->patchs + i;
      if (P3_sphere_raypick (data, patch->sphere) == P3_FALSE) continue;
      /* raypick on tris */
      if (P3_land_tri_raypick_b (land, patch->tri_top,    data, rdata->option, parent) == P3_TRUE) return P3_TRUE;
      if (P3_land_tri_raypick_b (land, patch->tri_left,   data, rdata->option, parent) == P3_TRUE) return P3_TRUE;
      if (P3_land_tri_raypick_b (land, patch->tri_right,  data, rdata->option, parent) == P3_TRUE) return P3_TRUE;
      if (P3_land_tri_raypick_b (land, patch->tri_bottom, data, rdata->option, parent) == P3_TRUE) return P3_TRUE;
    }
  } else {
    if (data[3] == 0.0 && data[5] == 0.0) {
      int x1 = (int) (data[0] / land->scale_factor);
      int z1 = (int) (data[2] / land->scale_factor);
      if (x1 >= 0.0 && z1 >= 0.0 && x1 < land->nb_vertex_width && z1 < land->nb_vertex_depth) {
        P3_land_full_raypick_rect_b (land, x1, z1, x1 + 1, z1 + 1, data, rdata->option);
      }
    } else {
      if (data[6] < 0.0) {
        for (i = 0; i < land->nb_patchs; i++) {
          patch = land->patchs + i;
          if (P3_sphere_raypick (data, patch->sphere) == P3_FALSE) continue;
          /* raypick on tris with full accuracy */
          if (P3_land_full_raypick_rect_b (land,
                                      (int) patch->tri_top->v3->coord[0] / land->scale_factor,
                                      (int) patch->tri_top->v3->coord[2] / land->scale_factor,
                                      (int) patch->tri_bottom->v3->coord[0] / land->scale_factor,
                                      (int) patch->tri_bottom->v3->coord[2] / land->scale_factor,
                                      data, rdata->option) == P3_TRUE) return P3_TRUE;
      }
      } else {
        GLfloat x1; GLfloat z1; GLfloat x2; GLfloat z2;
        GLfloat f;
        if (rdata->option & P3_RAYPICK_HALF_LINE) {
          x1 = data[0];
          z1 = data[2];
        } else {
          x1 = data[0] - data[6] * data[3];
          z1 = data[2] - data[6] * data[5];
        }
        x2 = data[0] + data[6] * data[3];
        z2 = data[2] + data[6] * data[5];
        if (x1 > x2) {
          f = x1;
          x1 = x2;
          x2 = f;
        }
        if (z1 > z2) {
          f = z1;
          z1 = z2;
          z2 = f;
        }
        x1 /= land->scale_factor;
        z1 /= land->scale_factor;
        x2 /= land->scale_factor;
        z2 /= land->scale_factor;
        if (x2 < 0.0) return P3_FALSE;
        if (z2 < 0.0) return P3_FALSE;
        if (x1 >= (GLfloat) land->nb_vertex_width) return P3_FALSE;
        if (z1 >= (GLfloat) land->nb_vertex_depth) return P3_FALSE;
        if (x1 < 0.0) x1 = 0.0;
        if (z1 < 0.0) z1 = 0.0;
        x2++; z2++;
        if (x2 >= (GLfloat) land->nb_vertex_width) x2 = (GLfloat) land->nb_vertex_width - 1;
        if (z2 >= (GLfloat) land->nb_vertex_depth) z2 = (GLfloat) land->nb_vertex_depth - 1;
        return P3_land_full_raypick_rect_b (land, (int) x1, (int) z1, (int) x2, (int) z2, data, rdata->option);
      }
    }
  }
  return P3_FALSE;
}

static void P3_land_get_height_at_factors (P3_land_vertex* v1, P3_land_vertex* v2, P3_land_vertex* v3, GLfloat x, GLfloat z, GLfloat* k, GLfloat* t) {
  GLfloat u[2];
  GLfloat w[2];
  GLfloat q;
  GLfloat* ptr;
  ptr = v1->coord;
  u[0] = v2->coord[0] - ptr[0];
  u[1] = v2->coord[2] - ptr[2];
  w[0] = v3->coord[0] - ptr[0];
  w[1] = v3->coord[2] - ptr[2];
  q = 1.0 / (w[0] * u[1] - w[1] * u[0]);
  *t = (  (x - ptr[0]) * u[1] - (z - ptr[2]) * u[0]) * q;
  *k = (- (x - ptr[0]) * w[1] + (z - ptr[2]) * w[0]) * q;
}

GLfloat P3_land_get_height_at (P3_land* land, GLfloat x, GLfloat z, GLfloat** norm) {
  GLfloat k; GLfloat t;
  P3_land_vertex* v1;
  P3_land_vertex* v2;
  if (!(land->option & P3_LAND_INITED)) P3_land_init (land);
  if (land->option & P3_LAND_REAL_LOD_RAYPICK) {
    /* return the true height, the one of the land_tri */
    P3_land_patch* patch;
    P3_land_tri* tri;
    int i;
    for (i = 0; i < land->nb_patchs; i++) {
      patch = land->patchs + i;
      v1 = patch->tri_top->v3;
      v2 = patch->tri_bottom->v3;
      if (x >= v1->coord[0] && x <= v2->coord[0] && z >= v1->coord[2] && z<= v2->coord[2]) {

#define TRI_GET_HEIGHT_AT(tria) \
  P3_land_get_height_at_factors (tria->v1, tria->v2, tria->v3, x, z, &k, &t); \
  if (k <= 1.0 + P3_EPSILON && k >= -P3_EPSILON && t <= 1.0 + P3_EPSILON && t >= -P3_EPSILON && (k + t - 1.0) < P3_EPSILON) { \
    tri = tria; \
    while (1) { \
      P3_land_get_height_at_factors (tri->v1, tri->v2, tri->v3, x, z, &k, &t); \
      if (P3_land_tri_has_child (tri)) { \
        if (k >= t) { \
          tri = tri->right_child; \
        } else { \
          tri = tri->left_child; \
        } \
      } else { \
        if (norm != NULL) { *norm = tri->normal; } \
        return tri->v1->coord[1] + k * (tri->v2->coord[1] - tri->v1->coord[1]) + t * (tri->v3->coord[1] - tri->v1->coord[1]); \
      } \
    } \
  }

        TRI_GET_HEIGHT_AT (patch->tri_top)
        TRI_GET_HEIGHT_AT (patch->tri_left)
        TRI_GET_HEIGHT_AT (patch->tri_right)
        TRI_GET_HEIGHT_AT (patch->tri_bottom)

#undef TRI_GET_HEIGHT_AT

        break;
      }
    }
  } else {
    /* return interpolated height */
    int ix; int iz;
    GLfloat nx; GLfloat nz;
    P3_land_vertex* v3;
    nx = x / land->scale_factor;
    nz = z / land->scale_factor;
    ix = (int) nx;
    iz = (int) nz;
    if (ix >= 0 && iz >= 0 && ix < land->nb_vertex_width && iz < land->nb_vertex_depth) {
      if (((ix & 1) && (iz & 1)) || (!(ix & 1) && !(iz & 1))) {
        if (nz - iz > nx - ix) {
          v1 = P3_land_get_vertex (land, ix,     iz + 1);
          v2 = P3_land_get_vertex (land, ix,     iz);
          v3 = P3_land_get_vertex (land, ix + 1, iz + 1);
          P3_land_get_height_at_factors (v1, v2, v3, x, z, &k, &t);
          if (norm != NULL) { *norm = land->normals + 6 * (ix + iz * (land->nb_vertex_width - 1)); }
          return v1->coord[1] + k * (v2->coord[1] - v1->coord[1]) + t * (v3->coord[1] - v1->coord[1]);
        } else {
          v1 = P3_land_get_vertex (land, ix + 1, iz);
          v2 = P3_land_get_vertex (land, ix + 1, iz + 1);
          v3 = P3_land_get_vertex (land, ix,     iz);
          P3_land_get_height_at_factors (v1, v2, v3, x, z, &k, &t);
          if (norm != NULL) { *norm = land->normals + 6 * (ix + iz * (land->nb_vertex_width - 1)) + 3; }
          return v1->coord[1] + k * (v2->coord[1] - v1->coord[1]) + t * (v3->coord[1] - v1->coord[1]);
        }
      } else {
        if (nz - iz + nx - ix < 1) {
          v1 = P3_land_get_vertex (land, ix,     iz);
          v2 = P3_land_get_vertex (land, ix + 1, iz);
          v3 = P3_land_get_vertex (land, ix,     iz + 1);
          P3_land_get_height_at_factors (v1, v2, v3, x, z, &k, &t);
          if (norm != NULL) { *norm = land->normals + 6 * (ix + iz * (land->nb_vertex_width - 1)); }
          return v1->coord[1] + k * (v2->coord[1] - v1->coord[1]) + t * (v3->coord[1] - v1->coord[1]);
        } else {
          v1 = P3_land_get_vertex (land, ix + 1, iz + 1);
          v2 = P3_land_get_vertex (land, ix,     iz + 1);
          v3 = P3_land_get_vertex (land, ix + 1, iz);
          P3_land_get_height_at_factors (v1, v2, v3, x, z, &k, &t);
          if (norm != NULL) { *norm = land->normals + 6 * (ix + iz * (land->nb_vertex_width - 1)) + 3; }
          return v1->coord[1] + k * (v2->coord[1] - v1->coord[1]) + t * (v3->coord[1] - v1->coord[1]);
        }
      }
    }
  }
  if (norm != NULL) { *norm = NULL; }
  return -1.0;
}


/*===========+
 | VERTEX FX |
 +===========*/

void P3_land_check_vertex_options (P3_land* land) {
  if (!(land->option & P3_LAND_VERTEX_OPTIONS)) {
    land->option |= P3_LAND_VERTEX_OPTIONS;
    free (land->vertex_options);
    land->vertex_options = (char*) calloc (land->nb_vertex_width * land->nb_vertex_depth, sizeof (char));
  }
}

GLfloat* P3_land_check_color (P3_land* land, GLfloat* color) {
  if (!(land->option & P3_LAND_COLORED)) {
    GLfloat* whit;
    int nb = land->nb_vertex_width * land->nb_vertex_depth;
    int i;
    land->option |= P3_LAND_COLORED;
    land->vertex_colors = (GLfloat**) malloc (nb * sizeof (GLfloat*));
    whit = P3_land_register_color (land, white);
    for (i = 0; i < nb; i++) land->vertex_colors[i] = whit;
  }
  if (1.0 - color[3] > P3_EPSILON) {
    /* need vertex option to stock vertex alpha */
    P3_land_check_vertex_options (land);
  }
  return P3_land_register_color (land, color);
}

void P3_land_check_warfog (P3_land* land) {
  if (!(land->option & P3_LAND_WARFOG)) {
    GLfloat* color;
    int nb = land->nb_vertex_width * land->nb_vertex_depth;
    int i;
    /* need fx */
    P3_fx_init ();
    /* need vertex option */
    P3_land_check_vertex_options (land);
    /* need warfog */
    land->vertex_warfogs = (GLfloat**) malloc (nb * sizeof (GLfloat*));
    if (land->option & P3_LAND_COLORED) {
      for (i = 0; i < nb; i++) land->vertex_warfogs[i] = land->vertex_colors[i];
    } else {
      if (land->colors > 0) {
        color = land->colors;
      } else {
        color = P3_land_register_color (land, white);
      }
      for (i = 0; i < nb; i++) land->vertex_warfogs[i] = color;
    }
    land->option |= P3_LAND_WARFOG;
  }
}

/*
void P3_land_add_vertex_option_all (P3_land* land, int flag) {
  int i;
  int nb = land->nb_vertex_width * land->nb_vertex_depth;
  for (i = 0; i < nb; i++) {
    land->vertex_options[i] |= flag;
  }
}

void P3_land_rem_vertex_option_all (P3_land* land, int flag) {
  int i;
  int nb = land->nb_vertex_width * land->nb_vertex_depth;
  for (i = 0; i < nb; i++) {
    land->vertex_options[i] &= ~flag;
  }
}
*/

void P3_land_fx_all (P3_land* land, P3_fx* fx) {
  int nb = land->nb_vertex_width * land->nb_vertex_depth;
  int i;
  for (i = 0; i < nb; i++) fx->apply (fx, i);
}

void P3_land_fx_in_sphere (P3_land* land, P3_fx* fx, GLfloat sphere[4]) {
  int x1, z1, x2, z2;
  int i, j;
  int k, t;
  /* make box from sphere */
  x1 = (int) ((sphere[0] - sphere[3]) / land->scale_factor);
  x2 = (int) ((sphere[0] + sphere[3]) / land->scale_factor);
  z1 = (int) ((sphere[2] - sphere[3]) / land->scale_factor + 1.0);
  z2 = (int) ((sphere[2] + sphere[3]) / land->scale_factor + 1.0);
  if (x1 >= land->nb_vertex_width || z1 >= land->nb_vertex_depth || x2 < 0 || z2 < 0) return;
  if (x1 < 0) x1 = 0;
  if (z1 < 0) z1 = 0;
  if (x2 > land->nb_vertex_width) x2 = land->nb_vertex_width;
  if (z2 > land->nb_vertex_depth) z2 = land->nb_vertex_depth;
  /* test each vertex in the box */
  k = x1 + z1 * land->nb_vertex_width;
  for (j = z1; j < z2; j++) {
    t = k;
    for (i = x1; i < x2; i++) {
      if (P3_point_is_in_sphere (sphere, (land->vertices + t)->coord) == P3_TRUE) {
        fx->apply (fx, t);
      }
      t++;
    }
    k += land->nb_vertex_width;
  }
}

void P3_land_fx_in_cylinderY (P3_land* land, P3_fx* fx, GLfloat cylinder[3]) {
  GLfloat* f;
  GLfloat x; GLfloat z;
  int x1, z1, x2, z2;
  int i, j;
  int k, t;
  /* make box from cylinder */
  x1 = (int) ((cylinder[0] - cylinder[2]) / land->scale_factor);
  x2 = (int) ((cylinder[0] + cylinder[2]) / land->scale_factor);
  z1 = (int) ((cylinder[1] - cylinder[2]) / land->scale_factor + 1.0);
  z2 = (int) ((cylinder[1] + cylinder[2]) / land->scale_factor + 1.0);
  if (x1 >= land->nb_vertex_width || z1 >= land->nb_vertex_depth || x2 < 0 || z2 < 0) return;
  if (x1 < 0) x1 = 0;
  if (z1 < 0) z1 = 0;
  if (x2 > land->nb_vertex_width) x2 = land->nb_vertex_width;
  if (z2 > land->nb_vertex_depth) z2 = land->nb_vertex_depth;
  /* test each vertex in the box */
  k = x1 + z1 * land->nb_vertex_width;
  for (j = z1; j < z2; j++) {
    t = k;
    for (i = x1; i < x2; i++) {
      f = (land->vertices + t)->coord;
      x = f[0] - cylinder[0];
      z = f[2] - cylinder[1];
      if (x * x + z * z <= cylinder[2] * cylinder[2]) {
        fx->apply (fx, t);
      }
      t++;
    }
    k += land->nb_vertex_width;
  }
}


/*========+
 | SAVING |
 +========*/

static int P3_land_get_material_index (P3_xmesh* mesh, P3_material* material) {
  int i;
  for (i = 0; i < mesh->nb_materials; i++) {
    if (mesh->materials[i] == material) { return i; }
  }
  return -1;
}

void P3_land_get_data (P3_land* land, P3_chunk* chunk) {
  P3_land_vertex* v;
  int nb;
  int i;
  i = land->option & ~P3_LAND_INITED;
  if (land->nb_vertex_width != land->nb_vertex_depth) i |= P3_LAND_NON_SQUARRED;
  P3_chunk_save_int (chunk, i);
  P3_chunk_save_int (chunk, land->nb_vertex_width);
  if (land->nb_vertex_width != land->nb_vertex_depth) P3_chunk_save_int (chunk, land->nb_vertex_depth);
  P3_chunk_save_int (chunk, land->patch_size);
  P3_chunk_save_float (chunk, land->texture_factor);
  P3_chunk_save_float (chunk, land->scale_factor);
  P3_chunk_save_float (chunk, land->split_factor);
  P3_chunk_save_int (chunk, land->nb_colors);
  if (land->nb_colors > 0) {
    P3_chunk_save (chunk, land->colors, 4 * land->nb_colors * sizeof (GLfloat));
  }
  nb = land->nb_vertex_width * land->nb_vertex_depth;
  for (i = 0; i < nb; i++) {
    v = land->vertices + i;
    P3_chunk_save_float (chunk, v->coord[1]);
    P3_chunk_save_int (chunk, P3_xmesh_get_material_index ((P3_xmesh*) land, v->pack->material));
    if (land->option & P3_LAND_COLORED && land->nb_colors > 0) {
      P3_chunk_save_int (chunk, land->vertex_colors[v - land->vertices] - land->colors);
    }
  }
  if (land->option & P3_LAND_VERTEX_OPTIONS) {
    P3_chunk_add (chunk, land->vertex_options, nb * sizeof (char));
  }
}

void P3_land_set_data (P3_land* land, P3_chunk* chunk) {
  P3_land_vertex* v;
  int i;
  int n;
  int nb;
  land->class = &P3_class_land;
  land->nb_patchs = 0;
  land->patchs = NULL;
  land->option = P3_chunk_load_int (chunk);
  land->nb_vertex_width = P3_chunk_load_int (chunk);
  if (land->option & P3_LAND_NON_SQUARRED) {
    land->option -= P3_LAND_NON_SQUARRED;
    land->nb_vertex_depth = P3_chunk_load_int (chunk);
  } else {
    land->nb_vertex_depth = land->nb_vertex_width;
  }
  land->patch_size = P3_chunk_load_int (chunk);
  land->texture_factor = P3_chunk_load_float (chunk);
  land->scale_factor = P3_chunk_load_float (chunk);
  land->split_factor = P3_chunk_load_float (chunk);
  land->nb_colors = P3_chunk_load_int (chunk);
  nb = land->nb_vertex_width * land->nb_vertex_depth;
  if (land->option & P3_LAND_COLORED) {
    land->colors = (GLfloat*) malloc (4 * land->nb_colors * sizeof (GLfloat));
    land->vertex_colors = (GLfloat**) malloc (nb * sizeof (GLfloat*));
    P3_chunk_load (chunk, land->colors, 4 * land->nb_colors * sizeof (GLfloat));
  } else if (land->nb_colors > 0) {
    /* for compatibility with older version */
    chunk->nb += land->nb_colors * 3 * sizeof (GLfloat);
    land->colors = NULL;
    land->vertex_colors = NULL;
  } else {
    land->colors = NULL;
    land->vertex_colors = NULL;
  }
  if (land->nb_vertex_width == 0 || land->nb_vertex_depth == 0) {
    land->vertices = NULL;
    land->normals = NULL;
  } else {
    land->vertices = (P3_land_vertex*) malloc (nb * sizeof (P3_land_vertex));
    for (i = 0; i < nb; i++) {
      v = land->vertices + i;
      v->coord[1] = P3_chunk_load_float (chunk);
      n = P3_chunk_load_int (chunk);
      if (n == -1) {
        v->pack = P3_xpack_get (P3_FACE_TRIANGLE, NULL);
      } else {
        v->pack = P3_xpack_get (P3_FACE_TRIANGLE, land->materials[n]);
      }
      if (land->option & P3_LAND_COLORED) {
        land->vertex_colors[v - land->vertices] = land->colors + P3_chunk_load_int (chunk);
      } else if (land->nb_colors > 0) {
        /* for compatibility with older version */
        chunk->nb += sizeof (int);
      }
    }
    land->normals = (GLfloat*) malloc ((land->nb_vertex_width - 1) * (land->nb_vertex_depth - 1) * 6 * sizeof (GLfloat));
  }
  if (!(land->option & P3_LAND_COLORED) && land->nb_colors > 0) {
    /* for compatibility with older version */
    land->nb_colors = 0;
  }
  if (land->option & P3_LAND_VERTEX_OPTIONS) {
    land->vertex_options = (char*) malloc (nb * sizeof (char));
    P3_chunk_load (chunk, land->vertex_options, nb * sizeof (char));
  } else {
    land->vertex_options = NULL;
  }
  land->vertex_warfogs = NULL;
  P3_land_compute_coords (land);
}

void P3_land_init (P3_land* land) {
  if (land->patchs != NULL) {
    P3_land_free_patchs (land);
  }
  P3_land_compute_normals (land);
  P3_land_create_patchs (land);
  if (land->option & P3_LAND_VERTEX_OPTIONS) { P3_land_force_presence (land); }
  land->option |= P3_LAND_INITED;
}

