/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * light.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include <math.h>

#include "p3_base.h"
#include "util.h"
#include "math3d.h"
#include "frustum.h"
//#include "material.h"
//#include "glext.h"
#include "raypick.h"
#include "renderer.h"
#include "coordsys.h"
#include "light.h"

extern int          maxlights;
extern P3_light**   lights_gl;
extern int*         lights_gl_activated;
//extern P3_list*     lights_top;
//extern P3_list*     lights_deep;
//extern P3_chain*    lights_stack;
extern P3_renderer* renderer;


/* Top level light means that the light enlight all the other objects
 * whereas Deep light are stopped by portal.
 */

/* - general library functions - */

int P3_light_get_id (void) {
  int i;
  /* check if one light has not been set yet */
  for (i = 0; i < maxlights; i++) {
    if (lights_gl[i] == NULL) {
      return i;
    }
  }
  /* check if one light is disabled */
  for (i = 0; i < maxlights; i++) {
    if (lights_gl_activated[i] == P3_FALSE) {
      lights_gl[i]->id = -1;
      return i;
    }
  }
  /* return the first light id */
  lights_gl[0]->id = -1;
  return 0;
}

void P3_disable_all_lights (void) {
  int i;
  for (i = 0; i < maxlights; i++) {
    glDisable (GL_LIGHT0 + i);
    lights_gl_activated[i] = P3_FALSE;
  }
}

void P3_disable_deep_lights (void) {
  /* disable all non top level lights */
  int i;
  P3_light* light;
  for (i = 0; i < maxlights; i++) {
    light = lights_gl[i];
    if (light != NULL && !(light->option & P3_LIGHT_TOP_LEVEL) && lights_gl_activated[i] == P3_TRUE) {
      glDisable (GL_LIGHT0 + light->id);
      lights_gl_activated[light->id] = P3_FALSE;
    }
  }
/*
  for (i = 0; i < renderer->lights_deep->nb; i++) {
    light = (P3_light*) P3_list_get (renderer->lights_deep, i);
    if (light->id != -1) {
      glDisable (GL_LIGHT0 + light->id);
      *(lights_gl_activated + light->id) = P3_FALSE;
    }
  }
*/
}


#if 0

void P3_static_lights_off (void) {
  int i;
  /* disable all static lights */
  for (i = 0; i < maxlights; i++) {
    if (lights_gl_activated[i] == P3_TRUE && lights_gl[i]->option & P3_LIGHT_STATIC) {
      glDisable (GL_LIGHT0 + i);
    }
  }
}

void P3_static_lights_on (void) {
  int i;
  /* re-enable static lights */
  for (i = 0; i < maxlights; i++) {
    if (lights_gl_activated[i] == P3_TRUE && lights_gl[i]->option & P3_LIGHT_STATIC) {
      glEnable (GL_LIGHT0 + i);
    }
  }
}

#endif

/* - light functions - */

P3_class P3_class_light = { 
  P3_ID_LIGHT,
  (batch_func)     P3_light_batch,
  (render_func)    P3_light_render,
  (shadow_func)    0,
  (raypick_func)   0,
  (raypick_b_func) 0,
};

P3_light* P3_light_new(P3_light* light) {
  if (light == NULL) {
    light = (P3_light*) malloc (sizeof (P3_light));
  }
  light->id = -1;
  light->class = &P3_class_light;
  P3_coordsys_initialize ((P3_coordsys*) light);
  light->w = 1.0;
  light->angle    = 180.0;
  light->exponent = 0.0;
  light->constant  = 1.0;
  light->linear    = 0.0;
  light->quadratic = 0.0;
  /* ambient color */
  light->colors[ 0] = 0.0;
  light->colors[ 1] = 0.0;
  light->colors[ 2] = 0.0;
  light->colors[ 3] = 1.0;
  /* diffuse color */
  light->colors[ 4] = 1.0;
  light->colors[ 5] = 1.0;
  light->colors[ 6] = 1.0;
  light->colors[ 7] = 1.0;
  /* specular color */
  light->colors[ 8] = 1.0;
  light->colors[ 9] = 1.0;
  light->colors[10] = 1.0;
  light->colors[11] = 1.0;
/*
  if (type & P3_LIGHT_POINT) {
    light->angle = 180.0;
    light->position[2] = 0.0;
    light->position[3] = 1.0;
  }
  if (type & P3_LIGHT_DIRECTIONAL) {
    light->angle = 180.0;
    light->position[2] = 1.0;
    light->position[3] = 0.0;
  }
  if (type & P3_LIGHT_SPOT) {
    light->angle = 45.0;
    light->position[2] = 0.0;
    light->position[3] = 1.0;
  }
*/
  light->shadow_color = NULL;
  light->id = -1;
  light->radius = -1.0;
  P3_light_compute_radius (light);
  return light;
}

int P3_light_get_shadow_at (P3_light* light, GLfloat position[3]) {
  P3_raypick_data data;
  P3_coordsys* root;
  GLfloat* rdata = data.root_data;
  root = P3_coordsys_get_root ((P3_coordsys*) light);
  if (root != NULL && root->class->raypick_b != 0) {
    if (light->option & P3_LIGHT_DIRECTIONAL) {
//    if (fabs (light->w) < P3_EPSILON) {
      /* directional light */
      rdata[3] =   0.0;
      rdata[4] =   0.0;
      rdata[5] = - 1.0;
      P3_vector_by_matrix (rdata + 3, P3_coordsys_get_root_matrix ((P3_coordsys*) light));
      P3_vector_normalize (rdata + 3);
      rdata[6] = 100.0;
      P3_point_by_matrix_copy (rdata, position, P3_coordsys_get_root_matrix (light->parent));
      rdata[0] -= rdata[3] * rdata[6];
      rdata[1] -= rdata[4] * rdata[6];
      rdata[2] -= rdata[5] * rdata[6];
      rdata[6] -= 1.0;
    } else {
      P3_vector_from_points (rdata + 3, light->m + 12, position);
      if (light->parent == NULL) {
        memcpy (rdata, light->m + 12, 3 * sizeof (GLfloat));
      } else {
        P3_point_by_matrix_copy (rdata, light->m + 12, P3_coordsys_get_root_matrix (light->parent));
        P3_vector_by_matrix (rdata + 2, P3_coordsys_get_root_matrix (light->parent));
      }
      rdata[6] = P3_vector_length (rdata + 3) - 1.0;
      P3_vector_normalize (rdata + 3);
    }
    data.option = P3_RAYPICK_HALF_LINE;
    return P3_raypick_b ((P3_any_object*) root, &data);
  }
  return P3_FALSE;
}

GLfloat P3_light_get_spotlight_at (P3_light* light, GLfloat position[3]) {
  GLfloat v[3];
  GLfloat w[3];
  GLfloat m;
  if (fabs (light->angle - 180.0) < P3_EPSILON || fabs (light->w) < P3_EPSILON) {
    /* point or directional light */
    return 1.0;
  }
  v[0] = position[0] - light->m[12];
  v[1] = position[1] - light->m[13];
  v[2] = position[2] - light->m[14];
  /* lights are -z oriented */
  w[0] = - light->m[ 8];
  w[1] = - light->m[ 9];
  w[2] = - light->m[10];
  m = P3_vector_dot_product (v, w);
  if (m < 0.0) { m = 0.0; }
  if (m <= cos (light->angle)) {
    /* position is out of cone */
    return 0.0;
  } else {
    return pow (m, light->exponent);
  }
}

GLfloat P3_light_get_attenuation_at (P3_light* light, GLfloat position[3]) {
  GLfloat d;
  if (fabs (light->w) < P3_EPSILON) {
    /* directional light */
    return 1.0;
  }
  d = P3_point_distance_to (light->m + 12, position);
  return 1.0 / (light->constant + light->linear * d + light->quadratic * d * d);
}

void P3_light_get_color_at (P3_light* light, GLfloat position[3], GLfloat* normal, int shadow, GLfloat color[4]) {
  GLfloat f;
  GLfloat g;
  GLfloat angle;
  GLfloat attenuation;
  GLfloat spotlight;
  attenuation = P3_light_get_attenuation_at (light, position);
  spotlight = P3_light_get_spotlight_at (light, position);
  if (attenuation == 0.0f || spotlight == 0.0f) {
    color[0] = 0.0f;
    color[1] = 0.0f;
    color[2] = 0.0f;
    color[3] = 0.0f;
    return;
  }
  f = attenuation * spotlight;
  if (shadow == P3_TRUE && P3_light_get_shadow_at (light, position) == P3_TRUE) {
    g = 0.0f;
  } else {
    if (normal == NULL) { 
      angle = 1.0f;
    } else {
      GLfloat v[3];
      GLfloat n[3];
      memcpy (n, normal, 3 * sizeof (GLfloat));
      P3_vector_normalize (n);
      if (fabs (light->w) < P3_EPSILON) {
        /* directional light */
        v[0] =   0.0;
        v[1] =   0.0;
        v[2] = - 1.0;
        P3_vector_by_matrix (v, light->m);
      } else {
        P3_vector_from_points (v, light->m + 12, position);
        P3_vector_normalize (v);
      }
      angle = P3_vector_dot_product (n, v);
      if (angle > 0.0) {
        angle = 0.0f;
      } else {
        angle = - angle;
      }
    }
    g = angle;
  }
  f = attenuation * spotlight;
  color[0] = f * (light->colors[0] + g * light->colors[4]);
  color[1] = f * (light->colors[1] + g * light->colors[5]);
  color[2] = f * (light->colors[2] + g * light->colors[6]);
  color[3] = f * (light->colors[3] + g * light->colors[7]);
}

void P3_light_cast_into (P3_light* light, P3_coordsys* csys) {
  /* compute light direction into csys */
  if (light->option & P3_LIGHT_DIRECTIONAL) {
    /* cast light direction in object coordsys */
    light->data[0] = - light->m[ 8];
    light->data[1] = - light->m[ 9];
    light->data[2] = - light->m[10];
    if (light->parent != NULL && light->parent != csys) {
      P3_vector_by_matrix (light->data, P3_coordsys_get_root_matrix (light->parent));
      P3_vector_by_matrix (light->data, P3_coordsys_get_inverted_root_matrix (csys));
      P3_vector_normalize (light->data);
    }
  } else {
    /* cast light position in object coordsys */
    if (light->parent == NULL || light->parent == csys) {
      memcpy (light->data, light->m + 12, 3 * sizeof (GLfloat));
    } else {
      P3_point_by_matrix_copy (light->data, light->m + 12, P3_coordsys_get_root_matrix (light->parent));
      P3_point_by_matrix (light->data, P3_coordsys_get_inverted_root_matrix (csys));
    }
  }
}

void P3_light_list_cast_into (P3_list* lights, P3_coordsys* csys) {
  int i;
  for (i = 0; i < lights->nb; i++) P3_light_cast_into ((P3_light*) P3_list_get (lights, i), csys);
}

void P3_light_batch (P3_light* light, P3_instance* csys) {
  GLfloat data[4];
  if (light->option & P3_OBJECT_HIDDEN) return;
  P3_multiply_matrix (light->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) light));
  /* frustum test */
  if (light->radius > 0.0) {
    data[0] = light->m[12];
    data[1] = light->m[13];
    data[2] = light->m[14];
    data[3] = light->radius;
    light->frustum_data = -1;
    if (P3_sphere_in_frustum (P3_renderer_get_frustum (csys), data) != P3_TRUE) return;
  }
  if (light->option & P3_LIGHT_TOP_LEVEL) {
    P3_list_add (renderer->top_lights, light);
  } else {
    P3_list_add (renderer->c_context->lights, light);
  }
  P3_list_add (renderer->all_lights, light);
}

void P3_light_render (P3_light* light, P3_instance* csys) {
  static GLfloat p[4] = { 0.0, 0.0, 0.0, 1.0 };
  static GLfloat q[3] = { 0.0, 0.0, -1.0 };
  GLenum id;
  p[3] = light->w;
  if (!(light->option & P3_OBJECT_HIDDEN)) {
    glLoadMatrixf (light->render_matrix);
    if (light->id == -1) {
      light->id = P3_light_get_id ();
      lights_gl[light->id] = light;
      id = GL_LIGHT0 + light->id;
      glLightf  (id, GL_SPOT_EXPONENT, light->exponent);
      glLightf  (id, GL_SPOT_CUTOFF,   light->angle);
      glLightfv (id, GL_AMBIENT,       light->colors);
      glLightfv (id, GL_DIFFUSE,       light->colors + 4);
      glLightfv (id, GL_SPECULAR,      light->colors + 8);
      glLightf  (id, GL_CONSTANT_ATTENUATION,  light->constant);
      glLightf  (id, GL_LINEAR_ATTENUATION,    light->linear);
      glLightf  (id, GL_QUADRATIC_ATTENUATION, light->quadratic);
    } else {
      id = GL_LIGHT0 + light->id;
    }
    if (light->option & P3_LIGHT_DIRECTIONAL) {
//    if (fabs (light->w) < P3_EPSILON) {
      /* directional light */
      p[2] = 1.0;
      glLightfv (id, GL_POSITION, p);
      p[2] = 0.0;
    } else {
      /* positional light */
      glLightfv (id, GL_POSITION, p);
      glLightfv (id, GL_SPOT_DIRECTION, q);
    }
    glEnable (id);
    lights_gl_activated[light->id] = P3_TRUE;
  }
}

void P3_light_compute_radius (P3_light* light) {
  if (fabs (light->w) < P3_EPSILON) {
    /* directional light */
    light->radius = -1.0;
  } else if (light->linear != 0 && light->quadratic == 0) {
    light->radius = ((1.0 / P3_LIGHT_NULL_ATTENUATION) - light->constant) / light->linear;
  } else if (light->linear == 0 && light->quadratic == 0) {
    /* directional infinite light */
// TO DO ok ?
    light->radius = -1.0;
  }
// TO DO other cases
}

void P3_light_get_data (P3_light* l, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, l->option);
  P3_chunk_save_float (chunk, l->w);
  P3_chunk_save (chunk, l->m, 19 * sizeof (GLfloat));
  P3_chunk_save (chunk, l->colors, 12 * sizeof (GLfloat));
  P3_chunk_save_float (chunk, l->radius);
  P3_chunk_save_float (chunk, l->angle);
  P3_chunk_save_float (chunk, l->exponent);
  P3_chunk_save_float (chunk, l->linear);
  P3_chunk_save_float (chunk, l->constant);
  P3_chunk_save_float (chunk, l->quadratic);
  if (l->option & P3_LIGHT_SHADOW_COLOR) P3_chunk_save (chunk, l->shadow_color, 4 * sizeof (GLfloat));
}

void P3_light_set_data (P3_light* l, P3_chunk* chunk) {
  l->class = &P3_class_light;
  l->validity = P3_COORDSYS_INVALID;
  l->option = P3_chunk_load_int (chunk);
  l->w = P3_chunk_load_float (chunk);
  if (fabs (l->w) < P3_EPSILON) l->option |= P3_LIGHT_DIRECTIONAL;
  P3_chunk_load (chunk, l->m, 19 * sizeof (GLfloat));
  P3_chunk_load (chunk, l->colors, 12 * sizeof (GLfloat));
  l->radius    = P3_chunk_load_float (chunk);
  l->angle     = P3_chunk_load_float (chunk);
  l->exponent  = P3_chunk_load_float (chunk);
  l->linear    = P3_chunk_load_float (chunk);
  l->constant  = P3_chunk_load_float (chunk);
  l->quadratic = P3_chunk_load_float (chunk);
  if (l->option & P3_LIGHT_SHADOW_COLOR) {
    l->shadow_color = (GLfloat*) malloc (4 * sizeof (GLfloat));
    P3_chunk_load (chunk, l->shadow_color, 4 * sizeof (GLfloat));
  } else {
    l->shadow_color = NULL;
  }
}

/*
void P3_light_look_at (P3_light* light, GLfloat p[3]) {
// TO DO is length of direction change something ?
  light->direction[0] = p[0] - light->position[0];
  light->direction[1] = p[1] - light->position[1];
  light->direction[2] = p[2] - light->position[2];
}

void P3_light_rotate_lateral (P3_light* light, GLfloat angle) {
  P3_point_rotate_lateral (light->position,  angle);
  P3_point_rotate_lateral (light->direction, angle);
}

void P3_light_rotate_vertical (P3_light* light, GLfloat angle) {
  P3_point_rotate_vertical (light->position,  angle);
  P3_point_rotate_vertical (light->direction, angle);
}

void P3_light_rotate_incline (P3_light* light, GLfloat angle) {
  P3_point_rotate_incline (light->position,  angle);
  P3_point_rotate_incline (light->direction, angle);
}

void P3_light_rotate_axe (P3_light* light, GLfloat angle, GLfloat x, GLfloat y, GLfloat z) {
  P3_point_rotate_axe (light->position,  angle, x, y, z);
  P3_point_rotate_axe (light->direction, angle, x, y, z);
}

void P3_light_rotate (P3_light* light, GLfloat angle, GLfloat p1[3], GLfloat p2[3]) {
  P3_point_rotate (light->position, angle, p1, p2);
  P3_point_rotate_axe (light->direction, angle, p2[0] - p1[0], p2[1] - p1[1], p2[2] - p1[2]);
}

*/
