/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * material.c
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "gladd.h"
#include "util.h"
#include "renderer.h"
#include "material.h"

extern int engine_option;
extern int quality;
extern GLfloat white[4];
extern GLfloat black[4];
extern P3_material* current_material;
extern P3_renderer* renderer;
/* NULL material packs */
extern int NULL_nb_packs;
extern P3_xpack** NULL_packs;


/*==========+
 | MATERIAL |
 +==========*/

P3_material* P3_material_new (P3_material* m) {
  if (m == NULL) {
    m = (P3_material*) malloc (sizeof (P3_material));
  }
  m->option = 0;
  m->id = 0;
  m->image = NULL;
  m->shininess = 128.0;
  m->diffuse  = white;
  m->specular = black;
  m->emissive = black;
/*
  // diffuse color
  m->colors[0] = 1.0;
  m->colors[1] = 1.0;
  m->colors[2] = 1.0;
  m->colors[3] = 1.0;
  // specular
  m->colors[4] = 0.0;
  m->colors[5] = 0.0;
  m->colors[6] = 0.0;
  m->colors[7] = 1.0;
*/
  m->nb_packs = 0;
  m->packs = NULL;
  return m;
}

void P3_material_dealloc (P3_material* m) {
  P3_xpack* pack;
  P3_xpack* pak;
  P3_xpack* next;
  int i;
  if (m->id != 0) glDeleteTextures (1, &(m->id));
  if (m->option & P3_MATERIAL_DIFFUSE)  free (m->diffuse);
  if (m->option & P3_MATERIAL_SPECULAR) free (m->specular);
  if (m->option & P3_MATERIAL_EMISSIVE) free (m->emissive);
  for (i = 0; i < m->nb_packs; i++) {
    pack = m->packs[i];
    pak = pack->secondpass;
    while (pak != NULL) {
      next = pak->secondpass;
      free (pak);
      pak = next;
    }
    free (pack);
  }
  free (m->packs);
}

int P3_image_get_type (P3_image* img) {
  switch (img->nb_color) {
  case 1:
    return GL_LUMINANCE;
  case 3:
    return GL_RGB;
  case 4:
    return GL_RGBA;
  default:
    P3_error ("unknown image number of color (%i)", img->nb_color);
    return -1;
  }
}

int P3_image_get_internal_format (P3_image* img) {
  switch (quality) {
  case P3_QUALITY_LOW:
    switch (img->nb_color) {
    case 1: return GL_LUMINANCE;
    case 3: return GL_RGB;
    case 4: return GL_RGBA;
    }
  case P3_QUALITY_MEDIUM:
    switch (img->nb_color) {
    case 1: return GL_LUMINANCE8;
    case 3: return GL_RGB8;
    case 4: return GL_RGBA8;
    }
  case P3_QUALITY_HIGH:
    switch (img->nb_color) {
    case 1: return GL_LUMINANCE16;
    case 3: return GL_RGB16;
    case 4: return GL_RGBA16;
    }
  }
  return -1;
}

void P3_material_init (P3_material* m) {
  int img_type;
  int img_int_format;
  if (m->image != NULL && engine_option & P3_GL_INITED) {
    if (m->id == 0) glGenTextures (1, &(m->id));
    glBindTexture (GL_TEXTURE_2D, m->id);
// TO DO usefull ?
    //glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    if (m->option & P3_MATERIAL_CLAMP) {
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    } else {
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    }
    img_type = P3_image_get_type (m->image);
    img_int_format = P3_image_get_internal_format (m->image);
// TO DO ?
//      if (m->option & P3_MATERIAL_MASK) {
// HACK TO DO ?
//        glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
//        glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
//        glTexImage2D (GL_TEXTURE_2D, 0, img_int_format, m->image->width, m->image->height, 0, img_type, GL_UNSIGNED_BYTE, m->image->pixels);
//      } else {
    if (engine_option & P3_USE_MIPMAP) {
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      P3_build2Dmipmaps (m);
    } else {
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexImage2D (GL_TEXTURE_2D, 0, img_int_format, m->image->width, m->image->height, 0, img_type, GL_UNSIGNED_BYTE, m->image->pixels);
    }
//  }
  }
}

void P3_material_activate (P3_material* m) {
  if (m != current_material) {
    P3_material_inactivate (current_material);
    current_material = m;
    if (m == NULL) {
      glDisable (GL_TEXTURE_2D);
    } else {
      if (m->image == NULL) {
        glDisable (GL_TEXTURE_2D);
      } else {
        if (m->id == 0) { 
          P3_material_init (m); 
        } else {
          glBindTexture (GL_TEXTURE_2D, m->id);
        }
      }
      if (m->option & P3_MATERIAL_SEPARATE_SPECULAR) glLightModeli (GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);
      glMaterialf  (GL_FRONT_AND_BACK, GL_SHININESS, m->shininess);
      glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR,  m->specular);
      if (m->option & P3_MATERIAL_MASK) {
        glDisable (GL_ALPHA_TEST);
        glAlphaFunc (GL_NOTEQUAL, 0.0);
        glEnable (GL_ALPHA_TEST);
        glDepthMask (GL_TRUE); 
      }
      if (m->option & P3_MATERIAL_ADDITIVE_BLENDING) glBlendFunc (GL_SRC_ALPHA, GL_ONE);
    }
  }
  /* these values may have change due to some vertices */
  /* we must set that because GL_COLOR_MATERIAL is enable : ie material diffuse color is in fact glColor... */
  if (m == NULL) {
    glColor4fv (white);
    glMaterialfv (GL_FRONT_AND_BACK, GL_EMISSION, black);
  } else {
    glColor4fv (m->diffuse);
    glMaterialfv (GL_FRONT_AND_BACK, GL_EMISSION, m->emissive);
  }
}

void P3_material_inactivate (P3_material* m) {
  if (m == NULL) {
    glEnable (GL_TEXTURE_2D);
  } else {
    glBindTexture (GL_TEXTURE_2D, 0);
    if (m->image == NULL) glEnable (GL_TEXTURE_2D); 
    if (m->option & P3_MATERIAL_MASK) { 
      glDisable (GL_ALPHA_TEST);
      if (renderer->state == P3_RENDERER_STATE_ALPHA) {
        glDepthMask (GL_FALSE);
      }
    }
    if (m->option & P3_MATERIAL_SEPARATE_SPECULAR) glLightModeli (GL_LIGHT_MODEL_COLOR_CONTROL, GL_SINGLE_COLOR);
    if (m->option & P3_MATERIAL_ADDITIVE_BLENDING) glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }
}

void P3_material_compute_alpha (P3_material* m) {
  int i;
  GLubyte* ptr;
  m->option &= ~(P3_MATERIAL_ALPHA | P3_MATERIAL_MASK);
  if (m->option & P3_MATERIAL_ADDITIVE_BLENDING) { 
    m->option |= P3_MATERIAL_ALPHA;
  }
  if (m->diffuse[3] < 1.0 - P3_EPSILON) {
    m->option |= P3_MATERIAL_ALPHA;
  }
  if (m->image != NULL && m->image->nb_color == 4) {
    m->option |= P3_MATERIAL_ALPHA;
    ptr = m->image->pixels + 3;
    for (i = 0; i < m->image->width * m->image->height; i++) {
      if (*ptr != 0 && *ptr != 255) {
        i = -1;
        break;
      }
      ptr += 4;
    }
    if (i != -1) {
      m->option |= P3_MATERIAL_MASK;
    }
  }

printf ("Material option:\n");
if (m->option & P3_MATERIAL_ALPHA) { printf ("  alpha\n"); }
if (m->option & P3_MATERIAL_MASK)  { printf ("  mask\n"); }

}

void P3_material_get_data (P3_material* m, P3_chunk* chunk) {
  P3_chunk_save_int (chunk, m->option);
  P3_chunk_save_float (chunk, m->shininess);
  if (m->option & P3_MATERIAL_DIFFUSE)  P3_chunk_save (chunk, m->diffuse,  4 * sizeof (GLfloat));
  if (m->option & P3_MATERIAL_SPECULAR) P3_chunk_save (chunk, m->specular, 4 * sizeof (GLfloat));
  if (m->option & P3_MATERIAL_EMISSIVE) P3_chunk_save (chunk, m->emissive, 4 * sizeof (GLfloat));
  if (m->option & P3_MATERIAL_IMAGE) {
    P3_chunk_save_int (chunk, m->image->nb_color);
    P3_chunk_save_int (chunk, m->image->width);
    P3_chunk_save_int (chunk, m->image->height);
    P3_chunk_save (chunk, m->image->pixels, m->image->width * m->image->height * m->image->nb_color * sizeof (GLubyte));
  }
}

void P3_image_set_data (P3_image* image, P3_chunk* chunk) {
  int size;
  image->nb_color = P3_chunk_load_int (chunk);
  image->width    = P3_chunk_load_int (chunk);
  image->height   = P3_chunk_load_int (chunk);
  size = image->nb_color * image->width * image->height * sizeof (GLubyte);
  image->pixels = (GLubyte*) malloc (size);
  P3_chunk_load (chunk, image->pixels, size);
}

void P3_material_set_data (P3_material* m, P3_chunk* chunk) {
  m->option = P3_chunk_load_int (chunk);
  m->shininess = P3_chunk_load_float (chunk);
  if (m->option & P3_MATERIAL_DIFFUSE) {
    m->diffuse = (GLfloat*) malloc (4 * sizeof (GLfloat));
    P3_chunk_load (chunk, m->diffuse, 4 * sizeof (GLfloat));
  } else {
    m->diffuse = white;
  }
  if (m->option & P3_MATERIAL_SPECULAR) {
    m->specular = (GLfloat*) malloc (4 * sizeof (GLfloat));
    P3_chunk_load (chunk, m->specular, 4 * sizeof (GLfloat));
  } else {
    m->specular = black;
  }
  if (m->option & P3_MATERIAL_EMISSIVE) {
    m->emissive = (GLfloat*) malloc (4 * sizeof (GLfloat));
    P3_chunk_load (chunk, m->emissive, 4 * sizeof (GLfloat));
  } else {
    m->emissive = black;
  }
  m->id = 0;
  m->nb_packs = 0;
  m->packs = NULL;
}


/*======+
 | PACK |
 +======*/

P3_xpack* P3_xpack_get (int option, P3_material* material) {
  P3_xpack* pack;
  int opt;
  int i;
  opt = option & P3_PACK_OPTIONS;
  if (material == NULL) {
    /* look for existing pack */
    for (i = 0; i < NULL_nb_packs; i++) {
      pack = NULL_packs[i];
      if (pack->option == opt) return pack;
    }
    /* create a new pack */
    pack = (P3_xpack*) malloc (sizeof (P3_xpack));
    pack->material = NULL;
    pack->option = opt;
    pack->data = -1;
    pack->secondpass = NULL;
    pack->alpha = NULL;
    NULL_packs = (P3_xpack**) realloc (NULL_packs, (NULL_nb_packs + 1) * sizeof (P3_xpack*));
    NULL_packs[NULL_nb_packs] = pack;
    NULL_nb_packs++;
    return pack;
  } else {
    /* look for existing pack */
    for (i = 0; i < material->nb_packs; i++) {
      pack = material->packs[i];
      if (pack->option == opt) return pack;
    }
    /* create a new pack */
    pack = (P3_xpack*) malloc (sizeof (P3_xpack));
    pack->material = material;
    pack->option = opt;
    pack->data = -1;
    pack->secondpass = NULL;
    pack->alpha = NULL;
    material->packs = (P3_xpack**) realloc (material->packs, (material->nb_packs + 1) * sizeof (P3_xpack*));
    material->packs[material->nb_packs] = pack;
    (material->nb_packs)++;
    return pack;
  }
}

P3_xpack* P3_xpack_get_secondpass (P3_xpack* pack) {
  if (pack->secondpass == NULL) {
    P3_xpack* pak;
    /* create a new pack */
    pak = (P3_xpack*) malloc (sizeof (P3_xpack));
    pak->material = pack->material;
    if (pack->option & P3_PACK_SECONDPASS) {
      pak->option = pack->option | P3_PACK_SPECIAL;
    } else {
      pak->option = pack->option | P3_PACK_SECONDPASS;
    }
    pak->data = -1;
    pak->secondpass = NULL;
    pak->alpha = NULL;
    pack->secondpass = pak;
  }
  return pack->secondpass;
}

P3_xpack* P3_xpack_get_alpha (P3_xpack* pack) {
  if (pack->alpha == NULL) {
    P3_xpack* pak;
    /* create a new pack */
    pak = (P3_xpack*) malloc (sizeof (P3_xpack));
    pak->material = pack->material;
    pak->option = pack->option | P3_FACE_ALPHA;
    pak->data = -1;
    pak->secondpass = NULL;
    pak->alpha = NULL;
    pack->alpha = pak;
  }
  return pack->alpha;
}


