/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * xmesh_tree.c
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 **********************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "mesh.h"


/*======+
 | TREE |
 +======*/

void P3_xnode_dealloc (P3_xnode* node) {
  int i;
  for (i = 0; i < node->nb_child; i++) P3_xnode_dealloc (node->child[i]);
  free (node->faces);
  free (node->child);
  free (node);
}

static void P3_xmesh_face_create_sphere (P3_xmesh* mesh, P3_xface* face, GLfloat* sphere) {
  GLfloat p[12];
  memcpy (p,     mesh->vertex_coords[face->v1], 3 * sizeof (GLfloat));
  memcpy (p + 3, mesh->vertex_coords[face->v2], 3 * sizeof (GLfloat));
  memcpy (p + 6, mesh->vertex_coords[face->v3], 3 * sizeof (GLfloat));
  if (face->option & P3_FACE_TRIANGLE) {
    P3_sphere_from_points (sphere, p, 3);
  } else if (face->option & P3_FACE_QUAD) {
    memcpy (p + 9, mesh->vertex_coords[face->v4], 3 * sizeof (GLfloat));
    P3_sphere_from_points (sphere, p, 4);
  }
}

static P3_xnode* P3_xnode_new (void* face, GLfloat* sphere) {
  P3_xnode* node;
  node = (P3_xnode*) malloc (sizeof (P3_xnode));
  node->nb_faces = 1;
  node->faces = (void**) malloc (sizeof (void*));
  node->faces[0] = face;
  node->nb_child = 0;
  node->child = NULL;
  memcpy (node->sphere, sphere, 4 * sizeof (GLfloat));
  return node;
}

static void P3_xnode_add_node (P3_xnode* node, P3_xnode* add) {
  node->child = (P3_xnode**) realloc (node->child, (node->nb_child + 1) * sizeof (P3_xnode*));
  node->child[node->nb_child] = add;
  (node->nb_child)++;
}

static void P3_xnode_register_node (P3_xnode* node, P3_xnode* add) {
  GLfloat d;
  int i;
  int added = P3_FALSE;
  /* test if ancient nodes can be contained into the added node */
  i = 0;
  while (i < node->nb_child) {
    if (node->child[i] == NULL) {
      if (added == P3_FALSE) {
        node->child[i] = add;
        added = P3_TRUE;
      }
      i++;
    } else {
      d = P3_point_distance_to (add->sphere, node->child[i]->sphere);
      if (d + node->child[i]->sphere[3] <= add->sphere[3]) {
        /* add child i into added node */
        P3_xnode_add_node (add, node->child[i]);
        if (added == P3_FALSE) {
          node->child[i] = add;
          added = P3_TRUE;
          i++;
        } else {
          (node->nb_child)--;
          node->child[i] = node->child[node->nb_child];
          node->child[node->nb_child] = NULL;
        }
      } else {
        i++;
      }
    }
  }
  if (added == P3_FALSE) P3_xnode_add_node (node, add);
}

static void P3_xnode_added (P3_xnode* node, P3_xnode* new) {
  GLfloat d;
  int i;
  /* test if ancient nodes can be contained into the new one */
  i = 0;
  while (i < node->nb_child) {
    if (node->child[i] != NULL && new != node->child[i]) {
      d = P3_point_distance_to (new->sphere, node->child[i]->sphere);
      if (d + node->child[i]->sphere[3] <= new->sphere[3]) {
        /* add child i into new node */
        P3_xnode_add_node (new, node->child[i]);
        (node->nb_child)--;
        node->child[i] = node->child[node->nb_child];
        node->child[node->nb_child] = NULL;
      } else {
        i++;
      }
    } else {
      i++;
    }
  }
}

static int P3_xnode_gather (P3_xnode* node, int mode, GLfloat param) {
  int best1 = -1;
  int best2 = -1;
  P3_xnode* n;
  GLfloat best_sphere[4];
  GLfloat sphere[4];
  int i; int j;
  /* return P3_FALSE if no more gather are possible */
  /* try different technics... */
  if (mode == 0) {
    /* take the smallest sphere */
    GLfloat min_radius = 100000.0f;
    GLfloat radius = param * node->sphere[3];
    n = NULL;
    for (i = 0; i < node->nb_child; i++) {
      if (n == NULL || node->child[i]->sphere[3] < min_radius) { 
        best1 = i;
        n = node->child[i];
        min_radius = n->sphere[3]; 
      }
    }
    if (min_radius >= radius) {
      return P3_FALSE;
    } else {
      /* find the 1rst sphere to gather with that produce a sphere with a radius <= radius */
      for (i = 0; i < node->nb_child; i++) {
        if (i != best1) { 
          P3_sphere_from_2_spheres (best_sphere, n->sphere, node->child[i]->sphere);
          if (best_sphere[3] <= radius) {
            best2 = i;
            break;
          }
        }
      }
      return P3_FALSE;
    }
  } else {
    /* compute the best tree possible but very slow */
    /* find the 2 children that produce the smallest sphere */
    for (i = 0; i < node->nb_child; i++) {
      n = node->child[i];
      if (n != NULL) {
        for (j = i + 1; j < node->nb_child; j++) {
          if (node->child[j] != NULL) {
            P3_sphere_from_2_spheres (sphere, n->sphere, node->child[j]->sphere);
            if (best1 < 0 || sphere[3] < best_sphere[3]) {
              memcpy (best_sphere, sphere, 4 * sizeof (GLfloat));
              best1 = i;
              best2 = j;
            }
          }
        }
      }
    }
  }
  if (best_sphere[3] >= node->sphere[3]) { return P3_FALSE; }
  /* gather best1 and best2 */
  n = (P3_xnode*) malloc (sizeof (P3_xnode));
  n->nb_faces = 0;
  n->faces = NULL;
  n->nb_child = 2;
  n->child = (P3_xnode**) malloc (2 * sizeof (P3_xnode*));
  n->child[0] = node->child[best1];
  n->child[1] = node->child[best2];
  memcpy (n->sphere, best_sphere, 4 * sizeof (GLfloat));
  (node->nb_child)--;
  node->child[best1] = n;
  node->child[best2] = node->child[node->nb_child];
  P3_xnode_added (node, n);
  return P3_TRUE;
}

static void P3_xnode_add_face (P3_xnode* node, void* face, GLfloat* sphere) {
  P3_xnode* new;
  /* create a new node for face and add the new node to our node */
  new = P3_xnode_new (face, sphere);
  P3_xnode_register_node (node, new);
  node->child = (P3_xnode**) realloc (node->child, node->nb_child * sizeof (P3_xnode*));
}

static void P3_xnode_join (P3_xnode* n1, P3_xnode* n2) {
  int i;
  n1->faces = (void**) realloc (n1->faces, (n1->nb_faces + n2->nb_faces) * sizeof (void*));
  for (i = 0; i < n2->nb_faces; i++) {
    n1->faces[n1->nb_faces + i] = n2->faces[i];
  }
  n1->nb_faces += n2->nb_faces;
  n1->child = (P3_xnode**) realloc (n1->child, (n1->nb_child + n2->nb_child) * sizeof (P3_xnode*));
  for (i = 0; i < n2->nb_child; i++) {
    n1->child[n1->nb_child + i] = n2->child[i];
  }
  n1->nb_child += n2->nb_child;
}

static void P3_xnode_collapse_with_child (P3_xnode* node, GLfloat collapse) {
  int i;
  for (i = 0; i < node->nb_child; i++) {
    if (node->child[i]->sphere[3] > collapse * node->sphere[3]) {
      P3_xnode_join (node, node->child[i]);
      (node->nb_child)--;
      node->child[i] = node->child[node->nb_child];
    }
  }
}

static void P3_xnode_optimize (P3_xnode* node, GLfloat collapse, int mode, GLfloat param) {
  int i;
  while (node->nb_child > 2) {
    /* gather some children */
    if (P3_xnode_gather (node, mode, param) == P3_FALSE) break;
  }
  P3_xnode_collapse_with_child (node, collapse);
  node->child = (P3_xnode**) realloc (node->child, node->nb_child * sizeof (P3_xnode*));
  for (i = 0; i < node->nb_child; i++) {
    P3_xnode_optimize (node->child[i], collapse, mode, param);
  }
}

/*
static int P3_xnode_gather_useless_leaf (P3_xnode* node, P3_xnode* parent, int child_index) {
  int i;
  if (node->nb_child == 0) {
    if (node->nb_faces == 1 && parent != NULL && PTR_AS (node->faces[0], int) & P3_XFACE_HAS_SPHERE) {
      // add face to parent
      parent->faces = (void**) realloc (parent->faces, (parent->nb_faces + 1) * sizeof (void*));
      parent->faces[parent->nb_faces] = node->faces[0];
      (parent->nb_faces)++;
      // remove child from parent
      (parent->nb_child)--;
      parent->child[child_index] = parent->child[parent->nb_child];
      parent->child = (P3_xnode**) realloc (parent->child, parent->nb_child * sizeof (P3_xnode*));
      return P3_TRUE;
    }
  } else {
    i = 0;
    while (i < node->nb_child) {
      if (P3_xnode_gather_useless_leaf (node->child[i], node, i) == P3_FALSE) {
        i++;
      }
    }
  }
  return P3_FALSE;
}
*/

static void P3_xnode_register_inside_face (P3_xnode* node, void* face, GLfloat* sphere) {
  /* recurse to children */
  GLfloat d;
  int i;
  for (i = 0; i < node->nb_child; i++) {
    d = P3_point_distance_to (node->child[i]->sphere, sphere);
    if (d + sphere[3] <= node->child[i]->sphere[3]) {
      /* face is inside that child */
      P3_xnode_register_inside_face (node->child[i], face, sphere);
      return;
    }
  }
  /* no child found => face must be added to node */
  P3_xnode_add_face (node, face, sphere);
}

static P3_xnode* P3_xnode_register_face (P3_xnode* node, P3_xnode* parent, void* face, GLfloat* sphere) {
  GLfloat d;
  d = P3_point_distance_to (node->sphere, sphere);
  if (d + sphere[3] <= node->sphere[3]) {
    /* face is inside node */
    P3_xnode_register_inside_face (node, face, sphere);
  } else if (d + node->sphere[3] <= sphere[3]) {
    /* node is inside face */
    P3_xnode* n;
    n = (P3_xnode*) malloc (sizeof (P3_xnode));
    n->nb_faces = 1;
    n->faces = (void**) malloc (sizeof (void*));
    n->faces[0] = face;
    n->nb_child = 1;
    n->child = (P3_xnode**) malloc (sizeof (P3_xnode*));
    n->child[0] = node;
    memcpy (n->sphere, sphere, 4 * sizeof (GLfloat));
    return n;
  } else {
    if (parent == NULL) {
      /* create a new node with no face */
      P3_xnode* n;
      n = (P3_xnode*) malloc (sizeof (P3_xnode));
      n->nb_faces = 0;
      n->faces = NULL;
      n->nb_child = 2;
      n->child = (P3_xnode**) malloc (2 * sizeof (P3_xnode*));
      n->child[0] = node;
      n->child[1] = P3_xnode_new (face, sphere);
      P3_sphere_from_2_spheres (n->sphere, node->sphere, sphere);
      return n;
    } else {
      P3_xnode_add_face (parent, face, sphere);
    }
  }
  return node;
}

int P3_xnode_get_nb_level (P3_xnode* node) {
  int i; int nb = 0; int n;
  for (i = 0; i < node->nb_child; i++) {
    n = P3_xnode_get_nb_level (node->child[i]);
    if (n > nb) nb = n;
  }
  return nb + 1;
}

int P3_xnode_get_memory_size (P3_xnode* node) {
  int size;
  int i;
  size = 2 * sizeof (int) + 4 * sizeof (GLfloat) + (2 + node->nb_child + node->nb_faces) * sizeof (void*);
  for (i = 0; i < node->nb_child; i++) {
    size += P3_xnode_get_memory_size (node->child[i]);
  }
  return size;
}

P3_xnode* P3_xmesh_build_tree (P3_xmesh* mesh) {
  GLfloat sphere[4];
  P3_xnode* tree;
  void* face;

printf ("Building tree...\n");

  tree = NULL;
  face = mesh->faces;
  while (face < mesh->faces + mesh->faces_size) {
    P3_xmesh_face_create_sphere (mesh, face, sphere);
    if (tree == NULL) {
      tree = P3_xnode_new (face, sphere);
    } else {
      tree = P3_xnode_register_face (tree, NULL, face, sphere);
    }
    face += P3_xmesh_face_size (mesh, face);
    //printf ("1 face added ok. testing\n");
    //P3_xnode_get_nb_level (tree);
    //printf ("test finished\n");
  }

printf ("  %i levels\n", P3_xnode_get_nb_level (tree));
printf ("  [DONE]\n");

  return tree;
}

void P3_xmesh_optimize_tree (P3_xnode* tree, GLfloat collapse, int mode, GLfloat param) {
  int i;

printf ("Optimizing tree...\n");

  P3_xnode_optimize (tree, collapse, mode, param);
//  P3_xnode_gather_useless_leaf (tree, NULL, -1);

printf ("  %i levels\n", P3_xnode_get_nb_level (tree));
i = P3_xnode_get_memory_size (tree);
printf ("  size in memory : tree %i Ko (%i octets)\n", (int) (i / 1000.0f), i);
printf ("  [DONE]\n");

}
