/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**************************************************************************
 * portal.c : rectangle shape that links 2 worlds
 *
 * WARNING : there may be graphical bugs if a portal is seen through 
 * another portal (it may lack of clip planes)
 *
 * to change the size of a portal, you must scale the portal coordsys 
 * matrix
 *
 * lights don't go through portal. if you want a same light on the 2 sides, 
 * you must set twice the same light in the 2 worlds
 *
 * Copyright (C) 2001-2002 Bertrand 'blam' LAMY
 **************************************************************************/

#include "p3_base.h"
#include "math3d.h"
#include "util.h"
#include "coordsys.h"
#include "world.h"
#include "frustum.h"
#include "atmosphere.h"
#include "renderer.h"
#include "light.h"
#include "portal.h"

//extern int maxclipplanes;
//extern int nbclipplanes;
extern P3_renderer* renderer;
//extern P3_any_object* context_popper;


P3_class P3_class_portal = { 
  P3_ID_PORTAL,
  (batch_func)     P3_portal_batch,
  (render_func)    0,//P3_portal_render,
  (shadow_func)    P3_portal_shadow,
  (raypick_func)   P3_portal_raypick,
  (raypick_b_func) P3_portal_raypick_b,
};


P3_portal* P3_portal_new (P3_portal* p) {
  if (p == NULL) {
    p = (P3_portal*) malloc (sizeof (P3_portal));
  }
  p->class = &P3_class_portal;
  P3_coordsys_initialize ((P3_coordsys*) p);
  p->equation = NULL;
  p->beyond = NULL;
  p->beyond_name = NULL;
  p->coords = NULL;
  p->nbpoints = 0;
  return p;
}

void P3_portal_set_clipping_planes (P3_portal* portal) {
  GLfloat* m;
  /* we must put these points and vectors in the camera coordsys */
  GLfloat p1[3] = { -0.5, -0.5, 0.0 };
  GLfloat p2[3] = {  0.5,  0.5, 0.0 };
  GLfloat v1[3] = {  0.0,  1.0, 0.0 };
  GLfloat v2[3] = {  1.0,  0.0, 0.0 };
  m = P3_coordsys_get_root_matrix ((P3_coordsys*) portal);
  P3_point_by_matrix  (p1, m);
  P3_point_by_matrix  (p2, m);
  P3_vector_by_matrix (v1, m);
  P3_vector_by_matrix (v2, m);
  m = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) renderer->c_camera);
  P3_point_by_matrix  (p1, m);
  P3_point_by_matrix  (p2, m);
  P3_vector_by_matrix (v1, m);
  P3_vector_by_matrix (v2, m);
  /* compute planes equations */
  portal->equation[ 0] = (double) ( p1[1] * v1[2] - p1[2] * v1[1]);
  portal->equation[ 1] = (double) (-p1[0] * v1[2] + p1[2] * v1[0]);
  portal->equation[ 2] = (double) ( p1[0] * v1[1] - p1[1] * v1[0]);
  portal->equation[ 3] = (double) ( 0.0);
  portal->equation[ 4] = (double) (-p2[1] * v1[2] + p2[2] * v1[1]);
  portal->equation[ 5] = (double) ( p2[0] * v1[2] - p2[2] * v1[0]);
  portal->equation[ 6] = (double) (-p2[0] * v1[1] + p2[1] * v1[0]);
  portal->equation[ 7] = (double) ( 0.0);
  portal->equation[ 8] = (double) (-p1[1] * v2[2] + p1[2] * v2[1]);
  portal->equation[ 9] = (double) ( p1[0] * v2[2] - p1[2] * v2[0]);
  portal->equation[10] = (double) (-p1[0] * v2[1] + p1[1] * v2[0]);
  portal->equation[11] = (double) ( 0.0);
  portal->equation[12] = (double) ( p2[1] * v2[2] - p2[2] * v2[1]);
  portal->equation[13] = (double) (-p2[0] * v2[2] + p2[2] * v2[0]);
  portal->equation[14] = (double) ( p2[0] * v2[1] - p2[1] * v2[0]);
  portal->equation[15] = (double) ( 0.0);
  if (portal->option & P3_PORTAL_USE_5_CLIP_PLANES) {
    portal->equation[16] = (double) ( v1[1] * v2[2] - v1[2] * v2[1]);
    portal->equation[17] = (double) (-v1[0] * v2[2] + v1[2] * v2[0]);
    portal->equation[18] = (double) ( v1[0] * v2[1] - v1[1] * v2[0]);
    portal->equation[19] = (double) -(portal->equation[16] * p1[0] + portal->equation[17] * p1[1] + portal->equation[18] * p1[2]);
  }
}

void P3_portal_batch (P3_portal* portal, P3_instance* csys) {
  int i;
  GLfloat p;
  GLfloat* matrix;
  GLfloat* ptr;
  P3_frustum* f;
  P3_context* ctx;

// HACK
//printf("portal : batch\n");

  if (portal->option & P3_OBJECT_HIDDEN) return;
  portal->frustum_data = -1;
//  if (portal->option & P3_OBJECT_OUT_OF_VIEW) { portal->option -= P3_OBJECT_OUT_OF_VIEW; }

  /* visibility test : culling test */
  ptr = renderer->r_frustum->position;
  matrix = P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) portal);
  p = ptr[0] * matrix[2] + ptr[1] * matrix[6] + ptr[2] * matrix[10] + matrix[14];
  if (p < 0.0) { 
    /* camera is behind the portal : nothing to render */
    return;
  }
  /* visibility test : frustum test */
  if (portal->option & P3_PORTAL_INFINITE) {
    f = P3_renderer_get_frustum ((P3_instance*) portal);
    for (i = 2; i < 24; i += 3) {
      if (f->points[i] <= 0.0) { i = -1; break; }
    }
    if(i == -1) {
      return;
    }
  } else {
    GLfloat sphere[4];
    sphere[0] = portal->m[12];
    sphere[1] = portal->m[13];
    sphere[2] = portal->m[14];
    if (portal->m[16] > portal->m[17]) {
      sphere[3] = P3_sqrt_2 * portal->m[16];
    } else {
      sphere[3] = P3_sqrt_2 * portal->m[17];
    }
    f = P3_renderer_get_frustum ((P3_instance*) portal->parent);
    if (P3_sphere_in_frustum (f, sphere) == P3_FALSE) {
      return;
    }
  }
  /* load world beyond if necessary */
  if (portal->beyond == NULL) {
    if (portal->beyond_name == NULL) {
      return;
    } else {
      portal->beyond = P3_portal_load_beyond (portal);
    }
  }
  /* determine clipping planes */
  if (portal->equation != NULL) { P3_portal_set_clipping_planes (portal); }
  if (portal->option & P3_PORTAL_BOUND_ATMOSPHERE) {// || portal->option & P3_PORTAL_TELEPORTER) {
    /* compute render matrix to go to portal coordsys */
    P3_multiply_matrix (portal->render_matrix, renderer->c_camera->render_matrix, P3_coordsys_get_root_matrix ((P3_coordsys*) portal));
    P3_portal_computes_points (portal);
  }
  P3_list_add (renderer->portals, portal);
  /* avoid batching 2 times the same world */
  if (portal->beyond->option & P3_WORLD_BATCHED) { return; }
  /* try to know if world beyond has already been batched */
  if (P3_contains (renderer->c_camera->to_render, portal->beyond) == P3_TRUE) { return; }
  /* create a new context */
  ctx = renderer->c_context;
  portal->context = P3_renderer_get_context ();
  portal->context->atmosphere = portal->beyond->atmosphere;
  portal->context->portal     = portal;
  renderer->c_context = portal->context;
  /* make world beyond */
  portal->beyond->option |= P3_WORLD_BATCHED;
  P3_list_add (renderer->worlds_made, portal->beyond);
  P3_world_batch (portal->beyond, NULL);
  renderer->c_context = ctx;

// HACK
//printf("... done\n");

}

int P3_portal_shadow (P3_portal* p, P3_instance* inst, P3_light* light) {
  if (light->option & P3_LIGHT_TOP_LEVEL && p->beyond != NULL) {
    return P3_world_shadow (p->beyond, inst, light);
  }
  return P3_FALSE;
}

void P3_portal_computes_points (P3_portal* portal) {
  GLfloat* p1; GLfloat* p2;
  int nb; int i;
  GLfloat v;
  GLfloat f;
  P3_frustum* frustum;
  /* another hard stuff to draw the portal quad even if a part of the quad is below the camera
   * front plane (necessary to avoid blinking when the camera is very near the portal)
   */
  GLfloat p[12] = { -0.5, -0.5, 0.0, 0.5, -0.5, 0.0, 0.5, 0.5, 0.0, -0.5, 0.5, 0.0 };
  /* put the 4 points of the portal in the camera coordsys. render_matrix must be set !!! */
  P3_point_by_matrix (p,     portal->render_matrix);
  P3_point_by_matrix (p + 3, portal->render_matrix);
  P3_point_by_matrix (p + 6, portal->render_matrix);
  P3_point_by_matrix (p + 9, portal->render_matrix);
  /* find if there are point behind the camera front plane */
  free (portal->coords);
  if (p[2] > - renderer->c_camera->front || p[5] > - renderer->c_camera->front || p[8] > - renderer->c_camera->front || p[11] > - renderer->c_camera->front) {
    /* intersect portal with front plane */
    frustum = renderer->c_camera->frustum;
    p2 = (GLfloat*) malloc (4 * sizeof (GLfloat));
    p2[0] = - frustum->planes[0];
    p2[1] = - frustum->planes[1];
    p2[2] = - frustum->planes[2];
    p2[3] =   frustum->planes[3];
    P3_face_intersect_plane (p, 4, p2, &p1, &nb);
    free (p2);
    /* intersect portal with side planes */
    P3_face_intersect_plane (p1, nb, frustum->planes + 4, &p2, &nb);
    free (p1);
    P3_face_intersect_plane (p2, nb, frustum->planes + 8, &p1, &nb);
    free (p2);
    P3_face_intersect_plane (p1, nb, frustum->planes + 12, &p2, &nb);
    free (p1);
    P3_face_intersect_plane (p2, nb, frustum->planes + 16, &portal->coords, &(portal->nbpoints));
    free (p2);
    /* push portal points to the front plane 
     * in fact we must not draw the vertices with the camera front value as Z coordinate
     * cause some OpenGL won't draw the quad
     */
    p1 = portal->coords;
    f = - (renderer->c_camera->front + renderer->c_camera->back) * 0.5;
    for (i = 0; i < portal->nbpoints * 3; i += 3) {
      v = f / p1[i + 2];
      p1[i]     *= v;
      p1[i + 1] *= v;
      p1[i + 2] = f;
    }
  } else {
    portal->coords = NULL;
    portal->nbpoints = 0;
  }
  portal->coords = realloc (portal->coords, (portal->nbpoints + 4) * 3 * sizeof (GLfloat));
  nb = portal->nbpoints * 3;
  memcpy (portal->coords + nb, p, 12 * sizeof (GLfloat));
}

// TO DO not used (DEPRECATED)
void P3_portal_protect (P3_portal* portal) {
  int i;
  GLfloat* ptr;
  glDisable (GL_CULL_FACE);
  glColorMask (GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
  /* draw the part of the portal that is in front of the camera */
  glLoadIdentity ();
  ptr = portal->coords + portal->nbpoints * 3;
  glBegin (GL_QUADS);
  glVertex3fv (ptr);
  glVertex3fv (ptr + 3);
  glVertex3fv (ptr + 6);
  glVertex3fv (ptr + 9);
  glEnd ();
  /* draw the part of the portal that is behind the front plane of the camera */
  if (portal->nbpoints > 0) {
    glBegin (GL_POLYGON);
    for (i = 0; i < portal->nbpoints * 3; i += 3) {
      glVertex3fv (portal->coords + i);
    }
    glEnd ();
  }
  glEnable (GL_CULL_FACE);
  glColorMask (GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
}

void P3_atmosphere_clear_part (P3_portal* portal) {
  /* this function is here because it works similary to the P3_portal_protect function instead of that 
   * we draw the background color of the world beyond. 
   */
  int i;
  GLfloat* ptr;
  P3_atmosphere* atm = portal->beyond->atmosphere;
  /* draw the part of the portal that is in front of the camera */
  glLoadIdentity ();
  glDisable (GL_TEXTURE_2D);
  glDisable (GL_FOG);
  glDisable (GL_LIGHTING);
  glDepthMask (GL_FALSE);
  glColor4fv (atm->bg_color);
  glDisable (GL_CULL_FACE);
  ptr = portal->coords + portal->nbpoints * 3;
  glBegin (GL_QUADS);
  glVertex3fv (ptr);
  glVertex3fv (ptr + 3);
  glVertex3fv (ptr + 6);
  glVertex3fv (ptr + 9);
  glEnd ();
  /* draw the part of the portal that is behind the front plane of the camera */
  if (portal->nbpoints > 0) {
    glBegin (GL_POLYGON);
    for (i = 0; i < portal->nbpoints * 3; i += 3) {
      glVertex3fv (portal->coords + i);
    }
    glEnd ();
  }
  /* skyplane */
  if (atm->option & P3_ATMOSPHERE_SKYPLANE) {
    /* active clip planes */
    if (portal->equation == NULL) {
      portal->equation = (double*) malloc (16 * sizeof(double));
      P3_portal_set_clipping_planes (portal);
    }
    glClipPlane (GL_CLIP_PLANE0, portal->equation);
    glClipPlane (GL_CLIP_PLANE1, portal->equation + 4);
    glClipPlane (GL_CLIP_PLANE2, portal->equation + 8);
    glClipPlane (GL_CLIP_PLANE3, portal->equation + 12);
    glEnable (GL_CLIP_PLANE0);
    glEnable (GL_CLIP_PLANE1);
    glEnable (GL_CLIP_PLANE2);
    glEnable (GL_CLIP_PLANE3);
    /* draw sky */
    P3_atmosphere_draw_bg (atm);
    /* disable clip planes */
    glDisable (GL_CLIP_PLANE0);
    glDisable (GL_CLIP_PLANE1);
    glDisable (GL_CLIP_PLANE2);
    glDisable (GL_CLIP_PLANE3);
  }
  glEnable (GL_CULL_FACE);
  glDepthMask (GL_TRUE);
  glEnable (GL_TEXTURE_2D);
  glEnable (GL_FOG);
  glEnable (GL_LIGHTING);
}

void P3_portal_draw_fog (P3_portal* portal, P3_atmosphere* atm) {
  /* this function is here because it works similary to the P3_portal_protect function instead of that 
   * we draw the fog color of the parent world.
   */
  GLfloat* ptr;
  /* draw the part of the portal that is in front of the camera */
  glDisable (GL_TEXTURE_2D);
  glDisable (GL_FOG);
  glDisable (GL_LIGHTING);
//  glDepthMask (GL_FALSE);
  glDisable (GL_CULL_FACE);
//  glEnable (GL_BLEND);
  glLoadIdentity ();
  ptr = portal->coords + portal->nbpoints * 3;
  glBegin (GL_QUADS);
  glColor4f (atm->fog_color[0], atm->fog_color[1], atm->fog_color[2], P3_fog_factor_at (atm, ptr));
  glVertex3fv (ptr);
  glColor4f (atm->fog_color[0], atm->fog_color[1], atm->fog_color[2], P3_fog_factor_at (atm, ptr + 3));
  glVertex3fv (ptr + 3);
  glColor4f (atm->fog_color[0], atm->fog_color[1], atm->fog_color[2], P3_fog_factor_at (atm, ptr + 6));
  glVertex3fv (ptr + 6);
  glColor4f (atm->fog_color[0], atm->fog_color[1], atm->fog_color[2], P3_fog_factor_at (atm, ptr + 9));
  glVertex3fv (ptr + 9);
  glEnd ();
  /* draw the part of the portal that is behind the front plane of the camera
   * HACK supposed to be too near to have fog
   */
//  glDisable (GL_BLEND);
  glEnable (GL_CULL_FACE);
//  glDepthMask (GL_TRUE);
  glEnable (GL_TEXTURE_2D);
  glEnable (GL_FOG);
  glEnable (GL_LIGHTING);
}

/*
void P3_portal_render (P3_portal* portal, P3_instance* csys) {
  int i; int j;
  int nb = 4;

// HACK
//printf("portal : render\n");

  if (portal->option & P3_PORTAL_STATE_RENDERING) {
    portal->option -= P3_PORTAL_STATE_RENDERING;
    // disable clip planes
    if (portal->option & (P3_PORTAL_USE_4_CLIP_PLANES | P3_PORTAL_USE_5_CLIP_PLANES)) { 
      if (portal->option & P3_PORTAL_USE_5_CLIP_PLANES) { nb = 5; }
      for (i = 0; i < nb; i++) {
        j = nbclipplanes - 1;
        if (j < 0) { j = maxclipplanes - 1; }
        glDisable (GL_CLIP_PLANE0 + j);
        nbclipplanes--;
        if (nbclipplanes < 0) { 
          nbclipplanes = maxclipplanes - 1; 
printf("*** destroying too many clip planes\n");
        }
      }
    }
    if (portal->option & P3_PORTAL_STATE_ALPHA) {
      portal->option -= P3_PORTAL_STATE_ALPHA;
      // draw fog if necessary 
      if (portal->option & P3_PORTAL_BOUND_ATMOSPHERE) {
        P3_world* w;
        // get root atmosphere 
        w = (P3_world*) portal->parent;
        while (w != NULL && w->atmosphere == NULL) {
          w = (P3_world*) w->parent;
        }
        if (w->atmosphere->option & P3_ATMOSPHERE_FOG) {
          P3_portal_draw_fog (portal, ((P3_world*) portal->parent)->atmosphere);
        }
      }
      // protect z-buffer if necessary 
//      if (portal->option & P3_PORTAL_TELEPORTER) {
//        P3_portal_protect (portal);
//      }
    } else {
      portal->option |= P3_PORTAL_STATE_ALPHA;
    }
  } else {
    portal->option |= P3_PORTAL_STATE_RENDERING;
    // ask world beyond to clear a part of the screen if necessary 
    if (portal->option & P3_PORTAL_STATE_ALPHA) {
      portal->option -= P3_PORTAL_STATE_ALPHA;
    } else {
      portal->option |= P3_PORTAL_STATE_ALPHA;
//      if (portal->option & P3_PORTAL_BOUND_ATMOSPHERE && portal->beyond->atmosphere != NULL) { 
//        P3_atmosphere_clear_part (portal, portal->beyond->atmosphere); 
//      }
    }
    // enable clip planes 
    if (portal->option & (P3_PORTAL_USE_4_CLIP_PLANES | P3_PORTAL_USE_5_CLIP_PLANES)) { 
      glLoadIdentity ();  // the clip planes must be defined in the camera coordsys... so identity
      if (portal->option & P3_PORTAL_USE_5_CLIP_PLANES) { nb = 5; }
      for (i = 0; i < nb; i++) {
        glClipPlane (GL_CLIP_PLANE0 + nbclipplanes, portal->equation + i * 4);
        glEnable (GL_CLIP_PLANE0 + nbclipplanes);
        nbclipplanes++;
        if (nbclipplanes >= maxclipplanes) { 
          nbclipplanes = 0; 
printf("*** setting too many clip planes\n");
        }
      }
    }
  }

// HACK
//printf("... done\n");

}
*/

int P3_point_is_beyond_portal (P3_portal* portal, GLfloat* point, P3_coordsys* csys) {
  GLfloat p[3];
  if (csys != NULL) {
    P3_point_by_matrix_copy (p, point, P3_coordsys_get_root_matrix (csys));
    P3_point_by_matrix      (p, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) portal));
  } else {
    P3_point_by_matrix_copy (p, point, P3_coordsys_get_inverted_root_matrix ((P3_coordsys*) portal));
  }
  if (p[2] < 0.0 && p[0] <= 0.5 && p[0] >= -0.5 && p[1] <= 0.5 && p[1] >= -0.5) {
    return P3_TRUE; 
  }
  return P3_FALSE;
}

void P3_portal_raypick (P3_portal* portal, P3_raypick_data* data, P3_raypickable* parent) {
  P3_world_raypick (portal->beyond, data, parent);
}

int P3_portal_raypick_b (P3_portal* portal, P3_raypick_data* data, P3_raypickable* parent) {
  return P3_world_raypick_b (portal->beyond, data, parent);
}

void P3_portal_get_data (P3_portal* p, P3_chunk* chunk) {
  P3_chunk_save (chunk, p->m, 19 * sizeof (GLfloat));
  P3_chunk_save_int (chunk, p->option);
}

void P3_portal_set_data (P3_portal* p, P3_chunk* chunk) {
  P3_chunk_load (chunk, p->m, 19 * sizeof (GLfloat));
  p->option = P3_chunk_load_int (chunk);
}


/*
void P3_portal_connect (P3_portal* portal, P3_portal* portal2) {
  // compute the transformation matrix to link portal to another portal
  // it is different from P3_portal_set_transform_to because portal coordsys have scale 
  //  and orientation informations we want to pass through
  P3_coordsys* c = (P3_coordsys*) portal;
  P3_coordsys* d = (P3_coordsys*) portal2;
  GLfloat m[19];
  GLfloat n[19];
  GLfloat o[19];
//  if (portal->transform == NULL) { portal->transform = (GLfloat*) malloc (57 * sizeof (GLfloat)); }
  // find matrix to go inside the first portal
  P3_matrix_copy (o, c->m);
  P3_matrix_scale (o, 1.0 / c->m[16], 1.0 / c->m[17], 1.0);
  P3_matrix_invert (m, o);
  // find matrix to go outside the second portal
  P3_matrix_copy (n, d->m);
  P3_matrix_scale (n, 1.0 / d->m[16], 1.0 / d->m[17], 1.0);
  // this is because the portals must be z opposed
  n[ 0] = - n[ 0];
  n[ 1] = - n[ 1];
  n[ 2] = - n[ 2];
  n[ 8] = - n[ 8];
  n[ 9] = - n[ 9];
  n[10] = - n[10];
  // compute transform matrix and its inverse
  P3_multiply_matrix(portal->transform + 19, n, m);
  // this matrix is used for the render but is then multiplied by the beyond world coordsys matrix
  // that's why we need to compensate by multiplying by the inverse of this matrix
  P3_matrix_invert (m, portal->transform + 19);
  P3_multiply_matrix (portal->transform, m, P3_coordsys_get_inverted_matrix((P3_coordsys*) portal->beyond));
  // compute matrix for frustum. id to the camera matrix but as the frustum is multiplied by the world
  // beyond inverted matrix we must compensate
  P3_multiply_matrix (portal->transform + 38, portal->beyond->m, portal->transform + 19);
}

void P3_portal_set_transform_to (P3_portal* portal, P3_coordsys* coordsys) {
  // compute the transformation matrix to link portal to a coordsys = a space coordinate and orientation
  P3_coordsys* c = (P3_coordsys*) portal;
  GLfloat m[19];
  GLfloat n[19];
  GLfloat o[19];
// TO DO WARNING this has never been tested. better to use P3_portal_connect instead
  if (portal->transform == NULL) { portal->transform = (GLfloat*) malloc (57 * sizeof (GLfloat)); }
  // find matrix to go inside the portal
  P3_matrix_copy (o, c->m);
  P3_matrix_scale (o, 1.0 / c->m[16], 1.0 / c->m[17], 1.0);
  P3_matrix_invert (m, o);
  // find matrix to go outside the coordsys 
  P3_matrix_copy (n, coordsys->m);
  // compute transform matrix and its inverse 
  P3_multiply_matrix (portal->transform + 19, n, m);
  // this matrix is used for the render but is then multiplied by the beyond world coordsys matrix
  // that's why we need to compensate by multiplying by the inverse of this matrix 
  if (coordsys->parent == NULL) {
    P3_matrix_invert (portal->transform, portal->transform + 19);
  } else {
    P3_matrix_invert (m, portal->transform + 19);
    P3_multiply_matrix (portal->transform, m, P3_coordsys_get_inverted_matrix ((P3_coordsys*) coordsys->parent));
  }
}
*/

