/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************************************************
 * i_cal3d4p3.c : use Cal3D (http://cal3d.sourceforge.net)
 *              rocking animation lib with Soya
 * Copyright (C) 2003 Jean-Baptiste 'Jiba' LAMY
 *********************************************************/

//#include "../cal3d_wrapper.h" //#include <cal3d/cal3d_wrapper.h> // Buggous file ?

/*=============+
 | CAL3D SHAPE |
 +=============*/

static int PyP3Cal3DShape_Traverse (P3_cal3d_shape* self, visitproc visit, void* arg) {
	int err;
  int i;
  if (self->nb_materials > 0) {
    for (i = 0; i < self->nb_materials; i++) {
      err = visit ((PyObject*) self->materials[i], arg);
      if (err) { return err; }
    }
  }
	return 0;
}

static int PyP3Cal3DShape_Clear (P3_cal3d_shape* self) {
  int i;
  if (self->nb_materials > 0) {
    for( i = 0; i < self->nb_materials; i++) {
      Py_DECREF (self->materials[i]);
    }
    self->nb_materials = 0;
    free (self->materials);
  }
  return 0;
}

static void PyP3Cal3DShape_Dealloc (P3_cal3d_shape* self) {
  int i;
  PyObject_GC_UnTrack ((PyObject*) self);
  if (self->nb_materials > 0) {
    for (i = 0; i < self->nb_materials; i++) {
      Py_DECREF (self->materials[i]);
    }
    self->nb_materials = 0;
    free (self->materials);
  }
  CalCoreModel_Destroy (self->core_model);
  CalCoreModel_Delete  (self->core_model);
  self->ob_type->tp_free ((PyObject*) self);
}

static int PyP3Cal3DShape_Init (PyObject* self, PyObject* args, PyObject* kwds) {
  P3_cal3d_shape_new ((P3_cal3d_shape*) self);
	return 0;
}

static PyObject* PyP3Cal3DShape_LoadSkeleton (P3_cal3d_shape* self, PyObject* arg) {
  if (!CalCoreModel_LoadCoreSkeleton (self->core_model, PyString_AS_STRING (arg))) {
    P3_error (CalError_GetLastErrorDescription());
    P3_error ("CalCoreModel_LoadCoreSkeleton failed!");
    return PyInt_FromLong (0);
  }
  return PyInt_FromLong (1);
}

static PyObject* PyP3Cal3DShape_LoadMesh (P3_cal3d_shape* self, PyObject* arg) {
  int id = CalCoreModel_LoadCoreMesh (self->core_model, PyString_AS_STRING (arg));
  if (id == -1) { 
    P3_error (CalError_GetLastErrorDescription ()); 
    P3_error ("CalCoreModel_LoadCoreMesh failed!"); 
  }
  return PyInt_FromLong (id);
}

static PyObject* PyP3Cal3DShape_LoadMaterial (P3_cal3d_shape* self, PyObject* arg) {
  int id = CalCoreModel_LoadCoreMaterial (self->core_model, PyString_AS_STRING (arg));
  if (id == -1) { 
    P3_error (CalError_GetLastErrorDescription ()); 
    P3_error ("CalCoreModel_LoadCoreMaterial failed!"); 
  }
  return PyInt_FromLong (id);
}

static PyObject* PyP3Cal3DShape_LoadAnimation (P3_cal3d_shape* self, PyObject* arg) {
  int id = CalCoreModel_LoadCoreAnimation (self->core_model, PyString_AS_STRING (arg));
  if (id == -1) { 
    P3_error (CalError_GetLastErrorDescription()); 
    P3_error ("CalCoreModel_LoadCoreAnimation failed!"); 
  }
  return PyInt_FromLong (id);
}

static PyObject* PyP3Cal3DShape_BuildMaterials (P3_cal3d_shape* shape) {
  int i;
  struct CalCoreMaterial* material;
  
  if (shape->nb_materials > 0) {
    for (i = 0; i < shape->nb_materials; i++) { Py_DECREF (shape->materials[i]); }
    free (shape->materials);
  }
  if (shape->core_model == NULL) { 
    Py_INCREF (Py_None);
    return Py_None; 
  }
  
  shape->nb_materials = CalCoreModel_GetCoreMaterialCount (shape->core_model);
  shape->materials    = (P3_material**) malloc (shape->nb_materials * sizeof (P3_material*));
  for (i = 0; i < shape->nb_materials; i++) {
    CalCoreModel_CreateCoreMaterialThread (shape->core_model, i);
    CalCoreModel_SetCoreMaterialId (shape->core_model, i, 0, i);
    
    material = CalCoreModel_GetCoreMaterial (shape->core_model, i);
    CalCoreMaterial_SetUserData (material, (CalUserData) i);
    
    shape->materials[i] = (P3_material*) PyObject_CallMethod
      ((PyObject*) shape, "_get_material_4_cal3d", "sfffffffff",
       CalCoreMaterial_GetMapFilename (material, 0),
       1.0, // It seems that the Cal3D C version
       1.0, // does not support yet the
       1.0, // CalCoreMaterial_Get*Color functions.
       1.0,
       1.0,
       1.0,
       1.0,
       1.0,
       CalCoreMaterial_GetShininess (material));
    Py_INCREF (shape->materials[i]);
  }
  
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DShape_SetCellShading (P3_cal3d_shape* a, PyObject* args) {
  P3_material* shader;
  PyObject* tuple;
  GLfloat color[4];
  tuple = PySequence_Fast_GET_ITEM (args, 1);
  PY_TUPLE_FLOAT_TO_ARRAY_4 (color, tuple);
  shader = (P3_material*) PySequence_Fast_GET_ITEM (args, 0);
  if ((PyObject*) shader == Py_None) { shader = NULL; }
  P3_cal3d_shape_set_cell_shading (a, shader, color, (GLfloat) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

PY_GET_SET_ON_OPTION (Cal3DShape, P3_cal3d_shape*, ShadowCast, P3_CAL3D_SHADOW_CAST)

static PyGetSetDef PyP3Cal3DShape_GetSets[] = {
  { "shadow_cast", (getter) PyP3Cal3DShape_GetShadowCast, (setter) PyP3Cal3DShape_SetShadowCast, NULL },
  { NULL }
};

static PyMethodDef PyP3Cal3DShape_Methods[] = {
  { "set_cell_shading",      (PyCFunction) PyP3Cal3DShape_SetCellShading, METH_VARARGS },
  { "_load_skeleton",        (PyCFunction) PyP3Cal3DShape_LoadSkeleton,   METH_O },
  { "_load_mesh",            (PyCFunction) PyP3Cal3DShape_LoadMesh,       METH_O },
  { "_load_material",        (PyCFunction) PyP3Cal3DShape_LoadMaterial,   METH_O },
  { "_load_animation",       (PyCFunction) PyP3Cal3DShape_LoadAnimation,  METH_O },
  { "_build_materials",      (PyCFunction) PyP3Cal3DShape_BuildMaterials, METH_NOARGS },
  { NULL, NULL }
};

PyTypeObject PyP3Cal3DShape_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Cal3DShape",
  sizeof(P3_cal3d_shape),
  0,
  (destructor) PyP3Cal3DShape_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Cal3DShape_Traverse,/* tp_traverse */
  (inquiry) PyP3Cal3DShape_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Cal3DShape_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Cal3DShape_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Cal3DShape_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};


/*==============+
 | CAL3D VOLUME |
 +==============*/

static int PyP3Cal3DVolume_Init (PyObject* self, PyObject* args, PyObject* kwds) {
  P3_cal3d_volume_new ((P3_cal3d_volume*) self);
	return 0;
}

static int PyP3Cal3DVolume_Traverse (P3_cal3d_volume* w, visitproc visit, void* arg) {
	int err;
  if (w->shape != NULL) {
    err = visit ((PyObject*) w->shape, arg);
    if (err) { return err; }
  }
  if (w->parent != NULL) {
    err = visit ((PyObject*) w->parent, arg);
    if (err) { return err; }
  }
  if (w->attached_coordsys != NULL) {
    err = visit ((PyObject*) w->attached_coordsys, arg);
    if (err) { return err; }
  }
	return 0;
}

static int PyP3Cal3DVolume_Clear (P3_cal3d_volume* self) {
  Py_XDECREF (self->shape);
  Py_XDECREF (self->parent);
  Py_XDECREF (self->attached_coordsys);
  self->shape = NULL;
  self->parent = NULL;
  self->attached_coordsys = NULL;
  return 0;
}

static void PyP3Cal3DVolume_Dealloc (P3_cal3d_volume* self) {
  Py_XDECREF (self->shape);
  Py_XDECREF (self->parent);
  Py_XDECREF (self->attached_coordsys);

  P3_cal3d_volume_dealloc (self);
  
  PyObject_GC_UnTrack ((PyObject*) self);
  self->ob_type->tp_free ((PyObject*) self);
}

static PyObject* PyP3Cal3DVolume_GetShape (P3_cal3d_volume* volume, void* context) {
  if (volume->shape == NULL) {
    Py_INCREF (Py_None);
    return Py_None;
  } else {
    Py_INCREF ((PyObject*) volume->shape);
    return (PyObject*) volume->shape;
  }
}
static PyObject* PyP3Cal3DVolume_SetShape (P3_cal3d_volume* volume, PyObject* arg) {
  Py_XDECREF ((PyObject*) volume->shape);
  if (arg == Py_None || arg == NULL) {
    P3_cal3d_volume_set_shape (volume, NULL);
  } else {
    Py_INCREF (arg);
    P3_cal3d_volume_set_shape (volume, (P3_cal3d_shape*) arg);
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_Update (P3_cal3d_volume* volume, PyObject* arg) {
  volume->delta_time += (float) PyFloat_AS_DOUBLE(arg);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_SetAttached (P3_cal3d_volume* volume, PyObject* args) {
  P3_cal3d_volume_set_attached (volume, 
                                (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)), 
                                (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 1)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_IsAttached (P3_cal3d_volume* volume, PyObject* arg) {
  return PyInt_FromLong ((long) volume->attached_states[(int) PyInt_AS_LONG (arg)]);
}

static PyObject* PyP3Cal3DVolume_AttachAll (P3_cal3d_volume* volume) {
  P3_cal3d_volume_attach_all (volume);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_SetLodLevel (P3_cal3d_volume* volume, PyObject* arg) {
  if (volume->model != NULL) { 
    CalModel_SetLodLevel (volume->model, (float) PyFloat_AS_DOUBLE(arg)); 
  }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_BuildSubshapes (P3_cal3d_volume* volume) {
  P3_cal3d_volume_build_submeshes (volume);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_AnimateClearCycle (P3_cal3d_volume* volume, PyObject* args) {
  CalMixer_ClearCycle (CalModel_GetMixer (volume->model),
                       (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)),
                       (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_AnimateBlendCycle (P3_cal3d_volume* volume, PyObject* args) {
  CalMixer_BlendCycle (CalModel_GetMixer (volume->model),
                       (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)),
                       (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)),
                       (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_AnimateExecuteAction (P3_cal3d_volume* volume, PyObject* args) {
  CalMixer_ExecuteAction (CalModel_GetMixer (volume->model),
                          (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (args, 0)),
                          (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 1)),
                          (float) PyFloat_AS_DOUBLE (PySequence_Fast_GET_ITEM (args, 2)));
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_GetState (P3_cal3d_volume* w) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* attached_states;
  PyObject* tuple;
  int i;
  P3_cal3d_volume_get_data (w, chunk);
  tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  if (w->shape == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 1, Py_None);
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 2, Py_None);
  } else {
    Py_INCREF ((PyObject*) w->shape);
    PyTuple_SET_ITEM (tuple, 1, (PyObject*) w->shape);
// TO DO put in C ?
    attached_states = PyTuple_New (w->nb_attached_states);
    for (i = 0; i < w->nb_attached_states; i++) {
      PyTuple_SET_ITEM (attached_states, i, PyInt_FromLong((long) w->attached_states[i]));
    }
    PyTuple_SET_ITEM (tuple, 2, attached_states);
  }
  
  Py_INCREF ((PyObject*) w->attached_coordsys);
  PyTuple_SET_ITEM (tuple, 3, (PyObject*) w->attached_coordsys);
  
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Cal3DVolume_SetState (P3_cal3d_volume* w, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* attached_states;
  PyObject* s;
  int i;
  s = PySequence_Fast_GET_ITEM (args, 0);
  chunk->content = PyString_AS_STRING (s);
  P3_cal3d_volume_set_data (w, chunk);
  s = PySequence_Fast_GET_ITEM (args, 1);
  if (s == Py_None) {
    w->shape = NULL;
  } else {
    w->shape = (P3_cal3d_shape*) s;
    Py_INCREF ((PyObject*) w->shape);
    w->model = CalModel_New ();
    CalModel_Create (w->model, w->shape->core_model);
    attached_states = PySequence_Fast_GET_ITEM (args, 2);
    w->nb_attached_states = PySequence_Size (attached_states);
    w->attached_states = malloc (w->nb_attached_states * sizeof(int));
    for (i = 0; i < w->nb_attached_states; i++) {
      w->attached_states[i] = (int) PyInt_AS_LONG (PySequence_Fast_GET_ITEM (attached_states, i));
      if (w->attached_states[i] == 1) {
        CalModel_AttachMesh (w->model, i);
      }
    }
    P3_cal3d_volume_build_submeshes (w);
  }
  
  w->attached_coordsys = (P3_children) PySequence_Fast_GET_ITEM (args, 3);
  Py_INCREF ((PyObject*) w->attached_coordsys);
  
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject* PyP3Cal3DVolume_BoneId (P3_cal3d_volume* w, PyObject* arg) {
  char* name;
  int i;
  struct CalCoreSkeleton* core_skeleton;
  struct CalCoreBone*     core_bone;
  
  name = PyString_AS_STRING(arg);
  core_skeleton = CalSkeleton_GetCoreSkeleton(CalModel_GetSkeleton(w->model));
  i = 0;
  while(1) {
    core_bone = CalCoreSkeleton_GetCoreBone(core_skeleton, i);
    if (core_bone == NULL) {
      PyErr_SetString(PyExc_ValueError, "No bone with this name.");
      return NULL;
    }
    if (strcmp(name, CalCoreBone_GetName(core_bone)) == 0) {
      return PyInt_FromLong((long) i);
    }
    i++;
  }
}

static PyObject* PyP3Cal3DVolume_MoveToBone (P3_cal3d_volume* w, PyObject* args) {
  struct CalBone* bone;
  P3_instance* instance;
  float* trans;
  
  instance = ((P3_instance*) PySequence_Fast_GET_ITEM(args, 0));
  
  bone = CalSkeleton_GetBone(CalModel_GetSkeleton(w->model), (int) PyInt_AS_LONG(PySequence_Fast_GET_ITEM(args, 1)));
  
  P3_matrix_from_quaternion(instance->m, CalQuaternion_Get(CalBone_GetRotationAbsolute(bone)));
  
  trans = CalVector_Get(CalBone_GetTranslationAbsolute(bone));
  P3_matrix_translate(instance->m, trans[0], trans[1], trans[2]);
  
  Py_INCREF (Py_None);
  return Py_None;
}

PY_GET_SET_ON_OBJECT (Cal3DVolume, P3_cal3d_volume*, AttachedCoordsys, attached_coordsys, PyObject*)

PY_GET_SET_ON_FLOAT_ARRAY (Cal3DVolume, P3_cal3d_volume*, Sphere, sphere, 4)

static PyGetSetDef PyP3Cal3DVolume_GetSets[] = {
  PYP3_VISIBLE_GETSETS,
  PYP3_CHILD_GETSETS,
  PYP3_COORDSYS_GETSETS,
  { "shape",             (getter) PyP3Cal3DVolume_GetShape,            0,                                              NULL },
  { "attached_coordsys", (getter) PyP3Cal3DVolume_GetAttachedCoordsys, (setter) PyP3Cal3DVolume_SetAttachedCoordsys,   NULL },
  { "sphere",            (getter) PyP3Cal3DVolume_GetSphere,           (setter) PyP3Cal3DVolume_SetSphere,             NULL },
  { NULL }
};

static PyMethodDef PyP3Cal3DVolume_Methods[] = {
  PYP3_COORDSYS_FUNCS,
  { "_set_lod_level",            (PyCFunction) PyP3Cal3DVolume_SetLodLevel,             METH_O },
  { "set_shape",                 (PyCFunction) PyP3Cal3DVolume_SetShape,                METH_O },
  { "_update",                   (PyCFunction) PyP3Cal3DVolume_Update,                  METH_O },
  { "_set_attached",             (PyCFunction) PyP3Cal3DVolume_SetAttached,             METH_VARARGS },
  { "_is_attached",              (PyCFunction) PyP3Cal3DVolume_IsAttached,              METH_O },
  { "attach_all",                (PyCFunction) PyP3Cal3DVolume_AttachAll,               METH_NOARGS },
  { "_build_subshapes",          (PyCFunction) PyP3Cal3DVolume_BuildSubshapes,          METH_NOARGS },
  { "_animate_clear_cycle",      (PyCFunction) PyP3Cal3DVolume_AnimateClearCycle,       METH_VARARGS },
  { "_animate_blend_cycle",      (PyCFunction) PyP3Cal3DVolume_AnimateBlendCycle,       METH_VARARGS },
  { "_animate_execute_action",   (PyCFunction) PyP3Cal3DVolume_AnimateExecuteAction,    METH_VARARGS },
  { "_getstate",                 (PyCFunction) PyP3Cal3DVolume_GetState,                METH_NOARGS },
  { "_setstate",                 (PyCFunction) PyP3Cal3DVolume_SetState,                METH_O },
  { "_bone_id",                  (PyCFunction) PyP3Cal3DVolume_BoneId,                  METH_O },
  { "_move_to_bone",             (PyCFunction) PyP3Cal3DVolume_MoveToBone,              METH_VARARGS },
  { NULL, NULL }
};

PyTypeObject PyP3Cal3DVolume_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Cal3DVolume",
  sizeof(P3_cal3d_volume),
  0,
  (destructor) PyP3Cal3DVolume_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Cal3DVolume_Traverse,/* tp_traverse */
  (inquiry) PyP3Cal3DVolume_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Cal3DVolume_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Cal3DVolume_GetSets,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Cal3DVolume_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};




