/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**********************************************
 * Copyright (C) 2002-2003 Bertrand 'blam' LAMY
 **********************************************/

/*================================*
 * FACE: Python object definition *
 *================================*/

void P3_face_raypick (PyObject* face, P3_raypick_data* data, P3_raypickable* parent) {
  PyObject* o; PyObject* w; PyObject* c;
  GLfloat* raydata;
  GLfloat* p;
  GLfloat r;
  GLfloat normal[3];
  int nb; int i;
  int option;
  int double_sided = P3_FALSE;
  o = PyObject_GetAttrString (face, "vertices");
  nb = PySequence_Size (o);
  if (nb < 3) return;
  option = data->option;
  w = PyObject_GetAttrString (face, "double_sided");
  if (PyObject_IsTrue (w) == 1) {
    option &= ~P3_RAYPICK_CULL_FACE;
    double_sided = P3_TRUE;
  }
  Py_DECREF (w);
  /* get vertices coordinates */
  c = PyObject_GetAttrString (face, "parent");
  p = (GLfloat*) malloc (nb * 3 * sizeof (GLfloat));
  for (i = 0; i < nb; i++) {
    PyP3_GetPositionInto (PySequence_Fast_GET_ITEM (o, i), (P3_coordsys*) c, p + i * 3);
  }
  P3_face_normal (normal, p, p + 3, p + 6); 
  P3_vector_normalize (normal);
  raydata = P3_raypickable_get_raypick_data ((P3_raypickable*) c, data);
  if (nb == 3) {
    i = P3_triangle_raypick (raydata, p, p + 3, p + 6, normal, option, &r);
  } else if (nb == 4) {
    i = P3_quad_raypick (raydata, p, p + 3, p + 6, p + 9, normal, option, &r);
  } else {
// TO DO ?
    P3_error ("raypicking on face with more than 4 vertices is not supported");
    i = P3_FALSE;
  }
  if (data->ret_csys == NULL || fabs (r) < fabs (data->result)) {
    if (i == P3_RAYPICK_DIRECT) {
      data->result = r;
      data->ret_csys = c;
      memcpy (data->normal, normal, 3 * sizeof (GLfloat));
    } else if (i == P3_RAYPICK_INDIRECT) {
      data->result = r;
      data->ret_csys = c;
      if (double_sided == P3_TRUE) {
        data->normal[0] = -normal[0];
        data->normal[1] = -normal[1];
        data->normal[2] = -normal[2];
      } else {
        memcpy (data->normal, normal, 3 * sizeof (GLfloat));
      }
    }
  }
  free (p);
  Py_DECREF (c);
  Py_DECREF (o);
}

int P3_face_raypick_b (PyObject* face, P3_raypick_data* data, P3_raypickable* parent) {
  PyObject* o; PyObject* w; PyObject* c;
  GLfloat* raydata;
  GLfloat* p;
  GLfloat r; GLfloat normal[3];
  int nb; int i;
  int option;
  o = PyObject_GetAttrString (face, "vertices");
  nb = PySequence_Size (o);
  if (nb < 3) { return P3_FALSE; }
  option = data->option;
  if (option & P3_RAYPICK_CULL_FACE) {
    w = PyObject_GetAttrString (face, "double_sided");
    if (PyObject_IsTrue (w) == 1) option &= ~P3_RAYPICK_CULL_FACE;
    Py_DECREF (w);
  }
  /* get vertices coordinates */
  p = (GLfloat*) malloc (nb * 3 * sizeof (GLfloat));
  c = PyObject_GetAttrString (face, "parent");
  for (i = 0; i < nb; i++) {
    PyP3_GetPositionInto (PySequence_Fast_GET_ITEM (o, i), (P3_coordsys*) c, p + i * 3);
  }
  P3_face_normal (normal, p, p + 3, p + 6); 
  P3_vector_normalize (normal);
  raydata = P3_raypickable_get_raypick_data (parent, data);
  if (nb == 3) {
    i = P3_triangle_raypick (raydata, p, p + 3, p + 6, normal, option, &r);
  } else if (nb == 4) {
    i = P3_quad_raypick (raydata, p, p + 3, p + 6, p + 9, normal, option, &r);
  } else {
// TO DO ?
    P3_error ("raypicking on face with more than 4 vertices is not supported");
    i = P3_FALSE;
  }
  free (p);
  Py_DECREF (o);
  return i;
}

P3_class P3_class_face = { 
  P3_ID_FACE,
  (batch_func)     P3_shape_batch,
  (render_func)    P3_shape_render,
  (shadow_func)    0,
  (raypick_func)   P3_face_raypick,
  (raypick_b_func) P3_face_raypick_b,
};

static int PyP3Face_Init (PyObject* self, PyObject* args, PyObject* kwds) {
  ((P3_any_object*) self)->class = &P3_class_face;
	return 0;
}

PyTypeObject PyP3Face_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_Face",
  sizeof(P3_any_object),
  0,
  (destructor) PyP3Object_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) 0,/* tp_traverse */
  (inquiry) 0,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) 0,//&PyP3Renderable_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) 0,/* tp_getset */
  0,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Face_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_FREE,/* tp_free */
};


/*=====================================*
 * FACE: C functions about Python face *
 *=====================================*/

/*
int P3_face_get_option (P3_face* f1) {
  int opt;
  opt = 0;
  if (P3_face_is_smoothlit (f1))         { opt |= P3_R_SMOOTHLIT; }
  if (P3_face_is_alpha (f1))             { opt |= P3_R_ALPHA; }
  if (P3_face_is_double_sided (f1))      { opt |= P3_R_FRONT_AND_BACK; }
  if (P3_face_is_static_lit (f1))        { opt |= P3_R_STATIC_LIT; }
  if (P3_face_is_colored (f1))           { opt |= P3_R_COLORED; }
  if (!(P3_face_is_solid (f1)))          { opt |= P3_OBJECT_NON_SOLID; }
  if (!(P3_face_can_be_lit (f1)))        { opt |= P3_R_NEVER_LIT; }
  if (P3_face_get_material (f1) != NULL) { opt |= P3_R_TEXTURED; }
  return opt;
}
*/

P3_material* P3_face_get_material (PyObject* face) {
  PyObject* m;
  m = PyObject_GetAttrString (face, "material");
  Py_XDECREF (m);
  if (m == Py_None) {
    return NULL;
  } else {
    return (P3_material*) m;
  }
}

P3_coordsys* P3_face_get_coordsys (PyObject* face) {
  P3_coordsys* c;
  c = (P3_coordsys*) PyObject_GetAttrString (face, "parent");
  Py_XDECREF (c);
  if ((PyObject*) c == Py_None) {
    return NULL;
  } else {
    return c;
  }
}

void P3_face_get_box (PyObject* face, GLfloat* box, int* edited) {
  PyObject* vertex;
  int nbv;
  GLfloat coord[3];
  P3_coordsys* csys;
  int i;
  nbv = P3_face_get_vertices_number (face);
  for (i = 0; i < nbv; i++) {
    vertex = P3_face_get_vertex (face, i);
    csys = P3_vertex_get_coordsys (vertex);
    P3_vertex_get_coord (vertex, coord);
    P3_point_by_matrix (coord, P3_coordsys_get_root_matrix (csys));
    if (*edited == P3_FALSE) {
      *edited = P3_TRUE;
      memcpy (box,     coord, 3 * sizeof (GLfloat));
      memcpy (box + 3, coord, 3 * sizeof (GLfloat));
    } else {
      if (coord[0] < box[0]) { box[0] = coord[0]; }
      if (coord[1] < box[1]) { box[1] = coord[1]; }
      if (coord[2] < box[2]) { box[2] = coord[2]; }
      if (coord[0] > box[3]) { box[3] = coord[0]; }
      if (coord[1] > box[4]) { box[4] = coord[1]; }
      if (coord[2] > box[5]) { box[5] = coord[2]; }
    }
  }
}

P3_coordsys* P3_vertex_get_coordsys (PyObject* vertex) {
  P3_coordsys* c;
  c = (P3_coordsys*) PyObject_GetAttrString (vertex, "parent");
  Py_XDECREF (c);
  if ((PyObject*) c == Py_None) {
    return NULL;
  } else {
    return c;
  }
}

void P3_vertex_set_diffuse (PyObject* o, GLfloat* c) {
  PyObject* tuple;
  tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) c[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) c[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) c[2]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) c[3]));
  PyObject_SetAttrString (o, "color", tuple);
  Py_DECREF (tuple);
}

void P3_vertex_set_emissive (PyObject* o, GLfloat* c) {
  PyObject* tuple;
  tuple = PyTuple_New (4);
  PyTuple_SET_ITEM (tuple, 0, PyFloat_FromDouble ((double) c[0]));
  PyTuple_SET_ITEM (tuple, 1, PyFloat_FromDouble ((double) c[1]));
  PyTuple_SET_ITEM (tuple, 2, PyFloat_FromDouble ((double) c[2]));
  PyTuple_SET_ITEM (tuple, 3, PyFloat_FromDouble ((double) c[3]));
  PyObject_SetAttrString (o, "emissive", tuple);
  Py_DECREF (tuple);
}

PyObject* P3_face_new (P3_world* w, P3_material* material, int nb_v, ...) {
  va_list ap;
  PyObject* f;
  PyObject* list;
  int i;
  va_start (ap, nb_v);
  if (material == NULL) { material = (P3_material*) Py_None; }
  list = PyList_New (0);
  for (i = 0; i < nb_v; i++) {
    PyList_Append (list, va_arg (ap, PyObject*));
  }
  f = PyObject_CallMethod (P3Module, "new_face", "OOO", w, list, material);
#ifdef SAFE_MODE
  if (PyErr_Occurred () != NULL) { PyErr_Print (); }
#endif /* SAFE_MODE */
  va_end (ap);
  return f;
}

PyObject* P3_vertex_new (P3_world* w, GLfloat x, GLfloat y, GLfloat z) {
  PyObject* v;
  v = PyObject_CallMethod (P3Module, "new_vertex", "Offf", w, x, y, z);
  PyObject_SetAttrString (v, "color", Py_None);
#ifdef SAFE_MODE
  if (PyErr_Occurred () != NULL) { PyErr_Print (); }
#endif /* SAFE_MODE */
  return v;
}

#define FUNC_FACE_GET(name, attr) \
  int P3_face_##name (PyObject* face) { \
    PyObject* o; \
    int i = 0; \
    o = PyObject_GetAttrString (face, attr); \
    if (o != NULL) { \
      i = (PyObject_IsTrue (o) == 1); \
      Py_DECREF (o); \
    } \
    return i; \
  }
#define FUNC_FACE_IS(name, attr) \
  int P3_face_##name (PyObject* face) { \
    PyObject* o; \
    int i = 0; \
    o = PyObject_CallMethod (face, attr, NULL); \
    if (o != NULL) { \
      i = (PyObject_IsTrue (o) == 1); \
      Py_DECREF (o); \
    } \
    return i; \
  }
#define FUNC_FACE_SET(name, attr) \
  void P3_face_set_##name (PyObject* face, int value) { \
    PyObject* o; \
    o = PyInt_FromLong ((long) value); \
    PyObject_SetAttrString (face, attr, o); \
    Py_DECREF (o); \
  }
#define FUNC_FACE_TRY(name, attr) \
  int P3_face_##name (PyObject* face) { \
    PyObject* o; \
    int i = 0; \
    if (PyObject_HasAttrString (face, attr)) { \
      o = PyObject_GetAttrString (face, attr); \
      i = PyObject_IsTrue (o); \
      Py_XDECREF (o); \
    } \
    return i; \
  }

FUNC_FACE_GET (is_smoothlit, "smooth_lit")
FUNC_FACE_GET (is_double_sided, "double_sided")
FUNC_FACE_GET (is_solid, "solid")
FUNC_FACE_GET (is_static_lit, "static_lit")
FUNC_FACE_GET (can_be_lit, "lit")
FUNC_FACE_IS  (is_alpha, "is_alpha")
FUNC_FACE_IS  (is_colored, "is_colored")
FUNC_FACE_SET (smoothlit, "smooth_lit")
FUNC_FACE_SET (double_sided, "double_sided")
FUNC_FACE_SET (solid, "solid")
FUNC_FACE_SET (static_lit, "static_lit")
FUNC_FACE_SET (can_be_lit, "lit")
FUNC_FACE_SET (cell_shading, "cell_shading")
FUNC_FACE_TRY (cell_shading, "cell_shading")

#undef FUNC_FACE_GET
#undef FUNC_FACE_IS
#undef FUNC_FACE_SET
#undef FUNC_FACE_TRY

int P3_face_get_vertices_number (PyObject* face) {
  PyObject* o;
  int i = 0;
  o = PyObject_GetAttrString (face, "vertices");
  if (o != NULL) {
    i = PySequence_Size (o);
    Py_DECREF (o);
  }
  return i;
}

PyObject* P3_face_get_vertex (PyObject* face, int index) {
  PyObject* o;
  PyObject* v = NULL;
  o = PyObject_GetAttrString (face, "vertices");
  if (o != NULL) {
    v = PySequence_Fast_GET_ITEM (o, index);
    Py_DECREF (o);
  }
  return v;
}

void P3_face_add_vertex (PyObject* face, PyObject* vertex) {
  PyObject* o;
  o = PyObject_GetAttrString (face, "vertices");
  if (o != NULL) {
    Py_INCREF (vertex);
    PyList_Append (o, vertex);
    Py_DECREF (o);
  }
}

void P3_vertex_get_coord (PyObject* v, GLfloat* ptr) {
  PyObject* o;
  o = PyObject_GetAttrString (v, "x");
  if (o != NULL) {
    ptr[0] = (GLfloat) PyFloat_AS_DOUBLE (o);
    Py_DECREF (o);
  }
  o = PyObject_GetAttrString (v, "y");
  if (o != NULL) {
    ptr[1] = (GLfloat) PyFloat_AS_DOUBLE (o);
    Py_DECREF (o);
  }
  o = PyObject_GetAttrString (v, "z");
  if (o != NULL) {
    ptr[2] = (GLfloat) PyFloat_AS_DOUBLE (o);
    Py_DECREF (o);
  }
}

void P3_vertex_get_texcoord (PyObject* v, GLfloat* ptr) {
  PyObject* o;
  o = PyObject_GetAttrString (v, "tex_x");
  if (o != NULL) {
    ptr[0] = (GLfloat) PyFloat_AS_DOUBLE (o);
    Py_DECREF (o);
  }
  o = PyObject_GetAttrString (v, "tex_y");
  if (o != NULL) {
    ptr[1] = (GLfloat) PyFloat_AS_DOUBLE (o);
    Py_DECREF (o);
  }
}

void P3_vertex_get_diffuse (PyObject* v, GLfloat* ptr) {
  PyObject* o;
  o = PyObject_GetAttrString (v, "color");
  if (o != NULL && o != Py_None) {
    PY_TUPLE_FLOAT_TO_ARRAY_4 (ptr, o);
    Py_DECREF (o);
  } else {
    ptr[0] = 1.0;
    ptr[1] = 1.0;
    ptr[2] = 1.0;
    ptr[3] = 1.0;
  }
}

void P3_vertex_get_emissive (PyObject* v, GLfloat* ptr) {
  PyObject* o;
  if (PyObject_HasAttrString (v, "emissive")) {
    o = PyObject_GetAttrString (v, "emissive");
    if (o != NULL && o != Py_None) {
      PY_TUPLE_FLOAT_TO_ARRAY_4 (ptr, o);
      Py_DECREF (o);
      return;
    }
  }
  ptr[0] = 0.0;
  ptr[1] = 0.0;
  ptr[2] = 0.0;
  ptr[3] = 1.0;
}

int P3_vertex_is_morphing (PyObject* v) {
  PyObject* o;
  int i;
  if (PyObject_HasAttrString (v, "morphing") == 1) {
    o = PyObject_GetAttrString (v, "morphing");
    i = PyInt_AS_LONG (o);
    Py_XDECREF (o);
    return i;
  }
  return 0;
}

P3_material* P3_vertex_get_material (PyObject* v) {
  PyObject* m;
  if (!PyObject_HasAttrString (v, "material")) {
    return NULL;
  }
  m = PyObject_GetAttrString (v, "material");
  Py_XDECREF (m);
  if (m == Py_None) {
    return NULL;
  } else {
    return (P3_material*) m;
  }
}

void P3_vertex_set_texcoord (PyObject* v, GLfloat* uv) {
  PyObject* o;
  o = PyFloat_FromDouble ((double) uv[0]);
  PyObject_SetAttrString (v, "tex_x", o);
  Py_DECREF (o);
  o = PyFloat_FromDouble ((double) uv[1]);
  PyObject_SetAttrString (v, "tex_y", o);
  Py_DECREF (o);
}
