/*
 * P3 python wrapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************
 * Copyright (C) 2002 Bertrand 'blam' LAMY
 *****************************************/

/*========+
 | PORTAL |
 +========*/

/*---------+
 | Methods |
 +---------*/

static int PyP3Portal_Traverse (P3_portal* w, visitproc visit, void* arg) {
	int err;
  if (w->beyond != NULL) {
    err = visit ((PyObject*) w->beyond, arg);
    if (err) { return err; }
  }
  if (w->parent != NULL) {
    err = visit ((PyObject*) w->parent, arg);
    if (err) { return err; }
  }
	return 0;
}

static int PyP3Portal_Clear (P3_portal* self) {
  Py_XDECREF (self->beyond);
  Py_XDECREF (self->parent);
  self->beyond = NULL;
  self->parent = NULL;
  return 0;
}

static int PyP3Portal_Init (PyObject* self, PyObject* args, PyObject* kwds) {
  P3_portal_new ((P3_portal*) self);
	return 0;
}

static void PyP3Portal_Dealloc (P3_portal* w) {
  PyObject_GC_UnTrack ((PyObject*) w);
  Py_XDECREF (w->beyond);
  Py_XDECREF (w->parent);
  w->ob_type->tp_free ((PyObject*) w);
}

static PyObject* PyP3Portal_IsBeyond (P3_portal* p, PyObject* arg) {
  P3_coordsys* coordsys;
  GLfloat point[3];
  PyObject* o;
  o = PyObject_GetAttrString (arg, "x");
  point[0] = (GLfloat) PyFloat_AS_DOUBLE (o);
  Py_DECREF (o);
  o = PyObject_GetAttrString (arg, "y");
  point[1] = (GLfloat) PyFloat_AS_DOUBLE (o);
  Py_DECREF (o);
  o = PyObject_GetAttrString (arg, "z");
  point[2] = (GLfloat) PyFloat_AS_DOUBLE (o);
  Py_DECREF (o);
  coordsys = (P3_coordsys*) PyObject_GetAttrString (arg, "parent");
  if (P3_point_is_beyond_portal (p, point, coordsys) == P3_TRUE) {
    Py_DECREF (coordsys);
    return PyInt_FromLong (1);
  } else {
    Py_DECREF (coordsys);
    return PyInt_FromLong (0);
  }
}

static PyObject* PyP3Portal_GetState (P3_portal* p) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* tuple;
  P3_portal_get_data (p, chunk);
  tuple = PyTuple_New (2);
  PyTuple_SET_ITEM (tuple, 0, PyString_FromStringAndSize ((char*) chunk->content, chunk->nb));
  if (p->beyond == NULL) {
    Py_INCREF (Py_None);
    PyTuple_SET_ITEM (tuple, 1, Py_None);
  } else if (p->beyond_name == NULL) {
    Py_INCREF ((PyObject*) p->beyond);
    PyTuple_SET_ITEM (tuple, 1, (PyObject*) p->beyond);
  } else {
    Py_INCREF ((PyObject*) p->beyond_name);
    PyTuple_SET_ITEM (tuple, 1, (PyObject*) p->beyond_name);
  }
  P3_chunk_dealloc (chunk);
  return tuple;
}

static PyObject* PyP3Portal_SetState (P3_portal* p, PyObject* args) {
  P3_chunk* chunk = P3_chunk_new ();
  PyObject* o;
  o = PySequence_Fast_GET_ITEM (args, 0);
  chunk->content = PyString_AS_STRING (o);
  P3_portal_set_data (p, chunk);
  o = PySequence_Fast_GET_ITEM (args, 1);
  if (o == Py_None) {
    p->beyond_name = NULL;
    p->beyond = NULL;
  } else if (PyString_Check (o)) {
    Py_INCREF (o);
    p->beyond = NULL;
    p->beyond_name = o;
  } else {
    Py_INCREF (o);
    p->beyond = (P3_world*) o;
    p->beyond_name = NULL;
  }
  free (chunk);
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef PyP3Portal_Methods[] = {
//  PYP3_COORDSYS_FUNCS,
  { "is_beyond", (PyCFunction) PyP3Portal_IsBeyond, METH_O },
  { "_getstate", (PyCFunction) PyP3Portal_GetState, METH_NOARGS },
  { "_setstate", (PyCFunction) PyP3Portal_SetState, METH_O },
  { NULL, NULL }
};

/*---------+
 | Get Set |
 +---------*/

PY_GET_SET_ON_OPTION (Portal, P3_portal*, Infinite, P3_PORTAL_INFINITE)
PY_GET_SET_ON_OPTION (Portal, P3_portal*, BoundAtmosphere, P3_PORTAL_BOUND_ATMOSPHERE)

static PyObject* PyP3Portal_GetUseClipPlane (P3_portal* m, void* context) {
  if (m->option & P3_PORTAL_USE_4_CLIP_PLANES) {
    return PyInt_FromLong (1);
  } else if (m->option & P3_PORTAL_USE_5_CLIP_PLANES) {
    return PyInt_FromLong (2);
  } else {
    return PyInt_FromLong (0);
  }
}

static int PyP3Portal_SetUseClipPlane (P3_portal* m, PyObject* value, void* context) {
  if (PyInt_AS_LONG (value) == 1) {
    m->option |= P3_PORTAL_USE_4_CLIP_PLANES;
    m->equation = (double*) malloc (16 * sizeof(double));
  } else if (PyInt_AS_LONG (value) == 2) {
    m->option |= P3_PORTAL_USE_5_CLIP_PLANES;
    m->equation = (double*) malloc (20 * sizeof(double));
  } else {
    if (m->option & P3_PORTAL_USE_4_CLIP_PLANES) { m->option -= P3_PORTAL_USE_4_CLIP_PLANES; }
    if (m->option & P3_PORTAL_USE_5_CLIP_PLANES) { m->option -= P3_PORTAL_USE_5_CLIP_PLANES; }    
    free (m->equation);
    m->equation = NULL;
  }
  return 0;
}

static PyObject* PyP3Portal_GetBeyond (P3_portal* m, void* context) {
  if (m->beyond == NULL) {
    Py_INCREF (Py_None);
    return Py_None;
  } else {
    Py_INCREF ((PyObject*) m->beyond);
    return (PyObject*) m->beyond;
  }
}
static int PyP3Portal_SetBeyond (P3_portal* m, PyObject* value, void* context) {
  PyObject* o;
  Py_XDECREF ((PyObject*) m->beyond);
  Py_XDECREF ((PyObject*) m->beyond_name);
  m->beyond_name = NULL;
  if (value == Py_None || value == NULL) {
    m->beyond = NULL;
  } else {
    Py_INCREF (value);
    m->beyond = (P3_world*) value;
    o = PyObject_GetAttrString (value, "filename");
    if (o == Py_None) {
      Py_DECREF (o);
    } else {
      m->beyond_name = o;
    }
  }
  return 0;
}

static PyGetSetDef PyP3Portal_GetSets[] = {
//  PYP3_VISIBLE_GETSETS,
//  PYP3_CHILD_GETSETS,
//  PYP3_COORDSYS_GETSETS,
  { "beyond",            (getter) PyP3Portal_GetBeyond,          (setter) PyP3Portal_SetBeyond,          NULL },
  { "use_clip_plane",    (getter) PyP3Portal_GetUseClipPlane,    (setter) PyP3Portal_SetUseClipPlane,    NULL },
  { "infinite",          (getter) PyP3Portal_GetInfinite,        (setter) PyP3Portal_SetInfinite,        NULL },
  { "bound_atm",         (getter) PyP3Portal_GetBoundAtmosphere, (setter) PyP3Portal_SetBoundAtmosphere, NULL },
  { NULL }
};

/*------+
 | Type |
 +------*/

PyTypeObject PyP3Portal_Type = {
  PyObject_HEAD_INIT(NULL)
  0,
  "_soya._Portal",
  sizeof(P3_portal),// + PyGC_HEAD_SIZE,
  0,
  (destructor) PyP3Portal_Dealloc,/* tp_dealloc */
  0,/* tp_print */
  0,/* tp_getattr */
  0,/* tp_setattr */
  0,/* tp_compare */
  0,/* tp_repr */
  0,/* tp_as_number */
  0,/* tp_as_sequence */
  0,/* tp_as_mapping */
  0,/* tp_hash */
  0,/* tp_call */
  0,/* tp_str */
  PYP3_GENERIC_GETATTR,/* tp_getattro */
  0,/* tp_setattro */
  0,/* tp_as_buffer */
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE | Py_TPFLAGS_HAVE_GC,/* tp_flags */
  0,/* tp_doc */
  (traverseproc) PyP3Portal_Traverse,/* tp_traverse */
  (inquiry) PyP3Portal_Clear,/* tp_clear */
  0,/* tp_richcompare */
  0,/* tp_weaklistoffset */
  0,/* tp_iter */
  0,/* tp_iternext */
  (PyMethodDef*) &PyP3Portal_Methods,/* tp_methods */
  0,/* tp_members */
  (PyGetSetDef*) &PyP3Portal_GetSets,/* tp_getset */
  &PyP3Element_Type,/* tp_base */
  0,/* tp_dict */
  0,/* tp_descr_get */
  0,/* tp_descr_set */
  0,/* tp_dictoffset */
  (initproc) PyP3Portal_Init,/* tp_init */
  PYP3_GENERIC_ALLOC,/* tp_alloc */
  (newfunc) PyP3Object_New,/* tp_new */
  PYP3_GENERIC_GC_FREE,/* tp_free */
};


