/*
 * P3
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*************************************************
 * text.c : define font and functions to draw text
 * Copyright (C) 2001-2003 Bertrand 'blam' LAMY
 *************************************************/

#include <SDL/SDL.h>

#ifdef HAVE_FREETYPE2
#include <freetype/freetype.h>
#endif /* HAVE_FREETYPE2 */

#include "p3_base.h"
#include "image.h"
#include "text.h"

extern int screen_w;
extern int screen_h;
extern int maxtexturesize;


/*=============*
 * RASTER FONT *
 *=============*/

void P3_font_raster_print_2i (P3_font_raster* font, char* s, int x, int y, int* rw, int* rh) {
  int nx;
  int i;
  int w;
  char c;
  if (font->call_list == -1) { P3_font_raster_init (font); }
  *rw = 0;
  *rh = font->height + 1;
  y += (font->height + 1);
  nx = x;
  glDisable (GL_TEXTURE_2D);
  glRasterPos2i (x, y);
  for (i = 0; i < strlen (s); i++) {
    c = s[i];
    if (c == '\n') {
      y += (font->height + 1);
      nx = x;
      glRasterPos2i (nx, y);
      *rh += font->height + 1;
      if (w > *rw) { *rw = w; }
      w = 0;
    } else if (c == ' ') {
      nx += font->width;
      glRasterPos2i (nx, y);
      w += font->width;
    } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
      glCallList (((unsigned char) c) + font->offset);
      nx += font->width;
      w += font->width;
    }
  }
  if (w > *rw) { *rw = w; }
  glEnable (GL_TEXTURE_2D);
}

int P3_font_raster_print_area (P3_font_raster* font, char* s, int x1, int y, int width, int height, int align) {
  char c;
  int nx;
  int i;
  int chars_per_line;
  int nb_chars;
  int chars_left;
//  int nb_lines;
  int y1;
  int rh;
  if (font->call_list == -1) { P3_font_raster_init (font); }
  glDisable (GL_TEXTURE_2D);
  y1 = y + (font->height + 1);
  chars_per_line = (int) floor (width / font->width);
  chars_left = strlen (s);
//  nb_lines = (int) floor (height / (font->height + 1));
  rh = 0;
  if (align == P3_TEXT_ALIGN_LEFT) {
    /* left align */
    while (1) {
      rh += font->height + 1;
//    for (j = 0; j < nb_lines; j++) {
      if (s[0] == ' ') { s++; chars_left--; }
      nx = x1;
      if (chars_left > chars_per_line) {
        nb_chars = chars_per_line;
        /* reverse search for a ' ' to cut */
        for (i = chars_per_line; i > 0; i--) {
          if (s[i] == ' ' || s[i] == '\n') {
            nb_chars = i;
            break;
          }
        }
      } else {
        nb_chars = chars_left;
      }
      /* draw nb_chars of s */
      glRasterPos2i (nx, y1);
      for (i = 0; i < nb_chars; i++) {
        c = s[i];
        if (c == ' ') {
          nx += font->width;
          glRasterPos2i (nx, y1);
        } else if (c == '\n') {
          /* ends this line */
          nx = x1;
          nb_chars = i + 1;
          break;
        } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
          glCallList (((unsigned char) c) + font->offset);
          nx += font->width;
        }
      }
      /* advance and change line */
      s += nb_chars;
      chars_left -= nb_chars;
      y1 += (font->height + 1);
      if (chars_left <= 0) { break; }
      if (height > 0 && y1 > y + height) { break; }
    }
  } else if (align == P3_TEXT_ALIGN_CENTER) {
    /* center align */
    char* str;
    int center = x1 + width / 2; 
    while (1) {
      rh += font->height + 1;
//    for (j = 0; j < nb_lines; j++) {
      if (s[0] == ' ' || s[0] == '\n') { s++; chars_left--; }
      nx = x1;
      if (chars_left > chars_per_line) {
        nb_chars = chars_per_line;
      } else {
        nb_chars = chars_left;
      }
      /* looks for an internal '\n' in s */
      str = memchr (s, '\n', nb_chars);
      if (str == NULL) {
        if (nb_chars == chars_per_line) {
          /* reverse search for a ' ' to cut */
          for (i = chars_per_line; i > 0; i--) {
            if (s[i] == ' ') {
              nb_chars = i;
              break;
            }
          }
        }
      } else {
        nb_chars = (str - s) / sizeof (char);
      }
      /* draw nb_chars of s */
      nx = center - (nb_chars * font->width) / 2;
      glRasterPos2i (nx, y1);
      for (i = 0; i < nb_chars; i++) {
        c = s[i];
        if (c == ' ') {
          nx += font->width;
          glRasterPos2i (nx, y1);
        } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
          glCallList (((unsigned char) c) + font->offset);
          nx += font->width;
        }
      }
      /* advance and change line */
      s += nb_chars;
      chars_left -= nb_chars;
      y1 += (font->height + 1);
      if (chars_left <= 0) { break; }
      if (height > 0 && y1 > y + height) { break; }
    }
  }
  glEnable (GL_TEXTURE_2D);
  return rh;
}

void P3_font_raster_print_3f (P3_font_raster* font, char* s, GLfloat x, GLfloat y, GLfloat z) {
  int nx;
  int i;
  char c;
  if (font->call_list == -1) { P3_font_raster_init (font); }
  y += (font->height + 1);
  nx = x;
  glDisable (GL_TEXTURE_2D);
  glRasterPos3f (x, y, z);
  for (i = 0; i < strlen (s); i++) {
    c = s[i];
    if (c == '\n') {
      y += (font->height + 1);
      nx = x;
      glRasterPos3f (nx, y, z);
    } else if (c == ' ') {
      nx += font->width;
      glRasterPos3f (nx, y, z);
    } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
      glCallList (((unsigned char) c) + font->offset);
      nx += font->width;
    }
  }
  glEnable (GL_TEXTURE_2D);
}

void P3_font_raster_init (P3_font_raster* font) {
  int i;
  GLubyte* ptr;
  int size;
/*
// HACK
for (i = 30; i < 60; i++) {
  printf("%i = %c\n", i, (char) i);
}
*/
  font->call_list = glGenLists (font->nb_chars);
  size = font->stored_width * font->height;
  ptr = font->chars;
  for (i = 0; i < font->nb_chars; i++) {
    glNewList (font->call_list + i, GL_COMPILE);
    glBitmap (font->width, font->height, 0.0, 0.0, font->width, 0.0, ptr);
    glEndList ();
    ptr += size;
  }
  font->offset = font->call_list - font->first_char;
}

P3_font_raster* P3_font_raster_from_image (P3_font_raster* font, P3_image* image, int w, int h, int first_char) {
  int i; int x; int y; int j;
  GLubyte* img_ptr;
  GLubyte* font_ptr;
  /* (w,h) size of a character in pixels for the image */
  if (font == NULL) {
    font = (P3_font_raster*) malloc (sizeof (P3_font_raster));
  }
  font->font_type = P3_FONT_RASTER;

  /* make black and white image from image */
  font->stored_width = (int) ceil ((float) w / 8.0);
  font->width = w;
  font->height = h;
  font->nb_chars = (int) (image->width / w);
  font->chars = (GLubyte*) calloc (font->nb_chars * font->stored_width * font->height, sizeof (GLubyte));

// TO DO if h < img_h

  font_ptr = font->chars;
  for (i = 0; i < font->nb_chars; i++) {
    for (y = font->height - 1; y >= 0; y--) {
//    for (y = 0; y < font->height; y++) {
      img_ptr = image->pixels + (i * w + y * image->width) * image->nb_color;
      j = 0;
      while (j < font->stored_width * 8) {
        *font_ptr = 0;
        for (x = 0; x < 8; x++) {
          if (x + j >= w) { break; }
          if (img_ptr[(x + j) * image->nb_color] >= 128) {
            *font_ptr += (1 << (7 - x));
          }
        }
        font_ptr++;
        j += 8;
      }
    }
  }

  font->call_list = -1;
  font->offset = 0;
  font->first_char = (int) first_char;
  font->last_char = (int) (font->nb_chars + font->first_char);
//  font->char_size = font->width * font->height / 8;
  return font;
}

void P3_font_raster_save (P3_font_raster* font, char* filename) {
  FILE* file;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  int i;
#endif
  file = fopen (filename, "wb");
  if (file == NULL) { 
    P3_error ("can't open file %s", filename); 
    return;
  }
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  i = SDL_Swap32 (font->font_type);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->first_char);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->nb_chars);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->stored_width);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->width);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->height);
  fwrite (&i, sizeof (int), 1, file);
#else
  fwrite (&(font->font_type),    sizeof (int), 1, file);
  fwrite (&(font->first_char),   sizeof (int), 1, file);
  fwrite (&(font->nb_chars),     sizeof (int), 1, file);
  fwrite (&(font->stored_width), sizeof (int), 1, file);
  fwrite (&(font->width),        sizeof (int), 1, file);
  fwrite (&(font->height),       sizeof (int), 1, file);
#endif
  fwrite (font->chars, sizeof (GLubyte), font->nb_chars * font->stored_width * font->height, file);
  fclose (file);
}

P3_font_raster* P3_font_raster_load (P3_font_raster* font, FILE* file) {
  if (font == NULL) font = (P3_font_raster*) malloc (sizeof (P3_font_raster));
  font->font_type = P3_FONT_RASTER;
  fread (&(font->first_char),   sizeof (int), 1, file);
  fread (&(font->nb_chars),     sizeof (int), 1, file);
  fread (&(font->stored_width), sizeof (int), 1, file);
  fread (&(font->width),        sizeof (int), 1, file);
  fread (&(font->height),       sizeof (int), 1, file);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (font->first_char);
  SDL_Swap32 (font->nb_chars);
  SDL_Swap32 (font->stored_width);
  SDL_Swap32 (font->width);
  SDL_Swap32 (font->height);
#endif
  font->chars = (GLubyte*) malloc (font->nb_chars * font->stored_width * font->height * sizeof (GLubyte));
  fread (font->chars, sizeof (GLubyte), font->nb_chars * font->stored_width * font->height, file);
  fclose (file);
  font->call_list = -1;
  font->offset = 0;
  font->last_char = (int) (font->nb_chars + font->first_char);
  return font;
}


/*==============*
 * TEXTURE FONT *
 *==============*/

P3_font_texture* P3_font_texture_from_image (P3_font_texture* font, P3_image* image, int first_char, GLubyte* color_limit) {
  int i; int j;
  int last_limit;
  int nb_limits;
  GLubyte* ptr;
  GLubyte* ptr2;
  if (font == NULL) font = (P3_font_texture*) malloc (sizeof (P3_font_texture));
  font->font_type = P3_FONT_TEXTURE;
  /* load the texture */
  if      (image->nb_color == 3) font->type = GL_RGB;
  else if (image->nb_color == 4) font->type = GL_RGBA;
  else                           font->type = GL_ALPHA;
  /* traverse img_pixels and find characters with the first line */
  ptr = image->pixels;
  /* first char */
  if (memcmp (image->pixels, color_limit, image->nb_color * sizeof (GLubyte)) == 0) {
    last_limit = 1;
    nb_limits = 1;
  } else {
    last_limit = 0;
    nb_limits = 0;
  }
  font->char_start = NULL;
  font->char_size = NULL;
  font->nb_chars = 0;
  /* following chars */
  for (i = 0; i < image->width; i++) {
    if (memcmp (ptr, color_limit, image->nb_color * sizeof (GLubyte)) == 0) {
      if (i != 0) {
        /* found a limit ! (first limit, if set, avoided) */
        j = font->nb_chars;
        (font->nb_chars)++;
        font->char_size = (int*) realloc (font->char_size, font->nb_chars * sizeof (int));
        font->char_size[j] = i - last_limit;
        last_limit = i + 1;
        nb_limits++;
      }
    }
    ptr += image->nb_color;
  }
  /* add last character if no limit at the end */
  if (memcmp (image->pixels + (image->width - 1) * image->nb_color, color_limit, image->nb_color * sizeof (GLubyte)) != 0) {
    (font->nb_chars)++;
    font->char_size = (int*) realloc (font->char_size, font->nb_chars * sizeof (int));
    font->char_size[font->nb_chars - 1] = image->width - last_limit;
  }
  font->width = image->width - nb_limits;
  font->char_line = (int*) calloc (font->nb_chars, sizeof (int));
  font->char_verti = 1.0;
  font->char_height = font->height;
  /* copy pixels (avoid limits) */
  font->pixels = (GLubyte*) malloc (font->width * font->height * image->nb_color * sizeof (GLubyte));
  ptr = image->pixels;
  ptr2 = font->pixels;
  for (j = 0; j < font->height; j++) {
    for (i = 0; i < image->width; i++) {
      if (memcmp (ptr, color_limit, image->nb_color * sizeof (GLubyte)) != 0) {
        memcpy (ptr2, ptr, image->nb_color * sizeof (GLubyte));
        ptr2 += image->nb_color;
      }
      ptr += image->nb_color;
    }
  }
  /* finalize */
  font->id = 0;
  font->first_char = (int) first_char;
  font->last_char = (int) font->nb_chars + font->first_char - 1;
  font->space = (int) font->width / font->nb_chars;
  return font;
}

void P3_font_texture_adjust (P3_font_texture* font) {
  int nb_color;
  GLubyte* n_p;
  int n_w;
  int n_h;
  int i; int j; int k;
  int cum_w;
  if      (font->type == GL_RGB)  { nb_color = 3; }
  else if (font->type == GL_RGBA) { nb_color = 4; }
  else { nb_color = 1; }

  if (font->width < maxtexturesize) {
    /* adjust width and height to be a power of 2 */
    i = 0;
    while (1) {
      if ((1 << i) >= font->width) {
        n_w = (1 << i);
        break;
      }
      i++;
    }
    i = 0;
    while (1) {
      if ((1 << i) >= font->height) {
        n_h = (1 << i);
        break;
      }
      i++;
    }
    /* copy pixels */
    n_p = (GLubyte*) calloc (n_w * n_h * nb_color, sizeof (GLubyte));
    for (j = 0; j < font->height; j++) {
      memcpy (n_p + j * n_w * nb_color, font->pixels + j * font->width * nb_color, font->width * nb_color * sizeof (GLubyte));
    }
    free (font->pixels);
    font->pixels = n_p;
    font->width  = n_w;
    font->height = n_h;
    /* compute char_start */
    font->char_verti = 1.0;
    font->char_line = (int*) calloc (font->nb_chars, sizeof (int));
    font->char_start = (GLfloat*) malloc ((font->nb_chars + 1) * sizeof (GLfloat));
    cum_w = 0;
    for (i = 0; i < font->nb_chars; i++) {
      font->char_start[i] = (GLfloat) cum_w / font->width;
      cum_w += font->char_size[i];
    }
    font->char_start[font->nb_chars] = (GLfloat) cum_w / font->width;

  } else {
    /* we must make several lines */
    int nb_lines;
    int* lines_size;

    lines_size = NULL;
    nb_lines = 0;
    cum_w = 0;
    /* cut chars into lines */
    for (i = 0; i < font->nb_chars; i++) {
      if (cum_w + font->char_size[i] > maxtexturesize) {
        lines_size = (int*) realloc (lines_size, (nb_lines + 1) * sizeof (int));
        lines_size[nb_lines] = cum_w;
        nb_lines++;
        cum_w = 0;
      }
      cum_w += font->char_size[i];
    }
    lines_size = (int*) realloc (lines_size, (nb_lines + 1) * sizeof (int));
    lines_size[nb_lines] = cum_w;
    nb_lines++;
    /* find new size for texture (must also be a power of 2) */
    n_w = maxtexturesize;
    n_h = font->height * nb_lines;
    while (1) {
      if ((1 << i) >= n_h) {
        n_h = (1 << i);
        break;
      }
      i++;
    }
    /* copy pixels */
    n_p = (GLubyte*) calloc (n_w * n_h * nb_color, sizeof (GLubyte));
    cum_w = 0;
    for (i = 0; i < nb_lines; i++) {
      for (j = 0; j < font->height; j++) {
        memcpy (n_p + (i * font->height + j) * n_w * nb_color, font->pixels + (j * font->width + cum_w) * nb_color, lines_size[i] * nb_color * sizeof (GLubyte));
      }
      cum_w += lines_size[i];
    }
    free (font->pixels);
    font->pixels = n_p;
    font->width  = n_w;
    font->char_verti = (float) font->height / (float) n_h;
    font->height = n_h;
    /* compute char_start */
    font->char_line = (int*) malloc (font->nb_chars * sizeof (int));
    font->char_start = (GLfloat*) malloc ((font->nb_chars + nb_lines) * sizeof (GLfloat));
    cum_w = 0;
    k = 0;
    j = 0;
    for (i = 0; i < font->nb_chars; i++) {
      font->char_start[j] = (GLfloat) cum_w / (GLfloat) font->width;
      j++;
      cum_w += font->char_size[i];
      if (cum_w > font->width) {
        /* new line */
        k++;
        font->char_start[j] = 0.0;
        j++;
        cum_w = font->char_size[i];
      }
      font->char_line[i] = k;
    }
    font->char_start[font->nb_chars] = (GLfloat) cum_w / (GLfloat) font->width;
  }
}

void P3_font_texture_init (P3_font_texture* font) {
  P3_font_texture_adjust (font);
  if (font->id == 0) { 
    glGenTextures (1, &(font->id));
  }
  glBindTexture (GL_TEXTURE_2D, font->id);
  glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexImage2D (GL_TEXTURE_2D, 0, font->type, font->width, font->height, 0, font->type, GL_UNSIGNED_BYTE, font->pixels);
}

#define P3_FONT_TEXTURE_PRINT_BEGIN \
  if (font->id == 0) { P3_font_texture_init (font); \
  } else { glBindTexture (GL_TEXTURE_2D, font->id); } \
  glDisable (GL_CULL_FACE); \
/* TO DO ?? */\
  glEnable (GL_TEXTURE_2D); \
  glEnable (GL_BLEND); \
  glBegin (GL_QUADS)

#define P3_FONT_TEXTURE_PRINT_END \
  glEnd (); \
  glEnable (GL_CULL_FACE); \
  glDisable (GL_BLEND); \
  glBindTexture (GL_TEXTURE_2D, 0)

#define P3_FONT_TEXTURE_PRINT_2I(draw_func) \
  int nx; int i; int cc; int line; GLfloat tx; char c; int w; \
  P3_FONT_TEXTURE_PRINT_BEGIN; \
  *rw = 0; *rh = font->char_height + 1; w = 0; \
  nx = x; \
  for (i = 0; i < strlen (s); i++) { \
    c = s[i]; \
    if (c == '\n') { \
      y += (font->char_height + 1); \
      nx = x; \
      *rh += font->char_height + 1; \
      if (w > *rw) { *rw = w; } \
      w = 0; \
    } else if (c == ' ') { \
      nx += font->space; \
      w += font->space; \
    } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) { \
      cc = ((unsigned char) c) - font->first_char; \
      line = font->char_line[cc]; \
      tx = font->char_start[cc + line]; \
      draw_func; \
      w += font->char_size[cc]; \
    } \
  } \
  if (w > *rw) { *rw = w; } \
  P3_FONT_TEXTURE_PRINT_END

void P3_font_texture_print_2i (P3_font_texture* font, char* s, int x, int y, int* rw, int* rh) {
  P3_FONT_TEXTURE_PRINT_2I ( \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height); \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    nx += font->char_size[cc]; \
    tx = font->char_start[cc + line + 1]; \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height) \
    );
}

void P3_font_texture_print_2i_s (P3_font_texture* font, char* s, int x, int y, float scale_x, float scale_y, int* rw, int* rh) {
  P3_FONT_TEXTURE_PRINT_2I ( \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height * scale_y); \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    nx += font->char_size[cc] * scale_x; \
    tx = font->char_start[cc + line + 1]; \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height * scale_y) \
    );
}

void P3_font_texture_print_2i_c (P3_font_texture* font, char* s, int x, int y, GLfloat* color1, GLfloat* color2, int* rw, int* rh) {
  P3_FONT_TEXTURE_PRINT_2I ( \
    glColor4fv (color2); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height); \
    glColor4fv (color1); \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    nx += font->char_size[cc]; \
    tx = font->char_start[cc + line + 1]; \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    glColor4fv (color2); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height) \
    );
}

void P3_font_texture_print_2i_sc (P3_font_texture* font, char* s, int x, int y, float scale_x, float scale_y, GLfloat* color1, GLfloat* color2, int* rw, int* rh) {
  P3_FONT_TEXTURE_PRINT_2I ( \
    glColor4fv (color2); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height * scale_y); \
    glColor4fv (color1); \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    nx += font->char_size[cc] * scale_x; \
    tx = font->char_start[cc + line + 1]; \
    glTexCoord2f (tx, line * font->char_verti); \
    glVertex2i (nx, y); \
    glColor4fv (color2); \
    glTexCoord2f (tx, (line + 1.0) * font->char_verti); \
    glVertex2i (nx, y + font->char_height * scale_y)
    );
}

int P3_font_texture_print_area (P3_font_texture* font, char* s, int x1, int y1, int width, int height, int align) {
  int nx; int y2; int y; int w; int rh;
  int i; int j; int k;
  char c; int chars_left; int line; int charac;
  GLfloat tx; GLfloat ty;
  P3_FONT_TEXTURE_PRINT_BEGIN;
  rh = 0;
  chars_left = strlen (s);
  y2 = y1 + height;
  if (align == P3_TEXT_ALIGN_LEFT) {
    /* left align */
    while (1) {
      if ((height >= 0 && y1 >= y2) || chars_left <= 0) { break; }
      rh += font->char_height + 1;
      /* take chars up to have a complete line */
      w = 0;
      k = -1;
      for (i = 0; i < chars_left; i++) {
        c = s[i];
        if (c == '\n') break;
        if (c == ' ') {
          w += font->space;
          k = i;
        } else {
          charac = ((unsigned char) c) - font->first_char;
          w += font->char_size[charac];
        }
        if (w >= width) {
          if (k != -1) {
            i = k;
          } else {
            k = -2;
          }
          if (i == 0) i = 1;
          break;
        }
      }
      /* draw i characters */
      nx = x1;
      y = y1 + font->char_height;
      ty = 1.0;
      if (height >= 0 && y > y2) {
        ty = 1.0 - (GLfloat) (y - y2) / (GLfloat) font->char_height;
        y = y2;
      }
      for (j = 0; j < i; j++) {
        c = s[j];
        if (c == ' ') {
          nx += font->space;
        } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
          charac = ((unsigned char) c) - font->first_char;
          line = font->char_line[charac];
          tx = font->char_start[charac + line];
          glTexCoord2f (tx, (line + ty) * font->char_verti);
          glVertex2i (nx, y);
          glTexCoord2f (tx, line * font->char_verti);
          glVertex2i (nx, y1);
          nx += font->char_size[charac];
          tx = font->char_start[charac + line + 1];
          glTexCoord2f (tx, line * font->char_verti);
          glVertex2i (nx, y1);
          glTexCoord2f (tx, (line + ty) * font->char_verti);
          glVertex2i (nx, y);
        }
      }
      /* move to next line */
      if (k == -2) {
        chars_left -= i;
        s += i;
      } else {
        chars_left -= (i + 1);
        s += (i + 1);
      }
      y1 += font->char_height + 1;
    }
  } else if (align == P3_TEXT_ALIGN_CENTER) {
    int last_w;
    /* center align */
    while (1) {
      if ((height >= 0 && y1 >= y2) || chars_left <= 0) { break; }
      rh += font->char_height + 1;
      /* take chars up to have a complete line */
      last_w = 0;
      w = 0;
      k = -1;
      for (i = 0; i < chars_left; i++) {
        c = s[i];
        if (c == '\n') {
          last_w = w;
          break;
        }
        if (c == ' ') {
          last_w = w;
          w += font->space;
          k = i;
        } else {
          charac = ((unsigned char) c) - font->first_char;
          w += font->char_size[charac];
        }
        if (w >= width) {
          if (k != -1) {
            i = k;
          } else {
            k = -2;
            last_w = w;
          }
          if (i == 0) i = 1;
          break;
        }
      }
      if (i == chars_left) last_w = w;
      /* draw i characters */
      nx = x1 + ((width - last_w) >> 1);
      y = y1 + font->char_height;
      ty = 1.0;
      if (height >= 0 && y > y2) {
        ty = 1.0 - (GLfloat) (y - y2) / (GLfloat) font->char_height;
        y = y2;
      }
      for (j = 0; j < i; j++) {
        c = s[j];
        if (c == ' ') {
          nx += font->space;
        } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
          charac = ((unsigned char) c) - font->first_char;
          line = font->char_line[charac];
          tx = font->char_start[charac + line];
          glTexCoord2f (tx, (line + ty) * font->char_verti);
          glVertex2i (nx, y);
          glTexCoord2f (tx, line * font->char_verti);
          glVertex2i (nx, y1);
          nx += font->char_size[charac];
          tx = font->char_start[charac + line + 1];
          glTexCoord2f (tx, line * font->char_verti);
          glVertex2i (nx, y1);
          glTexCoord2f (tx, (line + ty) * font->char_verti);
          glVertex2i (nx, y);
        }
      }
      /* move to next line */
      if (k == -2) {
        chars_left -= i;
        s += i;
      } else {
        chars_left -= (i + 1);
        s += (i + 1);
      }
      y1 += font->char_height + 1;
    }
  }
  P3_FONT_TEXTURE_PRINT_END;
  return rh;
}

char* P3_font_texture_get_size (P3_font_texture* font, char* s, int w, int* width, int* height, char* txt) {
  if (w < 0) {
    int i; char c; int w;
    *height = font->char_height + 1;
    *width = 0;
    w = 0;
    for (i = 0; i < strlen (s); i++) {
      c = s[i];
      if (c == '\n') {
        *height += font->char_height + 1;
        if (w > *width) { *width = w; }
        w = 0;
      } else if (c == ' ') {
        w += font->space;
      } else if (((unsigned char) c) >= font->first_char && ((unsigned char) c) < font->last_char) {
        w += font->char_size[((unsigned char) c) - font->first_char];
      }
    }
    if (w > *width) { *width = w; }
  } else {
    int cw; int charac;
    int i; int k;
    char c; int chars_left;
    int txt_size = strlen(s) + 1;
    char* txt_ptr = txt;
    *width = w;
    *height = 0;
    chars_left = strlen (s);
    while (1) {
      if (chars_left <= 0) { break; }
      *height += font->char_height + 1;
      /* take chars up to have a complete line */
      cw = 0;
      k = -1;
      for (i = 0; i < chars_left; i++) {
        c = s[i];
        if (c == '\n') break; 
        if (c == ' ') {
          cw += font->space;
          k = i;
        } else {
          charac = ((unsigned char) c) - font->first_char;
          cw += font->char_size[charac];
        }
        if (cw >= w) {
          if (k != -1) { 
            i = k; 
          } else {
            k = -2;
          }
          if (i == 0) i = 1;
          break;
        }
      }
      if (txt != NULL) {
        memcpy (txt_ptr, s, i * sizeof (char));
        txt_ptr += i;
        *txt_ptr = '\n';
        txt_ptr++;
      }
      /* move to next line */
      if (k == -2) {
        chars_left -= i;
        s += i;
        if (txt != NULL) {
          char* old = txt;
          txt_size++;
          txt = (char*) realloc (txt, (txt_size) * sizeof (char));
          txt_ptr = txt_ptr - old + txt;
        }
      } else {
        chars_left -= (i + 1);
        s += (i + 1);
      }
    }
    if (txt != NULL) { 
      txt_ptr--;
      *txt_ptr = '\0'; 
    }
  }
  return txt;
}

void P3_font_texture_dealloc (P3_font_texture* f) {
  if (f->id != 0) { glDeleteTextures (1, &(f->id)); }
  free (f->pixels);
  free (f->char_start);
  free (f->char_size);
  free (f->char_line);
}

void P3_font_texture_save (P3_font_texture* font, char* filename) {
  FILE* file;
  int nb_color;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  int i, j;
#endif
  /* WARNING a font texture must always been saved BEFORE adjustment, never after!!! */
  if (font->char_start != NULL) {
    P3_error ("texture font can't be saved cause it has been adjusted");
    return;
  }
  file = fopen (filename, "wb");
  if (file == NULL) { 
    P3_error ("can't open file %s", filename); 
    return;
  }
  if      (font->type == GL_RGB)  nb_color = 3;
  else if (font->type == GL_RGBA) nb_color = 4;
  else                            nb_color = 1;
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  i = SDL_Swap32 (font->font_type);
  fwrite (&i,  sizeof (int), 1, file);
  i = SDL_Swap32 (font->first_char);
  fwrite (&i, sizeof (int), 1, file);
  i = SDL_Swap32 (font->nb_chars);
  fwrite (&i,  sizeof (int), 1, file);
  i = SDL_Swap32 (font->space);
  fwrite (&i,  sizeof (int), 1, file);
  i = SDL_Swap32 (font->width);
  fwrite (&i,  sizeof (int), 1, file);
  i = SDL_Swap32 (font->height);
  fwrite (&i,  sizeof (int), 1, file);
  i = SDL_Swap32 (font->type);
  fwrite (&i,  sizeof (int), 1, file);
  fwrite (font->pixels, sizeof (GLubyte), font->width * font->height * nb_color, file);
  for (j = 0; j < font->nb_chars; j++) {
    i = SDL_Swap32 (font->char_size[j]);
    fwrite (f&i, sizeof (int), 1, file);
  }
#else
  fwrite (&(font->font_type),  sizeof (int), 1, file);
  fwrite (&(font->first_char), sizeof (int), 1, file);
  fwrite (&(font->nb_chars),   sizeof (int), 1, file);
  fwrite (&(font->space),      sizeof (int), 1, file);
  fwrite (&(font->width),      sizeof (int), 1, file);
  fwrite (&(font->height),     sizeof (int), 1, file);
  fwrite (&(font->type),       sizeof (int), 1, file);
  fwrite (font->pixels, sizeof (GLubyte), font->width * font->height * nb_color, file);
  fwrite (font->char_size, sizeof (int), font->nb_chars, file);
#endif
  fclose (file);
}

P3_font_texture* P3_font_texture_load (P3_font_texture* font, FILE* file) {
  int nb_color;
  if (font == NULL) font = (P3_font_texture*) malloc (sizeof (P3_font_texture));
  font->font_type = P3_FONT_TEXTURE;
  fread (&(font->first_char), sizeof (int), 1, file);
  fread (&(font->nb_chars),   sizeof (int), 1, file);
  fread (&(font->space),      sizeof (int), 1, file);
  fread (&(font->width),      sizeof (int), 1, file);
  fread (&(font->height),     sizeof (int), 1, file);
  fread (&(font->type),       sizeof (int), 1, file);
  if      (font->type == GL_RGB)  nb_color = 3;
  else if (font->type == GL_RGBA) nb_color = 4;
  else nb_color = 1;
  font->pixels = (GLubyte*) malloc (font->width * font->height * nb_color * sizeof (GLubyte));
  fread (font->pixels, sizeof (GLubyte), font->width * font->height * nb_color, file);
  font->char_size = (int*) malloc (font->nb_chars * sizeof (int));
  fread (font->char_size, sizeof (int), font->nb_chars, file);
  fclose (file);
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
  SDL_Swap32 (font->first_char);
  SDL_Swap32 (font->nb_chars);
  SDL_Swap32 (font->space);
  SDL_Swap32 (font->width);
  SDL_Swap32 (font->height);
  SDL_Swap32 (font->type);
  SDL_Swap32 (font->char_size);
#endif
  font->id = 0;
  font->last_char = (int) (font->nb_chars + font->first_char - 1);
  font->char_height = font->height;
  font->char_verti = 1.0;
  font->char_start = NULL;
  font->char_line = (int*) calloc (font->nb_chars, sizeof (int));
  return font;
}


/*============================*
 * FONT IMPORT FROM FREETYPE2 *
 *============================*/

P3_font_texture* P3_font_texture_from_freetype (P3_font_texture* font, char* filename, int pix_w, int pix_h, int from, int to) {

#ifdef HAVE_FREETYPE2

  int c;
  int i; int j; int k;
  int offset;
  GLubyte* pixels;
  int offset_max;
  int size_max;
  FT_Library library;
  FT_Face face;
  FT_Bitmap bitmap;
  /* load font */
  if (FT_Init_FreeType (&library)) {
    P3_error ("can't init FreeType");
    return NULL;
  }
  if (font == NULL) { font = (P3_font_texture*) malloc (sizeof (P3_font_texture)); }
  font->font_type = P3_FONT_TEXTURE;
  if (FT_New_Face (library, filename, 0, &face)) {
    P3_error ("can't open font %s", filename);
    return NULL;
  }
  if (face->face_flags & FT_FACE_FLAG_SCALABLE) {
    FT_Set_Pixel_Sizes (face, pix_w, pix_h);
  } else {
    FT_Set_Pixel_Sizes (face, 0, 0);
  }
  /* retrieve size */
  font->width = 0;
  font->nb_chars = 0;
  size_max = -1;
  offset_max = -1;
  for (c = from; c <= to; c++) {
    i = FT_Get_Char_Index (face, c);
    FT_Load_Glyph (face, i, 0);
    if (face->glyph->format != ft_glyph_format_bitmap) {
      FT_Render_Glyph (face->glyph, ft_render_mode_normal);
    }
    bitmap = face->glyph->bitmap;
    font->width += bitmap.width + 1;
//    font->width += bitmap.width + face->glyph->bitmap_left;
    if (size_max < 0 || size_max < bitmap.rows) { size_max = bitmap.rows; }
    if (offset_max < 0 || offset_max < face->glyph->bitmap_top) { offset_max = face->glyph->bitmap_top; }
    (font->nb_chars)++;
  }
  font->height = size_max + offset_max;
  font->type = GL_ALPHA;
  pixels = (GLubyte*) calloc (font->width * font->height, sizeof (GLubyte));
  font->char_size = (int*) malloc (font->nb_chars * sizeof (int));
  /* get pixels */
  k = 0;
  offset = 0;
  for (c = from; c <= to; c++) {
    i = FT_Get_Char_Index (face, c);
    FT_Load_Glyph (face, i, 0);
    if (face->glyph->format != ft_glyph_format_bitmap) {
      FT_Render_Glyph (face->glyph, ft_render_mode_normal);
    }
    bitmap = face->glyph->bitmap;
    font->char_size[k] = bitmap.width + 1;
//    font->char_size[k] = bitmap.width + face->glyph->bitmap_left;
    for (j = 0; j < bitmap.rows; j++) {
      memcpy (pixels + (j + offset_max - face->glyph->bitmap_top) * font->width + offset, bitmap.buffer + bitmap.pitch * j, bitmap.pitch);
    }
    offset += bitmap.width + 1;
//    offset += bitmap.width + face->glyph->bitmap_left;
    k++;
  }
  /* remove blank lines above and below */
  while (1) {
    for (i = 0; i < font->width; i++) {
      if (pixels[i] != 0) { i = -1; break; }
    }
    if (i == -1) { break; }
    (font->height)--;
    memmove (pixels, pixels + font->width, font->width * font->height * sizeof (GLubyte));
    pixels = (GLubyte*) realloc (pixels, font->width * font->height * sizeof (GLubyte));
  }
  while (1) {
    for (i = 0; i < font->width; i++) {
      if (pixels[i + (font->height - 1) * font->width] != 0) { i = -1; break; }
    }
    if (i == -1) { break; }
    (font->height)--;
    pixels = (GLubyte*) realloc (pixels, font->width * font->height * sizeof (GLubyte));
  }
  /* finalize */
  font->pixels = pixels;
  font->char_start = NULL;
  font->char_line = (int*) calloc (font->nb_chars, sizeof (int));
  font->char_verti = 1.0;
  font->char_height = font->height;
  font->id = 0;
  font->first_char = from;
  font->last_char = (int) (font->nb_chars + font->first_char - 1);
  font->space = (int) ((float) font->width / (float) font->nb_chars);
  /* dump */
  printf("\nFONT TEXTURE INFO:\n");
  printf("  - texture width  %i\n", font->width);
  printf("  - texture height %i\n\n", font->height);

#endif /* HAVE_FREETYPE2 */

  return font;
}

P3_font_raster* P3_font_raster_from_freetype (P3_font_raster* font, char* filename, int pix_w, int pix_h, int from, int to) {

#ifdef HAVE_FREETYPE2

  int c; int i; int j; int k;
  int size_max; int offset_max;
  int width;
  int skip_start; 
  int skip_end;
  GLubyte* ptr;
  GLubyte* pixels;
  FT_Library library;
  FT_Face face;
  FT_Bitmap bitmap;

  if (FT_Init_FreeType (&library)) {
    P3_error ("can't init FreeType");
    return NULL;
  }
  if (font == NULL) { font = (P3_font_raster*) malloc (sizeof (P3_font_raster)); }
  font->font_type = P3_FONT_RASTER;
  if (FT_New_Face (library, filename, 0, &face)) {
    P3_error ("can't open font %s", filename);
    return NULL;
  }  
  if (face->face_flags & FT_FACE_FLAG_SCALABLE) {
    FT_Set_Pixel_Sizes (face, pix_w, pix_h);
  } else {
    FT_Set_Pixel_Sizes (face, 0, 0);
  }
  /* find char width (constant) */
  font->width = 0;
  font->nb_chars = 0;
  size_max = 0;
  offset_max = 0;
  for (c = from; c <= to; c++) {
    i = FT_Get_Char_Index (face, c);
    FT_Load_Glyph (face, i, 0);
    FT_Render_Glyph (face->glyph, ft_render_mode_mono);
    bitmap = face->glyph->bitmap;
    k = bitmap.width + 1;
//    k = bitmap.width + face->glyph->bitmap_left;
    if (k > font->width) { font->width = k; }
    if (size_max < 0 || size_max < bitmap.rows) { size_max = bitmap.rows; }
    if (offset_max < 0 || offset_max < face->glyph->bitmap_top) { offset_max = face->glyph->bitmap_top; }
    (font->nb_chars)++;
  }
  font->stored_width = (int) ceil ((float) font->width / 8.0);
  width = font->nb_chars * font->stored_width;
  font->height = size_max + offset_max;
  font->chars = (GLubyte*) calloc (width * font->height, sizeof (GLubyte));
  /* get pixels */
  ptr = font->chars;
  pixels = (GLubyte*) malloc (font->stored_width * font->height * sizeof (GLubyte));
  for (c = from; c <= to; c++) {
    i = FT_Get_Char_Index (face, c);
    FT_Load_Glyph (face, i, 0);
    FT_Render_Glyph (face->glyph, ft_render_mode_mono);
    bitmap = face->glyph->bitmap;
    memset (pixels, 0, font->stored_width * font->height * sizeof (GLubyte));
    for (j = 0; j < bitmap.rows; j++) {
      memcpy (pixels + font->stored_width * (j + offset_max - face->glyph->bitmap_top), bitmap.buffer + bitmap.pitch * j, bitmap.pitch);
    }
    i = 0;
    for (j = font->height - 1; j >= 0; j--) {
      memcpy (ptr + font->stored_width * i, pixels + j * font->stored_width, font->stored_width);
      i++;
    }
    ptr += font->stored_width * font->height;
  }
  free (pixels);
  /* remove blank lines above and below */
  skip_start = -1;
  skip_end = -1;
  ptr = font->chars;
  pixels = (GLubyte*) calloc (font->stored_width, sizeof (GLubyte));
  for (i = 0; i < font->nb_chars; i++) {
    k = 0;
    for (c = 0; c < font->height; c++) {
      if (memcmp (ptr + c * font->stored_width, pixels, font->stored_width) == 0) { k++; } else { break; }
    }
    if (skip_start < 0 || skip_start > k) { skip_start = k; }
    k = 0;
    for (c = font->height - 1; c >= 0; c--) {
      if (memcmp (ptr + c * font->stored_width, pixels, font->stored_width) == 0) { k++; } else { break; }
    }
    if (skip_end < 0 || skip_end > k) { skip_end = k; }
    ptr += font->stored_width * font->height;
  }
  free (pixels);
  k = font->height - skip_start - skip_end;
  pixels = (GLubyte*) malloc (width * k * sizeof (GLubyte));
  for (i = 0; i < font->nb_chars; i++) {
    memcpy (pixels + font->stored_width * k * i, font->chars + font->stored_width * (skip_start + font->height * i), font->stored_width * k);
  }
  free (font->chars);
  font->chars = pixels;
  font->height = k;
  /* finalize */
  font->call_list = -1;
  font->offset = 0;
  font->first_char = (int) from;
  font->last_char = (int) (font->nb_chars + font->first_char);
  /* dump */
  printf("\nFONT RASTER INFO:\n");
  printf("  - width  %i\n", font->width);
  printf("  - height %i\n\n", font->height);

#endif /* HAVE_FREETYPE2 */

  return font;
}


