# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import math
import soya, soya.soya3d as soya3d, soya.widget as widget
from soya.math3d import Point, Vector

P  = Point()
V  = Vector()

class TravelingCamera(soya3d.Camera):
  def __init__(self, parent = None, target = None):
    soya3d.Camera.__init__(self, parent)
    
    self.travelings = []
    self.traveling  = None
    self.speed      = 0.3
    self._speed     = Vector()
    self.up         = Vector(self, 0.0, 1.0, 0.0)
    
  def add_traveling(self, traveling):
    self.travelings.append(traveling)
    self.traveling = traveling
    self._traveling_changed()
    
  def pop_traveling(self):
    self.travelings.pop()
    self.traveling = self.travelings[-1]
    self._traveling_changed()
    
  def remove_traveling(self, traveling):
    self.travelings.remove(traveling)
    self.traveling = self.travelings[-1]
    self._traveling_changed()
    
  def _traveling_changed(self):
    if not hasattr(self.traveling, "incline_as"):
      V.__init__(self.get_root(), 0.0, 1.0, 0.0)
      V.convert_to(self)
      V.z = 0.0
      if V.y or V.x:
        angle = self.up.angle_to(V)
        if V.x > 0: self.turn_incline(-angle)
        else:       self.turn_incline(angle)
        
  def begin_round(self):
    if self.traveling.smooth_move:
      self._speed.set_start_end(self, self.traveling.best_position(self))
      self._speed.__imul__(self.speed)
      
    else:
      self.move(self.traveling.best_position(self))
      
  def advance_time(self, proportion):
    if self.traveling.smooth_move:
      self.add_mul_vector(proportion, self._speed)
      
    if self.traveling.smooth_rotation:
      direction = self.traveling.best_direction(self)
      if not isinstance(direction, Vector):
        direction = self >> direction
      direction.convert_to(self)
      direction.normalize()
      if   direction.x < -0.02: direction.x = -0.02
      elif direction.x >  0.02: direction.x =  0.02
      if   direction.y < -0.02: direction.y = -0.02
      elif direction.y >  0.02: direction.y =  0.02
      direction.z = -1.0
      self.look_at(direction)
    else:
      if self.traveling.incline_as:
        dir = self.traveling.best_direction(self)
        if not isinstance(dir, Vector): dir = self >> dir
        cur = Vector(self, 0.0, 0.0, -1.0)
        axe = cur.cross_product(dir)
        axe.convert_to(self.parent)
        self.rotate_axe(cur.angle_to(dir), axe.x, axe.y, axe.z)
      else:
        self.look_at(self.traveling.best_direction(self))
        
    if self.traveling.incline_as:
      V.__init__(self.traveling.incline_as, 0.0, 1.0, 0.0)
      V.convert_to(self)
      V.z = 0.0
      if V.y or V.x:
        angle = self.up.angle_to(V)
        angle = min(2.0 * self.speed, angle) * proportion
        if V.x > 0: self.turn_incline(-angle)
        else:       self.turn_incline(angle)
        
  def zap(self):
    self.move   (self.traveling.best_position (self))
    self.look_at(self.traveling.best_direction(self))
    
    
class Traveling:
  incline_as = None
  def best_position (self, camera): raise NotImplementedError
  def best_direction(self, camera): raise NotImplementedError

class ThirdPersonTraveling(Traveling):
  """A Tomb-Raider-like traveling.
TARGET is a point in the character (or world) to follow."""
  def __init__(self, target = None, direction = None, smooth_move = 1, smooth_rotation = 0):
    self.target          = target
    self.incline_as      = target
    self._target         = Point()
    self.direction       = direction or Vector(None, 0.0, 1.0, 2.0)
    self.smooth_move     = smooth_move
    self.smooth_rotation = smooth_rotation
    self.distance        = 5.0
    self.offset_y        = 1.5
    self.offset_y2       = 0.0
    self.best            = Point()
    self.lateral_angle   = 0.0
    self.top_view        = 0.0
    
  def best_position(self, camera):
    root = camera.get_root()
    
    #self._target.clone(self.target)
    #self._target.convert_to(root)
    #self._target.y += self.offset_y
    self._target.__init__(self.target, 0.0, self.offset_y, 0.0)
    self._target.convert_to(root)
    
    best = self._best_position(root, camera, self._target)
    
    self.direction.set_start_end(self._target, best)
    
    return best
  
  def _best_position(self, root, camera, target):
    max_dist = 0.0
    px, py, pz = self.target.transform_point(camera.x, camera.y, camera.z, camera.parent)
    if self.lateral_angle:
      c = math.cos(self.lateral_angle)
      s = math.sin(self.lateral_angle)
      px, pz = c * px + s * pz, c * pz - s * px
      
    py -= self.top_view * self.distance
    
    if (abs(px) < 1.0) and (pz > 0.0): lat = 0
    #elif abs(px) > 1.0:                lat = cmp(px, 0.0)
    #else:                              lat = 0
    else:                              lat = cmp(px, 0.0)
    if abs(py - 3.0) < 1.0: vert = 0
    else:                   vert = cmp(py, 3.0)
    
    direction = self.direction
    
    p, max_dist = self._check(root, camera, target, direction)
    if (max_dist > self.distance - 0.1) and (lat == 0) and (vert == 0):
      return p
    else:
      self.best.clone(p)
      
    direction.convert_to(camera)
    
    direction.x += 0.3
    
    p, dist = self._check(root, camera, target, direction)
    if lat == -1: dist += 0.1
    if dist > max_dist: self.best.clone(p); max_dist = dist
    
    direction.x -= 0.6
    
    p, dist = self._check(root, camera, target, direction)
    if lat == 1: dist += 0.1
    if dist > max_dist: self.best.clone(p); max_dist = dist
    
    direction.x += 0.3
    
    direction.y += 0.2
    
    p, dist = self._check(root, camera, target, direction)
    if vert == -1: dist += 0.1
    if dist > max_dist: self.best.clone(p); max_dist = dist
    
    direction.y -= 0.4
    
    p, dist = self._check(root, camera, target, direction)
    if vert == 1: dist += 0.1
    if dist > max_dist: self.best.clone(p); max_dist = dist
    
    return self.best
  
  def _check(self, root, camera, target, direction):
    if not root.raypick(target, direction, self.distance, 2, P, V):
      direction.set_length(self.distance)
      P.__init__(target.parent, target.x, target.y, target.z)
      P.add_vector(direction)
      return P, self.distance
    
    return P, P.distance_to(target)
  
  def best_direction(self, camera):
    self._target.move(self.target)
    self._target.y += self.offset_y + self.offset_y2
    return self._target

  
class FixTraveling(Traveling):
  """A fixed view traveling.
POSITION is the position of the camera, and DIRECTION the direction to look at."""
  def __init__(self, position, direction, smooth_move = 1, smooth_rotation = 0):
    self.position        = position
    self.direction       = direction
    self.smooth_move     = smooth_move
    self.smooth_rotation = smooth_rotation
    
  def best_position(self, camera):
    return self.position or camera
  
  def best_direction(self, camera):
    return self.direction or Vector(camera, 0.0, 0.0, -1.0)



import _soya

class BlackBands(widget.Widget):
  """Black bands for 16/9 look."""
  
  def render(self):
    w = float(_soya.get_screen_width ())
    h = float(_soya.get_screen_height())
    t = h / 10.0
    _soya.glColor4f(0.0, 0.0, 0.0, 1.0)
    _soya.glBegin(_soya.GL_QUADS)
    
    _soya.glVertex2f(0.0, 0.0)
    _soya.glVertex2f(0.0, t)
    _soya.glVertex2f(w, t)
    _soya.glVertex2f(w, 0.0)
    
    _soya.glVertex2f(0.0, h - t)
    _soya.glVertex2f(0.0, h)
    _soya.glVertex2f(w, h)
    _soya.glVertex2f(w, h - t)
    
    _soya.glEnd()
    
BLACK_BANDS =  BlackBands()
 
