# Soya 3D
# Copyright (C) 2001-2002 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import math, random
import soya, _soya, soya.math3d as math3d, soya.soya3d as soya3d, soya.model as model

import os, os.path, cPickle as pickle
DEFAULT_MATERIAL = pickle.load(open(os.path.join(soya.DATADIR, "fx.data"), "rb"))
#DEFAULT_MATERIAL._alls["fx_default"] = DEFAULT_MATERIAL

class Particle(math3d.Point):
  def __init__(self, parent = None, x = 0.0, y = 0.0, z = 0.0, life = 1.0):
    math3d.Point.__init__(self, parent, x, y, z)
    self.life     = self.max_life = life
    self.mid_life = life / 2.0
    
  def advance_time(self, proportion = 1.0, system = None):
    self.life -= 0.05 * proportion
    if self.life < 0.0: return 0
    else:
      self.advance_position(proportion, system)
      self.advance_color   (proportion)
      return 1
  def advance_position(self): pass
  def advance_color   (self): pass
  
  def __repr__(self): return "<Particle %f %f %f in %s>" % (self.x, self.y, self.z, self.parent)
  
class Acceleration:
  def __init__(self, acceleration, speed = None):
    self.acceleration = acceleration
    self.set_xyz(0.0, 0.0, 0.0)
    self.speed = speed or math3d.Vector()
    
  def advance_position(self, proportion, system):
    a = system.cache.get((id(self.acceleration), self.speed.parent), None)
    if not a:
      a = self.speed.parent.transform_vector(self.acceleration.x, self.acceleration.y, self.acceleration.z, self.acceleration.parent)
      a = (
        a[0] * proportion,
        a[1] * proportion,
        a[2] * proportion,
      )
      system.cache[(id(self.acceleration), self.speed.parent)] = a
      
    _soya.acceleration_advance(self, a, proportion)
#     s.x += a[0]
#     s.y += a[1]
#     s.z += a[2]
    
#     self.x += s.x * proportion
#     self.y += s.y * proportion
#     self.z += s.z * proportion
    
    # Equivalent to (but faster than):
    #self.speed += self.acceleration * proportion
    #self       += self.speed        * proportion
  
class Fading:
  def __init__(self, colors):
    self.colors = colors
    #self._incr = self.max_life / (len(self.colors) - 1)
    self.color = list(self.colors[0])
    
  def set_colors(self, colors):
    self.colors = colors
    #self._incr = self.max_life / (len(self.colors) - 1)
    
  def advance_color(self, proportion):
    _soya.fading_advance(self)
    
#     i = len(self.colors) - int(self.life // self._incr) - 2
#     c1 = self.colors[i]
#     c2 = self.colors[i + 1]
    
#     f1 = (self.life % self._incr) / self._incr
#     f2 = 1.0 - f1
    
#     c = self.color
#     c[0] = c1[0] * f1 + c2[0] * f2
#     c[1] = c1[1] * f1 + c2[1] * f2
#     c[2] = c1[2] * f1 + c2[2] * f2
#     c[3] = c1[3] * f1 + c2[3] * f2
    
class AccelerationFadingParticle(Acceleration, Fading, Particle):
  def __init__(self, parent = None, colors = None, acceleration = None, speed = None):
    Particle.__init__(self, parent, life = 0.5 + random.random())
    Acceleration.__init__(self, acceleration, speed)
    Fading.__init__(self, colors)
    
class FireworkParticle(Acceleration, Fading, Particle):
  def __init__(self, parent = None, colors = None, acceleration = None, speed = None, generator = None):
    Particle.__init__(self, parent, life = 0.5 + random.random())
    Acceleration.__init__(self, acceleration, speed)
    Fading.__init__(self, colors)
    self.generator = generator
    self.nb_subparticles = 8
    self.width = self.height = 0.1
    
  def advance_time(self, proportion, system):
    alive = Particle.advance_time(self, proportion, system)
    if (not alive) and self.generator:
      for i in range(self.nb_subparticles):
        particle = self.generator()
        system.init_particle(particle)
        particle.move(self)
        system.particles.append(particle)
    return alive
    
class Generator:
  def __call__(self): pass
  def set_system(self, system):
    self.system = system
    
class FountainGenerator(Generator):
  def __init__(self, colors = ((0.6, 0.3, 0.9, 0.4), (0.2, 0.7, 0.8, 0.8), (0.6, 0.4, 0.8, 0.2)), acceleration = None, initial_speed = 0.15):
    if not acceleration: acceleration = math3d.Vector(None, 0.0, -0.022, 0.0)
    self.colors        = colors
    self.acceleration  = acceleration
    self.initial_speed = initial_speed
    
  def __call__(self):
    root = self.system.get_root()
    angle = random.random() * 3.1417
    speed = math3d.Vector(self.system, math.cos(angle), 2.0 + math.sqrt(random.random() * 50.0), math.sin(angle))
    speed.set_length(self.initial_speed * (1.5 + random.random()))
    speed.convert_to(root)
    particle = AccelerationFadingParticle(root, self.colors, self.acceleration, speed)
    particle.move(self.system)
    
    return particle
  
class FireworkGenerator(Generator):
  def __init__(self, colors = ((0.9, 0.7, 0.0, 0.4), (1.0, 1.0, 0.0, 1.0), (1.0, 1.0, 1.0, 1.0)), acceleration = None, initial_speed = 0.16, subgenerator = None):
    if not acceleration: acceleration = math3d.Vector(None, 0.0, -0.01, 0.0)
    self.colors        = colors
    self.acceleration  = acceleration
    self.initial_speed = initial_speed
    self.subgenerator  = subgenerator or FountainGenerator(colors = ((1.0, 1.0, 1.0, 1.0), (0.2, 0.7, 0.8, 0.8), (0.6, 0.4, 0.8, 0.2)), acceleration = math3d.Vector(None, 0.0, -0.03, 0.0), initial_speed = 0.2)
    
  def set_system(self, system):
    Generator.set_system(self, system)
    if not self.subgenerator is self:
      self.subgenerator.set_system(system)
      
  def __call__(self):
    root = self.system.get_root()
    angle = random.random() * 3.1417
    speed = math3d.Vector(self.system, math.cos(angle), 1.0 + math.sqrt(random.random() * 25.0), math.sin(angle))
    speed.set_length(self.initial_speed * (1.5 + random.random()))
    speed.convert_to(root)
    particle = FireworkParticle(root, self.colors, self.acceleration, speed, self.subgenerator)
    particle.move(self.system)
    
    return particle
  
class FireGenerator(Generator):
  def __init__(self, colors = ((0.9, 0.0, 0.1, 0.4), (0.9, 0.2, 0.1, 0.8), (0.8, 0.7, 0.0, 0.9), (0.6, 0.4, 0.1, 0.2)), acceleration = None, initial_speed = 0.08):
    if not acceleration: acceleration = math3d.Vector(None, 0.0, 0.01, 0.0)
    self.colors        = colors
    self.acceleration  = acceleration
    self.initial_speed = initial_speed
    
  def __call__(self):
    root = self.system.get_root()
    angle = random.random() * 3.1417
    speed = math3d.Vector(self.system, math.cos(angle), -math.sqrt(random.random() * 3.0), math.sin(angle))
    speed.set_length(self.initial_speed * 0.15 * (1.5 + random.random()))
    speed.convert_to(root)
    particle = AccelerationFadingParticle(root, self.colors, self.acceleration, speed)
    particle.move(self.system)
    
    return particle
  
class System(soya._CObj, soya3d.GraphicElement, _soya._Coordsys):
  def __init__(self, parent = None, generator = None, material = None, nb_particles = 50, removable = 0):
    _soya._Coordsys.__init__(self)
    if generator:
      self.generator     = generator
      self.generator.set_system(self)
      
    self.particles     = []
    self.nb_particles  = nb_particles
    self.material      = material
    self.lit           = 0
    self.name          = ""
    self.removable     = removable
    
    if parent: parent.add(self)
    
  def batch(self): return (1, None, self.material)
  
  def init_particle(self, particle): pass
  
  def advance_time(self, proportion = 1.0):
    self.cache = {}
    
    self.particles = filter(lambda particle: particle.advance_time(proportion, self), self.particles)
    if self.generator:
      for i in range(self.nb_particles - len(self.particles)):
        particle = self.generator()
        self.init_particle(particle)
        
        particle.advance_time(proportion, self)
        self.particles.append(particle)
        
      if self.removable: self.generator = None
    elif self.removable and not self.particles:
      self.parent.remove(self)
      
  def __repr__(self): return "<System %s, %s particles>" % (self.name, len(self.particles))
  
  def __getstate__(self):
    state = soya._CObj.__getstate__(self)
    del state["particles"]
    del state["cache"]
    state["nb_particles"] = len(self.particles)
    return state
  def __setstate__(self, state):
    nb_particles = state["nb_particles"]
    del state["nb_particles"]
    soya._CObj.__setstate__(self, state)
    self.particles = [self.generator() for i in xrange(nb_particles)]
    
    
class PointSystem(System):
  def __init__(self, parent = None, generator = None, material = None, nb_particles = 50, point_size = 10.0):
    System.__init__(self, parent, generator, material, nb_particles)
    self.point_size = point_size
    
  def render(self):
    if not self.lit: _soya.glDisable(_soya.GL_LIGHTING)
    
    _soya.glPointSize(self.point_size)
    
    root = self.get_root()
    
    _soya.glBegin(_soya.GL_POINTS)
    for particle in self.particles:
      _soya.glColor4f(*particle.color)
      _soya.glVertex3f(*self.transform_point(particle.x, particle.y, particle.z, particle.parent or root))
      
    _soya.glEnd()
    
    if not self.lit: _soya.glEnable(_soya.GL_LIGHTING)
    
class MultiPointSystem(System):
  def __init__(self, parent = None, generator = None, material = None, nb_particles = 50, point_size = 10.0, nb_points = 10, alpha_incr = -0.1):
    System.__init__(self, parent, generator, material, nb_particles)
    self.point_size = point_size
    self.nb_points  = nb_points
    self.alpha_incr = alpha_incr
    
  def init_particle(self, particle):
    particle.old_pos = []
    
  def render(self):
    if not self.lit: _soya.glDisable(_soya.GL_LIGHTING)
    
    _soya.glPointSize(self.point_size)
    
    root = self.get_root()
    
    _soya.glBegin(_soya.GL_POINTS)
    for particle in self.particles:
      particle.old_pos.insert(0, particle.copy())
      if len(particle.old_pos) > self.nb_points: del particle.old_pos[-1]
      
      color = list(particle.color)
      
      for pos in particle.old_pos:
        _soya.glColor4f(*color)
        _soya.glVertex3f(*self.transform_point(pos.x, pos.y, pos.z, pos.parent or root))
        color[3] = color[3] + self.alpha_incr
        
    _soya.glEnd()
    
    if not self.lit: _soya.glEnable(_soya.GL_LIGHTING)

class SpriteSystem(System):
  def __init__(self, parent = None, generator = None, material = None, nb_particles = 50, particle_width = 1.0, particle_height = 1.0):
    System.__init__(self, parent, generator, material, nb_particles)
    self.particle_width  = particle_width
    self.particle_height = particle_height
    
  def init_particle(self, particle):
    if not hasattr(particle, "width"):
      particle.width  = self.particle_width
      particle.height = self.particle_height
      
  def render(self):
    _soya.activate_material(self.material)
    if not self.lit: _soya.glDisable(_soya.GL_LIGHTING)
    
    _soya.draw_sprites(self, self.particles)
    
    if not self.lit: _soya.glEnable(_soya.GL_LIGHTING)
    
class MultiSpriteSystem(System):
  def __init__(self, parent = None, generator = None, material = None, nb_particles = 50, particle_width = 1.0, particle_height = 1.0, nb_sprites = 10, alpha_incr = -0.1):
    System.__init__(self, parent, generator, material, nb_particles)
    self.particle_width  = particle_width
    self.particle_height = particle_height
    self.nb_sprites      = nb_sprites
    self.alpha_incr      = alpha_incr
    
  def init_particle(self, particle):
    particle.old_pos = []
    if not hasattr(particle, "width"):
      particle.width   = self.particle_width
      particle.height  = self.particle_height
      
  def render(self):
    _soya.activate_material(self.material)
    if not self.lit: _soya.glDisable(_soya.GL_LIGHTING)
    
    for particle in self.particles:
      particle.old_pos.insert(0, particle.copy())
      if len(particle.old_pos) > self.nb_sprites: del particle.old_pos[-1]
      
    _soya.draw_multiple_sprites(self, self.particles, self.alpha_incr)
    
    if not self.lit: _soya.glEnable(_soya.GL_LIGHTING)
    

def _default():
  return DEFAULT_MATERIAL

def Fountain(parent = None, material = None):
  s = SpriteSystem(parent, FountainGenerator(), material or _default())
  s.particle_width = s.particle_height = 0.5
  return s

def Fire(parent = None, material = None):
  s = SpriteSystem(parent, FireGenerator(), material or _default())
  s.particle_width = s.particle_height = 0.5
  return s

def Firework(parent = None, material = None, nb_particles = 6):
  return SpriteSystem(parent, FireworkGenerator(), material or _default(), nb_particles = nb_particles)




class Particles(soya._CObj, soya3d.GraphicElement, _soya._Particles):
  def __init__(self, parent = None, material = None, nb_particles = 50, removable = 0):
    _soya._Particles.__init__(self)
    
    self.max_particles_per_round = 1000000
    self.nb_max   = nb_particles
    self.material = material
    self.lit      = 0
    self.name     = ""
    self.removable = removable
    
    if parent: parent.add(self)
    
  def regenerate(self):
    nb = 0
    i = self.nb
    while (i < self.nb_max):
      self.generate(i)
      i  += 1
      nb += 1
      if nb >= self.max_particles_per_round: break
    self._advance_time(1.0)
    
  def begin_round(self):
    if self.nb < self.nb_max and self.auto_generate_particle: self.regenerate()
    if self.removable and self.nb == 0:
      self.parent.remove(self)
      
  def advance_time(self, proportion = 1.0):
    self._advance_time(proportion)
    
  def __repr__(self): return "<Particles %s, %s particles maximum>" % (self.name, self.nb_max)
  
  def move(self, position):
    if (not position.parent) or (not self.parent) or (self.parent is position.parent):
      self.set_xyz(position.x, position.y, position.z)
    else:
      x, y, z = self.parent.transform_point(position.x, position.y, position.z, position.parent)
      self.set_xyz(x, y, z)



class FlagSubFire(Particles):
  def __init__(self, parent = None, material = None, nb_particles = 8, removable = 0):
    Particles.__init__(self, parent, material or _default(), nb_particles, removable)
    self.set_colors((1.0, 1.0, 1.0, 1.0), (0.2, 0.7, 0.8, 0.8), (0.2, 0.4, 0.8, 0.2))
    self.set_sizes((0.25, 0.25), (1.0, 1.0))
    
  def generate(self, index, position = None):
    #angle = random.random() * 3.1417
    #sx = math.cos(angle)
    #sy = 2.0 + math.sqrt(random.random() * 50.0)
    #sz = math.sin(angle)
    sx = random.random() - 0.5
    sy = 1.0 + random.random()
    sz = random.random() - 0.5
    l = (0.2 * (1.0 + random.random())) / math.sqrt(sx * sx + sy * sy + sz * sz)
    if position:
      self.set_particle_all(index, 0.5 + random.random(), position, (sx * l, sy * l, sz * l), (0.0, -0.03, 0.0))
    else:
      self.set_particle(index, 0.5 + random.random(), (sx * l, sy * l, sz * l), (0.0, -0.03, 0.0))


class Smoke(Particles):
  def __init__(self, parent = None, material = None, nb_particles = 12, removable = 0):
    Particles.__init__(self, parent, material or _default(), nb_particles, removable)
    self.set_colors((0.2, 0.2, 0.2, 1.0), (1.0, 1.0, 1.0, 0.0))
    self.set_sizes((0.25, 0.25), (1.0, 1.0))
    
  def generate(self, index, position = None):
    sx = random.random() - 0.5
    sy = 1.0 + random.random()
    sz = random.random() - 0.5
    l = (0.2 * (1.0 + random.random())) / math.sqrt(sx * sx + sy * sy + sz * sz) * 0.4
    if position:
      self.set_particle_all(index, 0.5 + random.random(), position, (sx * l, sy * l, sz * l), (0.0, -0.03, 0.0))
    else:
      self.set_particle(index, 1.0 + random.random(), (sx * l, sy * l, sz * l), (0.0, 0.0, 0.0))


class FlagFirework(soya._CObj, soya3d.GraphicElement, _soya._Particles):

  def __init__(self, parent = None, material = None, nb_particles = 6, removable = 0, subgenerator = None, nb_sub_particles = 8):
    _soya._Particles.__init__(self)
      
    self.nb_max   = nb_particles
    self.material = material or _default()
    self.lit      = 0
    self.name     = ""
    self.removable = removable
        
    if parent: parent.add(self)

    self.subgenerator = subgenerator or FlagSubFire(parent, nb_particles = nb_particles * nb_sub_particles)
    self.nb_sub_particles = nb_sub_particles
    self.set_colors((0.9, 0.7, 0.0, 0.4), (1.0, 1.0, 0.0, 1.0), (1.0, 1.0, 1.0, 1.0))
    self.auto_generate_particle = 1
    self.set_sizes((0.0, 0.0), (0.25, 0.25))

  def advance_time(self, proportion = 1.0):
    if self.nb < self.nb_max and self.auto_generate_particle: self.regenerate()
    self._advance_time(proportion)
    if self.removable and self.nb == 0:
      self.subgenerator.removable = 1
      self.parent.remove(self)

  def regenerate(self):
    i = self.nb
    while (i < self.nb_max):
      self.mygenerate(i)
      i = i + 1
    self._advance_time(1.0)

  def subgenerate(self, index):
    subg = self.subgenerator
    if subg:
      if subg.nb + self.nb_sub_particles > subg.nb_max:
        subg.nb_max = subg.nb + self.nb_sub_particles
      p = self.get_particle_position(index)
      i = 0
      while(i < self.nb_sub_particles):
        subg.generate(subg.nb, p)
        i = i + 1

  def mygenerate(self, index):
    sx = random.random() - 0.5
    sy = 1.0 + random.random()
    sz = random.random() - 0.5
    #angle = (random.random() - 0.5) * 3.1417
    #sx = math.cos(angle)
    #sy = 1.0 + math.sqrt(random.random() * 25.0)
    #sz = math.sin(angle)
    l = (0.2 * (1.8 + random.random())) / math.sqrt(sx * sx + sy * sy + sz * sz)
    self.set_particle(index, 0.5 + random.random(), (sx * l, sy * l, sz * l), (0.0, -0.03, 0.0))
    
  def generate(self, index):
    self.subgenerate(index)
    self.mygenerate(index)

  def __repr__(self): return "<FlagFirework %s, %s particles maximum>" % (self.name, self.nb_max)
  
  def move(self, position):
    if (not position.parent) or (not self.parent) or (self.parent is position.parent):
      self.set_xyz(position.x, position.y, position.z)
      self.subgenerator.set_xyz(position.x, position.y, position.z)
    else:
      x, y, z = self.parent.transform_point(position.x, position.y, position.z, position.parent)
      self.set_xyz(x, y, z)
      self.subgenerator.set_xyz(x, y, z)



